/*
 * types and functions to create, represent and browse
 * a concrete configuration description
 *
 * (c) 2004 Peppercon AG
 * tbr@peppercon.de
 */

//#include <pp/base.h>
#include <pp/vector.h>
#include <pp/hash.h>
#include <pp/strstream.h>
#include <pp/win32.h>
#include <pp/syms.h>
#include <pp/error.h>


#ifndef __PP_CD_H__
#define __PP_CD_H__

#define __TWEB_DEBUG 0

#ifdef __cplusplus
extern "C" {
#endif

/*
 * symbol types used in CDL, used to tag symbols of the abstract parse tree
 */
typedef enum pp_cd_as_id_e {
    AS_SECTION                = 0x0000000f,
    AS_SECTION_TYPES          = 0x00000001,
    AS_SECTION_CONFIGS        = 0x00000002,
    AS_SECTION_VALUES         = 0x00000003,

    AS_FUND_TYPE              = 0x000000f0,
    AS_FUND_TYPE_BOOL         = 0x00000010,
    AS_FUND_TYPE_INT          = 0x00000020,
    AS_FUND_TYPE_STRING       = 0x00000030,

    AS_TYPE                   = 0x00000f00,
    AS_TYPE_BOOL              = 0x00000100,
    AS_TYPE_FKT               = 0x00000200,
    AS_TYPE_ENUM              = 0x00000300,
    AS_TYPE_STRUCT            = 0x00000400,
    AS_TYPE_ALIAS             = 0x00000500,
    AS_TYPE_STRING_TMPL       = 0x00000600, 
    AS_TYPE_INT_TMPL          = 0x00000700,
    AS_TYPE_CHOICE_TMPL       = 0x00000800,
    AS_TYPE_VECTOR_TMPL       = 0x00000900,
    AS_TYPE_VECTOR_TMPL_SPEC  = 0x00000a00,
    AS_TYPE_VECTOR_TMPL_INST  = 0x00000b00,

    AS_EXPR                   = 0x0000f000,
    AS_EXPR_INT_LITERAL       = 0x00001000,
    AS_EXPR_STRING_LITERAL    = 0x00002000,
    AS_EXPR_BOOL_LITERAL      = 0x00003000,
    AS_EXPR_FKT_DECL          = 0x00004000,
    AS_EXPR_VALUE_LIST        = 0x00005000,
    AS_EXPR_ENUM_LITERAL      = 0x00006000,
    AS_EXPR_CHOICE_LITERAL    = 0x00007000,

    AS_CFG_KEY                = 0x000f0000,
    AS_CFG_KEY_SCALAR         = 0x00010000,
    AS_CFG_KEY_VECTOR         = 0x00020000,

    AS_MISC                   = 0xf0000000,
    AS_ENUMERATOR             = 0x10000000,
    AS_PROPERTY               = 0X20000000,
    AS_PROPERTY_TAG           = 0x30000000,
    AS_PROPERTY_STRING        = 0x40000000,
    AS_CD                     = 0x50000000
} pp_cd_as_id_t;

/* Macros for type checking operating on pp_cd_as_t */
#define PP_CD_IS_SECTION(asn, sec)    (((asn)->type & AS_SECTION)   == sec)
#define PP_CD_IS_FUND_TYPE(asn, fund) (((asn)->type & AS_FUND_TYPE) == fund)
#define PP_CD_IS_TYPE(asn, astype)    (((asn)->type & AS_TYPE)      == astype)
#define PP_CD_IS_EXPR(asn, expr)      (((asn)->type & AS_EXPR)      == expr)
#define PP_CD_IS_CFG_KEY(asn, cfgk)   (((asn)->type & AS_CFG_KEY)   == cfgk)
#define PP_CD_IS_MISC(asn, misc)      (((asn)->type & AS_MISC)      == misc)

/* Macros for type checking operating on pp_cd_as_type_t */
#define PP_CD_TYPE_IS_SECTION_CONFIGS(asn) \
    (((asn)->base.type & AS_SECTION) == AS_SECTION_CONFIGS)
#define PP_CD_TYPE_IS_SECTION_TYPES(asn) \
    (((asn)->base.type & AS_SECTION) == AS_SECTION_TYPES)
#define PP_CD_TYPE_IS_VECTOR_TEMPL(asn) \
    (((asn)->base.type & AS_TYPE) == AS_TYPE_VECTOR_TMPL)
#define PP_CD_IS_BASE_TYPE(asn, astype) \
    (((asn)->base.type & AS_TYPE) == astype)

#define PP_CD_COMP_DELI                 '.'

static const char PP_CD_COMP_DELI_STR[] =               ".";
static const char PP_CD_CHOICE_SELECT_COMP_STR[] =      "_c_";
static const char PP_CD_VECT_SIZE_COMP_STR[] =          "_s_";
static const char PP_CD_VECT_DEF_COMP_STR[] =           "_d_";
static const char PP_CD_VECT_ELEMS_COMP_STR[] =         "_e_";

extern pp_mallocator_t* pp_cd_malloc;

/*
 * qualified id
 */
typedef char* pp_cd_qualified_id_t; 
typedef const char* pp_cd_qualified_const_id_t; 

/*
 * enumerator list
 */
typedef vector_t pp_cd_enumerator_list_t;

/*
 * forward decl for abstrat syntax operation, see below for definition
 */
typedef struct pp_cd_as_op_s pp_cd_as_op_t;

/*
 * most basic symbol structure of abstract syntax tree
 */
typedef struct pp_cd_as_s {
    pp_cd_as_id_t type;
    int pos;
    int (*execute)(struct pp_cd_as_s* n, pp_cd_as_op_t* operation, void* ctx);
    void (*destroy)(pp_mallocator_t* a, struct pp_cd_as_s* n);
} pp_cd_as_t;

/*
 * sections
 */
typedef struct {
    pp_cd_as_t base;
    vector_t*  decls;
} pp_cd_as_section_t;

/*
 * the config description is the triple of types, configs and values
 */
typedef struct {
    pp_cd_as_t          base;
    // id of the corresponding cdl (product:fw_build_nr)
    char*               cdl_id;
    pp_mallocator_t*    alloc;
    pp_cd_as_section_t* types;
    pp_cd_as_section_t* configs;
    pp_cd_as_section_t* values;
    pp_hash_t*          symbols;
} pp_cd_as_cd_t;

/*
 * different properties and property list
 */
typedef vector_t pp_cd_taglist_t;

typedef union {
    char* text;
    pp_cd_taglist_t* tags;
} pp_cd_property_value_t;

typedef struct {
    pp_cd_as_t base;
    char* key;
    pp_cd_property_value_t value;
} pp_cd_as_property_t;

typedef vector_t pp_cd_proplist_t;

/*
 * variations of types
 */
typedef vector_t pp_cd_type_list_t;

typedef struct {
    pp_cd_as_t base;
    char* name;
    pp_cd_proplist_t* properties;
}  pp_cd_as_type_t;

typedef struct {
    pp_cd_as_type_t base;
    pp_cd_qualified_id_t ref_type_name;
    pp_cd_as_type_t* ref_type;
} pp_cd_as_type_alias_t;

typedef struct {
    pp_cd_as_type_t base;
    pp_cd_type_list_t* elements;
} pp_cd_as_type_struct_t;

typedef struct {
    pp_cd_as_type_t base;
    pp_cd_enumerator_list_t* elements;
} pp_cd_as_type_enum_t;

typedef struct {
    pp_cd_as_t base;
    char* value;
    pp_cd_proplist_t* properties;
} pp_cd_as_enumerator_t;

typedef struct {
    pp_cd_as_type_t base;
    char* regexp;
    int min;
    int max;
} pp_cd_as_type_string_tmpl_t;

typedef struct {
    pp_cd_as_type_t base;
    long long min;
    long long max;
} pp_cd_as_type_int_tmpl_t;

typedef struct {
    pp_cd_as_type_t base;
    pp_cd_qualified_id_t tmpl_type_name;
    pp_cd_as_type_t* tmpl_type;
    pp_cd_type_list_t* elements;
} pp_cd_as_type_choice_tmpl_t;

typedef struct {
    pp_cd_as_type_t base;
    pp_cd_qualified_id_t elem_type_name;
    pp_cd_as_type_t* elem_type;
    pp_cd_as_type_t* size_type;
    pp_cd_as_id_t idx_type;
    unsigned int bound;
} pp_cd_as_type_vector_tmpl_t;

typedef struct {
    pp_cd_as_type_t base;
    char* place_holder;
    pp_cd_as_type_vector_tmpl_t* vector_type;
} pp_cd_as_type_vector_tmpl_spec_t;

typedef struct {
    pp_cd_as_type_t base;
    pp_cd_qualified_id_t elem_type_name;
    pp_cd_qualified_id_t ref_type_name;
    pp_cd_as_type_vector_tmpl_t* ref_type;
} pp_cd_as_type_vector_tmpl_inst_t;

/*
 * value expressions
 */
typedef vector_t pp_cd_quali_cfg_key_t;

typedef union {
    int intidx;
    char* stridx;
} pp_cd_cfg_key_idx_value_t;

typedef struct {
    pp_cd_as_id_t type;
    pp_cd_cfg_key_idx_value_t value;
} pp_cd_cfg_key_idx_t;

typedef struct {
    pp_cd_as_id_t type;
    char* key;
    pp_cd_cfg_key_idx_t idx;
} pp_cd_cfg_key_t;

typedef vector_t pp_cd_value_expr_list_t;

typedef struct {
    pp_cd_as_type_t base;
    pp_cd_type_list_t* params;
} pp_cd_as_fkt_decl_t;

typedef struct {
   char* choice;
   pp_cd_value_expr_list_t* choicelist;
} pp_cd_choice_value_t;

typedef union {
    int   boolval;
    int   intval;
    char* stringval;
    pp_cd_choice_value_t choiceval;
    pp_cd_as_fkt_decl_t* fktval;
    pp_cd_value_expr_list_t* listval;
} pp_cd_literal_value_t;

typedef struct {
    pp_cd_as_t base;
    pp_cd_quali_cfg_key_t* qkey;
    pp_cd_as_type_t*       type;
    pp_cd_literal_value_t value;
} pp_cd_as_value_expr_t;

/*
 * general cd life cycle operations
 */
pp_cd_as_section_t* pp_cd_as_create_section(pp_mallocator_t* a, int pos,
					    pp_cd_as_id_t id, vector_t* decls);
pp_cd_as_cd_t* pp_cd_as_create_cd(pp_mallocator_t* a, int pos);
pp_cd_as_cd_t* pp_cd_as_merge_cd_section(pp_cd_as_cd_t* cd,
					 pp_cd_as_section_t* section);
void pp_cd_as_destroy(pp_mallocator_t* a, pp_cd_as_t* t);

/*
 * cd life cycle operations: types
 */
pp_cd_as_type_t* pp_cd_as_create_type(pp_mallocator_t* a, int pos, char* name,
				      pp_cd_as_id_t type);
pp_cd_as_type_t* pp_cd_as_create_alias(pp_mallocator_t* a, int pos, char* name,
				       pp_cd_qualified_id_t alias_type_name);
pp_cd_as_type_t* pp_cd_as_create_struct(pp_mallocator_t* a, int pos,
					char* name, vector_t* elements);
pp_cd_as_type_t* pp_cd_as_create_enum(pp_mallocator_t* a, int pos, char* name,
				      pp_cd_enumerator_list_t* elements);
pp_cd_as_type_t* pp_cd_as_create_string_tmpl(pp_mallocator_t* a, int pos,
					     char* name, char* reqexp,
					     int min, int max);
pp_cd_as_type_t* pp_cd_as_create_int_tmpl(pp_mallocator_t* a, int pos,
					  char* name, int min, int max);
pp_cd_as_type_t* pp_cd_as_create_choice_tmpl(pp_mallocator_t* a, int pos,
					     char* name,
					     pp_cd_qualified_id_t tmpl_name,
					     vector_t* elements);
pp_cd_as_type_t* pp_cd_as_create_vector_tmpl(pp_mallocator_t* a, int pos,
					     char* name, 
					     pp_cd_qualified_id_t elem_type,
					     pp_cd_as_id_t idx_type,int bound);
pp_cd_as_type_t* pp_cd_as_create_vector_tmpl_spec(pp_mallocator_t* a, int pos,
						  char* name, char* type_name,
						  pp_cd_as_type_t*vector_type);
pp_cd_as_type_t* pp_cd_as_create_vector_tmpl_inst(pp_mallocator_t* a, int pos,
						  char* name,
					    pp_cd_qualified_id_t elem_type,
					    pp_cd_qualified_id_t ref_type);
pp_cd_as_enumerator_t* pp_cd_as_create_enumerator(pp_mallocator_t* a, int pos,
						  char* value);
pp_cd_as_property_t* pp_cd_as_create_property(pp_mallocator_t* a, int pos,
					      char* key,
					      pp_cd_as_id_t type,
					      pp_cd_property_value_t value);

/*
 * cd life cycle operations: value expressions
 */
pp_cd_cfg_key_t* pp_cd_cfg_key_create(pp_mallocator_t* a, pp_cd_as_id_t type,
				      char* key, pp_cd_cfg_key_idx_t idx);
void pp_cd_cfg_key_destroy(pp_mallocator_t* a, pp_cd_cfg_key_t* key);
pp_cd_as_value_expr_t* pp_cd_as_create_value_expr(pp_mallocator_t* a, int pos,
						  pp_cd_quali_cfg_key_t* qkey,
						  pp_cd_as_id_t type,
						  pp_cd_literal_value_t value);
pp_cd_as_fkt_decl_t* pp_cd_as_create_fkt_decl(pp_mallocator_t* a,
					      int pos, char* name,
					      pp_cd_type_list_t* params,
					      pp_cd_proplist_t* properties);

/* 
 * abstract sytax operation 
 */
struct pp_cd_as_op_s {
    int (*exec_for_cd)(pp_cd_as_cd_t* as, pp_cd_as_op_t* op, void* ctx);
    int (*exec_for_section)(pp_cd_as_section_t* as, pp_cd_as_op_t* op,
			    void* ctx);
    int (*exec_for_type)(pp_cd_as_type_t* as, pp_cd_as_op_t* op, void* ctx);
    int (*exec_for_type_alias)(pp_cd_as_type_alias_t* as, pp_cd_as_op_t* op,
			       void* ctx);
    
    int (*exec_for_type_struct)(pp_cd_as_type_struct_t* as, pp_cd_as_op_t* op,
				void* ctx);
    int (*exec_for_type_enum)(pp_cd_as_type_enum_t* as, pp_cd_as_op_t* op,
			      void* ctx);
    int (*exec_for_type_string_tmpl)(pp_cd_as_type_string_tmpl_t* as,
				     pp_cd_as_op_t* op, void* ctx);
    int (*exec_for_type_int_tmpl)(pp_cd_as_type_int_tmpl_t* as,
				  pp_cd_as_op_t* op, void* ctx);
    
    int (*exec_for_type_choice_tmpl)(pp_cd_as_type_choice_tmpl_t* as,
				     pp_cd_as_op_t* op, void* ctx);
    int (*exec_for_type_vector_tmpl)(pp_cd_as_type_vector_tmpl_t* as,
				     pp_cd_as_op_t* op, void* ctx);
    int (*exec_for_type_vector_tmpl_spec)(pp_cd_as_type_vector_tmpl_spec_t* as,
					  pp_cd_as_op_t* op, void* ctx);
    int (*exec_for_type_vector_tmpl_inst)(pp_cd_as_type_vector_tmpl_inst_t* as,
					  pp_cd_as_op_t* op, void* ctx);
    
    int (*exec_for_enumerator)(pp_cd_as_enumerator_t* as, pp_cd_as_op_t* op,
			       void* ctx);
    int (*exec_for_property)(pp_cd_as_property_t* as, pp_cd_as_op_t* op,
			     void* ctx);
    int (*exec_for_value_expr)(pp_cd_as_value_expr_t* as, pp_cd_as_op_t* op,
			       void* ctx);
    int (*exec_for_fkt_decl)(pp_cd_as_fkt_decl_t* as, pp_cd_as_op_t* op,
		             void* ctx); // for as_op_php_array!
};

/*
 * and some concrete once
 */
extern pp_cd_as_op_t pp_cd_as_op_print;
extern pp_cd_as_op_t pp_cd_as_op_semcheck;
extern pp_cd_as_op_t pp_cd_as_op_tagsec;

extern pp_cd_as_op_t pp_cd_as_op_get_qid;
extern pp_cd_as_op_t pp_cd_as_op_type_compare;

/*
 * some utils
 */
pp_cd_as_cd_t* pp_cd_parse_cdl(pp_mallocator_t* a, const char* filename);
void pp_cd_destroy(void *cd);
void pp_cd_destroy_values(void *cd);
int pp_cd_execute_op(pp_cd_as_t* asn, pp_cd_as_op_t* op, void* ctx);
int pp_cd_op_list(vector_t* v, pp_cd_as_op_t* op, void* ctx);
int pp_cd_op_cd(pp_cd_as_cd_t*cd, pp_cd_as_op_t* op, void* ctx);
int pp_cd_op_section(pp_cd_as_cd_t*cd, pp_cd_as_op_t* op, void* ctx);
int pp_cd_nop(pp_cd_as_t*asn, pp_cd_as_op_t* op, void* ctx);
pp_cd_qualified_id_t pp_cd_qualified_id_create(pp_mallocator_t* a,
					       pp_cd_qualified_const_id_t scope,
					       const char* id);
pp_cd_qualified_id_t pp_cd_qualified_id_add(pp_mallocator_t* a,
					    pp_cd_qualified_id_t scope,
					    const char*id);
pp_cd_qualified_id_t
pp_cd_qualified_id_create_from_cfg_key(pp_mallocator_t* a, 
                                       pp_cd_qualified_id_t scope,
                                       const pp_cd_cfg_key_t *cfg_key);
pp_cd_qualified_id_t
pp_cd_qualified_id_add_from_cfg_key(pp_mallocator_t* a, 
                                    pp_cd_qualified_id_t scope,
                                    const pp_cd_cfg_key_t *cfg_key);
unsigned int pp_cd_qualified_id_count_comps(const char*id);
char* pp_cd_qualikey_to_qualiid(pp_cd_quali_cfg_key_t* qkey);
char* pp_cd_qualikey_to_string(const char* scope, pp_cd_quali_cfg_key_t* qkey);
void pp_cd_qid_push_scope(const char* id, char** scope);
void pp_cd_qid_pop_scope(char* scope);
void pp_cd_qid_pop_scope_n(char* scope, int n);
/*
void pp_cd_qkey_push_scope(pp_cd_quali_cfg_key_t* key, 
			   pp_cd_quali_cfg_key_t* scope);
void pp_cd_qkey_pop_scope_n(pp_cd_quali_cfg_key_t* scope, int n);
*/

const char* pp_cd_fund_type_name(pp_cd_as_id_t id);


int pp_cd_yyerrorf(const int pos, const char *format, ...);


/*
 * cd type browsing operations and validation and other utils
 */
/*
pp_cd_as_type_t* pp_cd_lookup_by_name(const char* typename);

pp_cd_as_type_t* pp_cd_lookup_by_id(const int id);

int pp_cd_lookup_id_by_name(const char* typename);
*/

typedef enum {
    PP_CD_RESOLVE_SIMPLE = 0x00,
    PP_CD_RESOLVE_ALIAS  = 0x01,
    PP_CD_RESOLVE_VECTOR = 0x02
} pp_cd_resolve_kind_t;

pp_cd_as_t* pp_cd_resolve_symbol(const pp_cd_as_cd_t* cd,
				 pp_cd_qualified_const_id_t symbol,
				 const pp_cd_resolve_kind_t t);

pp_cd_as_type_t* pp_cd_resolve_type_symbol(const pp_cd_as_cd_t* cd,
					   pp_cd_qualified_const_id_t symbol,
					   const pp_cd_resolve_kind_t t);

pp_cd_as_t* pp_cd_resolve_value_symbol(const pp_cd_as_cd_t* cd,
		                       pp_cd_qualified_const_id_t symbol,
			               const char *scope);
                                       
pp_cd_as_t* pp_cd_lookup_symbol(const pp_cd_as_cd_t* cd,
		                pp_cd_qualified_const_id_t symbol,
			        const char *scope);

pp_cd_as_t* pp_cd_resolve_sym_n_sec(const pp_cd_as_cd_t* cd,
				    pp_cd_qualified_const_id_t symbol,
				    const pp_cd_resolve_kind_t rt,
				    pp_cd_as_id_t* root_section);

pp_cd_as_type_t* pp_cd_resolve_type_sym_n_sec(const pp_cd_as_cd_t* cd,
					      pp_cd_qualified_const_id_t id,
					      const pp_cd_resolve_kind_t t,
					      pp_cd_as_id_t* root_section);

typedef enum {
    PP_CD_VALIDATION_ERROR_INTERNAL  = 0,
    PP_CD_VALIDATION_ERROR_INVALID   = 1,
    PP_CD_VALIDATION_ERROR_TOO_SHORT = 2,
    PP_CD_VALIDATION_ERROR_TOO_LONG  = 3,
    PP_CD_VALIDATION_ERROR_TOO_SMALL = 4,
    PP_CD_VALIDATION_ERROR_TOO_LARGE = 5
} pp_cd_validation_error_t;

int pp_cd_type_validate_value(const pp_cd_as_type_t* type, const char* value,
			      pp_cd_validation_error_t* err, int* err_param);
const char* pp_cd_val_err_msg(int validation_error);

char* pp_cd_type_prop_get(pp_cd_as_type_t* type, const char* prop);
char* pp_cd_fkt_decl_prop_get(pp_cd_as_fkt_decl_t* type, const char* prop);

int pp_cd_type_prop_has_flag(pp_cd_as_type_t* type, const char* prop,
			     const char* flag);

char* pp_cd_lookup_prop(const pp_cd_as_cd_t* cd, 
                        pp_cd_qualified_const_id_t qid,
                        const char* prop);
int pp_cd_prop_has_flag(const pp_cd_as_cd_t* cd, 
                           pp_cd_qualified_const_id_t qid,
                           const char* prop, const char* flag);

#ifdef PP_BOARD_PEMX

typedef struct {
    pp_cd_as_cd_t*      cd;     // the cd
    vector_t*           qids;   // vector of qualified ids
} pp_cd_cfg_key_desc_t;

void pp_cd_cfg_key_desc_destroy(void *v);

/**
 * Generates a vector of pp_cd_cfg_key_desc_t (cd_0, cd_1, ..., common)
 * corresponding to a given vector of cds (src_cd_0, src_cd_1, ...).
 * The common-cd holds qids that are present AND semantically identic in each
 * of the source cds. The remaining qids are represented by cd_0, cd_1, ...
 **/
vector_t* pp_cd_union_cds(vector_t *src_cds);

#endif // PP_BOARD_PEMX

#ifdef __cplusplus
};
#endif

#endif

