/*
 * The basic config library providing
 * the LocalProfileManager controlling the different kinds of
 * Local Profiles and offering the according access APIs
 *
 * (c) 2004 Peppercon AG
 * tbr@peppercon.de
 */

#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <unistd.h>
#include <sys/sem.h>

#include <pp/base.h>
#include <pp/base64.h>
#include <pp/tokenizer.h>
#include <pp/vector.h>
#include <pp/dict.h>
#include <pp/balloc.h>
#include <pp/profile.h>
#include <pp/shm.h>
#include <pp/cd.h>
#include <pp/ldap_prof.h>
#include <pp/um.h>
#include <pp/sem_keys.h>
#include <pp/intl.h>
#ifdef PP_BOARD_PEMX
# include <pp/config.h>
#else /* PP_BOARD_PEMX */
# include <liberic_config.h>
#endif /* PP_BOARD_PEMX */

#if defined(PP_FEAT_RPCCFG)
#include <pp/rpc_ripc.h>
#endif

#include <pp/cfg.h>
#include <pp/cfgx.h>

#include <liberic_pthread.h>

#include "cfg_intern.h"
#include "profile_cache.h"

/* debug options */
#if defined(PP_CFG_DEBUG)
#define noCFG_DEBUG_FILE_CODING
#endif /* PP_CFG_DEBUG */

#define CFG_COMP_DELI      PP_CD_COMP_DELI
#define CFG_COMP_DELI_STR  PP_CD_COMP_DELI_STR
#define CFG_COMP_VECT_LP   '['
#define CFG_COMP_VECT_RP   ']'
#define CFG_KEY_MAX_COMPS  20

const char* PP_CD_FNAME_DEFAULT = "/etc/code.cdl";
const char* PP_CD_OEM_SKEL_FNAME_DEFAULT = "/etc/oem_skel.cdl";
/* WARNING: be sure, PP_CD_OEM_FNAME_DEFAULT and the file included in 
            PP_CD_OEM_SKEL_FNAME_DEFAULT are the same!! */
const char* PP_CD_OEM_FNAME_DEFAULT = "/oem/oem.cdl";

const char* pp_cfg_comp_deli       = CFG_COMP_DELI_STR;
const char* pp_cfg_vect_elems_comp = PP_CD_VECT_ELEMS_COMP_STR;
const char* pp_cfg_vect_size_comp  = PP_CD_VECT_SIZE_COMP_STR;
const char* pp_cfg_vect_def_comp   = PP_CD_VECT_DEF_COMP_STR;
const char* pp_cfg_choice_select_comp = PP_CD_CHOICE_SELECT_COMP_STR;
const char* pp_cfg_boolean_true    = "yes";
const char* pp_cfg_boolean_false   = "no";
const char* pp_cfg_comp_wc_single  = "?";
    
const char* pp_profile_type_str[] = { "NONE",
                                      "LOCAL_CODE",
                                      "LOCAL_SYSTEM",
#if defined(PP_PROFILE_LOCAL_DEF_ENABLED)
                                      "LOCAL_DEF",
#endif /* PP_PROFILE_LOCAL_DEF_ENABLED */
                                      "LOCAL_OEM",
                                      "LOCAL",
#if defined(PP_PROFILE_LOCAL_TAG_ENABLED)
                                      "LOCAL_TAG",
#endif /* PP_PROFILE_LOCAL_TAG_ENABLED */
                                      "LDAP_DEV",
                                      "LDAP_USER",
                                      "EFFECTIVE"
                                    };

#define noCFG_USE_SHM
#ifdef CFG_USE_SHM
#include <pp/sem_keys.h>
#define CFG_SHM_SIZE               (64*1024)
#define CFG_SHM_IDX_CODE_PROFILE   0
#define CFG_SHM_IDX_MAP_SIZE       1
#endif /* CFG_USE_SHM */

static int pp_cfg_initialized = 0;

/* cfg structures */
/******************/
typedef struct {
    int (*get)(pp_profile_t* prof, vector_t* key_comps);
    pp_cfg_chg_kind_t (*set)(const char* value, pp_profile_t* prof,
			     const vector_t* key_comps);
} cfg_handler_t;

/* locally scoped variables      */
/*********************************/

/* our local profile cache */
static pp_profile_cache_t* prof_cache = NULL;

/* standard allocator */
static pp_mallocator_t* cfg_allocator;

#ifdef CFG_USE_SHM
/* our shared memory to keep certain shared things */
static pp_shm_t* cfg_shm;
#else
static pp_profile_t* cfg_code_profile = NULL;
#endif /* CFG_USE_SHM */

/* the oem profile */
static pp_profile_t* cfg_oem_profile = NULL;

/* the last component handlers */
static pp_dict_t* cfg_handlers_lastcomp;

/* our configuration description */
static pp_cd_as_cd_t* cfg_cd = NULL;

/*
 * local prototypes
 * ---------------------
 */
static cfg_entry_t* cfg_entry_create(pp_mallocator_t* a,
				     cfg_entry_type_t t, cfg_entry_value_t v);
static void cfg_entry_destroy(pp_mallocator_t* a, cfg_entry_t* e);
static cfg_handler_t* cfg_check_last_comp_handler(const vector_t* comps);
static cfg_entry_t* cfg_key_resolve_comp(int recursion, pp_dict_t* dict,
					 vector_t* comps,
					 unsigned int comp_idx,
					 unsigned int last_idx);
static cfg_entry_t* cfg_key_resolve_defaults(pp_profile_t* prof,
					     vector_t* comps);
static cfg_entry_t* cfg_key_resolve_exact_core(pp_dict_t** cfg_dict,
					       pp_dict_t* start_dict,
					       const vector_t* comps,
					       unsigned int comp_idx,
					       size_t end_idx);
/*
static int cfg_get_profile_type_stack_va(pp_profile_type_stack_t* 
                                         profile_type_stack, 
                                         pp_profile_type_t prof_type,
                                         const char* key, va_list args);
*/
static int cfg_get_profile_stack_va(vector_t** prof_stack, pp_profile_type_t prof_type,
		                    const char* key, va_list args);
static int cfg_get_from_prof(char** value, pp_profile_t* prof, vector_t *key_comps);
static int cfg_get_from_prof_core(char** value, pp_profile_t* prof,
				  vector_t* key_comps);
static int cfg_get_va(int force, char** value, pp_profile_type_t prof_type,
		      const char* key, va_list ap);
static int cfg_get_key_comps(int force, char** value, pp_profile_type_t prof_type,
			     vector_t *key_comps);
static int cfg_get_int_va(int force, int* number, pp_profile_type_t prof_type,
			  const char* key, va_list ap);
static int cfg_is_enabled_va(int force, int* is_enabled, 
                             pp_profile_type_t prof_type,
			     const char* key, va_list ap);
static int cfg_get_acl_va(pp_acl_t **acl, pp_profile_type_t prof_type,
			  const char* key, va_list ap);

static int cfg_set_va(const char* value, pp_profile_type_t prof_type,
		      const char* key, va_list ap);
static int cfg_set_key_comps(const char* value, pp_profile_type_t prof_type,
			     vector_t *key_comps);

static int cfg_set_int_va(const int number, pp_profile_type_t prof_type,
			  const char* key, va_list ap);
static int cfg_set_enabled_va(int is_enabled, pp_profile_type_t prof_type,
			      const char* key, va_list ap);
static int cfg_rename_va(const char* newkey, pp_profile_type_t prof_type,
			 const char* key, va_list ap);
static int cfg_copy_va(const char* newkey, pp_profile_type_t prof_type,
		       const char* key, va_list ap);
static int cfg_copy_recursive(int recursion, const pp_profile_t* prof,
                              pp_dict_t* dict, const char* key, 
                              const cfg_entry_t* element, int overwrite);
                              
static int cfg_remove_va(pp_profile_type_t prof_type, const char* key,
			 va_list ap);

static char* encode_cfg_key(int cprefix, vector_t* stack, 
                            char *cfg_key, size_t cfg_key_len);
static char* decode_cfg_key(const char* encoded_key, vector_t* stack, 
                            char *cfg_key, size_t cfg_key_len);

static pp_profile_t* load_profile(pp_profile_type_t type, const char* name);
static int diff_and_update_profile(pp_profile_t* prof, pp_profile_t* new_prof);
#ifndef PP_BOARD_PEMX
static void load_profile_option_cb(eric_config_option_t* option, va_list args);
static int save_profile(const pp_profile_t* prof, int lock);
#endif
static void save_profile_dict(int recursion, int cprefix, vector_t* cfgs,
                              vector_t* stack, const pp_dict_t* dict);
static const char* get_local_profile_name(pp_profile_type_t prof_type);
static int cfg_iter_create_va(pp_cfg_iter_t** iter, pp_cfg_iter_t* parent,
			      pp_profile_type_t type, cfg_iter_type_t itype, 
                              const char* key, va_list ap);
static int cfg_iter_create(pp_cfg_iter_t** iter, pp_cfg_iter_t* parent,
			   pp_profile_type_t type, cfg_iter_type_t itype, 
                           const vector_t* key_comps, u_int key_comps_sz,
                           pp_profile_t *p, pp_dict_t *dict);
static int cfg_iter_create_from_prof_stack(pp_cfg_iter_t** iter,
			                   pp_profile_type_t type, cfg_iter_type_t itype, 
                                           const vector_t* key_comps, u_int key_comps_sz,
                                           vector_t *ps);


static cfg_handler_t* cfg_handler_create(int (*get)(pp_profile_t*, vector_t*),
					 pp_cfg_chg_kind_t (*set)(const char*,
					     pp_profile_t*, const vector_t*));
static pp_cfg_chg_kind_t cfg_vector_set_size(const char* value,
					     pp_profile_t* prof,
					     const vector_t* key_comps);

static void cfg_key_expand_recursive(int r, vector_t* comps, 
                                     unsigned long cur_idx,
                                     key_comp_t *cur_comp_p,
                                     pp_strstream_t *strbuf_p,  
                                     const char* key, va_list ap);

static void profile_set_va(const char* value, pp_profile_t* prof, int notify,
                           const char* key, va_list ap);
                                             
static void* cfg_ext_chg_listener_thr_func(void* arg);

static int cfg_validate_va(const char* value, const char* key, va_list ap);

static int cfg_validate_key_comps(const char* value, const vector_t* key_comps);    

static int cfg_get_profile_va(const char* key, va_list ap, pp_profile_type_t *profile);

static int cfg_get_profile_key_comps(const vector_t* key_comps, pp_profile_type_t *profile);

static int cfg_get_prop_va(const char **value, const char *prop, 
                           const char* key, va_list va);


/**
 * dummy methods for the use without LDAP lib
 */
  
__attribute__ ((__weak__)) int 
pp_ldap_is_active(char** lds UNUSED, char** lds_user UNUSED) {
    return 0;
}

__attribute__ ((__weak__)) int 
pp_ldap_mini_is_active(char** lds_user UNUSED) {
    return 0;
}

__attribute__ ((__weak__)) int 
pp_ldap_authenticate(int *auth UNUSED, const char* moid UNUSED, 
                     const char* name UNUSED, const char* pw UNUSED,
                     char **groupname UNUSED) {
    return PP_ERR;
}

/**
 * even more aliases
 */
  
int
pp_cfg_get_profile_type_stack_core(pp_profile_type_stack_t *prof_type_stack, 
                                   pp_profile_type_t prof_type,
                                   const vector_t* key_comps)
__attribute__ ((weak, alias ("__pp_cfg_get_profile_type_stack_core")));

int
pp_cfg_get_profile_stack_core(vector_t** prof_stack, 
                              pp_profile_type_t prof_type,
                              const vector_t* key_comps)
__attribute__ ((weak, alias ("__pp_cfg_get_profile_stack_core")));

int
pp_cfg_get_profile_core(pp_profile_t** prof, pp_profile_type_t prof_type,
		        const vector_t *key_comps, int uid)
__attribute__ ((weak, alias ("__pp_cfg_get_profile_core")));

/*
 * utility methods
 * =======================
 */
static cfg_entry_t*
cfg_entry_create(pp_mallocator_t* a, cfg_entry_type_t t, cfg_entry_value_t v) {
    cfg_entry_t* e = pp_malloc(a, sizeof(cfg_entry_t));
    e->type = t;
    e->value = v;
    return e;
}

static void
cfg_entry_destroy(pp_mallocator_t* a, cfg_entry_t* e) {
    if(e != NULL) {
	switch(e->type) {
	  case CFG_STR: pp_free(a, e->value.string); break;
	  case CFG_DICT: pp_dict_destroy(e->value.dict); break;
	  default: assert(0);
	}
	pp_free(a, e);
    }
}

/*
 * returns a vector containing the extracted components of a config key
 * each component has a name and an 'alternative' information
 * the alternative informtion describes which components of the entire
 * key should be left out to find a possible default.
 *
 * memory management is a bit strange: in order to minimize malloc
 * calls all the strings are kept in a single, dynamically expanded buffer
 * managed pp_strstream, this buffer must be freed using the address
 * of the first key_component's name pointer.
 *
 * we also do some validation whether the seen components are allowed
 * validation functions will never return but abort the process, as they
 * detect programmer's faults
 */

#define CUR_KEY_COMP_INIT(an, isdx) cur_comp->name = (char*)strbuf->pos; \
                                    cur_comp->alt_next = (an);           \
                                    cur_comp->is_vector_idx = isdx;

#define CUR_KEY_COMP_DONE if ((unsigned int)cur_comp->name != strbuf->pos) { \
                              pp_strappendchr(strbuf, '\0');		     \
 	                      vector_add2(comps, cur_comp);                  \
	                      ++cur_idx;                                     \
                          }
                          
#define CFG_ALT_KEY_COMP -1

/* error management */
static int errno_base = 0; // triggers assertion failed if not registered
inline int pp_cfg_errno_base() { assert(errno_base > 0); return errno_base; }

static const char * pp_cfg_errors[] = {
    /* 0*/ N_("An internal error occurred."),
    /* 1*/ N_("Invalid configuration key."),
    /* 2*/ N_("Invalid configuration value."),
};

static const char* get_error_string(const int error) {
    return pp_cfg_errors[error - errno_base];
}


vector_t*
pp_cfg_key_expand(const char* key, ...) {
    vector_t* key_comps;
    va_list ap;
    va_start(ap, key);
    key_comps = cfg_key_expand(key, ap);
    va_end(ap);
    return key_comps;
}

static void cfg_key_expand_recursive(int r, vector_t* comps, 
                                     unsigned long cur_idx,
                                     key_comp_t *cur_comp,
                                     pp_strstream_t *strbuf,  
                                     const char* key, va_list ap) {
    char c;
    const char* p = key;
    assert(r < 5);

    while((c = *p++) != '\0') {
        switch (c) {
          case '%':
              switch (*p++) {
                case 'd':
#if !defined(NDEBUG)
                    pp_log("%s(): deprecated %%d in cfgkey found - use %%u\n", ___F);
#endif /* !NDEBUG */
                case 'u':
                    /* we possibly want to get negative ints some time... */
                    pp_strappendf(strbuf, "%u", va_arg(ap, unsigned int));
                    break;
                case 's':
                    cfg_key_expand_recursive(r + 1, comps, cur_idx, cur_comp,
                                             strbuf, va_arg(ap, char*), ap);
                    break;
                default: assert(0);
              }
              break;
          case CFG_COMP_VECT_LP:
              CUR_KEY_COMP_DONE;
              CUR_KEY_COMP_INIT(cur_idx + 2, 0);
              pp_strappendf(strbuf, "%s", pp_cfg_vect_elems_comp);
              CUR_KEY_COMP_DONE;
              CUR_KEY_COMP_INIT(0, 1);
              break;
          case CFG_COMP_DELI:
              CUR_KEY_COMP_DONE;
              CUR_KEY_COMP_INIT(0, 0);
              break;
          case CFG_COMP_VECT_RP:
              /* ignore */
              break;
          default:
              pp_strappendchr(strbuf, c);
        }
    }
}

vector_t*
cfg_key_expand(const char* key, va_list ap) {
    size_t s;
    const char* p = key;
    unsigned long cur_idx = 0;
    key_comp_t comp, *cur_comp = &comp, *cp;
    vector_t* comps;
    pp_strstream_t strstream, *strbuf = &strstream;
    comps = vector_new2(NULL, 32, sizeof(key_comp_t), NULL);
    if (p && *p) {
	pp_strstream_init(strbuf);
	CUR_KEY_COMP_INIT(0, 0);
        cfg_key_expand_recursive(0, comps, 0, cur_comp, strbuf, key, ap);
	CUR_KEY_COMP_DONE;
	
	// we simply remember the positions, need to change that to ptrs
	// and check whether alternatives do not point beyond bound
	s = vector_size(comps);
	for (cur_idx = 0; cur_idx < s; ++cur_idx) {
	    cp = vector_get2(comps, cur_idx);
	    cp->name = strbuf->buf + (unsigned long)cp->name;
            if(!strcmp(cp->name, pp_cfg_vect_elems_comp) && !cp->alt_next) {
                /* set alternative */
                cp->alt_next = cur_idx + 2;
            }
	    if (cp->alt_next >= (int)s) cp->alt_next = CFG_ALT_KEY_COMP;
	}
    }
    return comps;
}

vector_t* cfg_key_duplicate(const vector_t* comps) {
    vector_t *new_comps;
    unsigned int comps_sz, u, len;
    key_comp_t *key_comp, comp, *first_comp, *last_comp;
    char *buf;
    long offset;
    
    assert(comps && vector_size(comps));
    
    comps_sz = vector_size(comps);
    new_comps = vector_new2(NULL, comps_sz, sizeof(key_comp_t), NULL);

    /* get length of stream and duplicate */
    first_comp = (key_comp_t*)vector_get2(comps, 0);
    last_comp = (key_comp_t*)vector_get2(comps, comps_sz - 1);
    assert(first_comp->name);
    assert(last_comp->name);
    len = last_comp->name + strlen(last_comp->name) + 1 - first_comp->name;
    buf = (char*)malloc(len);
    memcpy(buf, first_comp->name, len);
    
    /* calc offset and set pointers */
    offset = (long)buf - (long)first_comp->name;
    for(u = 0; u < comps_sz; ++u) {
        key_comp = (key_comp_t*)vector_get2(comps, u);
        comp.name = (char*)((long)key_comp->name + offset);
        assert(comp.name < buf + len);
        comp.alt_next = key_comp->alt_next;
        comp.is_vector_idx = key_comp->is_vector_idx;
        vector_add2(new_comps, &comp);
    }
    assert(buf == ((key_comp_t*)vector_get2(new_comps, 0))->name);
    
    return new_comps;
}

/**
 * compares config keys in key_comp format
 * returns 1 if key_comps_1 and key_comps_2 are identical, 0 otherwhise
 */
int pp_cfg_key_compare(const vector_t *key_comps_1, 
                       const vector_t *key_comps_2) {
    unsigned int kc1_sz, kc2_sz, u;
    char *ptr1, *ptr2;
    
    kc1_sz = vector_size(key_comps_1);
    kc2_sz = vector_size(key_comps_2);
    if(kc1_sz != kc2_sz)
        return 0; /* vectors have to be of same size! */
    
    /* compare all vector elements */
    for(u = 0; u < kc1_sz; ++u) {
        for(ptr1 = ((key_comp_t*)vector_get2(key_comps_1, u))->name, 
            ptr2 = ((key_comp_t*)vector_get2(key_comps_2, u))->name;
            *ptr1 || *ptr2; ++ptr1, ++ptr2) {
            if(*ptr1 != *ptr2) {
                return 0; /* at least one char is different */
            }
        }
    }

    return 1;
}

void pp_cfg_key_comps_destroy(vector_t* comps) {
    if (vector_size(comps) > 0) {
	/* ATTENTION: This assumes that the name pointer of the first element
	 *            of a key component vector is a buffer containing all the
	 *            names of the other elements of the vector.
	 */
	free(((key_comp_t*)vector_get2(comps, 0))->name);
    }
    vector_delete(comps);
}

/*
 * TODO: do implement this, somewhen!!!
 * loop over all components and check whether they are valid
 * currently this includes:
 * - numeric vector indexes within bounds, if bounds exist
 * later:
 * - check with configuration description for valid keys
 */
#if 0
static int cfg_key_comps_validate(pp_profile_t* prof, vector_t* comps) {
    key_comp_t *cp;
    unsigned int i, s = vector_size(comps);
    char* p;
    int num_idx;
    
    for (i = 0; i < s; ++i) {
	cp = vector_get2(comps, cur_idx);
	if (cp->is_vector_idx) {
	    for (p = cp->name; *p != '\0'; ++p)
		if (!isdigit(*p))
		    goto non_num;
	    // this is a numeric index, so check for bounds
	    
	    
	    num_idx = atoi(cp->name);
	    
	non_num:
	}
    }
}
#endif

/* checks for a last component handler and returns it, if found */
static cfg_handler_t*
cfg_check_last_comp_handler(const vector_t* key_comps) {
    key_comp_t* kc = (key_comp_t*)vector_get2(key_comps,
					      vector_size(key_comps) - 1);
    if (kc->is_vector_idx)
	return NULL;

    return pp_dict_search(cfg_handlers_lastcomp, kc->name);
}

static cfg_entry_t*
cfg_key_resolve_comp(int recursion, pp_dict_t* dict, vector_t* comps,
		     unsigned int comp_idx, unsigned int last_idx) {
    key_comp_t* comp;
    cfg_entry_t* cfge;
    vector_t *alt_comps;

    assert(recursion++ < CFG_KEY_MAX_COMPS);
    
    comp = (key_comp_t*)vector_get2(comps, comp_idx);
    cfge = (cfg_entry_t*)pp_dict_search(dict, comp->name);
    
    if (comp_idx < last_idx) { // not last comp?
	if (cfge) {
	    assert(cfge->type == CFG_DICT);
	    cfge = cfg_key_resolve_comp(recursion, cfge->value.dict,
					comps, comp_idx + 1, last_idx);
	}
	if (!cfge) {
            if (comp->alt_next > 0) {
	        cfge = cfg_key_resolve_comp(recursion, dict, comps,
		    			    comp->alt_next, last_idx);
            } else if (comp->alt_next == CFG_ALT_KEY_COMP) {
                /**
                 * hack... ;-)
                 * we want to resolve a vector, e.g. log.event._e_.sth which has
                 * no alternative key - set to CFG_ALT_KEY_COMP (usually -1) by
                 * cfg_key_expand - because it would point beyond last_idx
                 * instead, we want to resolve log.event._d_
                 * to not destroy the comps vector, copy it, modify it, resolve
                 * and delete that vector again...
                 **/ 
                alt_comps = cfg_key_duplicate(comps);
                comp = (key_comp_t*)vector_get2(alt_comps, comp_idx);
                assert(!strcmp(comp->name, pp_cfg_vect_elems_comp));
                assert(strlen(pp_cfg_vect_elems_comp) ==
                       strlen(pp_cfg_vect_def_comp));
                strcpy(comp->name, pp_cfg_vect_def_comp);
                cfge = cfg_key_resolve_comp(recursion, dict, alt_comps,
		    			    comp_idx, comp_idx);
                pp_cfg_key_comps_destroy(alt_comps);
            }
	}
    }
    return cfge;
}

static cfg_entry_t*
cfg_key_resolve_defaults(pp_profile_t* prof, vector_t* comps){
    assert(vector_size(comps) > 0);
    assert(prof->type != PP_PROFILE_NONE);
    /**
     * for LDAP user profiles, strip off first three key components (user._e_.tweb)
     * as they are not stored in a LDAP user profile
     */
    return cfg_key_resolve_comp(0, prof->configs, comps, 
                                prof->type == PP_PROFILE_LDAP_USER ? 3 : 0,
                                vector_size(comps) - 1);
}

cfg_entry_t*
pp_cfg_key_resolve_force(pp_dict_t** cfg_dict, pp_profile_t* prof, 
                         const vector_t* comps) {
    cfg_entry_t  sce  = { CFG_DICT, { dict: prof->configs } };
    cfg_entry_t* cfge = &sce;
    pp_dict_t*   dict = NULL;
    key_comp_t*  comp;
    int i, comp_idx;
    int comps_size = vector_size(comps);

    assert(comps_size > 0);

    for (comp_idx = 0; comp_idx < comps_size; ++comp_idx) {
	
	comp = (key_comp_t*)vector_get2(comps, comp_idx);
	assert(strlen(comp->name) > 0);

	assert(cfge->type == CFG_DICT);
	dict = cfge->value.dict;
	
	if (NULL == (cfge = (cfg_entry_t*)pp_dict_search(dict, comp->name))) {
	    cfg_entry_value_t v;
	    cfg_entry_type_t  t;
	    if (comp_idx == comps_size - 1) { // last comp?
		t = CFG_STR;
		v.string = NULL;
	    } else {
		t = CFG_DICT;
		v.dict = pp_dict_str_new_with_alloc(prof->allocator);
	    }
	    cfge = cfg_entry_create(prof->allocator, t, v);
	    i = pp_dict_insert(dict, comp->name, cfge,
			       (pp_dict_del_func)cfg_entry_destroy, 0);
	    assert(i == 0); // entry should be empty
	}
    }
    if (cfg_dict) *cfg_dict = dict;
    return cfge;
}

static cfg_entry_t*
cfg_key_resolve_exact_core(pp_dict_t** cfg_dict, pp_dict_t* start_dict,
			   const vector_t* comps, unsigned int comp_idx, 
                           size_t end_idx) {
    cfg_entry_t  sce  = { CFG_DICT, { dict: start_dict } };
    cfg_entry_t* cfge = &sce;
    key_comp_t*  comp;
    pp_dict_t*   dict = start_dict;

    if(end_idx > 0) {
        for (; comp_idx < end_idx; ++comp_idx) {
            
            comp = (key_comp_t*)vector_get2(comps, comp_idx);
            assert(strlen(comp->name) > 0);
    
            assert(cfge->type == CFG_DICT);
            dict = cfge->value.dict;
            
            if (NULL == (cfge = (cfg_entry_t*)pp_dict_search(dict, comp->name))) {
#if defined(PP_CFG_DEBUG)
                printf("%s: pp_dict_search failed for key '%s'\n",
                       ___F, comp->name);
                printf("valid keys are\n");
                pp_ditr_t* iter = pp_ditr_new(dict);
                while(pp_ditr_next(iter)) {
                    printf(" - %s\n", (char*)pp_ditr_key(iter));
                }
                pp_ditr_destroy(iter);
#endif /* PP_CFG_DEBUG */
                break;
            }
        }
    }
    
    if (cfg_dict) *cfg_dict = dict;
    return cfge;
}   

cfg_entry_t*
pp_cfg_key_resolve_exact(pp_dict_t** cfg_dict, pp_profile_t* prof,
		      vector_t* comps) {
    /**
     * for LDAP user profiles, strip off first three key components (user._e_.tweb)
     * as they are not stored in a LDAP user profile
     */
    return cfg_key_resolve_exact_core(cfg_dict, prof->configs, comps, 
                                      prof->type == PP_PROFILE_LDAP_USER ? 3 : 0,
				      vector_size(comps));
}

int
cfg_get_from_prof_stack(char** value, vector_t* profile_stack,
			vector_t *key_comps) {
    unsigned long i;
    int ret = PP_ERR;
    pp_profile_t* p;
    
    for (i = 0; i < vector_size(profile_stack); ++i) {
	p = vector_get(profile_stack, i);
	pp_mutex_lock(&p->mtx);
	ret = cfg_get_from_prof(value, p, key_comps);
	pp_mutex_unlock(&p->mtx);
	if (ret != PP_ERR) {
	    /* we've found it */
	    break;
	}
    }
    /* we havn't found it */
    return ret;
}

static int
cfg_get_from_prof(char** value, pp_profile_t* prof, vector_t *key_comps) {
    int ret;
    cfg_handler_t* ch;

    // check if last component has a registered handler
    if (NULL != (ch = cfg_check_last_comp_handler(key_comps))
	&& NULL != ch->get) {
	ret = ch->get(prof, key_comps); // call handler
    } else {                      // no handler registered
	ret = cfg_get_from_prof_core(value, prof, key_comps);
    }
    
    return ret;
}

static int
cfg_get_from_prof_core(char** value, pp_profile_t* prof, vector_t* key_comps) {
    cfg_entry_t* e;
    int ret;

#if !defined(PP_CFG_NO_VALIDATION_DEBUG) && !defined(NDEBUG)
    { /* assert config key is valid */
        char cfg_key[MAX_OPT_KEY_LEN], qid[MAX_OPT_KEY_LEN];
        pp_cd_as_type_t *type;
        pp_cd_as_cd_t *cd = pp_cfg_get_cd();
        
        pp_cfg_get_key_from_key_comps(key_comps, cfg_key, MAX_OPT_KEY_LEN);
        cfg_get_qid_from_key_comps(key_comps, qid, MAX_OPT_KEY_LEN);
        
        if((type = pp_cd_resolve_type_symbol(cd, qid,
                                             PP_CD_RESOLVE_ALIAS)) == NULL) {
            pp_log("%s: Config key '%s' (qid '%s') is invalid\n",
                   ___F, cfg_key, qid);
//            assert(0);
        }
    }
#endif /* PP_CFG_DEBUG || !NDEBUG*/

    if (NULL != (e = cfg_key_resolve_defaults(prof, key_comps))) {
	assert(e->type == CFG_STR || !value);
        if(value) {
            assert(e->value.string);
            *value = strdup(e->value.string);
        }
        ret = prof->type;
    } else {
	ret = PP_ERR;
    }
    return ret;
}

void
pp_profile_set(const char* value, pp_profile_t* prof,
	       const char* key, va_list ap) {
    profile_set_va(value, prof, 1, key, ap);
}

static void 
profile_set_va(const char* value, pp_profile_t* prof, int notify,
	       const char* key, va_list ap) {
    vector_t* key_comps = cfg_key_expand(key, ap);
    pp_profile_set_core(value, prof, key_comps, notify);
    pp_cfg_key_comps_destroy(key_comps);
}

void
pp_profile_set_core(const char* value, pp_profile_t* prof,
		    const vector_t *key_comps, int notify) {
    pp_cfg_chg_kind_t kind;

    kind = cfg_set_to_prof_no_notify(value, prof, key_comps);
    if(notify && kind != PP_CFG_CHG_NOOP) {
        /* only notify if really something changed... */
        cfg_fire_change(key_comps, kind);
    }
}

pp_cfg_chg_kind_t
cfg_set_to_prof_no_notify(const char* value, pp_profile_t* prof,
			  const vector_t* key_comps) {
    pp_cfg_chg_kind_t ret;
    cfg_handler_t* ch;
    
    // check if last component has a registered handler
    if (NULL != (ch = cfg_check_last_comp_handler(key_comps))
	&& NULL != ch->set) {
	ret = ch->set(value, prof, key_comps); // call handler
    } else {
	ret = cfg_set_to_prof_core(value, prof, key_comps);
    }
    return ret;
}

pp_cfg_chg_kind_t
cfg_set_to_prof_core(const char* value, pp_profile_t* prof,
		     const vector_t* key_comps) {
    pp_cfg_chg_kind_t ret;
    cfg_entry_t* e;
    assert(value != NULL && strlen(value)<MAX_OPT_VALUE_LEN);
    
//    PRINT_KEY_VALUE(key_comps, value)
	
    e = pp_cfg_key_resolve_force(NULL, prof, key_comps);

    assert(e->type == CFG_STR);
    if (e->value.string != NULL) {
        /* variable is already set, check if it changed */
        if(strcmp(e->value.string, value)) {
            /* changed */
            pp_free(prof->allocator, e->value.string);
            ret = PP_CFG_CHG_MOD;
        } else {
            /* value is equal, do nothing */
            return PP_CFG_CHG_NOOP;
        }
    } else {
	ret = PP_CFG_CHG_ADD;
    }

#if !defined(PP_CFG_NO_VALIDATION_DEBUG) && !defined(NDEBUG)
    cfg_validate_key_comps(value, key_comps);
#endif /* PP_CFG_DEBUG || !NDEBUG*/

    e->value.string = pp_strdup(prof->allocator, value);
    return ret;
}

void
pp_profile_set_int(const int number, pp_profile_t* p, const char* key, va_list ap)
{
    char * value;
    CFG_INT_2_STRVAL(value, number);
    pp_profile_set(value, p, key, ap);
}

/* FIXME: (rwa) not used currently - remove? */
#if 0
void
pp_profile_set_uint(const u_int number, pp_profile_t* p, const char* key, va_list ap)
{
    char * value;
    CFG_UINT_2_STRVAL(value, number);
    pp_profile_set(value, p, key, ap);
}
#endif /* 0 */

void
pp_profile_set_bool(const int boolval,  pp_profile_t* p, const char* key, va_list ap)
{
    const char * value;
    CFG_BOOL_2_STRVAL(value, boolval);
    pp_profile_set(value, p, key, ap);
}

/* FIXME: (rwa) not used currently - remove? */
#if 0
void
cfg_set_float_to_prof(const float number, pp_profile_t* p, const char* key, va_list ap)
{
    char * value;
    CFG_FLOAT_2_STRVAL(value, number);
    pp_profile_set(value, p, key, ap);
}
#endif /* 0 */

static int
cfg_remove_from_prof(pp_profile_t* prof, const char* key, va_list ap) {
    vector_t* key_comps = cfg_key_expand(key, ap);
    int ret = cfg_remove_from_prof_core(prof, key_comps, 
                                        vector_size(key_comps) - 1);
    
    cfg_fire_change(key_comps, PP_CFG_CHG_REM);
    pp_cfg_key_comps_destroy(key_comps);
    
    return ret;
}
    
int cfg_remove_from_prof_core(pp_profile_t* prof, 
                              vector_t *key_comps, int last_comp_idx) {
    int ret;
    key_comp_t* last_comp;
    cfg_entry_t *cfge;
    
    if (NULL != (cfge = cfg_key_resolve_exact_core(NULL, prof->configs,
                                                   key_comps, 0,
                                                   last_comp_idx)) &&
        cfge->type == CFG_DICT) {
	last_comp = (key_comp_t*)vector_get2(key_comps, last_comp_idx);
	ret = pp_dict_remove(cfge->value.dict, last_comp->name, NULL);
    
        /* remove parent dict if empty */
        if(last_comp_idx) {
            if(NULL != (cfge = cfg_key_resolve_exact_core(NULL, prof->configs,
                                                          key_comps, 0,
                                                          last_comp_idx)) &&
               cfge->type == CFG_DICT) {
		pp_ditr_t* ditr;
                --last_comp_idx;
                ditr = pp_ditr_new(cfge->value.dict);
                if(!pp_ditr_first(ditr)) {
                    /* dict is empty */
                    ret = cfg_remove_from_prof_core(prof, key_comps, 
                                                    last_comp_idx);
                }
                pp_ditr_destroy(ditr);
            }
        }
    } else {
	ret = PP_ERR;
    }
    
#if defined(PP_CFG_DEBUG)
    assert(pp_profile_validate_type(PP_PROFILE_LOCAL) == PP_SUC);
#endif /* PP_CFG_DEBUG */
    return ret;
}

#ifdef PP_FEAT_RPCCFG
/**
 * serialize_key_comps - Serializes a vector with key_comp_t elements.
 *			 (e.g. for transmitting it over a network)
 *
 * Parameters:
 *	buf		- A pointer to a pointer to a buffer containing
 *			  the serialized data. Will be allocated by the
 *			  rountine and must be freed by the caller.
 *	buf_size_t	- A pointer to a variable containing the size of
 *			  the allocated buffer.
 *	key_comps	- A vector containing the key components.
 *
 * Return value:	PP_SUC on success; PP_ERR otherwise
 */
static int
serialize_key_comps(void ** buf, size_t * buf_size_p, const vector_t * key_comps)
{
    size_t key_comps_cnt = vector_size(key_comps);
    size_t i;
    size_t buf_size = sizeof(size_t);
    pp_buf_ptr_t buf_ptr;

    assert(buf);
    assert(buf_size_p);
    assert(key_comps);

    /* determine the size of the serialized data */
    for (i = 0; i < key_comps_cnt; ++i) {
	key_comp_t * comp = vector_get2(key_comps, i);
	size_t len = strlen(comp->name) + 1;
	buf_size += sizeof(size_t) + len + sizeof(int) + sizeof(int);
    }

    /* allocate and fill the buffer with serialized data */
    if ((buf_ptr.bp_void = malloc(buf_size)) != NULL) {
	*buf = buf_ptr.bp_void;
	*buf_size_p = buf_size;

	/* store comp vector size */
	*buf_ptr.bp_size_t++ = key_comps_cnt;

	/* interate over comp vector */
	for (i = 0; i < key_comps_cnt; ++i) {
	    key_comp_t * comp = vector_get2(key_comps, i);

	    /* store length of comp->name (includes final NUL char) */
	    size_t comp_name_len = strlen(comp->name) + 1;
	    *buf_ptr.bp_size_t++ = comp_name_len;

	    /* store comp->name */
	    memcpy(buf_ptr.bp_void, comp->name, comp_name_len);
	    buf_ptr.bp_void += comp_name_len;

	    /* store comp->alt_next */
	    *buf_ptr.bp_int++ = comp->alt_next;

	    /* store comp->is_vector_idx */
	    *buf_ptr.bp_int++ = comp->is_vector_idx;
	}
	return PP_SUC;
    }
    return PP_ERR;
}

/**
 * deserialize_key_comps - Deserializes a buffer containing a serialized
 *			   vector with key_comp_t elements.
 *
 * Parameters:
 *	buf		- A pointer to a buffer containing the serialized
 *			  data.
 *
 * Return value:	On success the deserialized key component is
 *			returned. In case of an error NULL is returned.
 */
static vector_t*
deserialize_key_comps(const void * buf, size_t buf_size)
{
    pp_buf_ptr_const_t buf_ptr = { .bp_void = buf};
    vector_t * key_comps;
    size_t key_comps_cnt = *buf_ptr.bp_size_t++;
    size_t i;
    pp_strstream_t strstream;

    /* sanity check */
    if (key_comps_cnt > 64) return NULL;

    /* allocate the key component vector */
    key_comps = vector_new2(NULL, key_comps_cnt, sizeof(key_comp_t), NULL);
    if (key_comps == NULL) return NULL;

    pp_strstream_init(&strstream);

    for (i = 0; i < key_comps_cnt; ++i) {
	key_comp_t key_comp;

	/*
	 * NOTE: We need to be compatible to cfg_key_expand() regarding
	 *       memory management!
	 */

	/* deserialize a key component */
	size_t name_len = *buf_ptr.bp_size_t++;
	/*
	 * Because the memeory location may change we only remember
	 * the relative position for now. This will be fixed afterwards.
	 */
	key_comp.name = (char *)strstream.pos;
	pp_strwrite(&strstream, buf_ptr.bp_void, name_len);
	buf_ptr.bp_void += name_len;
	key_comp.alt_next = *buf_ptr.bp_int++;
	key_comp.is_vector_idx = *buf_ptr.bp_int++;

	/* add the deserialized key component to the vector */
	vector_add2(key_comps, &key_comp);
    }

    PP_ASSERT(buf_size, ((u_long)buf_ptr.bp_void - (u_long)buf) == buf_size);

    /*
     * Fix the component name pointers and check whether alternatives
     * do not point beyond bound.
     */
    for (i = 0; i < key_comps_cnt; ++i) {
	key_comp_t * key_comp_p = vector_get2(key_comps, i);
	key_comp_p->name = strstream.buf + (unsigned long)key_comp_p->name;
	if ((size_t)key_comp_p->alt_next >= key_comps_cnt) {
	    key_comp_p->alt_next = CFG_ALT_KEY_COMP;
	}
    }

    return key_comps;
}

#endif /* PP_FEAT_RPCCFG */

/**
 * get magic:
 *
 * get var from config
 * => ret == PP_ERR or real prof type (!= PP_PROFILE_EFFECTIVE / -NONE)
 * if tx, lookup var in tx log
 *     if the cfg_var is stored in tx-log an assigned profile type is the same
 *     (for mode explicit profile type) or at least the same (if profile type
 *     is PP_PROFILE_EFFECTIVE) than the profile type requestet, return tx-var
 *     value instead
 * if still no value and forced mode, return default
 */
static int
cfg_get_va(int force, char** value, pp_profile_type_t prof_type, 
           const char* key, va_list ap) {
    vector_t* key_comps;
    int ret;

    assert(pp_cfg_initialized);
   
    /* try to get key from cfg */
    key_comps = cfg_key_expand(key, ap);

#ifdef PP_FEAT_RPCCFG
    if (!pp_rpc_am_i_master()) {
	char qid[MAX_OPT_KEY_LEN];
        pp_cd_as_cd_t *cd = pp_cfg_get_cd();
	cfg_get_qid_from_key_comps(key_comps, qid, MAX_OPT_KEY_LEN);
	/* check if the key has the "rpc_keep_local" flag */
	if (!pp_cd_prop_has_flag(cd, qid, "flags", "rpc_keep_local")) {
	    if (key_comps && value) {
		void * buf;
		size_t buf_size;
		if (PP_SUCCED(ret = serialize_key_comps(&buf, &buf_size, key_comps))) {
		    ret = pp_rpc_cfg_get(force, value, prof_type, buf, buf_size);
		    free(buf);
		} else if (force) {
		    *value = CFG_FORCE_DEFAULT_RETVAL;
		}
		pp_cfg_key_comps_destroy(key_comps);
		return ret;
	    }
	}
    }
#endif /* PP_FEAT_RPCCFG */
    ret = cfg_get_key_comps(force, value, prof_type, key_comps);
    pp_cfg_key_comps_destroy(key_comps);
    return ret;
}

#if 0 // old implementation
static int
cfg_get_key_comps(int force, char** value, pp_profile_type_t prof_type, vector_t *key_comps)
{
    int ret = PP_ERR;
    pp_cfg_tx_t* tx;

    assert(pp_cfg_initialized);
   
    tx = pp_cfg_tx_get_thread_ctx();

    if(tx && tx->commit_in_progress) {
        ret = cfg_tx_get_key_comps(tx, value, prof_type, key_comps);
	if (ret != PP_ERR) {
            /**
             * FIXME:
             * - *value may be NULL if cfg key is to be deleted
             *   (e.g. reset to defaults applied)
             * - value may be received from profile lower than it would have
             *   been received from config if PP_PROFILE_EFFECTIVE was
             *   requested
             */
	    goto finish;
	} else {
	    /* cfg var not in tx change log -> fall back to config */
	    /* build profile stack from locked profiles in tx */
	    tx = NULL;
	}
    }
    
    ret = cfg_get_key_comps_no_tx_check(value, prof_type, key_comps);

    /* check if we are running in transaction context... */
    if(tx) {
        char *tx_val;
        int tx_ret;
        pp_profile_type_t tx_prof_type;
        
        /* calculate tx_prof_type */
        if(ret == PP_ERR) {
            /* we didn't find key in cfg, use prof_type */
            tx_prof_type = prof_type;
        } else {
            /* we got a key, use returned prof_type */
            tx_prof_type = (pp_profile_type_t)ret;
        }
    
        /* get key from tx
         * return value tx_ret (profile type) must not be PP_ERR and at least
         * the same than the read cfg profile type ret since
         * - if tx_prof_type != PP_PROFILE_EFFECTIVE, tx_ret == ret | PP_ERR
         * - else tx_ret may be any profile type between NONE and EFFECTIVE if
         *   the key is set in tx_log or PP_ERR otherwhise */         
        if((tx_ret = cfg_tx_get_key_comps(tx, &tx_val, tx_prof_type,
                                          key_comps)) != PP_ERR &&
           (tx_ret >= ret)) {
            if(value) {
                /* care for get_prof_type where value is NULL */
                if(ret != PP_ERR) {
                    /* we already got a value, free it! */
                    free(*value);
                }
                 *value = tx_val;
            }
            ret = tx_ret;
        }
    }

 finish:
    if(ret == PP_ERR && force) {
#if !defined(NDEBUG)    
        char cfg_key[MAX_OPT_KEY_LEN];
        pp_log("config key '%s' not found in profile %s, "
               "returning default value\n", 
               pp_cfg_get_key_from_key_comps(key_comps, cfg_key,
                                             MAX_OPT_KEY_LEN),
               pp_profile_type_str[PP_REAL_PROFILE(prof_type)]);
//        assert(0);
#endif /* !NDEBUG */
        *value = CFG_FORCE_DEFAULT_RETVAL;
    }
    
    return ret;
}
#else // new implementation
/**
 * cfg_get_key_comps - get value vor cfg key specified by key_comps vector
 * 
 * get value (and cfg profile type) from configuration
 * if tx 
 *     get tx value (and tx profile type) from transaction context
 *     if tx profile type higher or equal to cfg profile type
 *         if tx value or profile type is not a stack
 *             set value = tx value
 *         else // tx value == NULL,
 *              // means cfg key is to be deleted in tx profile type 
 *             get value from profile stack below tx profile type
 * if forced and value == NULL
 *     set value = global system default key (0 for int, "" for string)
 */
static int
cfg_get_key_comps(int force, char** value, pp_profile_type_t prof_type, 
                  vector_t *key_comps)
{
    int ret, val_prof;
    pp_cfg_tx_t* tx;
    char *_value = NULL;
    pp_profile_type_t cfg_prof_type;
    int get_val = value != NULL;

    assert(pp_cfg_initialized);
    
    if((val_prof = cfg_get_key_comps_no_tx_check(get_val ? &_value : NULL,
                                                 prof_type,
                                                 key_comps)) != PP_ERR) {
        /* we got _value from profile type "val_prof" */
        cfg_prof_type = (pp_profile_type_t)val_prof;
    } else {
        /* we didn't get a value, set profile type to none */
        cfg_prof_type = PP_PROFILE_NONE;
    }

    tx = pp_cfg_tx_get_thread_ctx();
    
    if(tx) {
        /* so we have a transaction assigned to thread. perhaps we already set
           a newer value to key? */
        char *tx_val = NULL;
        
        if((ret = cfg_tx_get_key_comps(tx, get_val ? &tx_val : NULL, prof_type,
                                       key_comps)) != PP_ERR) {
            /* we found cfg key in transaction, lets see, if its profile type
               is higher or equal to cfg profile type */
            pp_profile_type_t tx_prof_type = (pp_profile_type_t)ret;
            
            if(tx_prof_type >= cfg_prof_type) {
                /* tx_val in tx is from higher or equal profile than _value,
                   check if we want to use tx_val */
                if(tx_val || !PP_PROFILE_IS_STACK(prof_type)) {
                    /* we got a value or requested explicit profile layer
                       -> use that tx_value nevertheless it may be NULL
                          (in that case, return PP_ERR!) */
                    free(_value);
                    _value = tx_val;
                    val_prof = _value ? ret : PP_ERR;
                } else if(tx_prof_type == cfg_prof_type) {
                    /* tx_val was NULL and we requested stack, test if we have
                       to search profile stack once more below tx_prof_type.
                       we know, tx_prof_type > PP_PROFILE_LOCAL_CODE, as code
                       profile is not writeable and would never be set (within
                       a transaction or outside) */
                    /* no need to free(tx_val);! */
                    free(_value);
                    val_prof = cfg_get_key_comps_no_tx_check(get_val ? &_value 
                                                                     : NULL,
                                             PP_PROFILE_STACK(tx_prof_type - 1),
                                                             key_comps);
                }
                /* else: if tx_prof_type > cfg_prof_type use val_prof and 
                         _value, no need to free tx_val == NULL */
            } else {
                /* tx_val is received from profile lower than _value,
                   use val_prof and _value */
                free(tx_val);
            }
        }
        /* else: we did not get tx_val from tx, use val_prof and _value */
    }
    /* else: no transaction, use val_prof and _value */
    
    if(value) {
        /* return the value */
        if(_value) {
            *value = _value;
        } else if(force) {
#if !defined(NDEBUG)    
            char cfg_key[MAX_OPT_KEY_LEN];
            pp_log("config key '%s' not found in profile %s, "
                   "returning global default value \"\"\n", 
                   pp_cfg_get_key_from_key_comps(key_comps, cfg_key,
                                                 MAX_OPT_KEY_LEN),
                   pp_profile_type_str[PP_REAL_PROFILE(prof_type)]);
//            assert(0);
#endif /* !NDEBUG */
            *value = CFG_FORCE_DEFAULT_RETVAL;
        }
    } else {
        /* it seems, only prof type was requested, no need for value */
        free(_value);
    }

    /* assert we return a value if forced */
    assert(!force || *value);
    
    /* assert we return a value on successful request */
    assert(val_prof == PP_ERR || !value || *value);
    
    return val_prof;
}
#endif

int
cfg_get_key_comps_no_tx_check(char** value, pp_profile_type_t prof_type, 
			      vector_t *key_comps) {
    int ret;
    vector_t* ps = NULL;
    if (PP_ERR == (ret = pp_cfg_get_profile_stack_core(&ps, prof_type, key_comps))) {
	goto bail;
    }
    ret = cfg_get_from_prof_stack(value, ps, key_comps);
    vector_delete(ps);
 bail:
    return ret;
}

/**
 * returns true (1) for writable profiles
 * (currently PP_PROFILE_LOCAL is the only one...)
 */ 
static int
cfg_profile_is_writable(pp_profile_type_t prof_type) {
    return ((prof_type == PP_PROFILE_LOCAL) || 
            (prof_type == PP_PROFILE_SYSTEM)) ? 1 : 0;
}

static int
cfg_set_va(const char* value, pp_profile_type_t prof_type, const char* key,
	   va_list ap) {
    vector_t *key_comps;
    va_list va;
    int ret;
    
    assert(pp_cfg_initialized);

    va_copy(va, ap);
    key_comps = cfg_key_expand(key, va);
    va_end(va);
    
#ifdef PP_FEAT_RPCCFG
    char qid[MAX_OPT_KEY_LEN];
    pp_cd_as_cd_t *cd = pp_cfg_get_cd();
    cfg_get_qid_from_key_comps(key_comps, qid, MAX_OPT_KEY_LEN);
    /* check if the key has the "rpc_keep_local" flag */
    if (!pp_cd_prop_has_flag(cd, qid, "flags", "rpc_keep_local")) {
	if (key_comps) {
	    void * buf;
	    size_t buf_size;
	    if (PP_SUCCED(serialize_key_comps(&buf, &buf_size, key_comps))) {
		pp_rpc_cfg_set(value, prof_type, buf, buf_size);
		free(buf);
	    }
	}
    }
#endif /* PP_FEAT_RPCCFG */

    if (value == NULL) return cfg_remove_va(prof_type, key, ap);

    ret = cfg_set_key_comps(value, prof_type, key_comps);

    /* do not destroy key comps in tx context, cfg_tx_log_entry_destroy cares about! */
    if (pp_cfg_tx_get_thread_ctx() == NULL) {
	pp_cfg_key_comps_destroy(key_comps);
    }

    return ret;
}

static int
cfg_set_key_comps(const char* value, pp_profile_type_t prof_type, vector_t *key_comps)
{
    pp_profile_type_stack_t pts = PP_PROFILE_TYPE_STACK_T_INITIALIZER;
    pp_profile_t* p;
    pp_cfg_tx_t* tx;
    int uid = -1;
    int effective = PP_PROFILE_IS_STACK(prof_type);
    int ret;

    assert(pp_cfg_initialized);

    if(effective) {
        if(PP_ERR == (ret = pp_cfg_get_profile_type_stack_core(&pts, prof_type, 
                                                               key_comps))) {
            goto bail;
        }
        prof_type = pts.profile_type_stack[0];
    }
    
    if(!cfg_profile_is_writable(prof_type)) {
        char cfg_key[MAX_OPT_KEY_LEN];
        pp_log("%s(); attempt to set '%s=%s' to read only profile %s\n",
               ___F, 
               pp_cfg_get_key_from_key_comps(key_comps, cfg_key,
                                             MAX_OPT_KEY_LEN),
               value, pp_profile_type_str[PP_REAL_PROFILE(prof_type)]);
        // TODO? set errno?
        ret = PP_ERR;
        goto bail;
    }
    
    /* check if we are running in transaction context... */
    /* TODO: use thread specific variable instead of hash table */
    if(NULL != (tx = pp_cfg_tx_get_thread_ctx())) {
	/* TODO: should we lock new profiles we write to during tx_commit? */
        cfg_tx_set_key_comps(tx, value, prof_type, key_comps);
        ret = PP_SUC;
        goto bail;
    }

    /* for local profiles, we do not need to specify a key as they are all contained in
       the same profile, so key / args may be NULL in some cases */
    pp_cfg_is_user_key(&uid, key_comps);
    if (PP_ERR != (ret = pp_cfg_get_profile_core(&p, prof_type, 
                                                 key_comps, uid))) {
	pp_mutex_lock(&p->mtx);
	pp_profile_set_core(value, p, key_comps, 1);
	pp_mutex_unlock(&p->mtx);
	pp_profile_release(p);
    }
    
 bail:
    return ret;
}

static int
cfg_remove_va(pp_profile_type_t prof_type, const char* key, va_list ap)
{
    int ret;
    pp_profile_t* p;
    pp_cfg_tx_t* tx;
    pp_profile_type_stack_t pts = PP_PROFILE_TYPE_STACK_T_INITIALIZER;
    vector_t *key_comps;
    va_list va;
    int effective = PP_PROFILE_IS_STACK(prof_type);
    
    assert(pp_cfg_initialized);
    
    va_copy(va, ap);
    key_comps = cfg_key_expand(key, va);
    va_end(va);
    
    if(effective) {
        if(PP_ERR == (ret = pp_cfg_get_profile_type_stack_core(&pts, prof_type, 
                                                               key_comps))) {
            goto bail;
        }
        prof_type = pts.profile_type_stack[0];
    }
    
    if(!cfg_profile_is_writable(prof_type)) {
        char cfg_key[MAX_OPT_KEY_LEN];
        pp_log("%s(); attempt to remove '%s' from read only profile %s\n", ___F, 
               pp_cfg_get_key_from_key_comps(key_comps, cfg_key,
                                             MAX_OPT_KEY_LEN),
               pp_profile_type_str[PP_REAL_PROFILE(prof_type)]);
        // TODO? set errno?
        ret = PP_ERR;
        goto bail;
    }
    
    tx = pp_cfg_tx_get_thread_ctx();
    if(tx && !tx->commit_in_progress) {
        cfg_tx_remove_va(tx, prof_type, key, ap);
        ret = PP_SUC;
        goto bail;
    }
    
    va_copy(va, ap);
    if (PP_ERR != (ret = pp_cfg_get_profile(&p, prof_type, key, va))) {
	pp_mutex_lock(&p->mtx);
	ret = cfg_remove_from_prof(p, key, ap);
	pp_mutex_unlock(&p->mtx);
	pp_profile_release(p);
    }
    va_end(va);

 bail:
    pp_cfg_key_comps_destroy(key_comps);
    return ret;
}

static int
cfg_get_int_va(int force, int* number, pp_profile_type_t prof_type,
	       const char* key, va_list ap)
{
    int error;
    char *value = NULL;

    if(cfg_get_va(force, &value, prof_type, key, ap) != PP_ERR || force) {
        *number = pp_strtol_10(value, 0, &error);
        free(value);
        return error ? PP_ERR : PP_SUC;
    }
    return PP_ERR;
}

static int
cfg_set_int_va(const int number, pp_profile_type_t prof_type,
	       const char* key, va_list ap)
{
    char * value;
    CFG_INT_2_STRVAL(value, number);

    return cfg_set_va(value, prof_type, key, ap);
}

static int
cfg_get_uint_va(int force, u_int* number, pp_profile_type_t prof_type,
		const char* key, va_list ap)
{
    int error;
    char *value = NULL;

    if(cfg_get_va(force, &value, prof_type, key, ap) != PP_ERR || force) {
        *number = pp_strtoul_10(value, 0, &error);
        free(value);
        return error ? PP_ERR : PP_SUC;
    }
    return PP_ERR;
}

static int
cfg_set_uint_va(const u_int number, pp_profile_type_t prof_type,
		const char* key, va_list ap)
{
    char * value;
    CFG_UINT_2_STRVAL(value, number);

    return cfg_set_va(value, prof_type, key, ap);
}

static int
cfg_get_float_va(int force, float* number, pp_profile_type_t prof_type,
		 const char* key, va_list ap)
{
    int error;
    char *value = NULL;

    if(cfg_get_va(force, &value, prof_type, key, ap) != PP_ERR || force) {
        *number = pp_strtof(value, 0.0, &error);
        free(value);
        return error ? PP_ERR : PP_SUC;
    }
    return PP_ERR;
}

static int
cfg_set_float_va(const float number, pp_profile_type_t prof_type,
		 const char* key, va_list ap)
{
    char * value;
    CFG_FLOAT_2_STRVAL(value, number);

    return cfg_set_va(value, prof_type, key, ap);
}

static int
cfg_is_enabled_va(int force, int* is_enabled, pp_profile_type_t prof_type,
		  const char* key, va_list ap)
{
    int ret;
    char *value = NULL;

    if (PP_ERR != (ret = cfg_get_va(force, &value, prof_type, key, ap)) || force) {
	*is_enabled = !pp_strcmp_safe(value, pp_cfg_boolean_true);
        free(value);
    }
    return ret;
}

static int
cfg_set_enabled_va(int is_enabled, pp_profile_type_t prof_type,
		   const char* key, va_list ap)
{
    const char * value;
    CFG_BOOL_2_STRVAL(value, is_enabled);

    return cfg_set_va(value, prof_type, key, ap);
}

static int 
cfg_get_acl_va(pp_acl_t **acl, pp_profile_type_t prof_type,
               const char* key, va_list ap) {
    int ret = PP_ERR;
    pp_profile_t* p;
    pp_acl_t *_acl;
    vector_t *key_comps;
    vector_t* ps = NULL;
    int uid, gid = -1;
    
    key_comps = cfg_key_expand(key, ap);
    
    if((PP_SUC == pp_cfg_is_user_key(&uid, key_comps) &&
        PP_ERR == pp_cfg_get_int(&gid, "user[%u].group_id", uid)) ||
       PP_ERR == pp_cfg_is_group_key(&gid, key_comps)) {
        /* TODO! errno? */
        goto bail;
    }
    
    if (PP_ERR == pp_cfg_get_profile_stack_core(&ps, prof_type, key_comps)) {
        goto bail;
    }
    
    /* get the first prof, our user appears */
    p = (pp_profile_t*)vector_get(ps, 0);
    
    if(PP_ERR != (ret = pp_profile_get_acl(&_acl, p, gid)))
        *acl = _acl;
    
    vector_delete(ps);
 bail:
    pp_cfg_key_comps_destroy(key_comps);
    return ret;
}

static int
cfg_rename_va(const char* newkey, pp_profile_type_t prof_type,
	      const char* key, va_list args) {
    int ret = PP_ERR;
    cfg_entry_t* e;
    pp_dict_t* d;
    key_comp_t* last_comp;
    va_list va;
    vector_t* key_comps;
    u_int key_comps_sz;
    pp_profile_t* p = NULL;
    int i;
    pp_profile_type_stack_t pts = PP_PROFILE_TYPE_STACK_T_INITIALIZER;
    int effective = PP_PROFILE_IS_STACK(prof_type);
    
    assert(pp_cfg_initialized);
    
    va_copy(va, args);
    key_comps = cfg_key_expand(key, va);
    va_end(va);
    key_comps_sz = vector_size(key_comps);

    if(effective) {
        if(PP_ERR == (ret = pp_cfg_get_profile_type_stack_core(&pts, prof_type, 
                                                               key_comps))) {
            goto bail;
        }
        prof_type = pts.profile_type_stack[0];
    }
    
    if(!cfg_profile_is_writable(prof_type)) {
        char cfg_key[MAX_OPT_KEY_LEN];
        pp_log("%s(); attempt to rename '%s' in read only profile %s\n", ___F, 
               pp_cfg_get_key_from_key_comps(key_comps, cfg_key,
                                             MAX_OPT_KEY_LEN),
               pp_profile_type_str[PP_REAL_PROFILE(prof_type)]);
        // TODO? set errno?
	ret = PP_ERR;
        goto bail;
    }
    
    if (PP_ERR == (ret = pp_cfg_get_profile(&p, prof_type, key, args))) {
        goto bail;
    }

    if (NULL != (e = pp_cfg_key_resolve_exact(&d, p, key_comps))) {
	last_comp = (key_comp_t*)vector_get2(key_comps, key_comps_sz - 1);
        /* remove 'key' from dict and save it... */
        pp_dict_remove(d, last_comp->name, (void*)&e);
        /* ... to insert it as 'newkey' again... */
        i = pp_dict_insert(d, newkey, e, (pp_dict_del_func)cfg_entry_destroy, 0);
        assert(i == 0); // entry should be empty
    }

 bail:
    pp_profile_release(p);
    pp_cfg_key_comps_destroy(key_comps);
    return ret;
}

static int
cfg_copy_va(const char* newkey, pp_profile_type_t prof_type,
            const char* key, va_list ap) {
    int ret;
    cfg_entry_t* e;
    pp_dict_t* d;
    vector_t* key_comps = cfg_key_expand(key, ap);
    pp_profile_t* p = NULL;
    pp_profile_type_stack_t pts = PP_PROFILE_TYPE_STACK_T_INITIALIZER;
    int effective = PP_PROFILE_IS_STACK(prof_type);
    int uid = -1;

    if(effective) {
        if(PP_ERR == (ret = pp_cfg_get_profile_type_stack_core(&pts, prof_type,
                                                               key_comps))) {
            goto bail;
        }
        prof_type = pts.profile_type_stack[0];
    }

    if(!cfg_profile_is_writable(prof_type)) {
        char cfg_key[MAX_OPT_KEY_LEN];
        pp_log("%s(); attempt to copy '%s' in read only profile %s\n", ___F,
               pp_cfg_get_key_from_key_comps(key_comps, cfg_key,
                                             MAX_OPT_KEY_LEN),
               pp_profile_type_str[PP_REAL_PROFILE(prof_type)]);
        // TODO? set errno?
        ret = PP_ERR;
        goto bail;
    }
    
    pp_cfg_is_user_key(&uid, key_comps);
    if (PP_ERR == (ret = pp_cfg_get_profile_core(&p, prof_type,
                                                 key_comps, uid))) {
        goto bail;
    }

    if (NULL != (e = pp_cfg_key_resolve_exact(&d, p, key_comps))) {
        ret = cfg_copy_recursive(0, p, d, newkey, e, 1);
    }

 bail:
    pp_profile_release(p);
    pp_cfg_key_comps_destroy(key_comps);
    return ret;
}

/**
 * inserts a copy of element in a given dictionary with new elementname key
 * profile is only used to get allocator
 */
static int 
cfg_copy_recursive(int recursion, const pp_profile_t* prof,  pp_dict_t* dict, 
                   const char* key, const cfg_entry_t* element, int overwrite) {
    int ret = PP_SUC;
    cfg_entry_t *newe, *iter_data;
    cfg_entry_value_t v;
    pp_ditr_t* iter;
    int i;
    const char *iter_key;
    
    assert(recursion++ < CFG_KEY_MAX_COMPS);
    assert(dict);
    assert(key && *key);
    assert(element);
    
    /* check if key exists and overwrite is set */
    if(NULL != pp_dict_search(dict, key) && !overwrite)
        return PP_ERR;
    
    switch (element->type) {
        case CFG_DICT:
            /* create new dict and copy recursive */
            v.dict = pp_dict_str_new_with_alloc(prof->allocator);
            newe = cfg_entry_create(prof->allocator, CFG_DICT, v);
            i = pp_dict_insert(dict, key, newe, 
                               (pp_dict_del_func)cfg_entry_destroy, overwrite);
            assert(i == 0);
    
            iter = pp_ditr_new(element->value.dict);
            while(ret != PP_ERR && pp_ditr_next(iter)) {
                iter_key = pp_ditr_key(iter);
                iter_data = (cfg_entry_t*)pp_ditr_data(iter);
                ret = cfg_copy_recursive(recursion, prof, newe->value.dict, 
                                         iter_key, iter_data, overwrite);
            }
            pp_ditr_destroy(iter);
            break;
        case CFG_STR:
            /* just copy the element and return */
            v.string = pp_strdup(prof->allocator, element->value.string);
            newe = cfg_entry_create(prof->allocator, CFG_STR, v);
            i = pp_dict_insert(dict, key, newe,
                               (pp_dict_del_func)cfg_entry_destroy, overwrite);
            assert(i == 0);
            break;
        default:
            assert(0);
    }
    
    return ret;
}

/*
 * encodes a cfgkey given by stack to compressed cfgkey format
 * cprefix is the number of common prefixes to be skipped
 * this could be done way more elegant, but its faster than string operations
 */
static char*
encode_cfg_key(int cprefix, vector_t* stack, char *cfg_key, size_t cfg_key_len) {
    int idx, stack_sz = vector_size(stack);
    size_t current_len, current_cfg_key_len;
    char *current_key;
    
    for(idx = 0; idx < cprefix; idx++)
        cfg_key[idx] = CFG_COMP_DELI;
    current_cfg_key_len = cprefix;
    for(idx = cprefix; idx < stack_sz; ++idx) {
        current_key = (char*)vector_get(stack, idx);
        assert(current_key && *current_key);
        current_len = strlen(current_key);
        if(current_cfg_key_len + current_len >= cfg_key_len)
            return NULL;
        memcpy(cfg_key + current_cfg_key_len, current_key, current_len);
        cfg_key[current_cfg_key_len + current_len] = CFG_COMP_DELI;
        current_cfg_key_len += current_len + 1;
    }
    cfg_key[current_cfg_key_len - 1] = '\0';

#if defined(PP_CFG_DEBUG) && defined(CFG_DEBUG_FILE_CODING)
    printf("%s: encoded '%s", ___F, (char*)vector_get(stack, 0));
    for(idx = 1; idx < stack_sz; ++idx) {
        printf(".%s", (char*)vector_get(stack, idx));
    }
    printf("' to '%s'\n", cfg_key);
#endif /* PP_CFG_DEBUG && CFG_DEBUG_FILE_CODING */

    return cfg_key;
}

/*
 * expands a cfgkey in compressed cfgkey format to "real" qids
 * skipped prefixes are recovered from stack of previous prefixes
 * this could be done way more elegant, but its faster than string operations
 */
static char*
decode_cfg_key(const char* encoded_key, vector_t* stack, 
               char *cfg_key, size_t cfg_key_len) {
    int cprefix = 0;
    vector_t* key_comps = cfg_key_expand(encoded_key, NULL);
    u_int idx;
    u_int stack_sz = vector_size(stack);
    u_int key_comps_sz = vector_size(key_comps);
    size_t current_len, current_cfg_key_len = 0;
    char *current_key;
    
    while(encoded_key[cprefix] == CFG_COMP_DELI)
        cprefix++;
    // prefix count may not be larger than actual number of prefixes in stack
    assert(cprefix <= (int)stack_sz); 
    for(idx = cprefix; idx < stack_sz; idx++)
        vector_remove(stack, cprefix);
    for(idx = 0; idx < key_comps_sz; idx++)
        vector_add(stack, strdup(((key_comp_t*)vector_get2(key_comps, idx))->name));
    stack_sz = cprefix + key_comps_sz;
    
    pp_cfg_key_comps_destroy(key_comps);
    
    for(idx = 0; idx < stack_sz; idx++) {
        current_key = (char*)vector_get(stack, idx);
        assert(current_key && *current_key);
        current_len = strlen(current_key);
        if(current_cfg_key_len + current_len >= cfg_key_len)
            return NULL;
        memcpy(cfg_key + current_cfg_key_len, current_key, current_len);
        cfg_key[current_cfg_key_len + current_len] = CFG_COMP_DELI;
        current_cfg_key_len += current_len + 1;
    }
    cfg_key[current_cfg_key_len - 1] = '\0';
        
#if defined(PP_CFG_DEBUG) && defined(CFG_DEBUG_FILE_CODING)
    printf("%s: decoded '%s' from %s\n", ___F, cfg_key, encoded_key);
#endif /* PP_CFG_DEBUG && CFG_DEBUG_FILE_CODING */

    return cfg_key;
}

#ifndef PP_BOARD_PEMX
static void
load_profile_option_cb(eric_config_option_t* option, va_list args) {
    pp_profile_t* p = va_arg(args, pp_profile_t*);
    vector_t* stack = va_arg(args, vector_t*);
    char cfg_key[MAX_OPT_KEY_LEN+1];
    char *decoded_cfg_key;
    
    if(NULL != (decoded_cfg_key = decode_cfg_key(option->key, stack, 
                                                 cfg_key, sizeof(cfg_key)))) {
        profile_set_va(option->value, p, 0, decoded_cfg_key, NULL);
    }
}
#endif

static pp_profile_t*
load_profile(pp_profile_type_t type, const char* name) {
    pp_profile_t* p;
#ifndef PP_BOARD_PEMX
    vector_t stack;
#endif /* PP_BOARD_PEMX */
    
    p = pp_profile_create(cfg_allocator, type, name);

#ifndef PP_BOARD_PEMX
    /* for all options */
    vector_new(&stack, 10, free);
    if (0 > eric_config_file_for_each_option(name,
		 (for_each_option_cb_t)load_profile_option_cb, p, &stack)) {
	pp_profile_release(p);
	return NULL;
    }
    vector_delete(&stack);
#endif /* PP_BOARD_PEMX */
    return p;
}

/*
 * this is a recursive function what smells for trouble,
 * however I think our cfg keys are not very deeply nested,
 * so it should be OK for the fkt to call itself, let's say, 10 times
 *
 * compression of cfg keys works as follows:
 * - if ce is a string, push "prefix.key, value" to cfgs vector
 * - if ce is a dictionary, call recursive 
 *   - for first element with prefix = "prefix.key"
 *   - for following elements with prefix = "prefix."
 *
 * eg. user._e_.fru.name=Friedrich Ulrich
 *     user._e_.tbr.name=Thomas Breitfeld
 *     user._e_.tweb.name=Thomas Weber
 * ==> user._e_.fru.name=Friedrich Ulrich
 *     ..tbr.name=Thomas Breitfeld
 *     ..tweb.name=Thomas Weber
 */
static void
save_profile_dict(int recursion, int cprefix, vector_t* cfgs, vector_t* stack,
                  const pp_dict_t* dict) {
    pp_ditr_t* ditr = pp_ditr_new(dict);
    const cfg_entry_t* ce;
    char *key, *encoded_cfg_key;
    char cfg_key[MAX_OPT_KEY_LEN+1];
    
    /* if dict is empty, return */
    if(pp_ditr_first(ditr) == 0 || pp_ditr_valid(ditr) == 0) {
        pp_ditr_destroy(ditr);
        return;
    }
    
    /* we better assert max recursions, you never know */
    ++recursion;
    assert(recursion < CFG_KEY_MAX_COMPS);
    
    for (pp_ditr_first(ditr); pp_ditr_valid(ditr); pp_ditr_next(ditr)) {

        key = pp_ditr_key(ditr);
        assert(key && *key);
        vector_add(stack, key);
#if defined(PP_CFG_DEBUG) && defined(CFG_DEBUG_FILE_CODING)
        printf("%s: recursion %d: key is '%s'\n", ___F, recursion, key);
#endif /* PP_CFG_DEBUG && CFG_DEBUG_FILE_CODING */
	
	ce = pp_ditr_data(ditr);
	switch (ce->type) {
	  case CFG_DICT:
	      save_profile_dict(recursion, cprefix, cfgs, stack, ce->value.dict);
	      break;
	  case CFG_STR:
#ifndef PP_BOARD_PEMX
              if(NULL != 
                 (encoded_cfg_key = encode_cfg_key(cprefix, stack, 
                                                   cfg_key, sizeof(cfg_key)))) {
                  vector_add(cfgs, eric_config_option_create(encoded_cfg_key,
                                                             ce->value.string));
              } else
#endif
              {
                  pp_log("%s(): INTENAL ERROR: could not encode config key\n",
                         ___F);
                  assert(0);
              }
	      break;
	  default:
	      assert(0);
	}
	
        vector_remove(stack, recursion - 1);
        cprefix = recursion - 1;
    }

    pp_ditr_destroy(ditr);
}

#ifndef PP_BOARD_PEMX
static int
save_profile(const pp_profile_t* prof, int lock) {
    vector_t cfgs, stack;
    int ret;

    vector_new(&cfgs, 200, eric_config_option_destroy);
    vector_new(&stack, 10, NULL);
    save_profile_dict(0, 0, &cfgs, &stack, prof->configs);
    vector_add(&cfgs, NULL);

    if (eric_config_file_write_core(prof->name,
			            (eric_config_option_t**)cfgs.elements,
                                    0, lock)) {
	ret = PP_ERR;
	goto bailout;
    }
    ret = PP_SUC;
    
 bailout:
    vector_delete(&cfgs);
    vector_delete(&stack);
    return ret;
}
#endif

static const char*
get_local_profile_name(pp_profile_type_t prof_type) {
    switch (prof_type) {
#if defined(PP_PROFILE_LOCAL_DEF_ENABLED)
      case PP_PROFILE_LOCAL_DEF:
	  return "default";
#endif /* PP_PROFILE_LOCAL_DEF_ENABLED */
      case PP_PROFILE_SYSTEM:
	  return "!system";
      case PP_PROFILE_LOCAL:
	  return "!config";
      default:
	  assert(0);
    }
    return NULL;
}


#ifdef CFG_USE_SHM
/*
 * put the code profile into shared memory
 * TODO:
 *    cd probably goes entirely into shared mem
 */
static int
cfg_shm_idx_map_init_fkt(void* idxmap[], pp_mallocator_t* shmalloc,
			 va_list ap) {
    char* cd_fname = va_arg(ap, char*);
    assert (cfg_cd == NULL);
    
    if (NULL == (cfg_cd = pp_cd_parse_cdl(pp_mallocator_heap(), cd_fname)))
	return PP_ERR;

    idxmap[CFG_SHM_IDX_CODE_PROFILE] = cfg_create_code_profile(shmalloc, cd);

    pp_cd_destroy(cd);
    return PP_SUC;
}
#else
static int
cfg_load_code_profile(const char* cd_fname) {
    assert(cfg_cd == NULL);
    
    if (NULL == (cfg_cd = pp_cd_parse_cdl(pp_mallocator_heap(), cd_fname)))
	return PP_ERR;

    cfg_code_profile = cfg_create_code_profile(pp_mallocator_heap(), cfg_cd);

    pp_cd_destroy_values(cfg_cd);

    return PP_SUC;
}
#endif /* CFG_USE_SHM */

static int
cfg_load_oem_profile(const char* cd_fname) {
    pp_cd_as_cd_t* cd_oem = NULL;
    FILE *f = NULL;
    
    cfg_oem_profile = pp_profile_create(pp_mallocator_heap(), 
                                        PP_PROFILE_LOCAL_OEM, "oem");

    if(cd_fname  && *cd_fname &&
       NULL != (f = fopen(PP_CD_OEM_FNAME_DEFAULT, "r")) &&
       NULL != (cd_oem = pp_cd_parse_cdl(pp_mallocator_heap(), cd_fname))) {
        cfg_oem_profile = cfg_load_profile(cfg_oem_profile, cd_oem);
        pp_cd_destroy(cd_oem);
    } else {
        // only for debugging
        //pp_log("No OEM configuration detected...\n");
    }           
    
    if(f)
        fclose(f);

    return PP_SUC;
}

/*
 * vector size key set callback method,
 * makes sure there are no more elements than the set size
 * i.e. all keys above the size will be deleted
 */
static pp_cfg_chg_kind_t
cfg_vector_set_size(const char* value, pp_profile_t* prof,
		    const vector_t* key_comps) {
    cfg_entry_t* cfge, *cfgpe;
    char* endptr, *key;
    long s, es;
    pp_ditr_t* iter;
    vector_t delvec;
    unsigned int i;
    int last_comp_idx = vector_size(key_comps) - 1;

    // lookup vector and get elements
    if (NULL != (cfgpe = cfg_key_resolve_exact_core(NULL, prof->configs,
					 	    key_comps, 0,
						    last_comp_idx))
	&& cfgpe->type == CFG_DICT
	&& NULL != (cfge = (cfg_entry_t*)pp_dict_search(cfgpe->value.dict,
						pp_cfg_vect_elems_comp))
	&& cfge->type == CFG_DICT) {
	
	s = strtol(value, &endptr, 10);
	assert(endptr != value);

	iter = pp_ditr_new(cfge->value.dict);
	vector_new(&delvec, 0, NULL);
	
	while(pp_ditr_next(iter)) {
	    key = (char*)pp_ditr_key(iter);
	    es = strtol(key, &endptr, 10);
	    if (endptr != key && es >= s)
		vector_add(&delvec, key);
	}
	    
	for (i = 0; i < vector_size(&delvec); ++i)
	    pp_dict_remove(cfge->value.dict, (char*)vector_get(&delvec, i),
			   NULL);
	vector_delete(&delvec);
        
        /* delete dict if empty */
        if(!pp_ditr_first(iter) && last_comp_idx) {
	    pp_dict_remove(cfgpe->value.dict, pp_cfg_vect_elems_comp, NULL);
        }
        pp_ditr_destroy(iter);
        
    }
    return cfg_set_to_prof_core(value, prof, key_comps);
}

/*
 * public methods
 * ============================= */

int pp_cfg_init(const char* cfg_cd_filename, const char* cfg_cd_oem_filename) {
    int ret;
    pthread_t cfg_ext_chg_listener_thrid;

    errno_base = pp_register_errnos(sizeof(pp_cfg_errors) /
                                    sizeof(*pp_cfg_errors), get_error_string);

    //cfg_allocator = pp_bnew(NULL, 64 * 1024, 0);
    cfg_allocator = pp_mallocator_heap();
    assert(prof_cache == NULL);
    prof_cache = pp_profile_cache_create(load_profile, diff_and_update_profile);
    
    cfg_notify_init();
    cfg_handlers_lastcomp = pp_dict_str_new();
    pp_cfg_register_handler(PP_CFG_HANDLER_LASTCOMP, pp_cfg_vect_size_comp,
			    NULL, cfg_vector_set_size);

#ifdef CFG_USE_SHM
    if (NULL == (cfg_shm = pp_shm_init(CFG_SHM_KEY, CFG_SHM_SIZE,
				       CFG_SHM_IDX_MAP_SIZE, 
				       cfg_shm_idx_map_init_fkt,
				       cfg_cd_filename))) {
	ret = PP_ERR;
    }
    ret = PP_SUC;
#else
    ret = cfg_load_code_profile(cfg_cd_filename);

    /**
     * TODO!
     * - check if oem-cdl really exists -> do not parse if not...
     * - modify parser... we already have cfg_cd and only want to parse oem
     *   value section, not types'n'configs...
     */
    ret |= cfg_load_oem_profile(cfg_cd_oem_filename);
#endif /* CFG_USE_SHM */

    ret |= pthread_key_create(&pp_cfg_tx_thread_ctx_key, 
                              pp_cfg_tx_rollback) == 0 ? PP_SUC : PP_ERR;

    /* start external change listener only in eric process! */
    if (pp_i_am_eric) {
	ret |= eric_pthread_create(&cfg_ext_chg_listener_thrid, 1, 32 * 1024,
				   cfg_ext_chg_listener_thr_func, NULL);
    }
    
    pp_cfg_initialized = 1;
    
    return ret;
}

void pp_cfg_cleanup() {
    assert(errno_base != 0);
    assert(prof_cache != NULL);

//    pthread_key_delete(pp_cfg_tx_thread_ctx_key);

    pp_cfg_initialized = 0;
    
#ifndef NDEBUG
    {
        /* check local prof rev_cnt */
        pp_profile_t *local = NULL;
        
        __pp_cfg_get_profile_core(&local, PP_PROFILE_LOCAL, NULL, -1);
        assert(local);
        if(local->ref_cnt != 2) {
            pp_log("ERROR on cleaning up cfg lib: local->ref_cnt = %d (!= 2)\n",
                   local->ref_cnt);
        }
// FIXME: MEMLEAK!!!        assert(local->ref_cnt == 2);
        pp_profile_release(local);
    }
#endif /* !NDEBUG */
    
    pp_cd_destroy(cfg_cd);
    cfg_cd = NULL;

    pp_dict_destroy(cfg_handlers_lastcomp);
    cfg_notify_cleanup();
    
    pp_profile_cache_destroy(prof_cache);

#ifdef PP_B_STATS
    pp_bstats(cfg_allocator, NULL);
#endif

#ifdef CFG_USE_SHM
    pp_shm_cleanup(cfg_shm);
#else
    assert(cfg_code_profile->ref_cnt == 1); // assure we delete the profile
    pp_profile_release(cfg_code_profile);
#endif

    assert(cfg_oem_profile->ref_cnt == 1); // assure we delete the profile
    pp_profile_release(cfg_oem_profile);
    
    pp_mallocator_destroy(cfg_allocator);
    prof_cache = NULL;

    pp_unregister_errnos(errno_base);
    errno_base = 0;
}

int pp_cfg_is_initialized(void)
{
    return pp_cfg_initialized;
}

/*
 * deep clean the shared memory, should be called without pp_cfg_init first
 */
void pp_cfg_destroy() {
#ifdef CFG_USE_SHM
    pp_shm_destroy(CFG_SHM_KEY);
#endif
}

/**
 * returns the active configuration description for
 * the initialiced cfg lib
 */
pp_cd_as_cd_t* pp_cfg_get_cd() {
    return cfg_cd;
}

/*
static int
cfg_get_profile_type_stack_va(pp_profile_type_stack_t *profile_type_stack, 
                              pp_profile_type_t prof_type,
                              const char* key, va_list args) {
    vector_t* key_comps = cfg_key_expand(key, args);
    int ret = pp_cfg_get_profile_type_stack_core(profile_type_stack, 
                                                 prof_type, key_comps);
    pp_cfg_key_comps_destroy(key_comps);
    return ret;
}
*/

int
pp_cfg_get_profile_stack(vector_t** prof_stack, pp_profile_type_t prof_type,
			 const char* key, ...) {
    int ret;
    va_list ap;
    va_start(ap, key);
    ret = cfg_get_profile_stack_va(prof_stack, prof_type, key, ap);
    va_end(ap);
    return ret;
}

static int
cfg_get_profile_stack_va(vector_t** prof_stack, pp_profile_type_t prof_type,
		         const char* key, va_list args) {
    vector_t* key_comps = cfg_key_expand(key, args);
    int ret = pp_cfg_get_profile_stack_core(prof_stack, prof_type, key_comps);
    pp_cfg_key_comps_destroy(key_comps);
    return ret;
}

int
pp_cfg_get_profile(pp_profile_t** prof, pp_profile_type_t prof_type,
		   const char* key, va_list args) {
    int ret = PP_ERR;
    vector_t* key_comps = NULL;
    int uid = -1;

    assert(!PP_PROFILE_IS_STACK(prof_type));

    /* for local profiles, we do not need to specify a key as they are all contained in
       the same profile, so key / args may be NULL in some cases */
    if(args) {
        key_comps = cfg_key_expand(key, args);
        pp_cfg_is_user_key(&uid, key_comps);
    }
    
    ret = pp_cfg_get_profile_core(prof, prof_type, key_comps, uid);

    if(key_comps) {
        pp_cfg_key_comps_destroy(key_comps);
    }

    return ret;
}

int
__pp_cfg_get_profile_type_stack_core(pp_profile_type_stack_t *prof_type_stack, 
                                     pp_profile_type_t prof_type,
                                     const vector_t* key_comps UNUSED) {
    pp_profile_type_t pts[(int)PP_PROFILE_LOCAL_HEAD + 1];
    pp_profile_type_t real_type = PP_REAL_PROFILE(prof_type);
    int effective = PP_PROFILE_IS_STACK(prof_type);
    int pnr, ret = PP_SUC;
    
    assert(real_type != PP_PROFILE_LDAP_USER);
    assert(real_type != PP_PROFILE_LDAP_DEV);
    
    if (real_type == PP_PROFILE_HEAD) {
        real_type = PP_PROFILE_LDAP_DEV - 1; // 1st profile below LDAP profiles
    }

    if (effective) {
        pp_profile_type_t ptype;

        for(ptype = real_type, pnr = 0; ptype >= PP_PROFILE_NONE; 
            --ptype, ++pnr) {
            pts[pnr] = ptype;
        }
    } else {
	pts[0] = real_type;
	pts[1] = PP_PROFILE_NONE;
        pnr = 2;
    }
    
//    if(ret != PP_ERR)
    {
        int pt_size = sizeof(pp_profile_type_t);
        // copy local profile type stack
        memcpy(prof_type_stack->profile_type_stack + 
               pt_size * prof_type_stack->size,
               pts, pt_size * pnr);
        prof_type_stack->size += pnr - 1;
    }

    return ret;
}

/*
 * this is the function creating the layered profile,
 * thus defining a certain priority. Highest priority comes
 * first (lowest index) in the vector
 * return value is PP_SUC, if at least one profile (!= NULL) is returned, 
 *                 PP_ERR otherwhise
 */
int
__pp_cfg_get_profile_stack_core(vector_t** prof_stack, 
                                pp_profile_type_t prof_type,
                                const vector_t* key_comps) {
    vector_t* ps;
    pp_profile_t* p;
    int ret = PP_ERR, i;
    pp_profile_type_stack_t pts = PP_PROFILE_TYPE_STACK_T_INITIALIZER;

    if(PP_ERR == __pp_cfg_get_profile_type_stack_core(&pts, prof_type, 
                                                      key_comps)) {
        return PP_ERR;
    }

    ps = vector_new(NULL, PP_PROFILE_COUNT, pp_profile_release);

    for (i = 0; pts.profile_type_stack[i] != PP_PROFILE_NONE; ++i) {
        if (PP_ERR != (ret = __pp_cfg_get_profile_core(&p,
                                                       pts.profile_type_stack[i],
                                                       key_comps, -1))) {
            assert(p);
            vector_add(ps, p);
        } else {
            vector_delete(ps);
            return ret;
        }
    }

    if(*prof_stack) {
        /* if prof_stack is already initialized, merge vectors */
        vector_addvec(*prof_stack, ps, NULL);
        vector_delete(ps);
    } else {
        /* else use the vector just created */
        *prof_stack = ps;
    }
    
    return ret;
}
    
int
__pp_cfg_get_profile_core(pp_profile_t** prof, pp_profile_type_t prof_type,
		          const vector_t* key_comps UNUSED, int uid UNUSED) {
    int ret = PP_SUC;
    pp_profile_type_t real_type = PP_REAL_PROFILE(prof_type);
    assert(!PP_PROFILE_IS_STACK(prof_type));
    switch (real_type) {
      case PP_PROFILE_LOCAL_CODE:
#ifdef CFG_USE_SHM
	  *prof = (pp_profile_t*)pp_shm_get_obj(cfg_shm,
						CFG_SHM_IDX_CODE_PROFILE);
#else
	  *prof = cfg_code_profile;
#endif
	  assert(*prof);
	  // just duplicate profile, since calling fkts will call release,
	  // no matter were the profiles actually comes from,
	  // however the Code-Profile will actually never be destroyed in shm
	  pp_profile_duplicate(*prof);
	  break;
      case PP_PROFILE_LOCAL_OEM:
          if(cfg_oem_profile != NULL) {
              *prof = cfg_oem_profile;
              assert(*prof);
              pp_profile_duplicate(*prof);
          } else {
              ret = PP_ERR;
          }
          break;
      case PP_PROFILE_SYSTEM:
      case PP_PROFILE_LOCAL:
	  if (!(*prof = pp_profile_cache_load(prof_cache, real_type,
				      get_local_profile_name(real_type))))
	      ret = PP_ERR;
	  break;
      default:
          pp_log("%s(): unsupported profile type %s requested\n", 
                 ___F, pp_profile_type_str[PP_REAL_PROFILE(prof_type)]);
	  assert(0);
	  break;
    }
    return ret;
}

/**
 * checks wether a given key components vector represents a indexed key with
 * base name vecname, else returns PP_ERR
 * the index id is set if specified
 */
static int cfg_is_indexed_key(int *idx, const vector_t *key_comps,
                       const char* vecname) {
    int ret = PP_ERR;
    char *key_comp_0, *key_comp_1, *key_comp_2;
    
    assert(vecname);
    
    /* an indexed key has at least 3 components: vecname._e_.id */
    if(!key_comps || vector_size(key_comps) < 3)
        goto bail;
        
    key_comp_0 = ((key_comp_t*)vector_get2(key_comps, 0))->name;
    key_comp_1 = ((key_comp_t*)vector_get2(key_comps, 1))->name;
    key_comp_2 = ((key_comp_t*)vector_get2(key_comps, 2))->name;
   
    if(!strcmp(vecname, key_comp_0) && 
       !strcmp(pp_cfg_vect_elems_comp, key_comp_1)) {
	int id = pp_strtol(key_comp_2, PP_ERR, 10, NULL);
        if(id == PP_ERR) {
            /* 3rd key comp has to be idx string. if not, return error */
            ret = PP_ERR;
        } else {
            if(idx != NULL) {
                *idx = id;
            }
            ret = PP_SUC;
        }
    }

 bail:
    return ret;
}

/**
 * checks wether a given key components vector represents a user key,
 * else returns PP_ERR
 * the user id is set if specified
 */
int pp_cfg_is_user_key(int *uid, const vector_t *key_comps) {
    return cfg_is_indexed_key(uid, key_comps, "user");
}

/**
 * checks wether a given key components vector represents a group key,
 * else returns PP_ERR
 * the group id is set if specified
 */
int pp_cfg_is_group_key(int *gid, const vector_t *key_comps) {
    return cfg_is_indexed_key(gid, key_comps, "group");
}

/*
 * flush changes back to persitent storage
 * pp_cfg_flush_t desides whehter it is just saved or also flashed
 * This function shouldn't be called in case no set operation was
 * called before
 */
int pp_cfg_save(pp_cfg_flush_t flush) {
    return pp_cfg_save_layer(PP_PROFILE_LOCAL, flush);
}

int pp_cfg_save_layer(pp_profile_type_t type, pp_cfg_flush_t flush) {
    pp_profile_t* p;
    int ret;

    if (PP_ERR == (ret = pp_cfg_get_profile(&p, type, NULL, NULL)))
	return ret;

    ret = cfg_save_profile(p, flush, 1);
    pp_profile_release(p);
    
    return ret;
}

int cfg_save_profile(pp_profile_t* p, pp_cfg_flush_t flush, int lock) {
    int ret = PP_ERR;
    
#ifndef PP_BOARD_PEMX
    if (PP_ERR != (ret = save_profile(p, lock))) {
	if (flush == DO_FLUSH) {
	    eric_config_flush();
            if(p->type == PP_PROFILE_LOCAL ||
               p->type == PP_PROFILE_SYSTEM) {
                profile_cache_update_timestamp(prof_cache, p->type,
                                               get_local_profile_name(p->type));
            }
        }
    }
#endif

    return ret;
}

/**
 * Get the string value of a config key
 *	NOTE: The get*_nodflt routines do not touch value in case of an error.
 *	      The other get routines always set value - even in case of an error.
 * @param value parameter for value, if *value == NULL, key was not found
 * @param prof  profile source, might be NULL if not requested
 * @param key   the config path to a specific property, inclusive
 *              format flags for the vector index
 *              %d - int
 *              %s - string
 * @arglist     paramters for format flags
 * @return      PP_ERR for error or
 *              PP_SUC for success
 * @example     if (err = pp_get_config(*buttonkey,
 *                                    "User[%s].KVMPort[%u].ButtonKey",
 *                                    "ronald", 4)) {
 *                  pp_log("pp_get_config failed: %s", pp_error_string(err));
 *                  return PP_SUC;
 *              }
 */
int
pp_cfg_get_with_prof(char** value, int* prof, const char* key, ...)
{
    int ret;
    va_list ap;
    va_start(ap, key);
    ret = cfg_get_va(1, value, PP_PROFILE_EFFECTIVE, key, ap);
    va_end(ap);
    if (prof) *prof = (ret != PP_ERR) ? ret : -1;
    return (ret != PP_ERR) ? PP_SUC : PP_ERR;
}

int
pp_cfg_get_with_prof_nodflt(char** value, int* prof, const char* key, ...)
{
    int ret;
    va_list ap;
    va_start(ap, key);
    ret = cfg_get_va(0, value, PP_PROFILE_EFFECTIVE, key, ap);
    va_end(ap);
    if (prof) *prof = (ret != PP_ERR) ? ret : -1;
    return (ret != PP_ERR) ? PP_SUC : PP_ERR;
}

int
pp_cfg_get_nodflt(char** value, const char* key, ...)
{
    int ret;
    va_list ap;
    va_start(ap, key);
    ret = cfg_get_va(0, value, PP_PROFILE_EFFECTIVE, key, ap);
    va_end(ap);
    return (ret != PP_ERR) ? PP_SUC : PP_ERR;
}

int
pp_cfg_get(char** value, const char* key, ...)
{
    int ret;
    va_list ap;
    va_start(ap, key);
    ret = cfg_get_va(1, value, PP_PROFILE_EFFECTIVE, key, ap);
    va_end(ap);
    return (ret != PP_ERR) ? PP_SUC : PP_ERR;
}

#ifdef PP_FEAT_RPCCFG
int
pp_cfg_get_with_serialized_key(int force, char** value, pp_profile_type_t prof_type,
			       const void * keybuf, size_t keybuf_size)
{
    vector_t * key_comps = deserialize_key_comps(keybuf, keybuf_size);
    int ret = PP_ERR;
    if (key_comps != NULL) {
	ret = cfg_get_key_comps(force, value, prof_type, key_comps);
	pp_cfg_key_comps_destroy(key_comps);
    } else {
	if (force) *value = CFG_FORCE_DEFAULT_RETVAL;
    }
    return (ret != PP_ERR) ? PP_SUC : PP_ERR;
}
#endif /* PP_FEAT_RPCCFG */

/**
 * Set the string value of a config key.
 * Works on PP_PROFILE_LOCAL.
 * @see pp_cfg_get
 */
int
pp_cfg_set(const char* value, const char* key, ...)
{
    int ret;
    va_list ap;
    va_start(ap, key);
    ret = cfg_set_va(value, PP_PROFILE_EFFECTIVE, key, ap);
    va_end(ap);
    return ret;
}

#ifdef PP_FEAT_RPCCFG
int
pp_cfg_set_with_serialized_key(const char* value, pp_profile_type_t prof_type,
			       const void * keybuf, size_t keybuf_size)
{
    vector_t * key_comps = deserialize_key_comps(keybuf, keybuf_size);
    int ret = PP_ERR;
    if (key_comps != NULL) {
	ret = cfg_set_key_comps(value, prof_type, key_comps);
	pp_cfg_key_comps_destroy(key_comps);
    }
    return ret;
}
#endif /* PP_FEAT_RPCCFG */

/**
 * Get the integer value of a config key
 * @see pp_cfg_get
 */
int
pp_cfg_get_int_nodflt(int* number, const char* key, ...)
{
    int ret;
    va_list ap;
    va_start(ap, key);
    ret = cfg_get_int_va(0, number, PP_PROFILE_EFFECTIVE, key, ap);
    va_end(ap);
    return (ret != PP_ERR) ? PP_SUC : PP_ERR;
}

int
pp_cfg_get_int(int* number, const char* key, ...)
{
    int ret;
    va_list ap;
    va_start(ap, key);
    ret = cfg_get_int_va(1, number, PP_PROFILE_EFFECTIVE, key, ap);
    va_end(ap);
    return (ret != PP_ERR) ? PP_SUC : PP_ERR;
}

/**
 * Set the integer value of a config key
 * Works on PP_PROFILE_LOCAL.
 * @see pp_cfg_get
 */
int
pp_cfg_set_int(const int number, const char* key, ...)
{
    int ret;
    va_list ap;
    va_start(ap, key);
    ret = cfg_set_int_va(number, PP_PROFILE_EFFECTIVE, key, ap);
    va_end(ap);
    return ret;
}

/**
 * Get the unsigned integer value of a config key
 * @see pp_cfg_get
 */
int
pp_cfg_get_uint_nodflt(u_int* number, const char* key, ...)
{
    int ret;
    va_list ap;
    va_start(ap, key);
    ret = cfg_get_uint_va(0, number, PP_PROFILE_EFFECTIVE, key, ap);
    va_end(ap);
    return (ret != PP_ERR) ? PP_SUC : PP_ERR;
}

int
pp_cfg_get_uint(u_int* number, const char* key, ...)
{
    int ret;
    va_list ap;
    va_start(ap, key);
    ret = cfg_get_uint_va(1, number, PP_PROFILE_EFFECTIVE, key, ap);
    va_end(ap);
    return (ret != PP_ERR) ? PP_SUC : PP_ERR;
}

/**
 * Set the unsigned integer value of a config key
 * Works on PP_PROFILE_LOCAL.
 * @see pp_cfg_get
 */
int
pp_cfg_set_uint(const u_int number, const char* key, ...)
{
    int ret;
    va_list ap;
    va_start(ap, key);
    ret = cfg_set_uint_va(number, PP_PROFILE_EFFECTIVE, key, ap);
    va_end(ap);
    return ret;
}

/**
 * Get the short value of a config key
 * @see pp_cfg_get
 */
int
pp_cfg_get_short_nodflt(short* number, const char* key, ...)
{
    int ret, val;
    va_list ap;
    va_start(ap, key);
    ret = cfg_get_int_va(0, &val, PP_PROFILE_EFFECTIVE, key, ap);
    va_end(ap);
    if (ret != PP_ERR && val >= SHRT_MIN && val <= SHRT_MAX) {
	*number = (short)val;
    } else {
	ret = PP_ERR;
    }
    return (ret != PP_ERR) ? PP_SUC : PP_ERR;
}

int
pp_cfg_get_short(short* number, const char* key, ...)
{
    int ret, val;
    va_list ap;
    va_start(ap, key);
    ret = cfg_get_int_va(1, &val, PP_PROFILE_EFFECTIVE, key, ap);
    va_end(ap);
    if (ret != PP_ERR && val >= SHRT_MIN && val <= SHRT_MAX) {
	*number = (short)val;
    } else {
	*number = 0;
	ret = PP_ERR;
    }
    return (ret != PP_ERR) ? PP_SUC : PP_ERR;
}

/**
 * Set the short value of a config key
 * Works on PP_PROFILE_LOCAL.
 * @see pp_cfg_get
 */
int
pp_cfg_set_short(const short number, const char* key, ...)
{
    int ret;
    va_list ap;
    va_start(ap, key);
    ret = cfg_set_int_va(number, PP_PROFILE_EFFECTIVE, key, ap);
    va_end(ap);
    return ret;
}

/**
 * Get the short value of a config key
 * @see pp_cfg_get
 */
int
pp_cfg_get_ushort_nodflt(u_short* number, const char* key, ...)
{
    int ret;
    u_int val;
    va_list ap;
    va_start(ap, key);
    ret = cfg_get_uint_va(0, &val, PP_PROFILE_EFFECTIVE, key, ap);
    va_end(ap);
    if (ret != PP_ERR && val <= USHRT_MAX) {
	*number = (u_short)val;
    } else {
	ret = PP_ERR;
    }
    return (ret != PP_ERR) ? PP_SUC : PP_ERR;
}

int
pp_cfg_get_ushort(u_short* number, const char* key, ...)
{
    int ret;
    u_int val;
    va_list ap;
    va_start(ap, key);
    ret = cfg_get_uint_va(1, &val, PP_PROFILE_EFFECTIVE, key, ap);
    va_end(ap);
    if (ret != PP_ERR && val <= USHRT_MAX) {
	*number = (u_short)val;
    } else {
	*number = 0;
	ret = PP_ERR;
    }
    return (ret != PP_ERR) ? PP_SUC : PP_ERR;
}

/**
 * Set the short value of a config key
 * Works on PP_PROFILE_LOCAL.
 * @see pp_cfg_get
 */
int
pp_cfg_set_ushort(const u_short number, const char* key, ...)
{
    int ret;
    va_list ap;
    va_start(ap, key);
    ret = cfg_set_uint_va(number, PP_PROFILE_EFFECTIVE, key, ap);
    va_end(ap);
    return ret;
}

/**
 * Get the float value of a config key
 * @see pp_cfg_get
 */
int
pp_cfg_get_float_nodflt(float* number, const char* key, ...)
{
    int ret;
    va_list ap;
    va_start(ap, key);
    ret = cfg_get_float_va(0, number, PP_PROFILE_EFFECTIVE, key, ap);
    va_end(ap);
    return (ret != PP_ERR) ? PP_SUC : PP_ERR;
}

int
pp_cfg_get_float(float* number, const char* key, ...)
{
    int ret;
    va_list ap;
    va_start(ap, key);
    ret = cfg_get_float_va(1, number, PP_PROFILE_EFFECTIVE, key, ap);
    va_end(ap);
    return (ret != PP_ERR) ? PP_SUC : PP_ERR;
}

/**
 * Set the float value of a config key
 * Works on PP_PROFILE_LOCAL.
 * @see pp_cfg_get
 */
int
pp_cfg_set_float(const float number, const char* key, ...)
{
    int ret;
    va_list ap;
    va_start(ap, key);
    ret = cfg_set_float_va(number, PP_PROFILE_EFFECTIVE, key, ap);
    va_end(ap);
    return ret;
}

/**
 * Get the boolean value of a config key
 * @param is_enabled will be set to true, if the property value
 *        is "yes", otherwise, even if key can't be found, it will
 *        be set to false
 * @see pp_cfg_get
 */
int
pp_cfg_is_enabled_nodflt(int* is_enabled, const char* key, ...)
{
    int ret;
    va_list ap;
    va_start(ap, key);
    ret = cfg_is_enabled_va(0, is_enabled, PP_PROFILE_EFFECTIVE, key, ap);
    va_end(ap);
    return (ret != PP_ERR) ? PP_SUC : PP_ERR;
}

int
pp_cfg_is_enabled(int* is_enabled, const char* key, ...)
{
    int ret;
    va_list ap;
    va_start(ap, key);
    ret = cfg_is_enabled_va(1, is_enabled, PP_PROFILE_EFFECTIVE, key, ap);
    va_end(ap);
    return (ret != PP_ERR) ? PP_SUC : PP_ERR;
}

/**
 * Set the boolean value of a config key
 * Works on PP_PROFILE_LOCAL.
 * @see pp_cfg_is_enabled
 */
int
pp_cfg_set_enabled(int is_enabled, const char* key, ...)
{
    int ret;
    va_list ap;
    va_start(ap, key);
    ret = cfg_set_enabled_va(is_enabled, PP_PROFILE_EFFECTIVE, key, ap);
    va_end(ap);
    return ret;
}

int
pp_cfg_get_binary(char **value, size_t *size, const char *key, ...)
{
    int ret;
    char *temp;
    va_list ap;
    va_start(ap,key);
    ret = cfg_get_va(0, &temp, PP_PROFILE_EFFECTIVE, key, ap);
    va_end(ap);
    if (ret != PP_ERR) {
	*value = (char*)malloc((strlen(temp) + 3) / 4 * 3);
	ret = pp_base64_dec(temp, strlen(temp), *value, size);
	if (ret == PP_ERR) free(*value);
	free(temp);
    }
    if (ret == PP_ERR) {
	*value = NULL;
	*size = 0;
    }
    return ret;
}

int
pp_cfg_set_binary(const char *value, size_t size, const char *key, ...)
{
    int ret;
    size_t b64_size = pp_base64_size(size);
    char *temp = (char*)malloc(b64_size + 1);
    va_list ap;
    pp_base64_enc(value, size, temp);
    temp[b64_size] = '\0';
    va_start(ap,key);
    ret = cfg_set_va(temp, PP_PROFILE_EFFECTIVE, key, ap);
    va_end(ap);
    return ret;
}

/**
 * Get the acl of a config key
 * @see pp_cfg_get
 */
int
pp_cfg_get_acl(pp_acl_t **acl, const char* key, ...)
{
    int ret;
    va_list ap;
    va_start(ap, key);
    ret = cfg_get_acl_va(acl, PP_PROFILE_EFFECTIVE, key, ap);
    va_end(ap);
    return (ret != PP_ERR) ? PP_SUC : PP_ERR;
}

/**
 * Check whether the effective config value equals the default
 * @see pp_cfg_get
 */
int pp_cfg_is_default(const char* key, ...)
{
    int ret = 1;
    int prof;
    char *v_effective = NULL;
    va_list ap, aq;
    va_start(ap, key);
    va_copy(aq, ap);

    prof = cfg_get_va(0, &v_effective, PP_PROFILE_EFFECTIVE, key, ap);
    if (prof >= PP_PROFILE_LOCAL) {
	char *v_default = NULL;
	cfg_get_va(0, &v_default, PP_PROFILE_DEFAULTS, key, aq);
	if (pp_strcmp_safe(v_effective, v_default)) {
	    ret = 0;
	}
	free(v_default);
    }

    free(v_effective);
    va_end(aq);
    va_end(ap);
    return ret;
}

/**
 * Removes a config key from the configuration
 * Attention: this function can be used to delete whole sub-trees,
 * not just leaf elements, so be careful!
 * @see pp_cfg_get
 */
int
pp_cfg_remove(const char* key, ...)
{
    int ret;
    va_list ap;
    va_start(ap, key);
    ret = cfg_remove_va(PP_PROFILE_EFFECTIVE, key, ap);
    va_end(ap);
    return ret;
}

/**
 * renames the last name component of a config key
 * (could be extended to rename one path to another, however
 * currently not needed, maybe later)
 */
int
pp_cfg_rename(const char* newkey, const char* key, ...)
{
    int ret;
    va_list ap;
    va_start(ap, key);
    ret = cfg_rename_va(newkey, PP_PROFILE_EFFECTIVE, key, ap);
    va_end(ap);
    return ret;
}

/**
 * copies the last name component of a config key
 * refer to pp_cfg_rename
 */
int
pp_cfg_copy(const char* newkey, const char* key, ...)
{
    int ret;
    va_list ap;
    va_start(ap, key);
    ret = cfg_copy_va(newkey, PP_PROFILE_EFFECTIVE, key, ap);
    va_end(ap);
    return ret;
}

int
pp_cfg_copy_at_layer(pp_profile_type_t type, const char* newkey,
                     const char* key, ...)
{
    int ret;
    va_list ap;
    va_start(ap, key);
    ret = cfg_copy_va(newkey, type, key, ap);
    va_end(ap);
    return ret;
}

/**
 * copies cfg_entry to given dict
 */
int
pp_cfg_copy_entry(const pp_profile_t* prof, pp_dict_t *d, const char *key,
		  const cfg_entry_t *e, int overwrite)
{
    return cfg_copy_recursive(0, prof, d, key, e, overwrite);
}

/**
 * Compacts a vector in the sense, that the current size
 * of the vector will be shrinked to the smallest possible size
 * still holding all existing elements, i.e. not existing indexes
 * will be removed from the vector. The length member of the vector
 * will be set accordingly
 */
//int pp_cfg_compact_vector(const char* vectorkey, ...);

/*
 * layered profile API
 * these functions are identical to the ones above, accept that they
 * request the Profile Manager to operate on specifically named
 * and predefined profile
 */

int
pp_cfg_get_at_layer_with_prof_nodflt(pp_profile_type_t type, char** value,
			             int* prof, const char* key, ...)
{
    int ret;
    va_list ap;
    va_start(ap, key);
    ret = cfg_get_va(0, value, type, key, ap);
    va_end(ap);
    if (prof) *prof = (ret != PP_ERR) ? ret : -1;
    return (ret != PP_ERR) ? PP_SUC : PP_ERR;
}

int
pp_cfg_get_at_layer_nodflt(pp_profile_type_t type, char** value,
			   const char* key, ...)
{
    int ret;
    va_list ap;
    va_start(ap, key);
    ret = cfg_get_va(0, value, type, key, ap);
    va_end(ap);
    return (ret != PP_ERR) ? PP_SUC : PP_ERR;
}

int
pp_cfg_get_at_layer(pp_profile_type_t type, char** value,
		    const char* key, ...)
{
    int ret;
    va_list ap;
    va_start(ap, key);
    ret = cfg_get_va(1, value, type, key, ap);
    va_end(ap);
    return (ret != PP_ERR) ? PP_SUC : PP_ERR;
}

int
pp_cfg_set_at_layer(pp_profile_type_t type, const char* value,
		    const char* key, ...)
{
    int ret;
    va_list ap;
    va_start(ap, key);
    ret = cfg_set_va(value, type, key, ap);
    va_end(ap);
    return ret;
}

int
pp_cfg_get_int_at_layer_nodflt(pp_profile_type_t type, int* number,
			       const char* key, ...)
{
    int ret;
    va_list ap;
    va_start(ap, key);
    ret = cfg_get_int_va(0, number, type, key, ap);
    va_end(ap);
    return (ret != PP_ERR) ? PP_SUC : PP_ERR;
}

int
pp_cfg_get_int_at_layer(pp_profile_type_t type, int* number,
			const char* key, ...)
{
    int ret;
    va_list ap;
    va_start(ap, key);
    ret = cfg_get_int_va(1, number, type, key, ap);
    va_end(ap);
    return (ret != PP_ERR) ? PP_SUC : PP_ERR;
}


int
pp_cfg_set_int_at_layer(pp_profile_type_t type, int number,
			const char* key, ...)
{
    int ret;
    va_list ap;
    va_start(ap, key);
    ret = cfg_set_int_va(number, type, key, ap);
    va_end(ap);
    return ret;
}

int
pp_cfg_is_enabled_at_layer_nodflt(pp_profile_type_t type, int* is_enabled,
				  const char* key, ...)
{
    int ret;
    va_list ap;
    va_start(ap, key);
    ret = cfg_is_enabled_va(0, is_enabled, type, key, ap);
    va_end(ap);
    return (ret != PP_ERR) ? PP_SUC : PP_ERR;
}

int
pp_cfg_is_enabled_at_layer(pp_profile_type_t type, int* is_enabled,
			   const char* key, ...)
{
    int ret;
    va_list ap;
    va_start(ap, key);
    ret = cfg_is_enabled_va(1, is_enabled, type, key, ap);
    va_end(ap);
    return (ret != PP_ERR) ? PP_SUC : PP_ERR;
}
    
int
pp_cfg_set_enabled_at_layer(pp_profile_type_t type, int is_enabled,
			    const char* key, ...)
{
    int ret;
    va_list ap;
    va_start(ap, key);
    ret = cfg_set_enabled_va(is_enabled, type, key, ap);
    va_end(ap);
    return ret;
}

int
pp_cfg_remove_at_layer(pp_profile_type_t type, const char* key, ...)
{
    int ret;
    va_list ap;
    va_start(ap, key);
    ret = cfg_remove_va(type, key, ap);
    va_end(ap);
    return ret;
}

/*
 * Iterator API
 * =======================================================
 */
struct pp_cfg_iter_s {
    pp_cfg_iter_t*    parent; // parent iterator
    pp_profile_type_t ptype;  // profile type, we operate on
    pp_profile_t*     prof;   // the profile, just to access mutex
    pp_ditr_t*        ditr;   // the dictionary iterator
    cfg_entry_t*      cfge;   // the current config entry
    cfg_iter_type_t   type;   // the iterator type
};

int pp_cfg_iter_create(pp_cfg_iter_t** iter, const char* key, ...) {
    int ret;
    va_list ap;
    va_start(ap, key);
    ret = cfg_iter_create_va(iter, NULL, PP_PROFILE_EFFECTIVE, FLAT, key, ap);
    va_end(ap);
    return ret;
}

int pp_cfg_iter_create_x_dict(pp_cfg_iter_t** iter, const char* key, ...) {
    int ret;
    va_list ap;
    va_start(ap, key);
    ret = cfg_iter_create_va(iter, NULL, PP_PROFILE_EFFECTIVE, X_DICT, key, ap);
    va_end(ap);
    return ret;
}

int pp_cfg_iter_create_at_layer(pp_cfg_iter_t** iter, pp_profile_type_t type,
				const char* key, ...) {
    int ret;
    va_list ap;
    va_start(ap, key);
    ret = cfg_iter_create_va(iter, NULL, type, FLAT, key, ap);
    va_end(ap);
    return ret;
}

int pp_cfg_iter_create2(pp_cfg_iter_t** iter, pp_cfg_iter_t* parent,
			const char* key, ...) {
    int ret;
    va_list ap;
    va_start(ap, key);
    ret = cfg_iter_create_va(iter, parent, parent->ptype, FLAT, key, ap);
    va_end(ap);
    return ret;
}

int pp_cfg_iter_get_prof_id(const char* key, ...) {
    int ret;
    pp_cfg_iter_t* iter;
    va_list ap;
    va_start(ap, key);
    ret = cfg_iter_create_va(&iter, NULL, PP_PROFILE_EFFECTIVE, FLAT, key, ap);
    pp_cfg_iter_destroy(iter);
    va_end(ap);
    return ret;
}

static int cfg_iter_create_va(pp_cfg_iter_t** iter, pp_cfg_iter_t* parent,
				 pp_profile_type_t type, cfg_iter_type_t itype, 
                                 const char* key, va_list ap) {
    int ret = PP_ERR;
    pp_profile_t* p;
    vector_t* key_comps;
    vector_t *ps = NULL;
    int uid = -1;
    int key_comps_sz;

    key_comps = cfg_key_expand(key, ap);
    key_comps_sz = vector_size(key_comps);
    
    if (parent == NULL) {
        if(type != PP_PROFILE_EFFECTIVE) {
            pp_cfg_is_user_key(&uid, key_comps);
            if(PP_ERR == (ret = pp_cfg_get_profile_core(&p, type, key_comps, uid)))
                goto bail;
            if(PP_ERR == (ret = cfg_iter_create(iter, parent, type, itype,
                                                key_comps, key_comps_sz, p, p->configs)))
                pp_profile_release(p);
        } else {
            if(PP_ERR == pp_cfg_get_profile_stack_core(&ps, type, key_comps))
                goto bail;
            ret = cfg_iter_create_from_prof_stack(iter, type, itype,
                                                  key_comps, key_comps_sz, ps);
            vector_delete(ps);
        }
    } else {
	if (parent->cfge == NULL || parent->cfge->type != CFG_DICT) {
            ret = PP_ERR;
	    goto bail;
        }
	p = pp_profile_duplicate(parent->prof);
        ret = cfg_iter_create(iter, parent, type, itype, key_comps, key_comps_sz,
                              p, parent->cfge->value.dict);
    }

 bail:
    pp_cfg_key_comps_destroy(key_comps);
    return ret;
}
                              
static int 
cfg_iter_create_from_prof_stack(pp_cfg_iter_t** iter,
                                pp_profile_type_t type, cfg_iter_type_t itype, 
                                const vector_t* key_comps, u_int key_comps_sz,
                                vector_t *ps) {
    int i, ps_sz = vector_size(ps);
    pp_profile_t *prof;
    for (i = 0; i < ps_sz; ++i) {
	prof = vector_get(ps, i);
        if(PP_ERR != cfg_iter_create(iter, NULL, type, itype, key_comps, key_comps_sz,
                                     prof, prof->configs)) { /* we've found it */
            /* remove entry i from ps, so that profile is not deleted with vector */
            vector_remove_dont_delete(ps, i);
	    return PP_SUC;
	}
    }
    /* we havn't found it */
    return PP_ERR;
}

/**
 * returns PP_ERR in case of an error, the id of the profile, 
 * the iteratior is generated for otherwhise
 * WARNING: do not test for (PP_SUC == cfg_iter_create) as it will fail!!!
 */
static int cfg_iter_create(pp_cfg_iter_t** iter, pp_cfg_iter_t* parent,
			   pp_profile_type_t type, cfg_iter_type_t itype, 
                           const vector_t* key_comps, u_int key_comps_sz,
                           pp_profile_t *p, pp_dict_t *dict) {
    int comp_idx = 0, ret = PP_ERR;
    cfg_entry_t *e = NULL;
    
    if(parent == NULL) {
        pp_mutex_lock(&p->mtx);
        /* check if we want to create an iterator within an LDAP_USER prof and
           strip off first three key comps (user._e_.tweb) as they won't exit */
        if(p->type == PP_PROFILE_LDAP_USER)
            comp_idx = 3;
    }
    if((!key_comps_sz && !parent) || /* if no key_comps are given, iterate over dict */
       (NULL != (e = cfg_key_resolve_exact_core(NULL, dict, key_comps, 
                                                comp_idx, key_comps_sz))
	&& e->type == CFG_DICT)) {
	*iter = malloc(sizeof(pp_cfg_iter_t));
	(*iter)->parent = parent;
	(*iter)->ptype = type;
	(*iter)->prof = p;
	(*iter)->ditr = pp_ditr_new(!key_comps_sz ? dict : e->value.dict);
	(*iter)->cfge = NULL;
	(*iter)->type = itype;
        ret = p->id;
    } else {
	if (parent == NULL) pp_mutex_unlock(&p->mtx);
        ret = PP_ERR;
    }

    return ret;
}
                              
int pp_cfg_iter_next(pp_cfg_iter_t* iter,
		     const char** key, const char** value) {
    cfg_entry_t* e;
    pp_ditr_t* ditr = iter->ditr;
    int ret = pp_ditr_next(ditr);
    pp_cfg_iter_t* iter_int;
    char *ikey;
    
    if (ret) {
	iter->cfge = e = pp_ditr_data(ditr);
        ikey = (char*)pp_ditr_key(ditr);
	if (key)
	    *key = ikey;
	if (value) {
	    if (e->type == CFG_DICT) {
                if(iter->type == FLAT)
                    *value = NULL;
                else { // iter->type == X_DICT
                    // create new iterator for dict... (assure to delete it afterwards!)
                    if(PP_ERR == (ret = pp_cfg_iter_create2(&iter_int, iter, ikey)))
                        return ret;
                    // ... and get next key/value pair from new child iterator
                    ret = pp_cfg_iter_next(iter_int, key, value);
                }
	    } else {
		*value = e->value.string;
	    }
	}
    } else {
        if(iter->type == FLAT || !iter->parent)
	    iter->cfge = NULL;
        else { // iter->type == X_DICT
            // delete the child iterator, we created...
            iter_int = iter->parent;
            pp_cfg_iter_destroy(iter);
            // ... and return next key/value pair in parent iterator
            ret = pp_cfg_iter_next(iter_int, key, value);
        }
    }
    return ret;
}

void pp_cfg_iter_destroy(pp_cfg_iter_t* iter) {
    if (iter != NULL) {
	if (iter->parent == NULL)
	    pp_mutex_unlock(&iter->prof->mtx);
	pp_ditr_destroy(iter->ditr);
	pp_profile_release(iter->prof);
	free(iter);
    }
}

int pp_cfg_iter_get(pp_cfg_iter_t* iter, const char** value,
		    const char* key, ...) {
    vector_t* key_comps;
    cfg_entry_t* e;
    va_list ap;
    int ret;
    assert(key && strlen(key) > 0);
    
    va_start(ap, key);
    key_comps = cfg_key_expand(key, ap);
    if (iter->cfge->type == CFG_DICT &&
	NULL != (e = cfg_key_resolve_exact_core(NULL, iter->cfge->value.dict,
						key_comps, 0,
						vector_size(key_comps))) &&
	e->type == CFG_STR) {
	*value = e->value.string;
	ret = PP_SUC;
    } else {
	ret = PP_ERR;
    }
    pp_cfg_key_comps_destroy(key_comps);
    va_end(ap);
    return ret;
}

/*
 * cfg handler API,
 * i.e. funktion pointers that are registered behind a certain cfg key
 * ===================================================================
 */

void pp_cfg_register_handler(pp_cfg_handler_type_t type, const char* key,
			     int (*get)(pp_profile_t*, vector_t*),
			     pp_cfg_chg_kind_t (*set)(const char*,
						      pp_profile_t*,
						      const vector_t*)) {
    int r;
    cfg_handler_t* h;
    PP_ASSERT(type, type == PP_CFG_HANDLER_LASTCOMP); // other is not yet implemented

    h = cfg_handler_create(get, set);
    r = pp_dict_insert(cfg_handlers_lastcomp, key, h, pp_free, 0);
    assert(r == 0);
}

static cfg_handler_t* cfg_handler_create(int(*get)(pp_profile_t*, vector_t*),
					 pp_cfg_chg_kind_t (*set)(const char*,
				             pp_profile_t*, const vector_t*)) {
    cfg_handler_t* h = malloc(sizeof(cfg_handler_t));
    h->get = get;
    h->set = set;

    return h;
}

/**
 * checks if LDAP profile is active vor user 'name', if no name is given for device
 * returns 1 if true, 0 if false and PP_ERR in case of an error
 *
 * replaces eric_um_Principal_ldap_prof_is_active
 */
int pp_cfg_ldap_prof_is_active(int uid) {
    int ret = 0;
    vector_t *ps = NULL;
    
    if( uid >= 0 ) {
        if(PP_ERR != (ret = pp_cfg_get_profile_stack(&ps, PP_PROFILE_EFFECTIVE, 
                                                     "user[%u]", uid)) &&
           vector_size(ps) > 0 &&
           ((pp_profile_t*)vector_get(ps, 0))->type == PP_PROFILE_LDAP_USER)
            ret = 1;
    } else {
        if(PP_ERR != (ret = cfg_get_profile_stack_va(&ps, PP_PROFILE_EFFECTIVE, 
                                                     NULL, NULL)) &&
           vector_size(ps) > 0 &&
           ((pp_profile_t*)vector_get(ps, 0))->type == PP_PROFILE_LDAP_DEV)
            ret = 1;
    }
    
    vector_delete(ps);    
    return ret;
}

/**
 * replaces eric_um_get_profile_info
 */
int pp_cfg_get_profile_info(pp_cfg_profile_info_t** _pi, int uid) {
    pp_cfg_profile_info_t* pi = NULL;
    int ret = PP_SUC;
    vector_t *ps = NULL;
    pp_profile_t *p;

    pi = malloc(sizeof(pp_cfg_profile_info_t));
    memset(pi, 0, sizeof(pp_cfg_profile_info_t));

    pi->ldap_prof_is_active = 0;
    if(uid >= 0) {
        ret = pp_cfg_get_profile_stack(&ps, PP_PROFILE_EFFECTIVE, "user[%u]", uid);
    } else {
        ret = cfg_get_profile_stack_va(&ps, PP_PROFILE_EFFECTIVE, NULL, NULL);
    }

    if(ret != PP_ERR) {
        pi->local_tag = NULL;
        pp_cfg_get(&pi->local_tag, "profile_tag.local");
    
        p = (pp_profile_t*)vector_get(ps, 0);
        if(p->type == PP_PROFILE_LDAP_USER || p->type == PP_PROFILE_LDAP_DEV)
            pi->ldap_prof_is_active = 1;
        vector_delete(ps);    

/* TODO: errno magic? */
//        pi->ldap_error = pp_ldap_has_errno(errno) ? errno : PP_SUC;
    
/* TODO: once more, the tags... */
        pi->ldap_requested_tag = NULL;
/*
        pi->ldap_requested_tag = eric_um_get_option(ERIC_UM_GLOBAL_P,
                                                    ldap_profile_tag_key,
                                                    DEF_NO_DEFAULT, error);
*/
        if (pi->ldap_prof_is_active) {
    //	pi->ldap_active_tag = strdup(pe->ldap_tag_name);
            pi->ldap_active_tag = NULL;
            pi->ldap_profile_name = strdup(p->name);
        }
        
        *_pi = pi;
    }
    return ret;
}     

/**
 * replaces eric_um_get_profile_info
 */
int pp_cfg_profile_info_destroy(pp_cfg_profile_info_t* profile_info) {
    if (profile_info != NULL) {
	free(profile_info->local_tag);
	free(profile_info->ldap_requested_tag);
	free(profile_info->ldap_active_tag);
	free(profile_info->ldap_profile_name);
	free(profile_info);
    }
    return PP_SUC;
}

/**
 * returns the profile type id of the profile, the cfg_key is fetched from
 * if cfg_key is not stored in any accessible profile, PP_ERR is returned
 */
int pp_cfg_get_prof_type(const char* key, ...) {
    int ret;
    va_list ap;
    va_start(ap, key);
    ret = cfg_get_va(0, NULL, PP_PROFILE_EFFECTIVE, key, ap);
    va_end(ap);
    return ret;
}

static int cfg_dict_diff_ditr_recursive_sub(vector_t *diff, int recursion, 
                                            int changes, 
                                            pp_strstream_t *cfgkey,
                                            pp_profile_type_t type,
                                            pp_ditr_t *ditr,
                                            cfg_tx_op_type_t op_type) {
    char *key;
    const cfg_entry_t *ce;
    
    assert(diff);
    assert(cfgkey);
    assert(ditr);
    assert(pp_ditr_valid(ditr));
    
    /* we better assert max recursions, you never know */
    assert(recursion < 20);
    
    key = (char*)pp_ditr_key(ditr);
    ce = pp_ditr_data(ditr);
    assert(ce);
    
    pp_strappend(cfgkey, key);
    
    switch(ce->type) {
        case CFG_STR:
            changes++;
            vector_add(diff, 
                       cfg_tx_log_entry_create(op_type, type,
                                               pp_cfg_key_expand(pp_strstream_buf(cfgkey)),
                                               ce->value.string));
            break;
        case CFG_DICT:
            {
                pp_ditr_t *new_ditr = pp_ditr_new(ce->value.dict);
                int key_pos;
            
                pp_strappendchr(cfgkey, CFG_COMP_DELI);
                
                for(pp_ditr_first(new_ditr),
                    key_pos = pp_strstream_pos(cfgkey);
                    pp_ditr_valid(new_ditr);
                    pp_ditr_next(new_ditr), pp_strstream_seek(cfgkey, key_pos)){
                    changes = cfg_dict_diff_ditr_recursive_sub(diff, 
                                                               recursion + 1,
                                                               changes,
                                                               cfgkey, type,
                                                               new_ditr,
                                                               op_type);
                }
                pp_ditr_destroy(new_ditr);
            }
            break;
        default:
            assert(0); // should not happen...
    }
    
    return changes;
}

static int cfg_dict_diff_apply_ditr_recursive(vector_t *diff, int recursion, 
                                              int changes, 
                                              pp_strstream_t *cfgkey,
                                              pp_profile_type_t type,
                                              pp_ditr_t *ditr) {
    return cfg_dict_diff_ditr_recursive_sub(diff, recursion, changes, cfgkey,
                                            type, ditr, CFG_TX_OP_SET);
}

static int cfg_dict_diff_remove_ditr_recursive(vector_t *diff, int recursion, 
                                               int changes, 
                                               pp_strstream_t *cfgkey,
                                               pp_profile_type_t type,
                                               pp_ditr_t *ditr) {
    return cfg_dict_diff_ditr_recursive_sub(diff, recursion, changes, cfgkey,
                                            type, ditr, CFG_TX_OP_REM);
}

static int cfg_dict_diff_recursive(vector_t *diff, int recursion, int changes,
                                   pp_strstream_t *cfgkey,
                                   pp_profile_type_t type,
                                   const pp_dict_t *local_dict, 
                                   const pp_dict_t *new_local_dict) {
    pp_ditr_t *local_ditr, *new_local_ditr;
    int local_ditr_valid, new_local_ditr_valid, key_pos;
    char *local_key, *new_local_key;
    const cfg_entry_t *local_ce, *new_local_ce;

    assert(diff);
    assert(cfgkey);
    assert(local_dict);
    assert(new_local_dict);
    
    /* we better assert max recursions, you never know */
    assert(recursion < 20);
    
    local_ditr = pp_ditr_new(local_dict);
    new_local_ditr = pp_ditr_new(new_local_dict);
    local_ditr_valid = pp_ditr_first(local_ditr);
    new_local_ditr_valid = pp_ditr_first(new_local_ditr);
    key_pos = pp_strstream_pos(cfgkey);

    if(key_pos) {
        pp_strappendchr(cfgkey, CFG_COMP_DELI);
        key_pos++;
    }
    
    while(local_ditr_valid | new_local_ditr_valid) {
        /* reset cfgkey stream */
        pp_strstream_seek(cfgkey, key_pos);
        
        if(!local_ditr_valid) {
            /* all remaining entries in new_local_dict are new -> add them */

            changes = cfg_dict_diff_apply_ditr_recursive(diff, recursion,
                                                         changes, cfgkey, type,
                                                         new_local_ditr);
            new_local_ditr_valid = pp_ditr_next(new_local_ditr);
        } else if(!new_local_ditr_valid) {
            /* all remaining entries in local_dict are old -> delete them*/
            changes = cfg_dict_diff_remove_ditr_recursive(diff, recursion,
                                                          changes, cfgkey, type,
                                                          local_ditr);
            local_ditr_valid = pp_ditr_next(local_ditr);
        } else {
            /* both dicts are valid, go on... */
            int comp;
            
            local_key = (char*)pp_ditr_key(local_ditr);
            new_local_key = (char*)pp_ditr_key(new_local_ditr);
            local_ce = pp_ditr_data(local_ditr);
            new_local_ce = pp_ditr_data(new_local_ditr);
            
            /* compare the keys */
            if(!(comp = strcmp(local_key, new_local_key))) {
                /* pew... keys are identical... care about values */
                
                assert(local_ce->type == new_local_ce->type);
                
                pp_strappend(cfgkey, local_key);
                
                switch(local_ce->type) {
                    case CFG_STR:
                        if(strcmp(local_ce->value.string,
                                  new_local_ce->value.string)) {
                            /* values differ, apply change */
                            changes++;
                            vector_add(diff, 
                                       cfg_tx_log_entry_create(CFG_TX_OP_SET,
                                                               type,
                                                 pp_cfg_key_expand(pp_strstream_buf(cfgkey)),
                                                 new_local_ce->value.string));
                        }
                        /* else values are equal, that's all folks */
                        break;
                    case CFG_DICT:
                        changes = cfg_dict_diff_recursive(diff, recursion + 1,
                                                          changes, cfgkey, type,
                                                          local_ce->value.dict,
                                                      new_local_ce->value.dict);
                        break;
                    default:
                        assert(0); // should not happen...
                }
                local_ditr_valid = pp_ditr_next(local_ditr);
                new_local_ditr_valid = pp_ditr_next(new_local_ditr);
            } else {
                /* keys are not identical, evaluate comp */
                if(comp > 0) {
                    /* local_key > new_local_key 
                       => new_local_key added */

                    changes = cfg_dict_diff_apply_ditr_recursive(diff,
                                                                 recursion,
                                                                 changes,
                                                                 cfgkey, type,
                                                                new_local_ditr);
                    new_local_ditr_valid = pp_ditr_next(new_local_ditr);
                } else {
                    /* local_key < new_local_key 
                       => local_key removed*/
                    changes = cfg_dict_diff_remove_ditr_recursive(diff,
                                                                  recursion,
                                                                  changes,
                                                                  cfgkey, type,
                                                                  local_ditr);
                    local_ditr_valid = pp_ditr_next(local_ditr);
                }
            }
        }
    }        
    
    pp_ditr_destroy(local_ditr);
    pp_ditr_destroy(new_local_ditr);
    
    return changes;
}

static int cfg_profile_diff(vector_t *diff, const pp_profile_t *local_prof, 
                            const pp_profile_t *new_local_prof) {
    int changes;
    pp_strstream_t cfgkey = PP_STRSTREAM_INITIALIZER;
    
    assert(diff);
    assert(local_prof);
    assert(new_local_prof);
    
    changes = cfg_dict_diff_recursive(diff, 0, 0, &cfgkey, local_prof->type,
                                      local_prof->configs,
                                      new_local_prof->configs);

    pp_strstream_free(&cfgkey);

    return changes;
}

static int diff_and_update_profile(pp_profile_t* prof, pp_profile_t* new_prof) {
    int ret = PP_ERR;
    int tx_internal = 0;
    int changes;
    pp_cfg_tx_t* tx;
    
    if(!(tx = pp_cfg_tx_get_thread_ctx())) {
        /* seems no transaction is open, create a new one */
        tx = pp_cfg_tx_begin(0);
        tx_internal = 1;
    }
    
    pp_mutex_lock(&prof->mtx);
    pp_mutex_lock(&new_prof->mtx);

    /* generate diff of old config and current config */
    changes = cfg_profile_diff(tx->chg_log, prof, new_prof);
    profile_cache_update_timestamp(prof_cache, prof->type,
                                   get_local_profile_name(prof->type));

    pp_mutex_unlock(&prof->mtx);
    pp_mutex_unlock(&new_prof->mtx);

    /* apply diff to tx */
    assert(changes <= (int)vector_size(tx->chg_log));
    
    if(tx_internal) {
        pp_cfg_tx_commit_core(tx, DO_FLUSH, 0, 1, NULL);
    }
    
    return ret;
}

char *
pp_cfg_value_shellesc(const char * s)
{
    if (s) {
	char * esc_s = malloc(2 * strlen(s));
	char * dp = esc_s;

	if (esc_s) {
	    while (*s) {
		if (*s == '$' || *s == '\\' || *s == '"' || *s == '`') {
		    *dp++ = '\\';
		}
		*dp++ = *s++;
	    }
	    *dp = '\0';
	}

	return esc_s;
    }

    return NULL;
}

/*
 * transforms the key so that it can become a shell variable
 * .[] => _
 * if 'noidx' is true, all indices will be deleted.
 */
char*
pp_cfg_key_transform(const char* k, int noidx) {
    char *p, *n, c;
    int skip = 0;
    p = n = malloc(strlen(k) + 1);
    while((c = *k)) {
	if (!skip) {
	    switch(c) {
	      case ']':
		  break;
	      case '[':
		  if (noidx) {
		      skip = 1;
		      break;
		  }
	      case '.':
		  *p++ = '_';
		  break;
	      default:
		  *p++ = c;
	    }
	} else if (c == ']') {
	    skip = 0;
	}
	    k++;
    }
    *p = '\0';
    return n;
}

char* pp_cfg_get_key_from_key_comps(const vector_t *key_comps, 
                                    char *key, int length) {
    int idx, key_comps_sz, len;
    key_comp_t *first_comp, *last_comp;
    
    assert(key);
    assert(key_comps);
    assert(length);
    
    key_comps_sz = vector_size(key_comps);
    
    if(key_comps_sz > 0) {
        first_comp = (key_comp_t*)vector_get2(key_comps, 0);
        last_comp = (key_comp_t*)vector_get2(key_comps, key_comps_sz - 1);
        assert(first_comp->name);
        assert(last_comp->name);
        len = last_comp->name + strlen(last_comp->name) + 1 - first_comp->name;
        if(length > len) {
            memcpy(key, first_comp->name, len);
            for(idx = strlen(first_comp->name); idx < len - 1; ++idx)
                if(key[idx] == '\0')
                    key[idx] = CFG_COMP_DELI;
        } else {
            *key = '\0';
        }
    } else {
        *key = '\0';
    }
    
    return key;
}

char* cfg_get_qid_from_key_comps(const vector_t *key_comps, 
                                 char *qid, int length) {
    int idx, key_comps_sz;
    key_comp_t *comp;
    pp_strstream_t strstream, *strbuf = &strstream;
    
    assert(qid);
    assert(key_comps);
    assert(length);
    
    key_comps_sz = vector_size(key_comps);
    
    if(key_comps_sz > 0) {
	pp_strstream_init(strbuf);
        for(idx = 0; idx < key_comps_sz; ++idx) {
            comp = (key_comp_t*)vector_get2(key_comps, idx);
            assert(comp && comp->name);
            if(!strcmp(comp->name, PP_CD_VECT_ELEMS_COMP_STR)) {
                /* for vector elem comp key, skip following key (id) */
                ++idx;
            }
            if(pp_strstream_pos(strbuf)) {
                pp_strappendchr(strbuf, PP_CD_COMP_DELI);
            }
            pp_strappend(strbuf, comp->name);
        }
        snprintf(qid, length, pp_strstream_buf(strbuf)); 
        pp_strstream_free(strbuf);
    } else {
        *qid = '\0';
    }
    
    return qid;
}

const char *
pp_cfg_get_key_comp_name_at_idx(const vector_t* comps, u_int idx)
{
    assert(comps);
    key_comp_t * comp = (key_comp_t*)vector_get2(comps, idx);
    return (comp ? comp->name : NULL);
}

static void*
cfg_ext_chg_listener_thr_func(void* arg UNUSED)
{
    sleep(1); // wait a bit
    while (pp_cfg_initialized) {
        /* listen for CFG_EXT_CHANGE_KEY (blocking) */
        if (pp_sem_unlock_bl(eric_config_ext_change_sem_id) == 0) {
            pp_profile_t *prof = NULL;
            
            __pp_cfg_get_profile_core(&prof, PP_PROFILE_LOCAL, NULL, -1);
            pp_profile_release(prof);            
        }
    }
    
    return NULL;
}

static int 
cfg_local_layer_is_empty_recursive(int recursion, const pp_dict_t *dict, 
                                   pp_strstream_t *strbuf,
                                   const char **excludes) {
    int ret = PP_SUC, pos = pp_strstream_pos(strbuf);
    pp_ditr_t *iter = pp_ditr_new(dict);
    
    /* we better assert max recursions, you never know */
    assert(recursion++ < 20);

    for(pp_ditr_first(iter); ret == PP_SUC && pp_ditr_valid(iter);
        pp_ditr_next(iter)) {
        char *key;
        const cfg_entry_t *ce;
        int i;
         
        key = (char*)pp_ditr_key(iter);
        ce = pp_ditr_data(iter);
        assert(ce);
        
        pp_strappend(strbuf, key);
        
        switch(ce->type) {
            case CFG_STR:
                for(i = 0; excludes[i]; ++i) {
                    if(!strcmp(excludes[i], strbuf->buf)) {
                        /* buffered key is in excludes, so break */
                        break;
                    }
                }
                
                /* if excludes[i] != NULL, buffered key is in excludes */
                if(!excludes[i]) {
                    ret = PP_ERR;
                }
                
                break;
            case CFG_DICT:
                pp_strappendchr(strbuf, CFG_COMP_DELI);
                ret = cfg_local_layer_is_empty_recursive(recursion,
                                                         ce->value.dict,
                                                         strbuf, excludes);
                break;
            default:
                assert(0); // should not happen...
        }
        
        pp_strstream_seek(strbuf, pos);
    }
    pp_ditr_destroy(iter);

    return ret;
}

int pp_cfg_local_layer_is_empty(void) {
    pp_profile_t *prof;
    int ret;
    pp_strstream_t strstream, *strbuf = &strstream;
    const char *excludes[] = { "serial", "network.mac", "cd_version", NULL };
    
     /* last element of excludes must be NULL */
    assert(excludes[sizeof(excludes) / sizeof(char*) - 1] == NULL);
    
    if(PP_ERR != (ret = __pp_cfg_get_profile_core(&prof, PP_PROFILE_LOCAL,
                                                  NULL, -1))) {
        pp_strstream_init(strbuf);
        ret = cfg_local_layer_is_empty_recursive(0, prof->configs, 
                                                 strbuf, excludes);
        pp_strstream_free(strbuf);
        pp_profile_release(prof);
    }
    
    return ret;
}

int
pp_cfg_validate(const char* value, const char* key, ...) {
    int ret;
    va_list ap;

    va_start(ap, key);

    ret = cfg_validate_va(value, key, ap);

    va_end(ap);

    return (ret != PP_ERR) ? PP_SUC : PP_ERR;
}

static int
cfg_validate_va(const char* value, const char* key, va_list ap) {
    int ret;
    vector_t *key_comps;

    key_comps = cfg_key_expand(key, ap);

    ret = cfg_validate_key_comps(value, key_comps);
    
    pp_cfg_key_comps_destroy(key_comps);
    
    return ret;
}

static int
cfg_validate_key_comps(const char* value, const vector_t* key_comps) {
    int ret = PP_SUC;
    char cfg_key[MAX_OPT_KEY_LEN], qid[MAX_OPT_KEY_LEN];
    pp_cd_as_type_t *type;
    pp_cd_validation_error_t val_err;
    int val_err_param;

    pp_cd_as_cd_t *cd = pp_cfg_get_cd();
    
    pp_cfg_get_key_from_key_comps(key_comps, cfg_key, MAX_OPT_KEY_LEN);
    cfg_get_qid_from_key_comps(key_comps, qid, MAX_OPT_KEY_LEN);
    
    if((type = pp_cd_resolve_type_symbol(cd, qid,
                                         PP_CD_RESOLVE_ALIAS)) == NULL) {
        pp_log("%s: Config key '%s' (qid '%s') is invalid\n",
               ___F, cfg_key, qid);
        errno = PP_CFG_ERR_INVALID_KEY;
        ret = PP_ERR;
    } else if(*value ||
              !pp_cd_prop_has_flag(cd, qid, "flags", "allow_empty")) {
        /* validate value */
        while(PP_CD_IS_TYPE(&type->base, AS_TYPE_VECTOR_TMPL)) {
            type = ((pp_cd_as_type_vector_tmpl_t*)type)->elem_type;
        }
        if(!pp_cd_type_validate_value(type, value, 
                                      &val_err, &val_err_param)) {
            pp_log("%s: Config value '%s' for key '%s' (qid '%s') "
                   "is invalid, reason:\n", ___F, value, cfg_key, qid);
            pp_log(pp_cd_val_err_msg(val_err), cfg_key, val_err_param);
            pp_log("\n");
            errno = PP_CFG_ERR_INVALID_VALUE;
            ret = PP_ERR;
        }
    }

    return ret;
}

int pp_cfg_get_profile_from_key(pp_profile_type_t * profile, const char* key, ...) {
    int ret;
    va_list ap;

    va_start(ap, key);

    ret = cfg_get_profile_va(key, ap, profile);

    va_end(ap);

    return (ret != PP_ERR) ? PP_SUC : PP_ERR;
}

static int
cfg_get_profile_va(const char* key, va_list ap, pp_profile_type_t *profile) {
    int ret;
    vector_t *key_comps;

    key_comps = cfg_key_expand(key, ap);

    ret = cfg_get_profile_key_comps(key_comps, profile);
    
    pp_cfg_key_comps_destroy(key_comps);
    
    return ret;
}

static int
cfg_get_profile_key_comps(const vector_t* key_comps, pp_profile_type_t *profile) {
    int ret = PP_SUC;
    char cfg_key[MAX_OPT_KEY_LEN], qid[MAX_OPT_KEY_LEN];
    pp_cd_as_type_t *type;

    pp_cd_as_cd_t *cd = pp_cfg_get_cd();
    
    pp_cfg_get_key_from_key_comps(key_comps, cfg_key, MAX_OPT_KEY_LEN);
    cfg_get_qid_from_key_comps(key_comps, qid, MAX_OPT_KEY_LEN);
    
    if((type = pp_cd_resolve_type_symbol(cd, qid,
                                         PP_CD_RESOLVE_ALIAS)) == NULL) {
        pp_log("%s: Config key '%s' (qid '%s') is invalid\n",
               ___F, cfg_key, qid);
        errno = PP_CFG_ERR_INVALID_KEY;
        ret = PP_ERR;
    } else if(pp_cd_prop_has_flag(cd, qid, "flags", "sys_profile")) {
        *profile = PP_PROFILE_SYSTEM;
    } else {
        *profile = PP_PROFILE_LOCAL;
    }

    return ret;
}

/**
 * validate and set calls, feel free to add some by useing following API
 * (bit more conversion than it had to, but keeps API small)
 *
 * PP_CFG_VALIDATE_AND_SET(<method suffix>, <value datatype>, <value format>)
 * e.g. PP_CFG_VALIDATE_AND_SET(_int, int, "%d") creates
 *      int pp_cfg_validate_and_set_int(int value, const char *key, ...)
 */

#define PP_CFG_VALIDATE_AND_SET(_name_, _datatype_, _format_) \
    int pp_cfg_validate_and_set##_name_(_datatype_ value, const char *key, ...) { \
        int ret; \
        va_list ap, va; \
        char value_str[MAX_OPT_VALUE_LEN + 1]; \
        snprintf(value_str, MAX_OPT_VALUE_LEN + 1, _format_, value); \
        va_start(ap, key); \
        va_copy(va, ap); \
        if(PP_ERR != (ret = cfg_validate_va(value_str, key, va))) { \
            ret = cfg_set##_name_##_va(value, PP_PROFILE_EFFECTIVE, key, ap); \
        } \
        va_end(va); \
        va_end(ap); \
        return ret; \
    }

#define PP_CFG_VALIDATE_AND_SET_AT_LAYER(_name_, _layer_, _datatype_, _format_) \
    int pp_cfg_validate_and_set##_name_##_at_layer(pp_profile_type_t _layer_, _datatype_ value, const char *key, ...) { \
        int ret; \
        va_list ap, va; \
        char value_str[MAX_OPT_VALUE_LEN + 1]; \
        snprintf(value_str, MAX_OPT_VALUE_LEN + 1, _format_, value); \
        va_start(ap, key); \
        va_copy(va, ap); \
        if(PP_ERR != (ret = cfg_validate_va(value_str, key, va))) { \
            ret = cfg_set##_name_##_va(value, _layer_, key, ap); \
        } \
        va_end(va); \
        va_end(ap); \
        return ret; \
    }

//PP_CFG_VALIDATE_AND_SET(, const char*, "%s")
//PP_CFG_VALIDATE_AND_SET(_int, int, "%d")

int pp_cfg_get_perm(const char **perm, const char *key, ...) {
    int ret;
    va_list ap;
    va_start(ap, key);
    ret = cfg_get_prop_va(perm, "perm", key, ap);
    va_end(ap);
    return (ret != PP_ERR) ? PP_SUC : PP_ERR;
}

int pp_cfg_get_prop(const char **value, const char *prop,
                    const char *key, ...) {
    int ret;
    va_list ap;
    va_start(ap, key);
    ret = cfg_get_prop_va(value, prop, key, ap);
    va_end(ap);
    return (ret != PP_ERR) ? PP_SUC : PP_ERR;
}

static int cfg_get_prop_va(const char **value, const char* prop,
                           const char* key, va_list va) {
    vector_t *key_comps;
    char qid[MAX_OPT_KEY_LEN];
    const char *val;
    
    assert(pp_cfg_initialized);

    key_comps = cfg_key_expand(key, va);
    cfg_get_qid_from_key_comps(key_comps, qid, MAX_OPT_KEY_LEN);
    val = pp_cd_lookup_prop(pp_cfg_get_cd(), qid, prop);

    pp_cfg_key_comps_destroy(key_comps);
    
    if(val) {
        *value = strdup(val);
        return PP_SUC;
    }
    
    return PP_ERR;
}

#ifndef NDEBUG
static int
save_profile_direct(const pp_profile_t* prof, int lock, const char *filename) {
    vector_t cfgs, stack;
    int ret;

    vector_new(&cfgs, 200, eric_config_option_destroy);
    vector_new(&stack, 10, NULL);
    save_profile_dict(0, 0, &cfgs, &stack, prof->configs);
    vector_add(&cfgs, NULL);

    if (eric_config_file_write_nucleus(filename,
			               (eric_config_option_t**)cfgs.elements,
                                       0, lock)) {
	ret = PP_ERR;
	goto bailout;
    }
    ret = PP_SUC;
    
 bailout:
    vector_delete(&cfgs);
    vector_delete(&stack);
    return ret;
}

void pp_cfg_dump_profile(const char *cdl, const char *file) {
    pp_cd_as_cd_t* cd = NULL;
    
    if(!pp_cfg_initialized) {
        cfg_handlers_lastcomp = pp_dict_str_new();
        cfg_notify_init();
    }

    if((cd = pp_cd_parse_cdl(pp_mallocator_heap(), cdl)) == NULL) {
        printf("could not parse %s\n", cdl);
    } else {
        pp_profile_t* p = pp_profile_create(pp_mallocator_heap(),
                                            PP_PROFILE_LOCAL_CODE, "dump");

        cfg_load_profile(p, cd);
        save_profile_direct(p, 1, file);
        pp_profile_release(p);
    }
    
    pp_cd_destroy(cd);

    if(!pp_cfg_initialized) {
        pp_dict_destroy(cfg_handlers_lastcomp);
        cfg_notify_cleanup();
    }
}
#endif /* !NDEBUG */
