/*
 * The basic config library providing
 * config notifications
 *
 * (c) 2004 Peppercon AG
 * tbr@peppercon.de
 */

#define PRINT_KEY(key_comps) \
            int __idx, __kcsz = vector_size(key_comps); \
            if(__kcsz > 0) { \
                printf("%s", ((key_comp_t*)vector_get2(key_comps, 0))->name); \
                for(__idx = 1; __idx < __kcsz; __idx++) \
                    printf(".%s", ((key_comp_t*)vector_get2(key_comps, __idx))->name); \
                printf("\n"); \
            }

#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>

#include <pp/base.h>
#include <pp/vector.h>
#include <pp/dict.h>
#include <pp/profile.h>
#ifndef PP_BOARD_PEMX
    #include <liberic_pthread.h>
#endif

#include <pp/cfg.h>
#include <pp/cfgx.h>

#include "cfg_intern.h"

/* cfg notification structures */
/*******************************/
typedef struct {
    pp_dict_t* dict;
    vector_t* listeners;
} cfgl_entry_t;

/* locally scoped variables      */
/*********************************/

/* the notification listener root tree */
static cfgl_entry_t* cfg_listeners;
pthread_mutex_t cfg_listeners_mtx = PTHREAD_ERRORCHECK_MUTEX_INITIALIZER_NP;

/* the notification listener threads */
static LIST_HEAD(cfg_notify_thr_list);
static pthread_mutex_t cfg_notify_thr_mtx =
    PTHREAD_ERRORCHECK_MUTEX_INITIALIZER_NP;
static pthread_cond_t cfg_notify_thr_cond;

/* the listener uid's counter 
   start counting from 1 since uid is used as hash key and may not be 0! */
static int cfg_listener_uid = 1;

/* Config change notification static functions */
static cfgl_entry_t* cfgl_entry_create(void);
static void cfgl_entry_destroy(pp_mallocator_t* a, cfgl_entry_t*);
static void cfgl_entry_del_listener(int ridx, cfgl_entry_t* cle,
				    vector_t* clepath, vector_t* comps);
static void cfgl_entry_del_all_listeners(cfgl_entry_t* cle, vector_t* clepath,
					 vector_t* comps);
static void cfgl_entry_check_del(cfgl_entry_t* cle, int cidx,
				 vector_t* clepath, vector_t* comps);
static int cfg_rem_change_listener_at_key(pp_cfg_chg_cb cfg_chg_cb,
					  const char* cfgkey, va_list ap);
static int cfg_rem_change_listener_completely(pp_cfg_chg_cb cfg_chg_cb);
static int cfg_add_change_listener(cfgl_listener_type_t type,
                                   pp_cfg_chg_cb cfg_chg_cb,
	                           const char* cfgkey, va_list ap);

inline static int cfg_get_listener_uid(void);
static void cfg_rem_scheduled_listener(cfgl_listener_t* le);

static pp_cfg_chg_ctx_t*
cfg_chg_cb_ctx_create(char **retstr, pp_cfg_chg_kind_t kind,
                      vector_t *key_comp_vec, pp_cfg_tx_t* tx, 
                      unsigned int mode);
static void cfg_chg_cb_ctx_destroy(void *v);

/*
 * Config change notification API
 * =================================================================
 */

/*
 * private methods
 *******************/

void cfg_notify_init() {
    pthread_cond_init(&cfg_notify_thr_cond, NULL);
    cfg_listeners = cfgl_entry_create();
}

void cfg_notify_cleanup() {
    MUTEX_LOCK(&cfg_notify_thr_mtx);
    while (!list_empty(&cfg_notify_thr_list)) {
	pthread_cond_wait(&cfg_notify_thr_cond, &cfg_notify_thr_mtx);
    }
    MUTEX_UNLOCK(&cfg_notify_thr_mtx);
    
    cfgl_entry_destroy(NULL, cfg_listeners);
}

inline static int cfg_get_listener_uid(void) {
    // no need to lock, will be called in locked section only
    return cfg_listener_uid++;
}

static cfgl_entry_t* cfgl_entry_create(void) {
    // we do not create the dict and vector in here, to save memory
    // that will be done on the fly, ones needed by the add function
    cfgl_entry_t* cfgle = malloc(sizeof(cfgl_entry_t));
    memset(cfgle, 0, sizeof(cfgl_entry_t));
    return cfgle;
}   
    
static void cfgl_entry_destroy(pp_mallocator_t* a UNUSED,
			       cfgl_entry_t* cle) {
    if (cle != NULL) {
	pp_dict_destroy(cle->dict);
	vector_delete(cle->listeners);
	free(cle);
    }
}

// deletes a single listener of the last cle in clepath and checks
// whether some part of the tree can be cleaned up
static void cfgl_entry_del_listener(int ridx, cfgl_entry_t* cle,
				    vector_t* clepath, vector_t* comps) {
    cfgl_listener_t* le = vector_get2(cle->listeners, ridx);
    cfg_rem_scheduled_listener(le);
    vector_remove(cle->listeners, ridx);
    cfgl_entry_check_del(cle, vector_size(clepath) - 1, clepath, comps);
}

// delets all listerns of last cle in clepath and checks
// whether some part of the tree can be cleaned up
static void cfgl_entry_del_all_listeners(cfgl_entry_t* cle,
					 vector_t* clepath, vector_t* comps) {
    unsigned int i;
    for (i = 0; i < vector_size(cle->listeners); ++i)
	cfg_rem_scheduled_listener(vector_get2(cle->listeners, i));
    vector_delete(cle->listeners);
    cle->listeners = NULL;
    cfgl_entry_check_del(cle, vector_size(clepath) - 1, clepath, comps);
}

static void cfgl_entry_check_del(cfgl_entry_t* cle, int cidx,
				 vector_t* clepath, vector_t* comps) {
    int dont_del = 0;
    cfgl_entry_t* pcle;
    key_comp_t* cle_comp;
    
    // check wether listeners is empty
    if (cle->listeners != NULL) {
	if (vector_size(cle->listeners) > 0) {
	    dont_del |= 1; // we still need that entry, it has listeners
	} else {
	    vector_delete(cle->listeners);
	    cle->listeners = NULL;
	}
    }
	
    // check wether dict is empty
    if (cle->dict != NULL) {
	if (pp_dict_count(cle->dict) > 0) {
	    dont_del |= 1; // we still need that entry, it has childs
	} else {
	    pp_dict_destroy(cle->dict);
	    cle->dict = NULL;
	}
    }

    // remove node from parent if not prevented, and call recursion
    if (!dont_del) {
	pcle = vector_get(clepath, cidx);
	cle_comp = vector_get2(comps, cidx);
	pp_dict_remove(pcle->dict, cle_comp->name, NULL);
	if (--cidx >= 0) {
	    cfgl_entry_check_del(pcle, cidx, clepath, comps);
	}
    }
}

static int cfg_rem_change_listener_at_key(pp_cfg_chg_cb cfg_chg_cb,
					  const char* cfgkey, va_list ap) {
    int ret = PP_ERR;
    unsigned int comps_size, comp_idx, i, ls;
    vector_t* comps, *clepath;
    key_comp_t* comp;
    cfgl_entry_t* cle = cfg_listeners;
    cfgl_listener_t* le;

    clepath = vector_new(NULL, 20, NULL);
    comps = cfg_key_expand(cfgkey, ap);
    comps_size = vector_size(comps);
    assert(comps_size > 0);

    MUTEX_LOCK(&cfg_listeners_mtx);
    
    // resolve path, return error if component is not found
    for (comp_idx = 0; comp_idx < comps_size; ++comp_idx) {
	vector_add(clepath, cle);
	comp = (key_comp_t*)vector_get2(comps, comp_idx);
	assert(strlen(comp->name) > 0);
	if (NULL == cle->dict ||
	    NULL == (cle = (cfgl_entry_t*)pp_dict_search(cle->dict,
							 comp->name))) {
	    goto bailout;
	}
    }

    // in case we have a callback, look for it, otherwise cleanup whole node
    if (cfg_chg_cb != NULL) {
	ls = vector_size(cle->listeners);
	for (i = 0; i < ls; ++i) {
	    le = (cfgl_listener_t*)vector_get2(cle->listeners, i);
	    if (cfg_chg_cb == le->cfg_chg_cb) {
		cfgl_entry_del_listener(i, cle, clepath, comps);
		ret = PP_SUC;
		break;
	    }
	}
    } else {
	cfgl_entry_del_all_listeners(cle, clepath, comps);
	ret = PP_SUC;
    }

 bailout:
    MUTEX_UNLOCK(&cfg_listeners_mtx);
    vector_delete(clepath);
    pp_cfg_key_comps_destroy(comps);
    return ret;
}

// removes the callback for every key
static int cfg_rem_change_listener_completely(pp_cfg_chg_cb cfg_chg_cb UNUSED){
    int implemented = 0;
    PP_ASSERT(implemented, implemented);
    return PP_SUC;
}

void cfg_fire_change(const vector_t* key_comps, pp_cfg_chg_kind_t kind) {
    vector_t* l;
    cfgl_keys_t *cfgl_keys;
    // lock until all threads have been started (their structure is
    // added to the thread list) so a change_listener_remove can't happen
    // until the listener is actually entered to the thread's structure,
    // and can be found by the remove operation there.
    MUTEX_LOCK(&cfg_listeners_mtx);
    l = cfg_lookup_change_listeners(key_comps);
    if (l != NULL) {
        cfgl_keys = malloc(sizeof(cfgl_keys_t));
        cfgl_keys->type = KEY_COMPS;
        cfgl_keys->value = cfg_key_duplicate(key_comps);
	cfg_notify_change_listeners(NULL, l, kind, cfgl_keys, NULL,
                                    CFGL_TYPE_ANY);
    }
    MUTEX_UNLOCK(&cfg_listeners_mtx);
}

static void
cfg_lookup_change_listeners_rec(const vector_t *key_comps, unsigned int idx,
                                cfgl_entry_t *cle, vector_t **listeners) {
    unsigned int key_comps_size;
    key_comp_t* key_comp;
    cfgl_entry_t *nle;
    
    key_comps_size = vector_size(key_comps);
    
    if (cle->listeners != NULL) {
        if (*listeners == NULL)
            *listeners = vector_new2(NULL, 20, sizeof(cfgl_listener_t),
                                     NULL);
        vector_addvec2(*listeners, cle->listeners, NULL);
    }

    if(idx < key_comps_size) {
	key_comp = (key_comp_t*)vector_get2(key_comps, idx);
	assert(key_comp->name && strlen(key_comp->name) > 0);
        idx++;
	if (NULL != cle->dict) {
            if(NULL != 
               (nle = (cfgl_entry_t*)pp_dict_search(cle->dict, key_comp->name)))
                cfg_lookup_change_listeners_rec(key_comps, idx, nle, listeners);
            if(NULL != 
               (nle = (cfgl_entry_t*)pp_dict_search(cle->dict,
                                                    pp_cfg_comp_wc_single)))
                cfg_lookup_change_listeners_rec(key_comps, idx, nle, listeners);
        }
    }
}

// this function may return NULL instead of an empty vector,
// just to make it a little more efficient, as the empty vector
// will probably happen more often than a full one
vector_t* cfg_lookup_change_listeners(const vector_t* key_comps) {
    vector_t* listeners = NULL;

    cfg_lookup_change_listeners_rec(key_comps, 0, cfg_listeners, &listeners);

    if (listeners != NULL) cfg_unify_change_listeners(listeners, NULL);
    return listeners;
}

static void cfg_unify_cfgl_keys(unsigned int dst, unsigned int src,
                                cfgl_keys_t *cfgl_keys) {
    vector_t *dst_vect, *src_vect;
    
    assert(cfgl_keys->type == CFGL_KEYS_HASH);
    
    if(NULL != (src_vect = pp_hash_get_entry_i(cfgl_keys->value, src)) &&
       NULL != (dst_vect = pp_hash_get_entry_i(cfgl_keys->value, dst))) {
        vector_addvec(dst_vect, src_vect, NULL);
        pp_hash_delete_entry_i(cfgl_keys->value, src);
    }
}

/**
 * - removes duplicates from the vector
 * - removes handlers for which cfg values have not changed
 * - keep handler for keys where
 *   - there is a value set in tx but not in cfg:
 *     -> new variable with no code profile default is set, e.g. user.email
 *   - there is a value set in tx with profile higher or equal than the profile
 *     assigned to the cfg value and the value changed:
 *     -> update, either override default or set another value
 *   - there is a NULL set in tx with profile higher or equal than the profile
 *     assigned to the cfg value and cfg value differs from default value:
 *     -> remove, e.g. set to defaults
 * - remove config keys from cfgl_keys which do not cause a change
 */
void cfg_unify_change_listeners(vector_t* listeners, cfgl_keys_t *cfgl_keys) {
    unsigned int i, k, val_changed = 1;
    cfgl_listener_t *chgl1, *chgl2;
    pp_cfg_tx_t *tx = pp_cfg_tx_get_thread_ctx();
    
    for (i = 0; i < vector_size(listeners);) {
	chgl1 = vector_get2(listeners, i);
        
        if(cfgl_keys) { /* check, whether config values really changed */
            char *val_cfg, *val_dflt, *val_tx = NULL;
            vector_t *key_comp_vec = NULL;
            int p_cfg, p_tx;
            vector_t *key_comps;
            pp_profile_type_t type = PP_PROFILE_EFFECTIVE;
            
            val_changed = 0;
            
            if(cfgl_keys->type == KEY_COMPS) {
                key_comp_vec = vector_new(NULL, 1, NULL);
                vector_add(key_comp_vec, cfgl_keys->value);
            } else  {
                key_comp_vec = pp_hash_get_entry_i(cfgl_keys->value,
                                                   chgl1->uid);
                assert(key_comp_vec);
            }
            for(k = 0; k < vector_size(key_comp_vec); ++k) {
                /* for all cfg keys in transaction ... */
                int cur_changed = 0;
                
                key_comps = vector_get(key_comp_vec, k);
                
                if((p_tx = cfg_tx_get_key_comps(tx, &val_tx, type, key_comps))
                        != PP_ERR) {
                    /* ATTENTION! 
                     * cfg_tx_get_key_comps will also return PP_SUC but NULL
                     * value for a cfg key which is marked to be removed! */
                    if((p_cfg = cfg_get_key_comps_no_tx_check(&val_cfg, type, key_comps))
                            != PP_ERR) {
                        /* ATTENTION! 
                         * cfg_get_key_comps may never return NULL value in 
                         * PP_SUC case, but pointer to '\0' */
                        if(p_tx >= p_cfg) {
                            /* tx commits to profile higher or equal to p_cfg */
                            if(!val_tx) {
                                /* cfg value is to be removed */
                                if(cfg_get_key_comps_no_tx_check(&val_dflt,
                                                     PP_PROFILE_DEFAULTS,
                                                     key_comps) != PP_ERR) {
                                    /* there is a default set for cfg var */
                                    if(pp_strcmp_safe(val_cfg, val_dflt)) {
                                        /* default differs from cfg value,
                                           effective value will be changed */
                                        cur_changed = 1;
                                    }
                                    free(val_dflt);
                                }
                            } else if(pp_strcmp_safe(val_cfg, val_tx)) {
                                /* value changed, usual case... */
                                cur_changed = 1;
                            }
                        }
                        free(val_cfg);
                    } else if(val_tx && *val_tx) {
                        /* cfg key is new, check if it is empty (== default) */
                        cur_changed = 1;
                    }    
                    free(val_tx);
                }
                    
                if(cur_changed) {
                    val_changed = 1;
                } else {
                    /* config key does not cause a change, remove from list! */
                    vector_remove(key_comp_vec, k);
                }
            }
            if(cfgl_keys->type == KEY_COMPS) {
                vector_delete(key_comp_vec);
            }
        }
                                       
        if(val_changed || !cfgl_keys) {
            /* either value has changed or do not care about values at all */
            for (k = i + 1; k < vector_size(listeners);) {
                chgl2 = vector_get2(listeners, k);
                if (chgl1->cfg_chg_cb == chgl2->cfg_chg_cb) {
                    if(chgl1->uid != chgl2->uid && cfgl_keys)
                        cfg_unify_cfgl_keys(chgl1->uid, chgl2->uid, cfgl_keys);
                    vector_remove(listeners, k);
                } else
                    k++;
            }
            ++i;
        } else {
            vector_remove(listeners, i);
        }
    }
}

int cfg_notify_change_listeners(pp_cfg_tx_t* tx, vector_t* listeners,
                                pp_cfg_chg_kind_t kind, cfgl_keys_t* cfgl_keys,
                                cfg_notify_thr_t** nthr, 
                                cfgl_listener_type_t type) {
    cfg_notify_thr_t* notify;
    int ret = PP_SUC;
    int threaded = nthr == NULL;
    /* it doesn't make sense to run threaded and return a notify thread object,
       as well as it not useful to run as thread and return nthr... */
    
    notify = malloc(sizeof(cfg_notify_thr_t));
    notify->listeners = listeners;
    notify->cfgl_keys = cfgl_keys;
    notify->kind = kind;
    notify->lidx = 0;
    MUTEX_CREATE_ERRORCHECKING(&notify->mtx);
    pthread_cond_init(&notify->cond, NULL);
    notify->retval = PP_SUC;
    notify->retstr = NULL;
    notify->threaded = threaded;
    notify->tx = tx;
    notify->type = type;
    INIT_LIST_HEAD(&notify->hook);

    MUTEX_LOCK(&cfg_notify_thr_mtx);

    if(threaded) {
        list_add(&notify->hook, &cfg_notify_thr_list);
#ifndef PP_BOARD_PEMX
        if (eric_pthread_create(&notify->thrid, 1, 128 * 1024,
                                cfg_notify_thr_func, (void*)notify))
#endif /* PP_BOARD_PEMX */
        {
            ret = PP_ERR;
            pp_log("cfg_fire_change: can't create thread\n");
            abort();
        }
    }
#ifndef PP_BOARD_PEMX
    else {
        *nthr = notify;
    }
#endif /* PP_BOARD_PEMX */
    
    MUTEX_UNLOCK(&cfg_notify_thr_mtx);
    
    return ret;
}

#ifndef PP_BOARD_PEMX
void* cfg_notify_thr_func(void* arg) {
    cfg_notify_thr_t* nthr = (cfg_notify_thr_t*)arg;
    cfgl_listener_t* le;
    pp_strstream_t strbuf;
    char *retstr = NULL;
    int retstr_log = (nthr->retstr != NULL) && !nthr->threaded;
    int ret;
    vector_t *key_comp_vec = NULL;
    unsigned int listeners_sz;
    pp_cfg_chg_ctx_t *ctx;

    // only log error strings if we want to return them
    if(retstr_log)
        pp_strstream_init(&strbuf);
    
    // execute callback, in case listener hasn't been removed
    MUTEX_LOCK(&nthr->mtx);
    if(nthr->cfgl_keys->type == KEY_COMPS) {
        key_comp_vec = vector_new(NULL, 1, 
                                  nthr->threaded ?
                                  (void(*)(void*))pp_cfg_key_comps_destroy :
                                  NULL);
        vector_add(key_comp_vec, nthr->cfgl_keys->value);
    }
    listeners_sz = vector_size(nthr->listeners);
    while (nthr->lidx < listeners_sz) {
	le = vector_get2(nthr->listeners, nthr->lidx);
	if (NULL == le->cfg_chg_cb || (nthr->type & le->type) != le->type) {
	    nthr->lidx++;
	    continue; // has been removed already or is of wrong type
	}
        if(nthr->cfgl_keys->type == CFGL_KEYS_HASH) {
            key_comp_vec = pp_hash_get_entry_i(nthr->cfgl_keys->value, le->uid);
            assert(key_comp_vec);
        }
	MUTEX_UNLOCK(&nthr->mtx);
//	if(PP_ERR == (ret = le->cfg_chg_cb(&retstr, nthr->kind, 
//                                           key_comp_vec))) {
    
#if defined(PP_CFG_DEBUG) && 0
        {
            /* print some hopefully helpful information */
            u_int u, key_comp_vec_sz = vector_size(key_comp_vec);
            char cfg_key[250];
            static const char *cfglt_name[] = { "undef", "CFGL_TYPE_WITHIN_TX",
                                                "CFGL_TYPE_POST_TX",
                                                "undef", "undef", "undef",
                                                "undef", "undef", "undef",
                                                "undef", "undef", "undef",
                                                "undef", "undef", "undef",
                                                "CFGL_TYPE_ANY" };
            
            printf("running change handler uid %d, type %s\nKey comps:\n",
                   le->uid, cfglt_name[le->type]);
            
            for(u = 0; u < key_comp_vec_sz; ++u) {
                pp_cfg_get_key_from_key_comps(vector_get(key_comp_vec, u),
                                              cfg_key, 250);
                printf("%s\n", cfg_key);
            }
        }
#endif /* PP_CFG_DEBUG */
    
        /* mode 0: do not free anything but struct itself */
        ctx = cfg_chg_cb_ctx_create(&retstr, nthr->kind, key_comp_vec, 
                                    nthr->tx, 0);
	if(PP_ERR == (ret = le->cfg_chg_cb(ctx))) {
            nthr->retval = PP_ERR;
        }
        if(retstr_log) {
            /* if we got a return string, append that, else append the
               assigned error string in case of an error*/
            if(retstr)
                pp_strappendf(&strbuf, "%s%s", 
                              pp_strstream_pos(&strbuf) ? "<br>" : "", retstr);
            else if(ret == PP_ERR)
                pp_strappendf(&strbuf, "%s%s", 
                              pp_strstream_pos(&strbuf) ? "<br>" : "",
                              pp_error_string(errno));
        }

        cfg_chg_cb_ctx_destroy(ctx);
        free(retstr);
        retstr = NULL;
	MUTEX_LOCK(&nthr->mtx);
	nthr->lidx++;
	pthread_cond_signal(&nthr->cond);
    }
    if(nthr->cfgl_keys->type == KEY_COMPS) {
        vector_delete(key_comp_vec);
    }
    if(retstr_log)
        *nthr->retstr = pp_strstream_buf(&strbuf);
    MUTEX_UNLOCK(&nthr->mtx);

    // remove myself from list of running threads and destroy structure
    if(nthr->threaded) {
        MUTEX_LOCK(&cfg_notify_thr_mtx);
        list_del(&nthr->hook);
        pthread_cond_signal(&cfg_notify_thr_cond);
        MUTEX_UNLOCK(&cfg_notify_thr_mtx);

        /* if threaded, clean up since that can't be done afterwards...
           if not, don't do so, as we possibly want to run cfg_notify_thr_func
           once more with other type (e.g. POST_TX!) */
        if(nthr->cfgl_keys->type == CFGL_KEYS_HASH) {
            pp_hash_delete_i(nthr->cfgl_keys->value);
        }
        // else already freed by vector_delete(key_comp_vec);
        free(nthr->cfgl_keys);
        nthr->cfgl_keys = NULL;
        vector_delete(nthr->listeners);

        free(nthr);
        nthr = NULL;
    }
    // else free in cfg_notify_change_listeners!

    return NULL;
}
#endif /* PP_BOARD_PEMX */

// check whether the given listener entry is about to be notified
// and wait until this operation has been completeted
static void cfg_rem_scheduled_listener(cfgl_listener_t* le) {
    struct list_head* pos;
    cfg_notify_thr_t* nthr;
    cfgl_listener_t* lle;
    unsigned int i;

    // we'll start over from the beginning once we had wait for
    // a thread to finish its callback to our listener in question
    // this is because we would like to unlook the notify->mtx during
    // wait and after that nothing need to be as it was before.
    // this seems to be strange, however, it won't happen frequently
    // and it will eventually terminate because of our listener uids
 startover:    
    
    MUTEX_LOCK(&cfg_notify_thr_mtx);
    list_for_each(pos, &cfg_notify_thr_list) {
	nthr = list_entry(pos, cfg_notify_thr_t, hook);
	MUTEX_LOCK(&nthr->mtx);

	// thread still bussy processing callbacks ?
	if (nthr->lidx < vector_size(nthr->listeners)) {
	    
	    // check current idx
	    lle = vector_get2(nthr->listeners, nthr->lidx);
	    if (lle->uid == le->uid) { // wait
/* TODO: assert(lle thread id != le thread id) */ 
		MUTEX_UNLOCK(&cfg_notify_thr_mtx);
/* FIXME:
   we have a possible deadlock here... assert, condition signaling is correct!
		pthread_cond_wait(&nthr->cond, &nthr->mtx);
   workaround for now:
*/
                {
                    struct timespec timer = {.tv_sec = 0, .tv_nsec = 10000000};
		    pthread_cond_timedwait(&nthr->cond, &nthr->mtx, &timer);
                }
		MUTEX_UNLOCK(&nthr->mtx);
		goto startover;
	    }
	    
	    // mark all that match and that are in front of current idx
	    for (i = nthr->lidx + 1; i < vector_size(nthr->listeners); ++i) {
		lle = vector_get2(nthr->listeners, i);
		if (lle->uid == le->uid)
		    lle->cfg_chg_cb = NULL;
	    }
	}
	MUTEX_UNLOCK(&nthr->mtx);
    }
    MUTEX_UNLOCK(&cfg_notify_thr_mtx);
}

/*
 * public methods
 *******************/

/**
 * Add a notification callback for a certain property.
 * The cfgkey specifies either a leave node or a structured
 * node. In case the cfgkey is a structured config entry the callback
 * will fire if any of the cfgkey's child entries have been changed.
 *
 * The callback will not tell which config has been changed
 * so the listener is responsible for keeping in mind what he's
 * registered for. There is also no way to figure out the old
 * value before the change happened. In case this is important
 * for the listener, he must keep that config value in it's state,
 * somehow.
 *
 * Generally it is intended to register a particular callback not
 * only for a single option but for a bunch of related config options.
 * Once the callback gets fired, the listener is expected to figure
 * out whether and how a particular option has changed.
 *
 * The callback gets directly fired after a set-operation in case
 * of executing the set-operation without transaction context or after
 * the commit-operation in case of an active transaction
 */
int pp_cfg_add_change_listener(pp_cfg_chg_cb cfg_chg_cb,
			       const char* cfgkey, ...) {
    va_list ap;
    int ret;

    va_start(ap, cfgkey);
    ret = cfg_add_change_listener(CFGL_TYPE_WITHIN_TX, cfg_chg_cb, cfgkey, ap);
    va_end(ap);
    
    return ret;
}

int pp_cfg_add_post_tx_change_listener(pp_cfg_chg_cb cfg_chg_cb,
			               const char* cfgkey, ...) {
    va_list ap;
    int ret;

    va_start(ap, cfgkey);
    ret = cfg_add_change_listener(CFGL_TYPE_POST_TX, cfg_chg_cb, cfgkey, ap);
    va_end(ap);
    
    return ret;
}

static int cfg_add_change_listener(cfgl_listener_type_t type,
                                   pp_cfg_chg_cb cfg_chg_cb,
		                   const char* cfgkey, va_list ap) {
    unsigned int comps_size, comp_idx;
    vector_t* comps;
    key_comp_t* comp;
    cfgl_entry_t* cle = cfg_listeners, *ncle;
    cfgl_listener_t tmple;
    
    comps = cfg_key_expand(cfgkey, ap);
    comps_size = vector_size(comps);
    assert(comps_size > 0);
    
#if defined(PP_CFG_DEBUG) || !defined(NDEBUG)
    {
        char cfg_key[255];

        cfg_get_qid_from_key_comps(comps, cfg_key, 255);
        
        if(!pp_cd_resolve_symbol(pp_cfg_get_cd(), cfg_key,
                                 PP_CD_RESOLVE_VECTOR)) {
            pp_log("Error resolving config key '%s'\n", cfg_key);
            assert(0);
/*
        } else {
            pp_log("Registering change listener for config key '%s'\n",
                   cfg_key);
*/
        }
    }
#endif /* PP_CFG_DEBUG || !NDEBUG*/

    MUTEX_LOCK(&cfg_listeners_mtx);
    
    // forthfully resolve path, i.e. create non existing components
    for (comp_idx = 0; comp_idx < comps_size; ++comp_idx) {

	comp = (key_comp_t*)vector_get2(comps, comp_idx);
	assert(strlen(comp->name) > 0);

	if ((NULL == cle->dict && (cle->dict = pp_dict_str_new())) || 
	     NULL == (ncle = (cfgl_entry_t*)pp_dict_search(cle->dict,
							   comp->name))) {
	    ncle = cfgl_entry_create();
	    pp_dict_insert(cle->dict, comp->name, ncle,
			   (pp_dict_del_func)cfgl_entry_destroy, 0);
	}
	cle = ncle;
    }

    // and add listener to the list of listeners
    if (NULL == cle->listeners)
	cle->listeners = vector_new2(NULL, 2, sizeof(cfgl_listener_t), NULL);
    tmple.cfg_chg_cb = cfg_chg_cb;
    tmple.uid = cfg_get_listener_uid();
    tmple.type = type;
    vector_add2(cle->listeners, &tmple);

    MUTEX_UNLOCK(&cfg_listeners_mtx);
    
    pp_cfg_key_comps_destroy(comps);
    return PP_SUC;
}

/**
 * Remove a notification callback. Normaly the same config key and
 * callback pointer as provided to the add function must be given to remove
 * exactly one entry
 * 
 * In case the config key is given only all listeners for this key
 * will be removed.
 *
 * In case a callback pointer is given only, this pointer will be
 * removed for all keys it is possibly registered
 */
int pp_cfg_rem_change_listener(pp_cfg_chg_cb cfg_chg_cb,
			       const char* cfgkey, ...) {
    va_list ap;
    int ret;
    
    if (cfgkey != NULL) {
	va_start(ap, cfgkey);
	ret = cfg_rem_change_listener_at_key(cfg_chg_cb, cfgkey, ap);
	va_end(ap);
    } else {
	ret = cfg_rem_change_listener_completely(cfg_chg_cb);
    }
    return ret;
}

static pp_cfg_chg_ctx_t* 
cfg_chg_cb_ctx_create(char **retstr, pp_cfg_chg_kind_t kind, 
                      vector_t *key_comp_vec, pp_cfg_tx_t* tx,
                      unsigned int mode) {
    pp_cfg_chg_ctx_t *ctx;
    
    ctx = (pp_cfg_chg_ctx_t*)malloc(sizeof(pp_cfg_chg_ctx_t));
    ctx->retstr = retstr;
    ctx->chg_kind = kind;
    ctx->key_comp_vec = key_comp_vec;
    ctx->tx = tx;
    ctx->mode = mode;
    
    return ctx;
}

static void cfg_chg_cb_ctx_destroy(void *v) {
    pp_cfg_chg_ctx_t *ctx = (pp_cfg_chg_ctx_t*)v;
    
    if(ctx->mode > 0) { // (1)
        if(ctx->mode & 1)
            free(ctx->retstr);
        ctx->mode = ctx->mode >> 1;
    }
    if(ctx->mode > 0) { // (2)
        if(ctx->mode & 1)
            vector_delete(ctx->key_comp_vec);
        ctx->mode = ctx->mode >> 1;
    }
    /* never destroy tx, it will take care about itself... */
    free(ctx);
}
