/**
 * Access Control Lists
 * This is a wrapper around the cfg-library providing better
 * interfaces for ACLs.
 *
 * Author: Thomas Breitfeld <thomas@peppercon.de>
 * Copyright 2003 Peppercon AG
 */

#ifndef _PP_ACL_H
#define _PP_ACL_H

#include <pthread.h>
#include <pp/base.h>
#include <pp/vector.h>

inline int pp_acl_errno_base(void);

#define PP_ACL_ERR_NO_ERROR                  0
#define PP_ACL_ERR_INTERNAL_ERROR            (pp_acl_errno_base() + 0)
#define PP_ACL_ERR_OBJECT_DOESNT_EXIST       (pp_acl_errno_base() + 1)
#define PP_ACL_ERR_OBJECT_PERM_DOESNT_EXIST  (pp_acl_errno_base() + 2)
#define PP_ACL_ERR_ACL_DOESNT_EXIST          (pp_acl_errno_base() + 3)

#define PP_ACL_MAX_PERM_LEN 32
#define PP_ACL_MAX_OBJ_LEN  64

/*
 * acl objects are the objects and the global kinda database
 * of things that can be managed in a acl for an particular user
 * i.e. each acl will point to entries in the acl_objects hash
 */
typedef struct pp_acl_objects_s {
    pp_hash_t* objects;
    pthread_mutex_t mtx;
} pp_acl_objects_t;

typedef struct pp_acl_object_entry_s {
    char * name;
    char * longname;
    /* TODO: no need for a hash as RAASIP defines permission to be unique */ 
    pp_hash_t * perms;
    char *type;
    char *ro_flag;
} pp_acl_object_entry_t;

typedef struct pp_acl_perm_entry_s {
    char* name;
} pp_acl_perm_entry_t;

typedef int (*pp_acl_custom_compare_cb_t)(const pp_acl_object_entry_t* x, 
                                          const pp_acl_object_entry_t* y);

extern const char* pp_acl_raasip_yes_str;
extern const char* pp_acl_raasip_no_str;
extern const char* pp_acl_raasip_none_str;
extern const char* pp_acl_raasip_view_str;
extern const char* pp_acl_raasip_control_str;
extern const char* pp_acl_raasip_deny_str;
extern const char* pp_acl_raasip_readonly_str;
extern const char* pp_acl_raasip_readwrite_str;

int  pp_acl_init(void);
void pp_acl_cleanup(void);

/*
 * acl objects define the general possibility what an ACL may contain
 */
pp_acl_objects_t* pp_acl_objects_create(void);
void pp_acl_objects_free(void* _objects);


vector_t* pp_acl_objects_get_all(char* filter_incl, char* filter_excl);
vector_t* pp_acl_objects_get_all_custom_compare(char* filter_incl, char* filter_excl,
                                          pp_acl_custom_compare_cb_t custom_compare);

char* pp_acl_objects_get_long_name(const char * name);
char* pp_acl_objects_get_read_only_flag(const char * name);

vector_t* pp_acl_objects_get_possible_perms(const char* name,
					    const char* prefix);

int pp_acl_objects_has_perm(const char* objname, const char* permname);

char* pp_acl_lookup_real_permission_name(const char * code_permission_name);

/**
 * Gets the RAASIP type of permission object (i.e. "admin", "user" or "port")
 *
 * @param type                  return value if lookup succeeded
 * @param objname               permission object to get type for
 *
 * @return                      PP_SUC on success, PP_ERR and errno otherwhise
 */
int pp_acl_objects_get_type(char **type, const char* objname);


/*
 * acl is a collection of entries pointing back to acl_objects
 */
typedef struct {
    pp_acl_objects_t*   objects;
    pp_hash_t*          acl;
    void*               parent;
    pp_mutex_t          mtx;
} pp_acl_t;

typedef struct {
    pp_acl_object_entry_t* object;
    pp_acl_perm_entry_t*   perm;
    int is_negative;
} pp_acl_entry_t;

void pp_acl_register_errnos(void);
void pp_acl_unregister_errnos(void);

pp_acl_t* pp_acl_create(void);

int pp_acl_add_entry(pp_acl_t* acl, const char* object, const char* perm,
		     const int negative);

int pp_acl_del_entry(pp_acl_t* acl, const char* object, const char* perm);

int pp_acl_del_all_entries(pp_acl_t* acl, const char* object);

pp_acl_t* pp_acl_clone(pp_acl_t* acl);

void pp_acl_destroy(pp_acl_t* acl);

int pp_acl_union(pp_acl_t* acl, pp_acl_t* other, int optimistic);

pp_hash_t* pp_acl_sort_perms_by_obj(pp_acl_t* acl);

#ifndef NDEBUG
void pp_acl_print(pp_acl_t* acl);
#endif /* !NDEBUG */

const char* pp_acl_get_error_string(const int error);

int pp_acl_store_to_cfg(pp_acl_t* acl, int gid);

int pp_acl_accumulate_all_permissions(pp_acl_t* usr_acl, vector_t* grp_acls,
				      const char* object, const char* perm);

int pp_acl_accumulate_all_permissions_biased(pp_acl_t* usr_acl,
					     vector_t* grp_acls,
					     const char* object,
					     const char* perm,
					     int pos_usr_bias,
					     int neg_usr_bias,
					     int pos_grp_bias,
					     int neg_grp_bias);

void pp_acl_accumulate_permission(const pp_acl_t* acl,
				  const char* object, const char* perm,
				  int* pos, int* neg);
                                  
#endif /* _PP_ACL_H */
