/*
 * The basic config library providing
 * the LocalProfileManager controlling the different kinds of
 * Local Profiles and offering the according access APIs
 *
 * (c) 2004 Peppercon AG
 * tbr@peppercon.de
 */

#ifdef __cplusplus
extern "C" {
#endif

#include <pp/base.h>
#include <pp/vector.h>
#include <pp/profile.h>
#include <pp/cd.h>
#include <pp/dict.h>

#ifndef __PP_CFG_H__
#define __PP_CFG_H__

/*
 * some constants
 */
extern const char* pp_cfg_comp_deli;
extern const char* pp_cfg_vect_elems_comp;
extern const char* pp_cfg_vect_size_comp;
extern const char* pp_cfg_choice_select_comp;
extern const char* pp_cfg_boolean_true;
extern const char* pp_cfg_boolean_false;
extern const char* pp_cfg_comp_wc_single;


/* cfg structures */
/******************/
typedef enum {
    CFG_STR,
    CFG_DICT
} cfg_entry_type_t;

typedef union {
    char* string;
    pp_dict_t* dict;
} cfg_entry_value_t;

typedef struct {
    cfg_entry_type_t type;
    cfg_entry_value_t value;
} cfg_entry_t;

/*
 * library management
 * ======================
 */

extern const char* PP_CD_FNAME_DEFAULT;

extern const char* PP_CD_OEM_SKEL_FNAME_DEFAULT;

int pp_cfg_init(const char* cfg_cd_filename, const char* cfg_cd_oem_filename);

void pp_cfg_cleanup(void);

int pp_cfg_is_initialized(void);

void pp_cfg_destroy(void);

/**
 * Returns errno base for um errors.
 * 
 * @return                      errno base for um errors
 */
int pp_cfg_errno_base(void);

// known errors, not complete yet
// Attention: This must match the pp_um_errors array in cfg.c!
#define PP_CFG_ERR_NO_ERROR             0
#define PP_CFG_ERR_INTERNAL_ERROR       (pp_cfg_errno_base() +  0)
#define PP_CFG_ERR_INVALID_KEY          (pp_cfg_errno_base() +  1)
#define PP_CFG_ERR_INVALID_VALUE        (pp_cfg_errno_base() +  2)
//#define PP_CFG_ERR_(pp_cfg_errno_base() + )

/**
 * returns the active configuration description for
 * the initialiced cfg lib
 */
pp_cd_as_cd_t* pp_cfg_get_cd(void);

/*
 * most basic (and in some wide semantic sense backward compatible)
 * config operations API
 * These funktions will operate on PP_PROFILE_LOCAL
 * =================================================================
 */

/**
 * Get the string value of a config key
 *	NOTE: The get*_nodflt routines do not touch value in case of an error.
 *	      The other get routines always set value - even in case of an error.
 * @param value parameter for value, if *value == NULL, key was not found
 * @param prof  profile source, might be NULL if not requested
 * @param key   the config path to a specific property, inclusive
 *              format flags for the vector index
 *              %d - int
 *              %s - string
 * @arglist     paramters for format flags
 * @return      PP_ERR for error or
 *              PP_SUC for success
 * @example     if (err = pp_get_config(*buttonkey,
 *                                    "User[%s].KVMPort[%u].ButtonKey",
 *                                    "ronald", 4)) {
 *                  pp_log("pp_get_config failed: %s", pp_error_string(err));
 *                  return 0;
 *              }
 */
int pp_cfg_get_with_prof(char** value, int* prof, const char* key, ...) __attribute__ ((format (printf, 3, 4)));
int pp_cfg_get_with_prof_nodflt(char** value, int* prof, const char* key, ...) __attribute__ ((format (printf, 3, 4)));
int pp_cfg_get_nodflt(char** value, const char* key, ...) __attribute__ ((format (printf, 2, 3)));
int pp_cfg_get(char** value, const char* key, ...) __attribute__ ((format (printf, 2, 3)));
#ifdef PP_FEAT_RPCCFG
int pp_cfg_get_with_serialized_key(int force, char** value, pp_profile_type_t prof_type, const void * keybuf, size_t keybuf_size);
#endif

/**
 * Set the string value of a config key.
 * Works on PP_PROFILE_LOCAL.
 * @see pp_cfg_get
 */
int pp_cfg_set(const char* value, const char* key, ...) __attribute__ ((format (printf, 2, 3)));
#ifdef PP_FEAT_RPCCFG
int pp_cfg_set_with_serialized_key(const char* value, pp_profile_type_t prof_type, const void * keybuf, size_t keybuf_size);
#endif

/**
 * Get the short value of a config key
 * @see pp_cfg_get
 */
int pp_cfg_get_short_nodflt(short* number, const char* key, ...) __attribute__ ((format (printf, 2, 3)));
int pp_cfg_get_short(short* number, const char* key, ...) __attribute__ ((format (printf, 2, 3)));

/**
 * Set the short value of a config key
 * Works on PP_PROFILE_LOCAL.
 * @see pp_cfg_get
 */
int pp_cfg_set_short(const short number, const char* key, ...) __attribute__ ((format (printf, 2, 3)));

/**
 * Get the unsigned short value of a config key
 * @see pp_cfg_get
 */
int pp_cfg_get_ushort_nodflt(u_short* number, const char* key, ...) __attribute__ ((format (printf, 2, 3)));
int pp_cfg_get_ushort(u_short* number, const char* key, ...) __attribute__ ((format (printf, 2, 3)));

/**
 * Set the unsigned short value of a config key
 * Works on PP_PROFILE_LOCAL.
 * @see pp_cfg_get
 */
int pp_cfg_set_ushort(const u_short number, const char* key, ...) __attribute__ ((format (printf, 2, 3)));

/**
 * Get the integer value of a config key
 * @see pp_cfg_get
 */
int pp_cfg_get_int_nodflt(int* number, const char* key, ...) __attribute__ ((format (printf, 2, 3)));
int pp_cfg_get_int(int* number, const char* key, ...) __attribute__ ((format (printf, 2, 3)));

/**
 * Set the integer value of a config key
 * Works on PP_PROFILE_LOCAL.
 * @see pp_cfg_get
 */
int pp_cfg_set_int(const int number, const char* key, ...) __attribute__ ((format (printf, 2, 3)));

/**
 * Get the unsigned integer value of a config key
 * @see pp_cfg_get
 */
int pp_cfg_get_uint_nodflt(u_int* number, const char* key, ...) __attribute__ ((format (printf, 2, 3)));
int pp_cfg_get_uint(u_int* number, const char* key, ...) __attribute__ ((format (printf, 2, 3)));

/**
 * Set the unsigned integer value of a config key
 * Works on PP_PROFILE_LOCAL.
 * @see pp_cfg_get
 */
int pp_cfg_set_uint(const u_int number, const char* key, ...) __attribute__ ((format (printf, 2, 3)));

/**
 * Get the long value of a config key
 * @see pp_cfg_get
 */
#define pp_cfg_get_long_nodflt(number, key, args...) pp_cfg_get_int_nodflt((int*)(void*)(number), key, ##args)
#define pp_cfg_get_long(number, key, args...) pp_cfg_get_int((int*)(void*)number, key, ##args)

/**
 * Set the long value of a config key
 * Works on PP_PROFILE_LOCAL.
 * @see pp_cfg_get
 */
#define pp_cfg_set_long(number, key, args...) pp_cfg_set_int((const int)(number), key, ##args)

/**
 * Get the unsigned long value of a config key
 * @see pp_cfg_get
 */
#define pp_cfg_get_ulong_nodflt(number, key, args...) pp_cfg_get_uint_nodflt((u_int*)(void*)(number), key, ##args)
#define pp_cfg_get_ulong(number, key, args...) pp_cfg_get_uint((u_int*)(void*)number, key, ##args)

/**
 * Set the unsigned long value of a config key
 * Works on PP_PROFILE_LOCAL.
 * @see pp_cfg_get
 */
#define pp_cfg_set_ulong(number, key, args...) pp_cfg_set_uint((const u_int)(number), key, ##args)

/**
 * Get the float value of a config key
 * @see pp_cfg_get
 */
int pp_cfg_get_float_nodflt(float* number, const char* key, ...) __attribute__ ((format (printf, 2, 3)));
int pp_cfg_get_float(float* number, const char* key, ...) __attribute__ ((format (printf, 2, 3)));

/**
 * Set the float value of a config key
 * Works on PP_PROFILE_LOCAL.
 * @see pp_cfg_get
 */
int pp_cfg_set_float(const float number, const char* key, ...) __attribute__ ((format (printf, 2, 3)));

/**
 * Get the boolean value of a config key
 * @param is_enabled will be set to true, if the property value
 *        is "yes", otherwise, even if key can't be found, it will
 *        be set to false
 * @see pp_cfg_get
 */
int pp_cfg_is_enabled_nodflt(int* is_enabled, const char* key, ...) __attribute__ ((format (printf, 2, 3)));
int pp_cfg_is_enabled(int* is_enabled, const char* key, ...) __attribute__ ((format (printf, 2, 3)));

/**
 * Set the boolean value of a config key
 * Works on PP_PROFILE_LOCAL.
 * @see pp_cfg_is_enabled
 */
int pp_cfg_set_enabled(int is_enabled, const char* key, ...) __attribute__ ((format (printf, 2, 3)));

/**
 * Get a binary value from a base64 encoded config key
 */
int pp_cfg_get_binary(char **value, size_t *size, const char* key, ...) __attribute__ ((format (printf, 3, 4)));

/**
 * Set a binary value in a config key encoded in base64
 */
int pp_cfg_set_binary(const char *value, size_t size, const char* key, ...) __attribute__ ((format (printf, 3, 4)));

/**
 * Get the acl of a config key
 * @see pp_cfg_get
 */
int pp_cfg_get_acl(pp_acl_t **acl, const char* key, ...) __attribute__ ((format (printf, 2, 3)));

/**
 * Check whether the effective config value equals the default
 * @see pp_cfg_get
 */
int pp_cfg_is_default(const char* key, ...) __attribute__ ((format (printf, 1, 2)));

/**
 * Removes a config key from the configuration
 * Attention: this function can be used to delete whole sub-trees,
 * not just leaf elements, so be careful!
 * @see pp_cfg_get
 */
int pp_cfg_remove(const char* key, ...) __attribute__ ((format (printf, 1, 2)));
/**
 * renames the last name component of a config key
 * (could be extended to rename one path to another, however
 * currently not needed, maybe later)
 * @see pp_cfg_get
 */
int pp_cfg_rename(const char* newkey, const char* key, ...) __attribute__ ((format (printf, 2, 3)));

/**
 * copies the last name component of a config key
 * @see pp_cfg_rename
 */
int pp_cfg_copy(const char* newkey, const char* key, ...) __attribute__ ((format (printf, 2, 3)));
int pp_cfg_copy_at_layer(pp_profile_type_t type, const char* newkey, 
                         const char* key, ...) 
                         __attribute__ ((format (printf, 3, 4)));

int pp_cfg_copy_entry(const pp_profile_t* prof, pp_dict_t *d, 
                      const char *key, const cfg_entry_t *e, int overwrite);

/*
 * some more vector utility functions
 * ==================================
 */

/**
 * Compacts a vector in the sense, that the current size
 * of the vector will be shrinked to the smallest possible size
 * still holding all existing elements, i.e. not existing indexes
 * will be removed from the vector. The length member of the vector
 * will be set accordingly
 */
//int pp_cfg_compact_vector(const char* vectorkey, ...);

/*
 * layered profile API
 * these functions are identical to the ones above, accept that they
 * request the Profile Manager to operate on specifically named
 * and predefined profile
 * =================================================================
 */

int pp_cfg_get_at_layer_with_prof_nodflt(pp_profile_type_t type, char** value,
	   		                 int* prof, const char* key, ...)
                                        __attribute__ ((format (printf, 4, 5)));
int pp_cfg_get_at_layer_nodflt(pp_profile_type_t type, char** value,
			       const char* key, ...) __attribute__ ((format (printf, 3, 4)));
int pp_cfg_get_at_layer(pp_profile_type_t type, char** value,
			const char* key, ...) __attribute__ ((format (printf, 3, 4)));
int pp_cfg_set_at_layer(pp_profile_type_t type, const char* value,
			const char* key, ...) __attribute__ ((format (printf, 3, 4)));
int pp_cfg_get_int_at_layer_nodflt(pp_profile_type_t type, int* number,
				   const char* key, ...) __attribute__ ((format (printf, 3, 4)));
int pp_cfg_get_int_at_layer(pp_profile_type_t type, int* number,
			    const char* key, ...) __attribute__ ((format (printf, 3, 4)));
int pp_cfg_set_int_at_layer(pp_profile_type_t type, int number,
			    const char* key, ...) __attribute__ ((format (printf, 3, 4)));
int pp_cfg_is_enabled_at_layer_nodflt(pp_profile_type_t type, int* is_enabled,
				      const char* key, ...) __attribute__ ((format (printf, 3, 4)));
int pp_cfg_is_enabled_at_layer(pp_profile_type_t type, int* is_enabled,
			       const char* key, ...) __attribute__ ((format (printf, 3, 4)));
int pp_cfg_set_enabled_at_layer(pp_profile_type_t type, int is_enabled,
				const char* key, ...) __attribute__ ((format (printf, 3, 4)));
int pp_cfg_remove_at_layer(pp_profile_type_t type, const char* key, ...) __attribute__ ((format (printf, 2, 3)));
int pp_cfg_rename_at_layer(pp_profile_type_t type, const char* newkey,
			   const char* key, ...) __attribute__ ((format (printf, 3, 4)));
//int pp_cfg_compact_vector_at_layer(pp_profile_type_t type,
//				   const char* vectorkey, ...);

/*
 * Iterator API
 * =======================================================
 */
typedef enum {
    FLAT,       // iterator looping over one dict
    X_DICT,     // iterator looping recursive over a dict and its child dicts
} cfg_iter_type_t;

typedef struct pp_cfg_iter_s pp_cfg_iter_t;

int pp_cfg_iter_create(pp_cfg_iter_t** iter, const char* key, ...) __attribute__ ((format (printf, 2, 3)));

int pp_cfg_iter_create2(pp_cfg_iter_t** iter, pp_cfg_iter_t* parent, 
                        const char* key, ...) __attribute__ ((format (printf, 3, 4)));

int pp_cfg_iter_create_at_layer(pp_cfg_iter_t** iter, pp_profile_type_t type,
				const char* key, ...) __attribute__ ((format (printf, 3, 4)));

int pp_cfg_iter_create_x_dict(pp_cfg_iter_t** iter, const char* key, ...) __attribute__ ((format (printf, 2, 3)));
                                
int pp_cfg_iter_next(pp_cfg_iter_t* iter, const char** key, const char** value);

void pp_cfg_iter_destroy(pp_cfg_iter_t* iter);

int pp_cfg_iter_get(pp_cfg_iter_t* iter, const char** value, const char* key, ...) __attribute__ ((format (printf, 3, 4)));

int pp_cfg_iter_get_prof_id(const char* key, ...) __attribute__ ((format (printf, 1, 2)));

    
/*
 * Persistence
 * ======================
 */

typedef enum {
    DONT_FLUSH,
    DO_FLUSH
} pp_cfg_flush_t;

/*
 * flush changes back to persitent storage
 * pp_cfg_flush_t desides whehter it is just saved or also flashed
 */
int pp_cfg_save(pp_cfg_flush_t flush);
int pp_cfg_save_layer(pp_profile_type_t type, pp_cfg_flush_t flush);

/*
 * Config change notification API
 * =================================================================
 */

typedef enum {
    PP_CFG_CHG_UNDEF,
    PP_CFG_CHG_MOD,
    PP_CFG_CHG_ADD,
    PP_CFG_CHG_REM,
    PP_CFG_CHG_NOOP
} pp_cfg_chg_kind_t;

/**
 * type of transaction context
 */
typedef struct pp_cfg_tx_s pp_cfg_tx_t;

/* pp_cfg_chg_ctx_t mode: which pointers to free on cfg_chg_cb_destroy? */
typedef struct {
    unsigned int        mode;
    char**              retstr;         // (1)
    pp_cfg_chg_kind_t   chg_kind;
    vector_t*           key_comp_vec;   // (2)
    pp_cfg_tx_t*        tx;
} pp_cfg_chg_ctx_t;

/**
 * type of the config change callback function
 */
typedef int (*pp_cfg_chg_cb)(pp_cfg_chg_ctx_t *ctx);

/**
 * Add a notification callback for a certain property.
 * The cfgkey specifies either a leave node or a structured
 * node. In case the cfgkey is a structured config entry the callback
 * will fire if any of the cfgkey's child entries have been changed.
 *
 * The callback will not tell which config has been changed
 * so the listener is responsible for keeping in mind what he's
 * registered for. There is also no way to figure out the old
 * value before the change happened. In case this is important
 * for the listener, he must keep that config value in it's state,
 * somehow.
 *
 * Generally it is intended to register a particular callback not
 * only for a single option but for a bunch of related config options.
 * Once the callback gets fired, the listener is expected to figure
 * out whether and how a particular option has changed.
 *
 * The callback gets directly fired after a set-operation in case
 * of executing the set-operation without transaction context or after
 * the commit-operation in case of an active transaction
 */
int pp_cfg_add_change_listener(pp_cfg_chg_cb cfg_chg_db,
			       const char* cfgkey, ...);
int pp_cfg_add_post_tx_change_listener(pp_cfg_chg_cb cfg_chg_db,
			               const char* cfgkey, ...) __attribute__ ((format (printf, 2, 3)));

/**
 * Remove a notification callback. Normaly the same config key and
 * callback pointer as provided to the add function must be given to remove
 * exactly one entry
 * 
 * In case the config key is given only all listeners for this key
 * will be removed.
 *
 * In case a callback pointer is given only, this pointer will be
 * removed for all keys it is possibly registered
 */
int pp_cfg_rem_change_listener(pp_cfg_chg_cb cfg_chg_db,
			       const char* cfgkey, ...) __attribute__ ((format (printf, 2, 3)));

/*
 * Config transaction API
 * =================================================================
 */

/**
 * Thread specific transaction context key
 */
extern pthread_key_t pp_cfg_tx_thread_ctx_key;

/**
 * Start a config transaction.
 * @param assign_to_thread call pp_cfg_tx_set_thread_ctx for created tx
 * @return the transaction context
 */
pp_cfg_tx_t* pp_cfg_tx_begin(int assign_to_thread);

/**
 * Commits a config transaction
 *
 * The transaction context will be destroyed.
 *
 * in case there are listeners registered for any of the config keys
 * that was set during the transaction, their callback will be fired
 * exactly once per commit
 */
int pp_cfg_tx_commit(pp_cfg_tx_t* tx);
int pp_cfg_tx_commit_core(pp_cfg_tx_t* tx, pp_cfg_flush_t flush,
                          int flush_lock, int threaded, char **retstr);

/**
 * Discards all changes since the begin of this transaction,
 * i.e. they will not be written through to the profile
 *
 * The transaction context will be destroyed.
 */
void pp_cfg_tx_rollback(void* tx);

/**
 * Sets active tx in thread context
 * @param tx transaction to set
 */
void pp_cfg_tx_set_thread_ctx(pp_cfg_tx_t* tx);

/**
 * Gets active tx from thread context
 * @return current transaction set in thread context
 */
pp_cfg_tx_t* pp_cfg_tx_get_thread_ctx(void);

/**
 * Releases active tx in thread context
 * @param tx transaction to be released, NULL for any tx set
 */
void pp_cfg_tx_release_thread_ctx(pp_cfg_tx_t* tx);

/*
 * cfg handler API,
 * i.e. funktion pointers that are registered behind a certain cfg key
 * ===================================================================
 */

typedef enum {
    PP_CFG_HANDLER_ALL,
    PP_CFG_HANDLER_LASTCOMP,
} pp_cfg_handler_type_t;

/**
 * register key handler,
 * may be used to implement 'virtual keys', i.e. keys whose
 * value will be dynamically generated
 */
void pp_cfg_register_handler(pp_cfg_handler_type_t type, const char* key,
			     int (*get)(pp_profile_t*, vector_t*),
			     pp_cfg_chg_kind_t (*set)(const char*,
						      pp_profile_t*,
						      const vector_t*));

/**
 * checks wether a given key components vector represents a user key,
 * else returns PP_ERR
 * the user id is set if specified
 */
int pp_cfg_is_user_key(int *uid, const vector_t *key_comps);

/**
 * checks wether a given key components vector represents a group key,
 * else returns PP_ERR
 * the group id is set if specified
 */
int pp_cfg_is_group_key(int *gid, const vector_t *key_comps);

cfg_entry_t* pp_cfg_key_resolve_exact(pp_dict_t** cfg_dict, pp_profile_t* prof, 
                                      vector_t* comps);
                                      
cfg_entry_t* pp_cfg_key_resolve_force(pp_dict_t** cfg_dict, pp_profile_t* prof,
                                      const vector_t* comps);

vector_t* pp_cfg_key_expand(const char* key, ...) __attribute__ ((format (printf, 1, 2)));

void pp_cfg_key_comps_destroy(vector_t* comps);

/**
 * compares config keys in key_comp format
 * returns 1 if key_comps_1 and key_comps_2 are identical, 0 otherwhise
 */
int pp_cfg_key_compare(const vector_t *key_comps_1, 
                       const vector_t *key_comps_2);

const char * pp_cfg_get_key_comp_name_at_idx(const vector_t* comps, u_int idx);

char* pp_cfg_get_key_from_key_comps(const vector_t *key_comps, 
                                    char *key, int length);

/**
 * returns the profile type id of the profile, the cfg_key is fetched from
 * if cfg_key is not stored in any accessible profile, PP_ERR is returned
 */
int pp_cfg_get_prof_type(const char* key, ...) __attribute__ ((format (printf, 1, 2)));


int pp_cfg_ldap_prof_is_active(int uid);

char* pp_cfg_value_shellesc(const char * s);

/*
 * transforms the key so that it can become a shell variable
 * .[] => _
 * if 'noidx' is true, all indices will be deleted.
 */
char* pp_cfg_key_transform(const char* k, int noidx);

/*
 * indicates is local profile is "nearly" empty. Excludes from this rule are
 * fixed values, e.g. MAC address or serial number
 * returns PP_SUC on emptly local layer, PP_ERR in all other cases
 */
int pp_cfg_local_layer_is_empty(void);

/**
 * high level validation API
 * do not use that too often, as it contains a lot of (unnecessary) string
 * operations. however provides check of generic config key against config
 * description.
 * returns PP_SUC on validation success, PP_ERR otherwhise.
 */
int pp_cfg_validate(const char* value, const char* key, ...);

/**
 * get the profile where a config key is stored
 * NOTE: currently, this is parsed from types_n_config.cdl and only
 * determines between LOCAL and SYSTEM profile!
 * returns PP_SUC on validation success, PP_ERR otherwhise.
 */
int pp_cfg_get_profile_from_key(pp_profile_type_t * profile, const char* key, ...);

/*
 * validates and sets a key (at a given layer)
 * on PP_ERR, key is not set!
 */
//int pp_cfg_validate_and_set(const char *value, const char *key, ...);
//int pp_cfg_validate_and_set_int(int number, const char *key, ...);

/**
 * gets permission property assigned to a config key
 *
 * @see pp_cfg_get_prop
 */
int pp_cfg_get_perm(const char **perm, const char *key, ...)
    __attribute__ ((format (printf, 2, 3)));
 
/**
 * gets permission property assigned to a config key
 *
 * @param perm          permission identifier, set on PP_SUC
 * @param key           the config key
 *
 * @return              PP_ERR on error, PP_SUC and perm else
 */
int pp_cfg_get_prop(const char **value, const char *prop, const char *key, ...)
    __attribute__ ((format (printf, 3, 4)));


#ifndef NDEBUG
/**
 * dumps a profile to a file
 *
 * @param cdl           name of config description file to get profile from
 * @param file          name of file to dump into
 *
 * @return              PP_ERR on error, PP_SUC else
 */
void pp_cfg_dump_profile(const char *cdl, const char *file);
#endif /* !NDEBUG */
 
#endif /* __PP_CFG_H__ */

#ifdef __cplusplus
}
#endif

