/**
 * A profile is a Configuration Description instance,
 * i.e. it contains values of configuration keys
 *
 * (c) 2004 Peppercon AG
 * tbr@peppercon.de>
 */
#ifndef __PP_PROFILE_H__
#define __PP_PROFILE_H__

#include <pp/base.h>
#include <pp/dict.h>
#include <pp/acl.h>

#define noPP_CFG_DEBUG

#define noPP_PROFILE_LOCAL_DEF_ENABLED
#define noPP_PROFILE_LOCAL_TAG_ENABLED
 
/*
 * the profile stack
 * assert consistency of pp_profile_type_t and pp_profile_type_str
 */ 
enum {
    PP_PROFILE_NONE,        // invalid type, used to indicate errors
    PP_PROFILE_LOCAL_CODE,  // hard coded default prof, not writeable
    PP_PROFILE_SYSTEM,      // system configuration, will not be reset
//    PP_PROFILE_LOCAL_DEF,   // default profile in config fs
    PP_PROFILE_LOCAL_OEM,   // OEM profile in config fs
    PP_PROFILE_LOCAL,       // standard profile in config fs
    PP_PROFILE_LDAP_DEV,    // ldap device profile
    PP_PROFILE_LDAP_USER,   // ldap user profile
    PP_PROFILE_HEAD,        // highest prioritized profile
}; // was pp_profile_type_t;
               
typedef int pp_profile_type_t;

extern const char* pp_profile_type_str[];

/*
 * some special profiles
 */
#define PP_PROFILE_COUNT       (PP_PROFILE_HEAD + 1)
#define PP_PROFILE_DEFAULTS    PP_PROFILE_STACK((PP_PROFILE_LOCAL - 1))
#define PP_PROFILE_EFFECTIVE   PP_PROFILE_STACK(PP_PROFILE_HEAD)
#define PP_PROFILE_LOCAL_HEAD  (PP_PROFILE_LDAP_DEV - 1)

/*
 * profile handling macros
 */
#define PP_PROFILE_STACK(__type__)      ((__type__) | (1 << 16))
#define PP_REAL_PROFILE(__type__)       ((__type__) & 0xffff)
#define PP_PROFILE_IS_STACK(__type__)   (((__type__) >> 16) & 0x1)

/*
 * the profile struct
 */
typedef struct {
    int                 id;
    pp_profile_type_t   type;
    char*               name;
    pp_dict_t*          configs;
    int                 ref_cnt;
    pp_mallocator_t*    allocator;
    pp_mutex_t          mtx;
    pp_mutex_t          ref_cnt_mtx;
    pp_hash_i_t*        acl;
    pp_acl_t*           ldap_acl;
} pp_profile_t;

/*
 * create a new profile and initialice content to be empty
 */
pp_profile_t* pp_profile_create(pp_mallocator_t* a, pp_profile_type_t type,
                                const char* name);

/*
 * Reference counting for profile structure
 */
pp_profile_t* pp_profile_duplicate(pp_profile_t* prof);

void pp_profile_release(void * _p);

/*
 * Low level profile setting
 */

void pp_profile_set_core(const char* value, pp_profile_t* prof,
	                 const vector_t *key_comps, int notify);

void pp_profile_set(const char* value, pp_profile_t* p,
		    const char* key, va_list ap);

void pp_profile_set_int(const int number, pp_profile_t* p,
			const char* key, va_list ap);

void pp_profile_set_bool(const int boolval,  pp_profile_t* p,
			 const char* key, va_list ap);

void pp_profile_set_float(const float number,  pp_profile_t* p,
			  const char* key, va_list ap);

int pp_profile_union(pp_profile_t **prof, vector_t *gprofs, int forward);

int pp_profile_get_acl(pp_acl_t **acl, pp_profile_t *prof, int gid);

void pp_profile_set_acl(pp_acl_t *acl, pp_profile_t *prof, int gid);

void pp_profile_set_ldap_acl(pp_profile_t *prof,pp_acl_t *acl);

void pp_profile_release_acl(void *acl);

int pp_profile_invalidate_acl_cache(pp_profile_t *prof);

#ifdef PP_CFG_DEBUG
void pp_profile_print(const pp_profile_t* prof);

int pp_profile_validate_type(pp_profile_type_t type);
#endif /* PP_CFG_DEBUG */

#endif
