#include <pp/cim.h>
#include <pp/cimTypes.h>
#include "cim_common.h"
#include "instance.h"

void pp_cim_data_null(pp_cim_datatype_t type, int array, pp_cim_data_t *data)
{
    if (data->null)
        return;
    if (array) {
        unsigned int i;
        assert(data->types.array);
        for (i = 0; i < vector_size(data->types.array); i++) {
            pp_cim_data_t *d = vector_get(data->types.array, i);
            pp_cim_data_null(type, 0, d);
            free(d);
        }
        vector_delete(data->types.array);
        data->types.array = NULL;
    } else if (type == PP_CIM_STRING || type == PP_CIM_STRING_CONST) {
        assert(data->types.string);
        if (type == PP_CIM_STRING) free(data->types.string);
        data->types.string = NULL;
    }
    data->null = 1;
}

void pp_cim_data_delete(pp_cim_datatype_t type, int array, pp_cim_data_t *data)
{
    if (!data)
        return;
    pp_cim_data_null(type, array, data);
    free(data);
}

void pp_cim_data_copy(pp_cim_datatype_t type, int array,
    pp_cim_data_t *dst, pp_cim_data_t *src)
{
    *dst = *src;
    if (src->null) {
        return;
    } else if (array) {
        size_t size = vector_size(src->types.array);
        unsigned int i;
        assert(src->types.array);
        dst->types.array = vector_new(NULL, size, NULL);
        for (i = 0; i < size; i++) {
            pp_cim_data_t *d = malloc(sizeof(pp_cim_data_t));
            pp_cim_data_copy(type, 0, d, vector_get(src->types.array, i));
            vector_add(dst->types.array, d);
        }
    } else if (type == PP_CIM_STRING || type == PP_CIM_STRING_CONST) {
        assert(src->types.string_const);
        dst->types.string = strdup(src->types.string_const);
    }
}

pp_cim_data_t *pp_cim_data_dup(pp_cim_datatype_t type, int array, pp_cim_data_t *data)
{
    pp_cim_data_t *result = malloc(sizeof(pp_cim_data_t));
    pp_cim_data_copy(type, array, result, data);
    return result;
}

pp_cim_data_t *pp_cim_data_parse(pp_cim_datatype_t type, const char *string)
{
    pp_cim_data_t *data;
    long long ll;
    double d;
    char *c;
    if (!string)
        return NULL;
    data = malloc(sizeof(pp_cim_data_t));
    data->null = 0;
    switch (type) {
        case PP_CIM_BOOLEAN:
            if (strcasecmp(string, "t") == 0 ||
                strcasecmp(string, "1") == 0 ||
                strcasecmp(string, "true") == 0) {
                data->types.boolean = 1;
                return data;
            } else if (strcasecmp(string, "f") == 0 ||
                strcasecmp(string, "0") == 0 ||
                strcasecmp(string, "false") == 0) {
                data->types.boolean = 0;
                return data;
            }
            goto bail;
        case PP_CIM_SIGNED:
            ll = strtoll(string, &c, 10);
            if (c[0] != '\0')
                goto bail;
            data->types.signed_int = ll;
            return data;
        case PP_CIM_UNSIGNED:
            ll = strtoll(string, &c, 10);
            if (c[0] != '\0')
                goto bail;
            if (ll < 0)
                goto bail;
            data->types.unsigned_int = ll;
            return data;
        case PP_CIM_REAL:
            d = strtod(string, &c);
            if (c[0] != '\0')
                goto bail;
            data->types.real = d;
            return data;
        case PP_CIM_STRING:
        case PP_CIM_STRING_CONST:
            data->types.string = strdup(string);
            return data;
        default:
            ;
    }

bail:
    free(data);
    return NULL;
}

int pp_cim_data_equal(pp_cim_datatype_t type,
    pp_cim_data_t data1, pp_cim_data_t data2)
{
    if (data1.null && data2.null) {
        return 1;
    } else if (data1.null || data2.null) {
        return 0;
    }
    switch (type) {
        case PP_CIM_BOOLEAN:
            return (data1.types.boolean && data2.types.boolean) ||
                (!data1.types.boolean && !data2.types.boolean);
        case PP_CIM_SIGNED:
            return data1.types.signed_int == data2.types.signed_int;
        case PP_CIM_UNSIGNED:
            return data1.types.unsigned_int == data2.types.unsigned_int;
        case PP_CIM_REAL:
	    assert(0); /* not used yet; avoid testing floats for equality now - it's unsafe */
            return 0;
        case PP_CIM_STRING:
        case PP_CIM_STRING_CONST:
            assert(data1.types.string);
            assert(data2.types.string);
            return (strcmp(data1.types.string, data2.types.string) == 0);
        case PP_CIM_REFERENCE:
            assert(data1.types.reference);
            assert(data2.types.reference);
            return data1.types.reference == data2.types.reference;
        default:
            ;
    }
    return 0;
}

char *pp_cim_data_print(pp_cim_datatype_t type, pp_cim_data_t data)
{
    char *buf;
    int len;
    if (data.null)
        return NULL;
    switch (type) {
        case PP_CIM_BOOLEAN:
            return strdup(data.types.boolean ? "true" : "false");
        case PP_CIM_SIGNED:
            len = snprintf(NULL, 0, "%d", data.types.signed_int);
            buf = malloc(len + 1);
            sprintf(buf, "%d", data.types.signed_int);
            return buf;
        case PP_CIM_UNSIGNED:
            len = snprintf(NULL, 0, "%d", data.types.unsigned_int);
            buf = malloc(len + 1);
            sprintf(buf, "%d", data.types.unsigned_int);
            return buf;
        case PP_CIM_REAL:
            len = snprintf(NULL, 0, "%f", data.types.real);
            buf = malloc(len + 1);
            sprintf(buf, "%f", data.types.real);
            return buf;
        case PP_CIM_STRING:
        case PP_CIM_STRING_CONST:
            assert(data.types.string);
            return strdup(data.types.string);
        default:
            ;
    }
    return NULL;
}

