#include "cim_common.h"
#include "class.h"
#include "namespace.h"

/*
 * Create and register a new CIM class.
 *
 * @param desc  Class description structure
 *
 * @return  New CIM class structure, NULL in case of error.
 */
pp_cim_class_t *pp_cim_class_new(pp_cim_class_desc_t *desc)
{
    pp_cim_class_t *c;
    char *name;

    pthread_rwlock_wrlock(&pp_cim_repository_lock);
    name = strdup_lc(desc->cim_name);
    c = pp_hash_get_entry(pp_cim_repository, name);
    free(name);
    if (c) {
        printf("Warning: Redefinition of %s CIM class, ignored.\n",
            desc->cim_name);
        c = NULL;
        goto bail;
    }
    if (desc->ufct) {
        name = strdup_lc(desc->ufct);
        c = pp_hash_get_entry(pp_cim_repository_ufct, name);
        free(name);
        if (c)
            printf("Warning: UFcT %s is already in use.\n", desc->ufct);
    }

    c = malloc(sizeof(pp_cim_class_t));
    c->cim_name = strdup(desc->cim_name);
    c->ufct = desc->ufct ? strdup(desc->ufct) : NULL;
    c->dispname = desc->dispname ? strdup(desc->dispname) : NULL;

    c->superclass = NULL;
    if (desc->superclass) {
        name = strdup_lc(desc->superclass);
        c->superclass = pp_hash_get_entry(pp_cim_repository, name);
        free(name);
        if (!c->superclass) {
            printf("Warning: CIM class %s depends on unknown class %s.\n",
                desc->cim_name, desc->superclass);
            free(c);
            c = NULL;
            goto bail;
        }
        vector_add(c->superclass->subclasses, c);
        // Adopt superclass's mutex
        c->mutex = c->superclass->mutex;
    } else {
        // Top-level class: Create mutex
        c->mutex = malloc(sizeof(pthread_mutex_t));
        MUTEX_CREATE_RECURSIVE(c->mutex);
        pp_cim_debug("%s: mutex = %p\n", c->cim_name, (void *)c->mutex);
    }
    c->subclasses = vector_new(NULL, 0, NULL);
    c->association = desc->assoc;

    c->properties = vector_new(NULL, 0, NULL);
    pp_cim_property_t *p = desc->properties;
    while (p && p->name) {
        vector_add(c->properties, p);
        p++;
    }

    c->methods = vector_new(NULL, 0, NULL);
    pp_cim_method_t *m = desc->methods;
    while (m && m->name) {
        vector_add(c->methods, m);
        m++;
    }

    c->instances = vector_new(NULL, 5, NULL);

    c->update = desc->update;
    c->clp_update = desc->clp_update;
    c->clp_help = desc->clp_help;

    name = strdup_lc(c->cim_name);
    pp_hash_set_entry(pp_cim_repository, name, c, NULL);
    free(name);
    if (c->ufct) {
        name = strdup_lc(c->ufct);
        pp_hash_set_entry(pp_cim_repository_ufct, name, c, NULL);
        free(name);
    }

bail:
    pthread_rwlock_unlock(&pp_cim_repository_lock);
    return c;
}

/*
 * Delete and unregister a CIM class.
 *
 * @param c  CIM class structure
 */
void pp_cim_class_delete(pp_cim_class_t *c)
{
    unsigned int i;
    if (!c)
        return;
    if (vector_size(c->subclasses) > 0) {
        printf("Warning: Tried to delete class %s with %d subclass(es)!\n",
            c->cim_name, vector_size(c->subclasses));
        return;
    }
    pp_hash_delete_entry(pp_cim_repository, c->cim_name);
    if (c->superclass) {
        for (i = 0; i < vector_size(c->superclass->subclasses); i++) {
            pp_cim_class_t *c1 = vector_get(c->superclass->subclasses, i);
            if (c1 == c) {
                vector_remove(c->superclass->subclasses, i);
                break;
            }
        }
    } else {
        pthread_mutex_destroy(c->mutex);
        free(c->mutex);
    }
    vector_delete(c->subclasses);
    for (i = 0; i < vector_size(c->instances); i++) {
        pp_cim_instance_t *inst = vector_get(c->instances, i);
        pp_cim_instance_delete(inst);
    }
    vector_delete(c->instances);
    free(c->cim_name);
    free(c->ufct);
    free(c->dispname);
    free(c);
}

/*
 * Lock a CIM class.
 *
 * @param cim_class  CIM class structure
 */
void pp_cim_class_lock(pp_cim_class_t *cim_class)
{
    pthread_mutex_lock(cim_class->mutex);
}

/*
 * Release (unlock) a CIM class.
 *
 * @param cim_class  CIM class structure
 */
void pp_cim_class_release(pp_cim_class_t *cim_class)
{
    pthread_mutex_unlock(cim_class->mutex);
}

/*
 * Call class update methods.
 *
 * @param cim_class  CIM class structure, NULL for all registered classes
 */
void pp_cim_class_update(pp_cim_class_t *cim_class)
{
    if (!cim_class) {
        pp_cim_class_t *c;
        pp_hash_iterator_t it;
        pthread_rwlock_rdlock(&pp_cim_repository_lock);
        c = pp_hash_get_first_entry_r(pp_cim_repository, &it);
        while (c) {
            if (c->update) {
                pp_cim_class_lock(c);
                c->update(c);
                pp_cim_class_release(c);
            }
            c = pp_hash_get_next_entry_r(pp_cim_repository, &it);
        }
        pthread_rwlock_unlock(&pp_cim_repository_lock);
    } else {
        pp_cim_class_lock(cim_class);
        if (cim_class->update)
            cim_class->update(cim_class);
        pp_cim_class_release(cim_class);
    }
}

/*
 * List class property descriptions (including those inherited from
 * superclasses, not including private ones).
 *
 * @param cim_class  CIM class structure
 *
 * @return  Property description vector; element type: pp_cim_property_t
 */
vector_t *pp_cim_list_properties(pp_cim_class_t *cim_class)
{
    vector_t *hierarchy, *result;
    hierarchy = vector_new(NULL, 5, NULL);
    result = vector_new(NULL, vector_size(cim_class->properties), NULL);
    unsigned int i, j, k;
    while (cim_class) {
        vector_add(hierarchy, cim_class);
        cim_class = cim_class->superclass;
    }
    for (i = 0; i < vector_size(hierarchy); i++) {
        pp_cim_class_t *c = vector_get(hierarchy, vector_size(hierarchy) - i - 1);
        for (j = 0; j < vector_size(c->properties); j++) {
            pp_cim_property_t *p = vector_get(c->properties, j);
            if (!p->priv) {
                for (k = 0; k < vector_size(result); k++) {
                    pp_cim_property_t *p1 = vector_get(result, k);
                    if (strcasecmp(p->name, p1->name) == 0) {
                        // property overridden by subclass
                        vector_remove(result, k);
                    }
                }
                vector_add(result, p);
            }
        }
    }
    vector_delete(hierarchy);
    return result;
}

/*
 * Get class property description by name.
 *
 * @param cim_class  CIM class structure
 * @param property   Property name
 *
 * @return  Property description
 */
pp_cim_property_t *pp_cim_class_get_property(pp_cim_class_t *cim_class,
    const char *property)
{
    unsigned int i;
    vector_t *props = pp_cim_list_properties(cim_class);
    pp_cim_property_t *ret = NULL;
    for (i = 0; i < vector_size(props); i++) {
        pp_cim_property_t *p = vector_get(props, i);
        if (strcasecmp(p->name, property) == 0) {
            ret = p;
            break;
        }
    }
    vector_delete(props);
    return ret;
}

/*
 * List class method fingerprints (including those inherited from
 * superclasses, not including private ones).
 *
 * @param cim_class  CIM class structure
 *
 * @return  Property description vector; element type: pp_cim_property_t
 */
vector_t *pp_cim_list_methods(pp_cim_class_t *cim_class)
{
    vector_t *hierarchy, *result;
    hierarchy = vector_new(NULL, 5, NULL);
    result = vector_new(NULL, vector_size(cim_class->methods), NULL);
    unsigned int i, j, k;
    while (cim_class) {
        vector_add(hierarchy, cim_class);
        cim_class = cim_class->superclass;
    }
    for (i = 0; i < vector_size(hierarchy); i++) {
        pp_cim_class_t *c = vector_get(hierarchy, vector_size(hierarchy) - i - 1);
        for (j = 0; j < vector_size(c->methods); j++) {
            pp_cim_method_t *m = vector_get(c->methods, j);
            if (!m->priv) {
                for (k = 0; k < vector_size(result); k++) {
                    pp_cim_method_t *m1 = vector_get(result, k);
                    if (strcasecmp(m->name, m1->name) == 0) {
                        // method overridden by subclass
                        vector_remove(result, k);
                    }
                }
                vector_add(result, m);
            }
        }
    }
    vector_delete(hierarchy);
    return result;
}

/*
 * Get method fingerprint by name.
 *
 * @param cim_class  CIM class structure
 * @param method     Method name
 *
 * @return  Method fingerprint
 */
pp_cim_method_t *pp_cim_class_get_method(pp_cim_class_t *cim_class,
    const char *method)
{
    unsigned int i;
    vector_t *methods = pp_cim_list_methods(cim_class);
    pp_cim_method_t *ret = NULL;
    for (i = 0; i < vector_size(methods); i++) {
        pp_cim_method_t *m = vector_get(methods, i);
        if (strcasecmp(m->name, method) == 0) {
            ret = m;
            break;
        }
    }
    vector_delete(methods);
    return ret;
}

static void pp_cim_get_instances_recursive(pp_cim_class_t *c, vector_t *v)
{
    unsigned int i;
    for (i = 0; i < vector_size(c->subclasses); i++) {
        pp_cim_class_t *c1 = vector_get(c->subclasses, i);
        pp_cim_get_instances_recursive(c1, v);
    }
    for (i = 0; i < vector_size(c->instances); i++) {
        pp_cim_instance_t *inst = vector_get(c->instances, i);
        vector_add(v, inst);
    }
}

/*
 * Get all instances of the given class and its subclasses.
 *
 * @param cim_class  CIM class structure
 *
 * @return  Instance vector; must be deleted by caller.
 */
vector_t *pp_cim_get_instances(pp_cim_class_t *cim_class)
{
    vector_t *result;
    result = vector_new(NULL, vector_size(cim_class->instances), NULL);
    pp_cim_get_instances_recursive(cim_class, result);
    return result;
}

