#ifndef _PP_CIM_H
#define _PP_CIM_H

#include <pp/cimTypes.h>
#include <pp/clp.h>


// Init and deinit
int pp_cim_init(void);
void pp_cim_cleanup(void);

// List all known CIM classes
vector_t *pp_cim_list_classes(void);
pp_cim_class_t *pp_cim_class_lookup(const char *cim_name);
pp_cim_class_t *pp_cim_class_lookup_ufct(const char *ufct);
void pp_cim_class_lock(pp_cim_class_t *cim_class);
void pp_cim_class_release(pp_cim_class_t *cim_class);
void pp_cim_class_update(pp_cim_class_t *cim_class);

vector_t *pp_cim_get_instances(pp_cim_class_t *cim_class);
vector_t *pp_cim_list_properties(pp_cim_class_t *cim_class);
pp_cim_property_t *pp_cim_class_get_property(pp_cim_class_t *cim_class,
	const char *property);
vector_t *pp_cim_list_methods(pp_cim_class_t *cim_class);
pp_cim_method_t *pp_cim_class_get_method(pp_cim_class_t *cim_class,
	const char *method);

// Relative UFiP lookup
pp_cim_instance_t *pp_cim_lookup_relative(pp_cim_instance_t *start, vector_t *ufip);
// Absolute UFiP lookup
pp_cim_instance_t *pp_cim_lookup_absolute(vector_t *ufip);
// Find all instaces matching a given CLP target
vector_t *pp_cim_clp_lookup(pp_clp_target_t *target, vector_t *default_target);
// Unlock all classes locked by pp_cim_clp_lookup
void pp_cim_clp_done(vector_t *instances);
// Find associations referencing a given instance
vector_t *pp_cim_find_associations(pp_cim_instance_t *instance,
	const char *name);
// Get an instance's UFiT
pp_clp_ufit_t *pp_cim_get_ufit(pp_cim_instance_t *instance);
// Get an instance's UFiP
int pp_cim_get_ufip(pp_cim_instance_t *instance, vector_t *ufip);
// Check whether one class is a subclass of another
int pp_cim_is_subclass(pp_cim_class_t *sub, pp_cim_class_t *super);

// Delete a CIM instance, delete on zero reference counter
void pp_cim_instance_unregister(pp_cim_instance_t *instance);
// Wait for zero reference counter, delete instance
void pp_cim_instance_delete(pp_cim_instance_t *instance);
// Increase an instance's reference counter
void pp_cim_instance_tag(pp_cim_instance_t *instance);
// Decrease an instance's reference counter, delete on zero
void pp_cim_instance_untag(pp_cim_instance_t *instance);
// Check if an instance belongs to or is inherited from a given class
int pp_cim_qualifies_as(pp_cim_instance_t *instance, const char *name);
// Check if an instance belongs to or is inherited from a given class
int pp_cim_qualifies_as_ufct(pp_cim_instance_t *instance, const char *ufct);
// Get instance property
pp_cim_propval_t *pp_cim_get_property(pp_cim_instance_t *instance,
	const char *property);
// Set instance property
int pp_cim_set_property(pp_cim_instance_t *instance,
	const char *property, pp_cim_data_t data);
// Get all properties from an instance
vector_t *pp_cim_get_properties(pp_cim_instance_t *instance);
// Set all instance properties at once
int pp_cim_set_properties(pp_cim_instance_t *instance,
	vector_t *names, vector_t *properties);
// Get CIM method by name
pp_cim_methodptr_t *pp_cim_get_method(pp_cim_instance_t *instance,
	const char *method);
// Get value map entry for index
const char *pp_cim_map_value(pp_cim_valmap_t *valmap, int idx);
// Update CIM proerties.
void pp_cim_update_properties(pp_cim_instance_t *instance);
// commit the data set via pp_cim_get_propert[y,ies]
void pp_cim_commit_properties(pp_cim_instance_t *instance);
// check the authorization of a cim instance
int pp_cim_get_authorization(pp_cim_instance_t *instance, const char *auth_string);

// Execute a CIM method call
int pp_cim_method_call_exec(pp_cim_method_call_t *call, pp_cim_data_t *result);

// Check whether an instance supports the CLP Reset command or not
int pp_cim_clp_reset_supported(pp_cim_instance_t *instance);
// Map CLP Reset command to a CIM method call
pp_cim_method_call_t *pp_cim_clp_map_reset(pp_cim_instance_t *instance);
// Check whether an instance supports the CLP Set command or not
int pp_cim_clp_set_supported(pp_cim_instance_t *instance);
// Map CLP Set command to a CIM method call
pp_cim_method_call_t *pp_cim_clp_map_set(pp_cim_instance_t *instance,
	    pp_clp_property_value_t *prop);
// Check whether an instance supports the CLP Start command or not
int pp_cim_clp_start_supported(pp_cim_instance_t *instance);
// Map CLP Start command to a CIM method call
pp_cim_method_call_t *pp_cim_clp_map_start(pp_cim_instance_t *instance);
// Check whether an instance supports the CLP Stop command or not
int pp_cim_clp_stop_supported(pp_cim_instance_t *instance);
// Map CLP Stop command to a CIM method call
pp_cim_method_call_t *pp_cim_clp_map_stop(pp_cim_instance_t *instance);
// Check whether a class supports the CLP create command or not
pp_cim_method_t *pp_cim_clp_create_supported(pp_cim_class_t *cim_class);
// Check whether a class supports to CLP delete command or not
int pp_cim_clp_delete_supported(pp_cim_instance_t *instnace);
// Map CLP delete command to a CIM method call
pp_cim_methodptr_t *pp_cim_clp_map_delete(pp_cim_instance_t *instance);

/* CIM data type handling */
// copy a CIM data structure
void pp_cim_data_copy(pp_cim_datatype_t type, int array,
	pp_cim_data_t *dst, pp_cim_data_t *src);
// Duplicate a CIM data structure
pp_cim_data_t *pp_cim_data_dup(pp_cim_datatype_t type, int array,
	pp_cim_data_t *data);
// Check if two CIM data structures are equal
int pp_cim_data_equal(pp_cim_datatype_t type,
	pp_cim_data_t data1, pp_cim_data_t data2);
// Convert string to CIM data structure
pp_cim_data_t *pp_cim_data_parse(pp_cim_datatype_t type, const char *string);
// Convert CIM data structure to string
char *pp_cim_data_print(pp_cim_datatype_t type, pp_cim_data_t data);

/* Delete functions for CIM structures */
// Delete a CIM class
void pp_cim_class_delete(pp_cim_class_t *c);
// Clean a CIM data structure
void pp_cim_data_null(pp_cim_datatype_t type, int array, pp_cim_data_t *data);
// Delete a CIM data structure
void pp_cim_data_delete(pp_cim_datatype_t type, int array, pp_cim_data_t *data);
// Delete a CIM property/value structure
void pp_cim_propval_delete(pp_cim_propval_t *propval);
// Delete a method structure
void pp_cim_method_delete(pp_cim_method_t *method);

// Create a new CIM method call structure
pp_cim_method_call_t *pp_cim_method_call_new(pp_cim_methodptr_t *methodptr,
	pp_cim_instance_t *instance);
// Delete a method call structure
void pp_cim_method_call_delete(pp_cim_method_call_t *call);

// Get CLP help message for class
char *pp_cim_clp_help_class(pp_cim_class_t *cim_class, int verbose);
// Get CLP help message for instance
char *pp_cim_clp_help_instance(pp_cim_instance_t *instance, int verbose);

#endif // _PP_CIM_H

