#ifndef _PP_CIMTYPES_H
#define _PP_CIMTYPES_H

#include <pthread.h>
#include <pp/base.h>
#include <pp/vector.h>

// forward declarations
typedef struct pp_cim_class_s pp_cim_class_t;
typedef struct pp_cim_instance_s pp_cim_instance_t;
typedef struct pp_cim_provider_s pp_cim_provider_t;

// CIM class
struct pp_cim_class_s {
    char *cim_name;         // CIM class name (e.g. CIM_NumericSensor)
    char *ufct;                   // User Friendly Class Tag (e.g. nsensor)
    char *dispname;               // display name (e.g. for WS-Management)
    pp_cim_class_t *superclass;   // superclass
    vector_t *subclasses;         // subclasses; element type: pp_cim_class_t
    int association;              // association class (0 = no, !0 = yes)
    vector_t *properties;         // property declarations; element type: pp_cim_property_t
    vector_t *methods;            // method fingerprints; element type: pp_cim_method_t
    vector_t *instances;          // instance vector; element type: pp_cim_instance_t
    pthread_mutex_t *mutex;       // class mutex
    void (*update)(pp_cim_class_t *cim_class);     // optional: find new instances
    void (*clp_update)(pp_cim_class_t *cim_class); // optional: find new CLP children
    char *(*clp_help)(pp_cim_class_t *cim_class, int verbose); // optional: get CLP help message for class
};

// CIM instance
struct pp_cim_instance_s {
    pp_cim_class_t *cim_class;    // CIM class
    char *alias;                  // CLP alias (e.g. nvoltsensor)
    int instance_id;              // CLP namespace instance suffix
    pp_cim_provider_t *provider;  // provider methods
    vector_t *prop_table;         // property table; element type: pp_cim_propval_t
    vector_t *method_table;       // virtual method table; element type: pp_cim_methodptr_t
    pp_cim_instance_t *parent;    // parent in CLP namespace tree
    pp_cim_instance_t *next;      // next sibling in CLP namespace tree
    pp_cim_instance_t *children;  // children in CLP namespace tree
    pthread_mutex_t mutex;        // instance mutex
    pthread_rwlock_t refcount;    // reference counter (delete protection)
};

// CIM data types
typedef enum {
    PP_CIM_VOID,
    PP_CIM_BOOLEAN,
    PP_CIM_SIGNED,
    PP_CIM_UNSIGNED,
    PP_CIM_REAL,
    PP_CIM_STRING,
    PP_CIM_STRING_CONST,
    PP_CIM_REFERENCE
} pp_cim_datatype_t;

// CIM data union
typedef struct {
    int null;
    union {
	int boolean;
	int signed_int;
	unsigned int unsigned_int;
	double real;
	char *string;
	const char *string_const; /* evil const hack! bad design! */
	vector_t *array; // element type: pp_cim_data_t
	pp_cim_instance_t *reference;
    } types;
} pp_cim_data_t;

// CIM property value map
typedef struct {
    int min;
    int max;
    const char *value;
} pp_cim_valmap_t;

// CIM property declaration structure
typedef struct {
    const char *name;
    pp_cim_datatype_t type;
    int array;
    pp_cim_valmap_t *valmap;
    int key;
    int priv;
    int required;
    int writable;
    pp_cim_data_t defvalue;
} pp_cim_property_t;

// CIM property value structure
typedef struct {
    pp_cim_property_t *property;
    pp_cim_data_t data;
    int changed;
} pp_cim_propval_t;

// argument structure for CIM method calls
typedef struct {
    const char *name;
    pp_cim_datatype_t type;
    int array;
    pp_cim_valmap_t *valmap;
    int in;
    int out;
} pp_cim_method_arg_t;

// CIM method structure; argument element type: pp_cim_method_arg_t
typedef struct {
    const char *name;
    pp_cim_datatype_t result;
    int num_args;
    pp_cim_method_arg_t *args;
    int priv;
    int (*defptr)(pp_cim_instance_t *instance, vector_t *args, pp_cim_data_t *result);
} pp_cim_method_t;

// CIM method pointer
typedef struct {
    pp_cim_method_t *method;
    int (*ptr)(pp_cim_instance_t *instance, vector_t *args, pp_cim_data_t *result);
} pp_cim_methodptr_t;

// CIM method call
typedef struct {
    pp_cim_methodptr_t *methodptr;
    pp_cim_instance_t *instance;
    vector_t *args; // element type: pp_cim_data_t
} pp_cim_method_call_t;

#endif // _PP_CIMTYPES_H

