#include "cim_common.h"
#include "instance.h"
#include "namespace.h"
#include "provider_common.h"

int instance_counter = 0;

/*
 * Create a new CIM instance from template.
 *
 * @param cim_class  CIM class name
 * @param provider   Provider reference
 *
 * @return  New CIM instance, NULL in case of error
 *
 * Note: The new instance will be registered in the CIM instance repository
 *       and must be released with pp_cim_instance_unregister().
 */
pp_cim_instance_t *pp_cim_instance_new(const char *cim_class,
    pp_cim_provider_t *provider)
{
    pp_cim_instance_t *i;
    pp_cim_class_t *c;
    vector_t *classes;
    pp_cim_data_t d;
    unsigned int x, y, z;

    c = pp_cim_class_lookup(cim_class);
    if (!c) {
        printf("Warning: Unknown CIM class %s, no instance created.\n", cim_class);
        return NULL;
    }

    instance_counter++;
    i = malloc(sizeof(pp_cim_instance_t));
    i->cim_class = c;

    MUTEX_CREATE_RECURSIVE(&i->mutex);

    pthread_rwlock_init(&i->refcount, NULL);
    pthread_rwlock_rdlock(&i->refcount);

    i->alias = NULL;
    i->instance_id = -1;
    i->provider = provider;

    // Get inheritance hierarchy
    classes = vector_new(NULL, 5, NULL);
    for (c = i->cim_class; c; c = c->superclass)
        vector_add(classes, c);

    // Create instance property table
    i->prop_table = vector_new(NULL, vector_size(i->cim_class->properties),
        (vector_elem_del_func_simple)pp_cim_propval_delete);
    for (x = 0; x < vector_size(classes); x++) {
        c = vector_get(classes, vector_size(classes) - x - 1);
        for (y = 0; y < vector_size(c->properties); y++) {
            int shadowed = 0;
            pp_cim_property_t *prop = vector_get(c->properties, y);
            pp_cim_propval_t *propval = NULL;
            for (z = 0; z < vector_size(i->prop_table); z++) {
                pp_cim_propval_t *pv = vector_get(i->prop_table, z);
                if (strcasecmp(pv->property->name, prop->name) == 0) {
                    // property overwritten by descendent class
                    shadowed = 1;
                    if (prop->type == PP_CIM_STRING && !pv->data.null) {
                        free(pv->data.types.string);
                    }
                    pv->data.null = 1;
                    propval = pv;
                    break;
                }
            }
            if (!shadowed) {
                propval = malloc(sizeof(pp_cim_propval_t));
                vector_add(i->prop_table, propval);
            }
            propval->property = prop;
            propval->data = prop->defvalue;
            propval->changed = 0;
            if ((prop->type == PP_CIM_STRING || prop->type == PP_CIM_STRING_CONST)  && !prop->defvalue.null) {
                propval->data.types.string = strdup(prop->defvalue.types.string_const);
            }
        }
        c = c->superclass;
    }

    // Replace CreationClassName property for derived classes
    if (i->cim_class->superclass) {
        d.null = 0;
        d.types.string = strdup(cim_class);
        provider_common_set_property(i, "CreationClassName", d, 1);
        free(d.types.string);
    }

    // Create method table
    i->method_table = vector_new(NULL, vector_size(i->cim_class->methods),
        (vector_elem_del_func_simple)free);
    for (x = 0; x < vector_size(classes); x++) {
        c = vector_get(classes, vector_size(classes) - x - 1);
        for (y = 0; y < vector_size(c->methods); y++) {
            int shadowed = 0;
            pp_cim_method_t *meth = vector_get(c->methods, y);
            pp_cim_methodptr_t *mptr = NULL;
            for (z = 0; z < vector_size(i->method_table); z++) {
                pp_cim_methodptr_t *m = vector_get(i->method_table, z);
                if (strcasecmp(m->method->name, meth->name) == 0) {
                    // method overwritten by descendent class
                    shadowed = 1;
                    mptr = m;
                    break;
                }
            }
            if (!shadowed) {
                mptr = malloc(sizeof(pp_cim_methodptr_t));
                vector_add(i->method_table, mptr);
            }
            mptr->method = meth;
            mptr->ptr = meth->defptr;
        }
        c = c->superclass;
    }

    vector_delete(classes);

    i->parent = NULL;
    i->next = NULL;
    i->children = NULL;

    // Register instance in CIM repository
    vector_add(i->cim_class->instances, i);
    pp_cim_class_release(i->cim_class);

    return i;
}

/*
 * Delete a CIM instance from memory.
 *
 * @param instance  CIM instance
 */
void pp_cim_instance_free(pp_cim_instance_t *instance)
{
    const char *cim_name;
    if (!instance)
        return;
    cim_name = instance->cim_class->cim_name;
    pthread_mutex_destroy(&instance->mutex);
    pthread_rwlock_destroy(&instance->refcount);
    free(instance->alias);
    vector_delete(instance->prop_table);
    vector_delete(instance->method_table);
    free(instance);

    instance_counter--;
    pp_cim_debug("%s instance deleted, %d total instances remaining.\n", cim_name, instance_counter);
}

/*
 * Wait for zero reference counter, delete instance
 *
 * @param instance  CIM instance
 */
void pp_cim_instance_delete(pp_cim_instance_t *instance)
{
    pthread_mutex_lock(&instance->mutex);
    pthread_rwlock_unlock(&instance->refcount);
    pthread_mutex_unlock(&instance->mutex);
    // wait for zero reference counter
    pthread_rwlock_wrlock(&instance->refcount);
    pthread_rwlock_unlock(&instance->refcount);
    pp_cim_instance_free(instance);
}

/*
 * Unregister a CIM instance, delete on zero reference counter
 *
 * @param instance  CIM instance
 */
void pp_cim_instance_unregister(pp_cim_instance_t *instance)
{
    vector_t *inst_vector;
    unsigned int i;
    if (!instance)
        return;
    pp_cim_class_lock(instance->cim_class);
    inst_vector = instance->cim_class->instances;
    for (i = 0; i < vector_size(inst_vector); i++) {
        pp_cim_instance_t *inst = vector_get(inst_vector, i);
        if (inst == instance) {
            vector_remove(inst_vector, i);
            break;
        }
    }
    pp_cim_class_release(instance->cim_class);
    pp_cim_instance_untag(instance);
}

/*
 * Increase a CIM instance's reference counter (protect against deletion)
 *
 * @param instance  CIM instance
 */
void pp_cim_instance_tag(pp_cim_instance_t *instance)
{
    pthread_rwlock_rdlock(&instance->refcount);
}

/*
 * Decrease a CIM instance's reference counter, delete on zero
 *
 * @param instance  CIM instance
 */
void pp_cim_instance_untag(pp_cim_instance_t *instance)
{
    int del;
    pthread_mutex_lock(&instance->mutex);
    pthread_rwlock_unlock(&instance->refcount);
    del = (pthread_rwlock_trywrlock(&instance->refcount) == 0);
    pthread_mutex_unlock(&instance->mutex);
    if (del) {
        pthread_rwlock_unlock(&instance->refcount);
        pp_cim_instance_free(instance);
    }
}

/*
 *
 *
 *
 */
void pp_cim_instance_remove_from_list(pp_cim_instance_t *instance)
{
    pp_cim_instance_t *p = instance->parent->children;
    pp_cim_instance_t *t;
    t = p;
    while (p) {
        if (p == instance) {
            // we got it
            if (t == p) {
                // first one
                instance->parent->children = p->next;
            } else {
                t->next = p->next;
            }
            break;
        }
        t = p;
        p = p->next;
    }
}

#ifdef PP_CIM_DEBUG
/*
 * Dump a CIM instance to stdout (debugging only)
 *
 * @param instance  CIM instance
 */
void pp_cim_instance_dump(pp_cim_instance_t *instance)
{
    unsigned int i;
    printf("CIM instance dump: %s\n", instance->cim_class->cim_name);
    for (i = 0; i < vector_size(instance->prop_table); i++) {
        pp_cim_propval_t *pv = vector_get(instance->prop_table, i);
        char *c = pp_cim_data_print(pv->property->type, pv->data);
        printf("  %s = %s\n", pv->property->name, c);
        free(c);
    }
}
#endif

/*
 * Clone a CIM property/value structure.
 *
 * @param propval  CIM property/value to duplicate
 *
 * @return  New property/value structure
 *
 * Note: The new structure must be freed with pp_cim_property_delete.
 */
pp_cim_propval_t *pp_cim_propval_dup(pp_cim_propval_t *propval)
{
    pp_cim_propval_t *p;
    if (!propval)
        return NULL;
    p = malloc(sizeof(pp_cim_propval_t));
    *p = *propval;
    if (p->property->type == PP_CIM_STRING_CONST) p->property->type = PP_CIM_STRING;
    pp_cim_data_copy(propval->property->type, propval->property->array,
        &p->data, &propval->data);
    return p;
}

/*
 * Delete a CIM property/value structure.
 *
 * @param propval  CIM property/value to delete
 */
void pp_cim_propval_delete(pp_cim_propval_t *propval)
{
    if (!propval)
        return;
    pp_cim_data_null(propval->property->type, propval->property->array,
        &propval->data);
    free(propval);
}

/*
 * Create a new CIM method call structure.
 *
 * @param methodptr  Pointer to method to call
 * @param instance   Instance to use for method call
 *
 * @return  New method call structure
 *
 * Note: The new structure must be freed with pp_cim_method_call_delete.
 */
pp_cim_method_call_t *pp_cim_method_call_new(pp_cim_methodptr_t *methodptr,
    pp_cim_instance_t *instance)
{
    pp_cim_method_call_t *c = malloc(sizeof(pp_cim_method_call_t));
    c->methodptr = methodptr;
    c->instance = instance;
    if (methodptr) {
        c->args = vector_new(NULL, methodptr->method->num_args, NULL);
    } else {
        c->args = NULL;
    }
    return c;
}

/*
 * Delete a CIM method call structure.
 *
 * @param call  Method call structure
 */
void pp_cim_method_call_delete(pp_cim_method_call_t *call)
{
    if (!call)
        return;
    if (call->methodptr && call->args) {
        unsigned int i;
        for (i = 0; i < vector_size(call->args); i++) {
            pp_cim_data_t *d = vector_get(call->args, i);
            pp_cim_data_delete(call->methodptr->method->args[i].type,
                call->methodptr->method->args[i].array, d);
        }
    }
    vector_delete(call->args);
    free(call);
}

