#include <pp/cim.h>
#include <pp/cimTypes.h>
#include <pp/cim_provider.h>
#include "cim_common.h"
#include "instance.h"
#include "provider_common.h"
#include "provider_computersystem.h"
#include "provider_system.h"
#include <pp/ipmi.h>

#include "valmap_computersystem.h"

static void computersystem_update(pp_cim_instance_t *instance);
static pp_cim_method_call_t *computersystem_map_reset(
    pp_cim_instance_t *instance);
static pp_cim_method_call_t *computersystem_map_start(
    pp_cim_instance_t *instance);
static pp_cim_method_call_t *computersystem_map_stop(
    pp_cim_instance_t *instance);
static char *computersystem_clp_help_instance(pp_cim_instance_t *instance,
    int verbose);
static char *computersystem_clp_help_class(pp_cim_class_t *cim_class,
    int verbose);

static int computersystem_chassiscontrol(pp_cim_instance_t *instance,
    vector_t *args, pp_cim_data_t *result);
static int computersystem_bmc_reset(pp_cim_instance_t *instance,
    vector_t *args, pp_cim_data_t *result);

// provider API functions
static pp_cim_provider_t provider =
{
    .deinit = provider_common_deinit,
    .update = computersystem_update,
    .commit = provider_common_commit,
    .authorize = provider_common_authorize,
    .get_property = provider_common_get_property,
    .set_property = provider_common_set_property,
    .get_properties = provider_common_get_properties,
    .set_properties = provider_common_set_properties,
    .get_method = provider_common_get_method,
    .call_method = provider_common_call_method,
    .clp_map_reset = computersystem_map_reset,
    .clp_map_set = NULL, // 'set' command not supported
    .clp_map_start = computersystem_map_start,
    .clp_map_stop = computersystem_map_stop,
    .clp_help = computersystem_clp_help_instance
};

// class properties
// { <name>, <type>, <array>, <valmap>, <key>, <priv>, <required>, <writable>, { <null>, { <defvalue> } } }
static pp_cim_property_t properties[] =
{
    {"Dedicated", PP_CIM_UNSIGNED, 1, dedicated_valmap, 0, 0, 0, 0, {1, {0}}},
    {"_BMC", PP_CIM_BOOLEAN, 0, NULL, 0, 1, 0, 0, {1, {0}}},
    {.name = NULL}
};

// CIM_ComputerSystem::_ChassisControl() arguments
// { <name>, <type>, <array>, <valmap>, <in>, <out> }
static pp_cim_method_arg_t computersystem_chassiscontrol_args[] =
{
    {"SubCommand", PP_CIM_SIGNED, 0, NULL, 1, 0}
};

// class methods
// { <name>, <result>, <num_args>, <args>, <priv>, <defptr> }
static pp_cim_method_t methods[] =
{
    {"_ChassisControl", PP_CIM_VOID, 1, computersystem_chassiscontrol_args, 1, computersystem_chassiscontrol},
    {"_BMC_Reset", PP_CIM_VOID, 0, NULL, 1, computersystem_bmc_reset},
    {.name = NULL}
};

// class description
pp_cim_class_desc_t pp_cim_computersystem_desc =
{
    .cim_name = "CIM_ComputerSystem",
    .ufct = "system",
    .dispname = "CIM Computer System",
    .superclass = "CIM_System",
    .assoc = 0,
    .properties = properties,
    .methods = methods,
    .update = NULL,
    .clp_update = NULL,
    .clp_help = computersystem_clp_help_class
};

// Create a new CIM_ComputerSystem instance
pp_cim_instance_t *pp_cim_computersystem_new()
{
    pp_cim_instance_t *i = pp_cim_instance_new("CIM_ComputerSystem", &provider);
    return i;
}

static void computersystem_update(pp_cim_instance_t *instance)
{
    pp_cim_propval_t *propval;
    pp_ipmi_return_t ipmi_ret;
    int ret, err;

    memset(&ipmi_ret, 0, sizeof(ipmi_ret));
    propval = instance->provider->get_property(instance, "Name");
    if (propval->data.null) {
	int device_id;
	pp_cim_data_t cs_name;
	char name[100];
	cs_name.null = 0;
	cs_name.types.string = name;

	ret = pp_ipmi_send_command(PP_IPMI_CMD_BMC, PP_IPMI_BMC_SUBCMD_INFO,
		NULL, &ipmi_ret, &err, NULL);
	if (ret) {
	    printf("IPMI failure: %s.\n", pp_ipmi_get_error_string(err));
	    goto bail;
	}
	device_id = ipmi_ret.data.bmc_info.device_id;

	pp_cim_propval_delete(propval);
	propval = instance->provider->get_property(instance, "_BMC");
	if (propval->data.types.boolean) {
	    // BMC Name according to IPMI CIM mapping 0.60
	    snprintf(name, sizeof(name), "IPMI Controller %d", device_id);
	} else {
	    // Host system Name according to IPMI CIM mapping 0.60
	    pp_ipmi_cleanup_ret(&ipmi_ret);
	    ret = pp_ipmi_send_command(PP_IPMI_CMD_BMC, PP_IPMI_BMC_SUBCMD_GET_SYSTEM_GUID,
		    NULL, &ipmi_ret, &err, NULL);
	    if (ret) {
		printf("IPMI failure: %s.\n", pp_ipmi_get_error_string(err));
		goto bail;
	    }
	    unsigned char *d = ipmi_ret.data.bmc_system_guid.guid;
	    snprintf(name, sizeof(name), "Unknown.IPMI BMC Device ID.%d:"
		    "%02x%02x%02x%02x%02x%02x-%02x%02x-%02x%02x-%02x%02x-%02x%02x%02x%02x",
		    device_id,
		    d[0], d[1], d[2], d[3], d[4], d[5], d[6], d[7],
		    d[8], d[9], d[10], d[11], d[12], d[13], d[14], d[15]);
	}

	provider_common_set_property(instance, "Name", cs_name, 1);
    }

bail:
    pp_ipmi_cleanup_ret(&ipmi_ret);
    pp_cim_propval_delete(propval);
}

static pp_cim_method_call_t *computersystem_map_reset(
    pp_cim_instance_t *instance)
{
    pp_cim_propval_t *propval;
    pp_cim_methodptr_t *methodptr;
    pp_cim_method_call_t *call;
    pp_cim_data_t *arg;

    propval = instance->provider->get_property(instance, "_BMC");
    if (propval->data.types.boolean) {
        // BMC reset
        methodptr = instance->provider->get_method(instance, "_BMC_Reset");
        call = pp_cim_method_call_new(methodptr, instance);
    } else {
        // Host system reset
        methodptr = instance->provider->get_method(instance, "_ChassisControl");
        call = pp_cim_method_call_new(methodptr, instance);

        arg = malloc(sizeof(pp_cim_data_t));
        arg->null = 0;
        arg->types.signed_int = PP_IPMI_CHASSIS_RESET;
        vector_add(call->args, arg);
    }
    pp_cim_propval_delete(propval);

    return call;
}

static pp_cim_method_call_t *computersystem_map_start(
    pp_cim_instance_t *instance)
{
    pp_cim_propval_t *propval;
    pp_cim_methodptr_t *methodptr;
    pp_cim_method_call_t *call;
    pp_cim_data_t *arg;

    propval = instance->provider->get_property(instance, "_BMC");
    if (propval->data.types.boolean) {
        // NOP
        call = pp_cim_method_call_new(NULL, instance);
    } else {
        // Host system power up
        methodptr = instance->provider->get_method(instance, "_ChassisControl");
        call = pp_cim_method_call_new(methodptr, instance);

        arg = malloc(sizeof(pp_cim_data_t));
        arg->null = 0;
        arg->types.signed_int = PP_IPMI_CHASSIS_POWER_UP;
        vector_add(call->args, arg);
    }
    pp_cim_propval_delete(propval);

    return call;
}

static pp_cim_method_call_t *computersystem_map_stop(
    pp_cim_instance_t *instance)
{
    pp_cim_propval_t *propval;
    pp_cim_methodptr_t *methodptr;
    pp_cim_method_call_t *call;
    pp_cim_data_t *arg;

    propval = instance->provider->get_property(instance, "_BMC");
    if (propval->data.types.boolean) {
        // NOP
        call = pp_cim_method_call_new(NULL, instance);
    } else {
        // Host system power down
        methodptr = instance->provider->get_method(instance, "_ChassisControl");
        call = pp_cim_method_call_new(methodptr, instance);

        arg = malloc(sizeof(pp_cim_data_t));
        arg->null = 0;
        arg->types.signed_int = PP_IPMI_CHASSIS_POWER_DOWN;
        vector_add(call->args, arg);
    }
    pp_cim_propval_delete(propval);

    return call;
}

static const char clp_help_message_instance_bmc[] =
    "This instance stores information about the IPMI Baseboard Management\r\n"
    "Controller (BMC). It supports the following CLP operations:\r\n"
    "  - Controller reset:  RESET\r\n";

static const char clp_help_message_instance_host[] =
    "This instance stores information about the host system. It supports\r\n"
    "the following CLP operations:\r\n"
    "  - Power on:      START\r\n"
    "  - Power off:     STOP\r\n"
    "  - System reset:  RESET\r\n";

static char *computersystem_clp_help_instance(pp_cim_instance_t *instance,
    int verbose UNUSED)
{
    pp_cim_propval_t *propval;
    const char *message;

    propval = instance->provider->get_property(instance, "_BMC");
    if (propval->data.types.boolean) {
        message = clp_help_message_instance_bmc;
    } else {
        message = clp_help_message_instance_host;
    }
    pp_cim_propval_delete(propval);

    return strdup(message);
}

static const char clp_help_message_class[] =
    "The CIM_ComputerSystem class is used to store information about\r\n"
    "computer system, e.g. the management controller or the host system.\r\n";

static char *computersystem_clp_help_class(pp_cim_class_t *cim_class UNUSED,
    int verbose UNUSED)
{
    return strdup(clp_help_message_class);
}

static int computersystem_chassiscontrol(pp_cim_instance_t *instance UNUSED,
    vector_t *args, pp_cim_data_t *result UNUSED)
{
    pp_ipmi_parameter_t ipmi_param;
    pp_cim_data_t *arg = vector_get(args, 0);
    int ret, err;

    pp_cim_debug("CIM_ComputerSystem::_ChassisControl(%d)\n", arg->types.signed_int);

    memset(&ipmi_param, 0, sizeof(ipmi_param));
    ipmi_param.data.chassis_power = arg->types.signed_int;
    ret = pp_ipmi_send_command(PP_IPMI_CMD_CHASSIS,
        PP_IPMI_CHASSIS_SUBCMD_POWER_CONTROL, &ipmi_param, NULL,
        &err, NULL);

    if (ret) {
        printf("IPMI failure: %s.\n", pp_ipmi_get_error_string(err));
        return PP_ERR;
    }

    return PP_SUC;
}

static int computersystem_bmc_reset(pp_cim_instance_t *instance UNUSED,
    vector_t *args UNUSED, pp_cim_data_t *result UNUSED)
{
    pp_ipmi_parameter_t ipmi_param;
    int ret, err;

    pp_cim_debug("CIM_ComputerSystem::_BMC_Reset()\n");

    memset(&ipmi_param, 0, sizeof(ipmi_param));
    ipmi_param.data.bmc_reset = PP_IPMI_BMC_RESET_WARM;
    ret = pp_ipmi_send_command(PP_IPMI_CMD_BMC,
        PP_IPMI_BMC_SUBCMD_RESET, &ipmi_param, NULL,
        &err, NULL);

    if (ret) {
        printf("IPMI failure: %s.\n", pp_ipmi_get_error_string(err));
        return PP_ERR;
    }

    return PP_SUC;
}

