#include <pp/cim.h>
#include <pp/cimTypes.h>
#include <pp/cim_provider.h>
#include "cim_common.h"
#include "instance.h"
#include "provider_common.h"
#include "provider_group.h"
#include "provider_memberofcollection.h"

static char *group_clp_help_instance(pp_cim_instance_t *instance, int verbose);
static char *group_clp_help_class(pp_cim_class_t *cim_class, int verbose);
static void group_clp_update(pp_cim_class_t *class);

// class properties
// { <name>, <type>, <array>, <valmap>, <key>, <priv>, <required>, <writable>, { <null>, { <defvalue> } } }
static pp_cim_property_t properties[] =
{
    {"CreationClassName", PP_CIM_STRING_CONST, 0, NULL, 1, 0, 1, 0, {1, {0}}},
    {"Name", PP_CIM_STRING_CONST, 0, NULL, 1, 0, 1, 0, {1, {0}}},
    {"_ID", PP_CIM_UNSIGNED, 0, NULL, 0, 1, 0, 0, {1, {0}}},
    {.name = NULL}
};

// provider API functions
static pp_cim_provider_t provider =
{
    .deinit = provider_common_deinit,
    .update = provider_common_update,
    .commit = provider_common_commit,
    .authorize = provider_common_authorize,
    .get_property = provider_common_get_property,
    .set_property = provider_common_set_property,
    .get_properties = provider_common_get_properties,
    .set_properties = provider_common_set_properties,
    .get_method = provider_common_get_method,
    .call_method = provider_common_call_method,
    .clp_map_reset = NULL, // 'reset' command not supported
    .clp_map_set = NULL, // 'set' command not supported
    .clp_map_start = NULL, // 'start' command not supported
    .clp_map_stop = NULL, // 'stop' command not supported
    .clp_help = group_clp_help_instance
};
// class description
pp_cim_class_desc_t pp_cim_group_desc =
{
    .cim_name = "CIM_Group",
    .ufct = "group",
    .dispname = "CIM Group",
    .superclass = "CIM_Collection",
    .assoc = 0,
    .properties = properties,
    .methods = NULL, // no methods
    .update = NULL,
    .clp_update = group_clp_update,
    .clp_help = group_clp_help_class
};

pp_cim_instance_t *pp_cim_group_new()
{
    pp_cim_instance_t *i = pp_cim_instance_new("CIM_Group", &provider);
    return i;
}

typedef struct
{
    int id;
    const char *name;
} group_t;

static group_t groups[] =
{
    {1, "IPMI Callback"},
    {2, "IPMI User"},
    {3, "IPMI Operator"},
    {4, "IPMI Administrator"},
    {5, "IPMI OEM"},
    {15, "IPMI No Access"},
    {-1, NULL}
};

// static lookup table for better performance
#define MAX_GIDS 16
static pp_cim_instance_t *group_instances[MAX_GIDS];

vector_t *pp_cim_group_discovery()
{
    vector_t *result = vector_new(NULL, 6, NULL);
    pp_cim_instance_t *inst;
    pp_cim_data_t d = {0, {0}};
    group_t *group = groups;

    memset(group_instances, 0, sizeof(group_instances));

    while (group->name) {
        inst = pp_cim_group_new();
        d.types.unsigned_int = group->id;
        provider_common_set_property(inst, "_ID", d, 1);
        d.types.string_const = group->name;
        provider_common_set_property(inst, "Name", d, 1);
        vector_add(result, inst);
        if (group->id < MAX_GIDS)
            group_instances[group->id] = inst;
        group++;
    }

    return result;
}

pp_cim_instance_t *pp_cim_group_getbyid(int group_id)
{
    if (group_id >= MAX_GIDS)
        return NULL;
    return group_instances[group_id];
}

int pp_cim_group_get_id(pp_cim_instance_t *instance)
{
    pp_cim_propval_t *propval =
        instance->provider->get_property(instance, "_ID");
    int id = propval->data.types.unsigned_int;
    pp_cim_propval_delete(propval);
    return id;
}

static const char clp_help_message_instance[] =
    "BMC accounts associated with this instance (via CIM_MemberOfCollection)\r\n"
    "are granted the \"%s\" privilege level.\r\n";

static char *group_clp_help_instance(pp_cim_instance_t *instance,
    int verbose UNUSED)
{
    int id = pp_cim_group_get_id(instance);
    group_t *group = groups;
    pp_strstream_t *str;

    while (group->name) {
        if (group->id == id)
            break;
        group++;
    }
    if (!group->name)
        return NULL;

    str = pp_strstream_init(NULL);
    pp_strappendf(str, clp_help_message_instance, group->name);
    return pp_strstream_buf_and_free(str);
}

static const char clp_help_message_class[] =
    "The CIM_Group class is used to store information about account groups,\r\n"
    "e.g. all BMC accounts which are granted a certain IPMI privilege level.\r\n"
    "Accounts can be associated with groups via the CIM_MemberOfCollection\r\n"
    "association class.\r\n";

static char *group_clp_help_class(pp_cim_class_t *cim_class UNUSED,
    int verbose UNUSED)
{
    return strdup(clp_help_message_class);
}

static void group_clp_update(pp_cim_class_t *class UNUSED)
{
    pp_cim_class_t *c =
        pp_cim_class_lookup("CIM_MemberOfCollection");
    if (c->update)
        c->update(c);
    pp_cim_class_release(c);
}

