#include <pp/cim.h>
#include <pp/cimTypes.h>
#include <pp/cim_provider.h>
#include <pp/ipmi.h>
#include "cim_common.h"
#include "instance.h"
#include "provider_common.h"
#include "provider_numericsensor.h"
#include "provider_logicaldevice.h"
#include "provider_sensor.h"

#include "valmap_numericsensor.h"

static void nsensor_update(pp_cim_instance_t *instance);
static void nsensor_commit(pp_cim_instance_t *instance);
static pp_cim_method_call_t *nsensor_clp_map_set(pp_cim_instance_t *instance,
    pp_clp_property_value_t *prop);
static char *numericsensor_clp_help_instance(pp_cim_instance_t *instance,
    int verbose);
static char *numericsensor_clp_help_class(pp_cim_class_t *cim_class, int verbose);

static int nsensor_clp_setthreshold(pp_cim_instance_t *instance,
    vector_t *args, pp_cim_data_t *result);
static pp_cim_method_call_t *map_set_threshold(pp_cim_instance_t *instance,
    pp_ipmi_threshold_type_t type, const char *value);

static int get_reading(pp_ipmi_sdr_list_entry_t *sdr_entry,
    pp_cim_propval_t *result);

// provider API functions
static pp_cim_provider_t provider =
{
    .deinit = provider_common_deinit,
    .update = nsensor_update,
    .commit = nsensor_commit,
    .authorize = provider_common_authorize,
    .get_property = provider_common_get_property,
    .set_property = provider_common_set_property,
    .get_properties = provider_common_get_properties,
    .set_properties = provider_common_set_properties,
    .get_method = provider_common_get_method,
    .call_method = provider_common_call_method,
    .clp_map_reset = pp_cim_logicaldevice_map_reset,
    .clp_map_set = nsensor_clp_map_set,
    .clp_map_start = NULL, // 'start' command not supported
    .clp_map_stop = NULL, // 'stop' command not supported
    .clp_help = numericsensor_clp_help_instance
};

// class properties
// { <name>, <type>, <array>, <valmap>, <key>, <priv>, <required>, <writable>, { <null>, { <defvalue> } } }
static pp_cim_property_t properties[] =
{
    {"BaseUnits", PP_CIM_UNSIGNED, 0, baseunits_valmap, 0, 0, 1, 0, {1, {0}}},
    {"UnitModifier", PP_CIM_SIGNED, 0, NULL, 0, 0, 1, 0, {1, {0}}},
    {"RateUnits", PP_CIM_UNSIGNED, 0, rateunits_valmap, 0, 0, 1, 0, {1, {0}}},
    {"CurrentReading", PP_CIM_SIGNED, 0, NULL, 0, 0, 1, 0, {1, {0}}},
    {"NominalReading", PP_CIM_SIGNED, 0, NULL, 0, 0, 1, 0, {1, {0}}},
    {"NormalMax", PP_CIM_SIGNED, 0, NULL, 0, 0, 0, 0, {1, {0}}},
    {"NormalMin", PP_CIM_SIGNED, 0, NULL, 0, 0, 0, 0, {1, {0}}},
    {"MaxReadable", PP_CIM_SIGNED, 0, NULL, 0, 0, 0, 0, {1, {0}}},
    {"MinReadable", PP_CIM_SIGNED, 0, NULL, 0, 0, 0, 0, {1, {0}}},
    {"LowerThresholdNonCritical", PP_CIM_SIGNED, 0, NULL, 0, 0, 0, 1, {1, {0}}},
    {"UpperThresholdNonCritical", PP_CIM_SIGNED, 0, NULL, 0, 0, 0, 1, {1, {0}}},
    {"LowerThresholdCritical", PP_CIM_SIGNED, 0, NULL, 0, 0, 0, 1, {1, {0}}},
    {"UpperThresholdCritical", PP_CIM_SIGNED, 0, NULL, 0, 0, 0, 1, {1, {0}}},
    {"LowerThresholdFatal", PP_CIM_SIGNED, 0, NULL, 0, 0, 0, 1, {1, {0}}},
    {"UpperThresholdFatal", PP_CIM_SIGNED, 0, NULL, 0, 0, 0, 1, {1, {0}}},
    {.name = NULL}
};

// CIM_NumericSensor::CLP_SetThreshold() arguments
// { <name>, <type>, <array>, <valmap>, <in>, <out> }
static pp_cim_method_arg_t nsensor_clp_setthreshold_args[] =
{
    {"Type", PP_CIM_SIGNED, 0, NULL, 1, 0},
    {"Value", PP_CIM_SIGNED, 0, NULL, 1, 0}
};

// class methods
// { <name>, <result>, <num_args>, <args>, <priv>, <defptr> }
static pp_cim_method_t methods[] =
{
    {"CLP_SetThreshold", PP_CIM_VOID, 2, nsensor_clp_setthreshold_args, 1, nsensor_clp_setthreshold},
    {.name = NULL}
};

// class description
pp_cim_class_desc_t pp_cim_numericsensor_desc =
{
    .cim_name = "CIM_NumericSensor",
    .ufct = "nsensor",
    .dispname = "CIM Numeric Sensor",
    .superclass = "CIM_Sensor",
    .assoc = 0,
    .properties = properties,
    .methods = methods,
    .update = NULL,
    .clp_update = NULL,
    .clp_help = numericsensor_clp_help_class
};

// Create a new CIM_NumericSensor instance
pp_cim_instance_t *pp_cim_numericsensor_new()
{
    pp_cim_instance_t *i = pp_cim_instance_new("CIM_NumericSensor", &provider);
    return i;
}

pp_cim_instance_t *pp_cim_numericsensor_getbyid(unsigned int id)
{
    pp_cim_class_t *cim_class = pp_cim_class_lookup("CIM_NumericSensor");
    vector_t *instances = pp_cim_get_instances(cim_class);
    pp_cim_data_t d;
    pp_cim_instance_t *result = NULL;
    unsigned int i;

    d.null = 0;
    d.types.unsigned_int = id;

    for (i = 0; i < vector_size(instances); i++) {
        pp_cim_instance_t *inst = vector_get(instances, i);
        pp_cim_propval_t *pv = provider_common_get_property(inst, "_IPMI_ID");
        if (pv) {
            if (pp_cim_data_equal(PP_CIM_UNSIGNED, pv->data, d)) {
                pp_cim_instance_tag(inst);
                result = inst;
            }
            pp_cim_propval_delete(pv);
        }
        if (result)
            break;
    }

    vector_delete(instances);
    pp_cim_class_release(cim_class);

    return result;
}

void nsensor_update_sdr(pp_cim_instance_t *instance,
    pp_ipmi_sdr_list_entry_t *sdr_entry)
{
    unsigned int i;

    if (sdr_entry->type != PP_IPMI_SENSOR_TYPE_FULL) {
        printf("Warning: Unhandled sensor type %d.\n", sdr_entry->type);
        return;
    }

    for (i = 0; i < vector_size(instance->prop_table); i++) {
        pp_cim_propval_t *pv = vector_get(instance->prop_table, i);
        pp_cim_property_t *p = pv->property;
        if (strcasecmp(p->name, "BaseUnits") == 0) {
            pv->data.null = 0;
            pv->data.types.unsigned_int = sdr_entry->data.full.cim_base_unit;
        } else if (strcasecmp(p->name, "UnitModifier") == 0) {
            pv->data.null = 0;
            pv->data.types.signed_int = sdr_entry->data.full.cim_unit_modifier;
        } else if (strcasecmp(p->name, "RateUnits") == 0) {
            pv->data.null = 0;
            pv->data.types.unsigned_int = sdr_entry->data.full.cim_rate_units;
        } else {
            get_reading(sdr_entry, pv);
        }
    }
}

static void nsensor_update(pp_cim_instance_t *instance)
{
    pp_ipmi_sdr_list_entry_t *sdr_entry;
    pp_ipmi_return_t *ipmi_ret = malloc(sizeof(pp_ipmi_return_t));
    pp_ipmi_parameter_t ipmi_param;
    pp_cim_propval_t *propval;
    int ret, err;

    provider_common_update(instance);

    memset(&ipmi_param, 0, sizeof(ipmi_param));
    ipmi_param.data.sensor_get_by_id_list = vector_new2(NULL, 1,
        sizeof(int), NULL);

    propval = instance->provider->get_property(instance, "_IPMI_ID");
    vector_add2(ipmi_param.data.sensor_get_by_name_list,
        &propval->data.types.unsigned_int);
    pp_cim_propval_delete(propval);

    ret = pp_ipmi_send_command(PP_IPMI_CMD_SENSOR,
        PP_IPMI_SENSOR_SUBCMD_GET_BY_ID, &ipmi_param, ipmi_ret,
        &err, NULL);

    vector_delete(ipmi_param.data.sensor_get_by_id_list);

    sdr_entry = vector_get(ipmi_ret->data.sensor_get_list, 0);
    sensor_update_sdr(instance, sdr_entry);
    nsensor_update_sdr(instance, sdr_entry);

    pp_ipmi_cleanup_ret(ipmi_ret);
    free(ipmi_ret);
}

static void nsensor_commit(pp_cim_instance_t *instance)
{
    pp_cim_propval_t *pv;
    pp_ipmi_parameter_t ipmi_param;
    char *sensorname;
    int err;
    unsigned int i;

    pv = instance->provider->get_property(instance, "_IPMI_Name");
    if (!pv || !pv->data.types.string)
        return;
    sensorname = strdup(pv->data.types.string);
    pp_cim_propval_delete(pv);

    memset(&ipmi_param, 0, sizeof(ipmi_param));
    ipmi_param.data.sensor_threshold_list = vector_new2(NULL, 1,
        sizeof(pp_ipmi_sensor_set_th_param_t), NULL);

    for (i = 0; i < vector_size(instance->prop_table); i++) {
        pp_ipmi_sensor_set_th_param_t threshold;
        int type = 0;
	const char *name;

        pv = vector_get(instance->prop_table, i);
        name = pv->property->name;

        if (!pv->changed) continue;
        pv->changed = 0;

        if (strcasecmp(name, "LowerThresholdNonCritical") == 0) {
            type = PP_IPMI_SENSOR_THRESHOLD_LOWER_NON_CRIT;
        } else if (strcasecmp(name, "UpperThresholdNonCritical") == 0) {
            type = PP_IPMI_SENSOR_THRESHOLD_UPPER_NON_CRIT;
        } else if (strcasecmp(name, "LowerThresholdCritical") == 0) {
            type = PP_IPMI_SENSOR_THRESHOLD_LOWER_CRIT;
        } else if (strcasecmp(name, "UpperThresholdCritical") == 0) {
            type = PP_IPMI_SENSOR_THRESHOLD_UPPER_CRIT;
        } else if (strcasecmp(name, "LowerThresholdFatal") == 0) {
            type = PP_IPMI_SENSOR_THRESHOLD_LOWER_NON_RECOV;
        } else if (strcasecmp(name, "UpperThresholdFatal") == 0) {
            type = PP_IPMI_SENSOR_THRESHOLD_UPPER_NON_RECOV;
        } else {
            continue;
        }

        threshold.sensor_name = sensorname;
        threshold.threshold_type = type;
        threshold.use_cim = 1;
        threshold.cim_value = pv->data.types.signed_int;
        vector_add2(ipmi_param.data.sensor_threshold_list, &threshold);
    }

    if (pp_ipmi_send_command(PP_IPMI_CMD_SENSOR,
        PP_IPMI_SENSOR_SUBCMD_THRESHOLD, &ipmi_param, NULL,
        &err, NULL)) {
        printf("IPMI failure: %s.\n", pp_ipmi_get_error_string(err));
    }

    vector_delete(ipmi_param.data.sensor_threshold_list);
    free(sensorname);
}

static pp_cim_method_call_t *nsensor_clp_map_set(pp_cim_instance_t *instance,
    pp_clp_property_value_t *prop)
{
    if (strcasecmp(prop->property, "LowerThresholdNonCritical") == 0) {
        return map_set_threshold(instance, PP_IPMI_SENSOR_THRESHOLD_LOWER_NON_CRIT, prop->value);
    } else if (strcasecmp(prop->property, "UpperThresholdNonCritical") == 0) {
        return map_set_threshold(instance, PP_IPMI_SENSOR_THRESHOLD_UPPER_NON_CRIT, prop->value);
    } else if (strcasecmp(prop->property, "LowerThresholdCritical") == 0) {
        return map_set_threshold(instance, PP_IPMI_SENSOR_THRESHOLD_LOWER_CRIT, prop->value);
    } else if (strcasecmp(prop->property, "UpperThresholdCritical") == 0) {
        return map_set_threshold(instance, PP_IPMI_SENSOR_THRESHOLD_UPPER_CRIT, prop->value);
    } else if (strcasecmp(prop->property, "LowerThresholdFatal") == 0) {
        return map_set_threshold(instance, PP_IPMI_SENSOR_THRESHOLD_LOWER_NON_RECOV, prop->value);
    } else if (strcasecmp(prop->property, "UpperThresholdFatal") == 0) {
        return map_set_threshold(instance, PP_IPMI_SENSOR_THRESHOLD_UPPER_NON_RECOV, prop->value);
    }
    return NULL;
}

static pp_cim_method_call_t *map_set_threshold(pp_cim_instance_t *instance,
    pp_ipmi_threshold_type_t type, const char *value)
{
    pp_cim_methodptr_t *methodptr;
    pp_cim_method_call_t *call;
    pp_cim_data_t *arg;
    int val;
    char *c;

    val = strtol(value, &c, 10);
    if (*c)
        return NULL;

    methodptr = instance->provider->get_method(instance, "CLP_SetThreshold");
    call = pp_cim_method_call_new(methodptr, instance);

    arg = malloc(sizeof(pp_cim_data_t));
    arg->null = 0;
    arg->types.signed_int = type;
    vector_add(call->args, arg);

    arg = malloc(sizeof(pp_cim_data_t));
    arg->null = 0;
    arg->types.signed_int = val;
    vector_add(call->args, arg);

    return call;
}

static const char clp_help_message[] =
    "The CIM_NumericSensor class is used to store information about threshold-\r\n"
    "based IPMI sensors and to query the current sensor state. It supports the\r\n"
    "following CLP operations:\r\n"
    "  - Change a threshold level:   SET LowerThresholdCritical=10\r\n"
    "  - Re-arm the numericsensor:   RESET\r\n";

static char *numericsensor_clp_help_instance(pp_cim_instance_t *instance UNUSED,
    int verbose UNUSED)
{
    return strdup(clp_help_message);
}

static char *numericsensor_clp_help_class(pp_cim_class_t *cim_class UNUSED,
    int verbose UNUSED)
{
    return strdup(clp_help_message);
}

// Method: CIM_NumericSensor::CLP_SetThreshold
static int nsensor_clp_setthreshold(pp_cim_instance_t *instance,
    vector_t *args, pp_cim_data_t *result UNUSED)
{
    pp_cim_data_t *arg;
    int type, value;
    pp_ipmi_parameter_t ipmi_param;
    pp_ipmi_sensor_set_th_param_t threshold;
    pp_cim_propval_t *pv;
    int ret, err;

    arg = vector_get(args, 0);
    type = arg->types.signed_int;
    arg = vector_get(args, 1);
    value = arg->types.signed_int;

    pp_cim_debug("CIM_NumericSensor::CLP_SetThreshold(%d, %d)\n", type, value);

    memset(&ipmi_param, 0, sizeof(ipmi_param));
    ipmi_param.data.sensor_threshold_list = vector_new2(NULL, 1,
        sizeof(pp_ipmi_sensor_set_th_param_t), NULL);

    pv = instance->provider->get_property(instance, "_IPMI_Name");
    threshold.sensor_name = strdup(pv->data.types.string);
    pp_cim_propval_delete(pv);
    threshold.threshold_type = type;
    threshold.use_cim = 1;
    threshold.cim_value = value;
    vector_add2(ipmi_param.data.sensor_threshold_list, &threshold);

    ret = pp_ipmi_send_command(PP_IPMI_CMD_SENSOR,
        PP_IPMI_SENSOR_SUBCMD_THRESHOLD, &ipmi_param, NULL,
        &err, NULL);

    if (ret) {
        printf("IPMI failure: %s.\n", pp_ipmi_get_error_string(err));
        return PP_ERR;
    }

    vector_delete(ipmi_param.data.sensor_threshold_list);
    free(threshold.sensor_name);

    return PP_SUC;
}

static int get_reading(pp_ipmi_sdr_list_entry_t *sdr_entry,
    pp_cim_propval_t *result)
{
    int reading;

    if (strcasecmp(result->property->name, "CurrentReading") == 0) {
        reading = sdr_entry->data.full.reading.cim_value;
    } else if (strcasecmp(result->property->name, "NominalReading") == 0) {
        reading = sdr_entry->data.full.nominal_reading.cim_value;
    } else if (strcasecmp(result->property->name, "NormalMax") == 0) {
        reading = sdr_entry->data.full.normal_maximum.cim_value;
    } else if (strcasecmp(result->property->name, "NormalMin") == 0) {
        reading = sdr_entry->data.full.normal_minimum.cim_value;
    } else if (strcasecmp(result->property->name, "MaxReadable") == 0) {
        reading = sdr_entry->data.full.maximum_sensor_range.cim_value;
    } else if (strcasecmp(result->property->name, "MinReadable") == 0) {
        reading = sdr_entry->data.full.minimum_sensor_range.cim_value;
    } else if (strcasecmp(result->property->name, "LowerThresholdNonCritical") == 0) {
        reading = sdr_entry->data.full.lower_non_critical.cim_value;
    } else if (strcasecmp(result->property->name, "UpperThresholdNonCritical") == 0) {
        reading = sdr_entry->data.full.upper_non_critical.cim_value;
    } else if (strcasecmp(result->property->name, "LowerThresholdCritical") == 0) {
        reading = sdr_entry->data.full.lower_critical.cim_value;
    } else if (strcasecmp(result->property->name, "UpperThresholdCritical") == 0) {
        reading = sdr_entry->data.full.upper_critical.cim_value;
    } else if (strcasecmp(result->property->name, "LowerThresholdFatal") == 0) {
        reading = sdr_entry->data.full.lower_non_recoverable.cim_value;
    } else if (strcasecmp(result->property->name, "UpperThresholdFatal") == 0) {
        reading = sdr_entry->data.full.upper_non_recoverable.cim_value;
    } else {
        return PP_ERR;
    }

    result->data.null = 0;
    result->data.types.signed_int = reading;
    return PP_SUC;
}

