/*
 * PP_OutletGroup class implementation
 *
 * Note: For demonstration purposes only, this class is not covered
 *       by any spec.
 */

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <pp/cim.h>
#include <pp/cimTypes.h>
#include <pp/cim_provider.h>
#include <pp_gpio.h>
#include <pp/ipmi.h>
#include "cim_common.h"
#include "instance.h"
#include "provider_common.h"
#include "provider_outletgroup.h"
#include "valmap_outlet.h"
#include "namespace.h"
#include "provider_component.h"
#include "provider_memberofcollection.h"

static void outletgroup_update(pp_cim_instance_t *instance);
static pp_cim_method_call_t *outletgroup_map_set(pp_cim_instance_t *instance,
    pp_clp_property_value_t *prop);
static int outletgroup_setpowerstate(pp_cim_instance_t *instance, vector_t *args,
    pp_cim_data_t *result);
static int outletgroup_setpowerondelay(pp_cim_instance_t *instance, vector_t *args,
    pp_cim_data_t *result);
static int outletgroup_addmember(pp_cim_instance_t *instance, vector_t *args,
    pp_cim_data_t *result);
static int outletgroup_delmember(pp_cim_instance_t *instance, vector_t *args,
    pp_cim_data_t *result);
static int outletgroup_clp_create(pp_cim_instance_t *instance, vector_t *args,
    pp_cim_data_t *result UNUSED);
static int outletgroup_clp_delete(pp_cim_instance_t *instance, vector_t *args UNUSED,
    pp_cim_data_t *result UNUSED);

// provider API functions
static pp_cim_provider_t provider =
{
    .deinit = provider_common_deinit,
    .update = outletgroup_update,
    .commit = provider_common_commit,
    .authorize = provider_common_authorize,
    .get_property = provider_common_get_property,
    .set_property = provider_common_set_property,
    .get_properties = provider_common_get_properties,
    .set_properties = provider_common_set_properties,
    .get_method = provider_common_get_method,
    .call_method = provider_common_call_method,
    .clp_map_reset = NULL, // 'reset' command not supported
    .clp_map_set = outletgroup_map_set,
    .clp_map_start = NULL, // 'start' command not supported
    .clp_map_stop = NULL // 'stop' command not supported
};

// class properties
// { <name>, <type>, <array>, <valmap>, <key>, <priv>, <required>, <writable>, { <null>, { <defvalue> } } }
static pp_cim_property_t properties[] =
{
    {"Name", PP_CIM_STRING_CONST, 0, NULL, 1, 0, 1, 0, {1, {0}}},
    {"powerState", PP_CIM_UNSIGNED, 0, outlet_valmap, 0, 1, 1, 1, {0, {.unsigned_int = 1}}},
    {"powerOnDelay", PP_CIM_UNSIGNED, 0, NULL, 0, 0, 1, 1, {0, {0}}},
    {.name = NULL}
};

// PP_OutletGroup::SetEnaboutletgroup() arguments
// { <name>, <type>, <array>, <valmap>, <in>, <out> }
static pp_cim_method_arg_t outletgroup_setpowerstate_args[] =
{
    {"powerState", PP_CIM_UNSIGNED, 0, outlet_valmap, 1, 0}
};

static pp_cim_method_arg_t outletgroup_setpowerondelay_args[] =
{
    {"powerOnDelay", PP_CIM_UNSIGNED, 0, NULL, 1, 0}
};


// class methods
// { <name>, <result>, <num_args>, <args>, <priv>, <defptr> }
static pp_cim_method_t methods[] =
{
    {"setPowerState", PP_CIM_VOID, 1, outletgroup_setpowerstate_args, 0, outletgroup_setpowerstate},
    {"setPowerOnDelay", PP_CIM_VOID, 1, outletgroup_setpowerondelay_args, 0, outletgroup_setpowerondelay},
    {"addMember", PP_CIM_VOID, 0, NULL, 0, outletgroup_addmember},
    {"delMember", PP_CIM_VOID, 0, NULL, 0, outletgroup_delmember},
    {"CLP_Create", PP_CIM_VOID, 0, NULL, 0, outletgroup_clp_create},
    {"CLP_Delete", PP_CIM_VOID, 0, NULL, 0, outletgroup_clp_delete},
    {.name = NULL}
};

// class description
pp_cim_class_desc_t pp_cim_outletgroup_desc =
{
    .cim_name = "CIM_OutletGroup",
    .dispname = "OUTLETGROUP",
    .ufct = "outletgroup",
    .superclass = "CIM_Collection",
    .assoc = 0,
    .properties = properties,
    .methods = methods,
    .update = NULL,
    .clp_update = NULL
};

// Create a new PP_OutletGroup instance
pp_cim_instance_t *pp_cim_outletgroup_new()
{
    pp_cim_instance_t *i = pp_cim_instance_new("CIM_OutletGroup", &provider);
    return i;
}

static int pp_cim_outletgroup_get_membership(int group, char *enable, char *mem)
{
    int err, ret;
    pp_ipmi_parameter_t params;
    //pp_ipmi_return_t *ipmi_ret = malloc(sizeof(pp_ipmi_return_t));
    pp_ipmi_return_t ipmi_ret;

    memset(&params, 0, sizeof(params));
    params.data.oem_pp_rpc_get_group_membership.group = group;
    ret = pp_ipmi_send_command(PP_IPMI_CMD_OEM_PP_RPC,
        PP_IPMI_OEM_PP_RPC_GET_GROUP_MEMBERSHIP,
        &params, &ipmi_ret, &err, NULL);
    if (ret) {
        printf("IPMI failure: %s.\n", pp_ipmi_get_error_string(err));
        //free(ipmi_ret);
        return 0;
    }
    if (enable != NULL)
        *enable = ipmi_ret.data.oem_pp_rpc_get_group_membership_rs.enable;

    if (mem != NULL) {
        mem[0] = ipmi_ret.data.oem_pp_rpc_get_group_membership_rs.mem[0];
        mem[1] = ipmi_ret.data.oem_pp_rpc_get_group_membership_rs.mem[1];
        mem[2] = ipmi_ret.data.oem_pp_rpc_get_group_membership_rs.mem[2];
    }
    pp_ipmi_cleanup_ret(&ipmi_ret);
    //free(ipmi_ret);

    /*
       if (mem != NULL) {
       printf("%s %d %x %x %x\n",
       __FUNCTION__, *enable, mem[0], mem[1], mem[2]);
       } else {
       printf("%s %d\n", __FUNCTION__, *enable);
       }
    */
    return 1;
}

static int pp_cim_outletgroup_set_membership(int group, char enable, char *mem)
{
    int err, ret;
    pp_ipmi_parameter_t params;

    if (mem != NULL)
        printf("%s %d %x %x %x\n", __FUNCTION__, enable, mem[0], mem[1], mem[2]);

    memset(&params, 0, sizeof(params));
    params.data.oem_pp_rpc_set_group_membership.group = group;
    params.data.oem_pp_rpc_set_group_membership.enable = enable;
    if (mem != NULL) {
        params.data.oem_pp_rpc_set_group_membership.mem[0] = mem[0];
        params.data.oem_pp_rpc_set_group_membership.mem[1] = mem[1];
        params.data.oem_pp_rpc_set_group_membership.mem[2] = mem[2];
    } else {
        params.data.oem_pp_rpc_set_group_membership.mem[0] = 0;
        params.data.oem_pp_rpc_set_group_membership.mem[1] = 0;
        params.data.oem_pp_rpc_set_group_membership.mem[2] = 0;
    }
    ret = pp_ipmi_send_command(PP_IPMI_CMD_OEM_PP_RPC,
        PP_IPMI_OEM_PP_RPC_SET_GROUP_MEMBERSHIP,
        &params, NULL, &err, NULL);
    if (ret) {
        printf("IPMI failure: %s.\n", pp_ipmi_get_error_string(err));
        return 0;
    }
    return 1;
}

static int pp_cim_outletgroup_set_enable(int group, char enable)
{
    // clear all member
    char mem[3] = {0, 0, 0};
    return pp_cim_outletgroup_set_membership(group, enable, mem);

}

void pp_cim_outletgroup_discovery(pp_cim_instance_t *instance, vector_t *outlets)
{
    int i;
    char enable;
    char mem[3];

    for (i = 0; i < 24; i++) {
        if (pp_cim_outletgroup_get_membership(i, &enable, mem) == 0)
            return;

        if (enable) {
            pp_cim_instance_t *outletgroup = pp_cim_outletgroup_new();

            pp_cim_data_t name, state;
            char buf[24];
            name.null = 0;
            state.null = 0;
            snprintf(buf, sizeof(buf), "OUTLETGROUP%d", i + 1);
            name.types.string_const = buf;
            state.types.unsigned_int = 1;
            provider_common_set_property(outletgroup, "Name", name, 1);
            provider_common_set_property(outletgroup, "powerState", state, 1);

            pp_cim_add_instance(outletgroup, instance);
            outletgroup->instance_id = i + 1;

            pp_cim_component_new("CIM_SystemDevice", instance, outletgroup);

            // create memberofcollection
            int j;
            for (j = 0; j < 24; j++) {
                if (mem[j/8] & (0x01 << (j % 8))) {
                    pp_cim_instance_t *outlet = vector_get(outlets, j);
                    pp_cim_memberofcollection_new(outletgroup, outlet);
                }
            }
        }
    }

}
/*
 * Looking for a new outletgroup.
 * if the id < 0, look for next one. Or look for the specific one.
 */
static pp_cim_instance_t *pp_cim_outletgroup_findnew(int id)
{
    pp_cim_instance_t *inst;
    pp_cim_data_t name, state;
    char buf[24];

    // find disable group
    int group_id = -1;
    char enable;
    if (id < 0) {
        // find next one
        int i;
        for (i = 0; i < 24; i++) {
            // get state from IPMI
            int ret = pp_cim_outletgroup_get_membership(i, &enable, NULL);
            if (ret == 0) {
                return NULL;
            }
            if (!enable) {
                group_id = i;
                break;
            }
        }
    } else {
        if (pp_cim_outletgroup_get_membership(id, &enable, NULL) == 0)
            return NULL;
        if (!enable) {
            group_id = id;
        }
    }

    // enable it
    if (group_id != -1) {
        pp_cim_outletgroup_set_enable(group_id, 1);
    } else {
        // no find specific one, or all outletgroup already enable
        //printf("not found\n");
        return NULL;
    }

    inst = pp_cim_outletgroup_new();

    inst->instance_id = group_id + 1;

    snprintf(buf, sizeof(buf), "OUTLETGROUP%d", group_id + 1);
    name.null = 0;
    state.null = 0;
    name.types.string_const = buf;
    state.types.unsigned_int = 1;
    provider_common_set_property(inst, "Name", name, 1);
    provider_common_set_property(inst, "powerState", state, 1);

    return inst;
}

static void outletgroup_update(pp_cim_instance_t *instance)
{
    pp_cim_data_t data;

    provider_common_update(instance);

    // power on delay of group
    // get state from IPMI
    int err, ret;
    pp_ipmi_parameter_t params;
    pp_ipmi_return_t ipmi_ret;

    memset(&params, 0, sizeof(params));
    params.data.oem_pp_rpc_get_group_power_on_delay.group = instance->instance_id - 1;
    //printf("flag1\n");
    ret = pp_ipmi_send_command(PP_IPMI_CMD_OEM_PP_RPC,
        PP_IPMI_OEM_PP_RPC_GET_GROUP_POWER_ON_DELAY,
        &params, &ipmi_ret, &err, NULL);
    if (ret) {
        printf("IPMI failure: %s.\n", pp_ipmi_get_error_string(err));
        return;
    }
    /* printf("%s: %d %d\n", __FUNCTION__, instance->instance_id,
       ipmi_ret->data.oem_pp_rpc_get_group_power_on_delay_rs.delay); */

    // set to property
    data.null = 0;
    data.types.unsigned_int = ipmi_ret.data.oem_pp_rpc_get_group_power_on_delay_rs.delay;
    pp_cim_set_property(instance, "powerOnDelay", data);

    pp_ipmi_cleanup_ret(&ipmi_ret);
}

static pp_cim_method_call_t *outletgroup_map_set(pp_cim_instance_t *instance,
    pp_clp_property_value_t *prop)
{
    pp_cim_data_t *arg;
    pp_cim_methodptr_t *methodptr;
    pp_cim_method_call_t *call;

    if (strcasecmp(prop->property, "powerState") == 0) {
        char value[32];
        methodptr = provider_common_get_method(instance, "setPowerState");
        call = pp_cim_method_call_new(methodptr, instance);

        // sywang: text to number
        value[1] = 0;
        if (!strcmp(prop->value, "on") || !strcmp(prop->value, "ON"))
            value[0] = '1';
        else if (!strcmp(prop->value, "off") || !strcmp(prop->value, "OFF"))
            value[0] = '2';
        else if (!strcmp(prop->value, "cycle") || !strcmp(prop->value, "CYCLE"))
            value[0] = '3';
        else {
            unsigned int i;
            for (i = 0; i < strlen(prop->value); i++) {
                if (i == 32) {
                    value[31] = 0;
                    break;
                }
                if (isdigit(prop->value[i]))
                    value[i] = prop->value[i];
                else {
                    value[0] = '0';
                    value[1] = 0;
                    break;
                }
                value[i+1] = 0;
            }
        }
        arg = pp_cim_data_parse(call->methodptr->method->args[0].type, value); //prop->value);
        // printf("%s: %d\n", __FUNCTION__, arg->types.unsigned_int);
        if (!arg)
            return NULL;
        vector_add(call->args, arg);
        return call;
    } else if (strcasecmp(prop->property, "powerOnDelay") == 0) {
        methodptr = provider_common_get_method(instance, "setPowerOnDelay");
        call = pp_cim_method_call_new(methodptr, instance);
        arg = pp_cim_data_parse(call->methodptr->method->args[0].type, prop->value);
        // printf("%s: %d\n", __FUNCTION__, arg->types.unsigned_int);
        if (!arg)
            return NULL;
        vector_add(call->args, arg);

        return call;
    }

    return NULL;
}

static int outletgroup_setpowerstate(pp_cim_instance_t *instance, vector_t *args,
    pp_cim_data_t *result UNUSED)
{
    pp_cim_data_t *arg;
    //  pp_cim_propval_t *pv;
    int state;

    if (vector_size(args) != 1)
        return PP_ERR;
    arg = vector_get(args, 0);
    state = arg->types.unsigned_int;

    // write to IPMI
    int err, ret;
    pp_ipmi_parameter_t params;

    memset(&params, 0, sizeof(params));
    params.data.oem_pp_rpc_set_group_state.group = instance->instance_id - 1;

    // value map again
    if (state == 1)
        params.data.oem_pp_rpc_set_group_state.status = 1;
    else
        params.data.oem_pp_rpc_set_group_state.status = 0;

    ret = pp_ipmi_send_command(PP_IPMI_CMD_OEM_PP_RPC,
        PP_IPMI_OEM_PP_RPC_SET_GROUP_STATE,
        &params, NULL, &err, NULL);
    if (ret) {
        printf("IPMI failure: %s.\n", pp_ipmi_get_error_string(err));
    }

    return PP_SUC;
}

static int outletgroup_setpowerondelay(pp_cim_instance_t *instance, vector_t *args,
    pp_cim_data_t *result UNUSED)
{
    pp_cim_data_t *arg;

    if (vector_size(args) != 1)
        return PP_ERR;
    arg = vector_get(args, 0);

    // write to IPMI
    int err, ret;
    pp_ipmi_parameter_t params;

    memset(&params, 0, sizeof(params));
    params.data.oem_pp_rpc_set_group_power_on_delay.group = instance->instance_id - 1;
    params.data.oem_pp_rpc_set_group_power_on_delay.delay = arg->types.unsigned_int;
    /* printf("%s : %d %d\n", __FUNCTION__,
       params.data.oem_pp_rpc_set_group_power_on_delay.group,
       params.data.oem_pp_rpc_set_group_power_on_delay.delay); */
    ret = pp_ipmi_send_command(PP_IPMI_CMD_OEM_PP_RPC,
        PP_IPMI_OEM_PP_RPC_SET_GROUP_POWER_ON_DELAY,
        &params, NULL, &err, NULL);
    if (ret) {
        printf("IPMI failure: %s.\n", pp_ipmi_get_error_string(err));
    }

    return PP_SUC;
}

static int outletgroup_addmember(pp_cim_instance_t *instance, vector_t *args,
    pp_cim_data_t *result UNUSED)
{
    pp_cim_data_t *arg;

    if (vector_size(args) != 1)
        return PP_ERR;
    arg = vector_get(args, 0);

    char enable;
    char value[3] = {0, 0, 0};
    char mem[3];
    int group = instance->instance_id - 1;
    char receptacle = arg->types.reference->instance_id - 1;

    if (pp_cim_outletgroup_get_membership(group, &enable, mem) == 0)
        return 0; // error

    int i = receptacle / 8;
    int j = receptacle % 8;
    value[i] = 0x01 << j;
    mem[0] = mem[0] | value[0];
    mem[1] = mem[1] | value[1];
    mem[2] = mem[2] | value[2];

    if (pp_cim_outletgroup_set_membership(group, enable, mem) == 0)
        return PP_ERR;
    else
        return PP_SUC;
}

static int outletgroup_delmember(pp_cim_instance_t *instance, vector_t *args,
    pp_cim_data_t *result UNUSED)
{
    pp_cim_data_t *arg;

    if (vector_size(args) != 1)
        return PP_ERR;
    arg = vector_get(args, 0);

    char enable;
    char value[3] = {0xff, 0xff, 0xff};
    char mem[3];
    int group = instance->instance_id - 1;
    char receptacle = arg->types.reference->instance_id - 1;

    if (pp_cim_outletgroup_get_membership(group, &enable, mem) == 0)
        return PP_ERR; // error

    int i = receptacle / 8;
    int j = receptacle % 8;
    value[i] = 0xff - (0x01 << j);
    mem[0] = mem[0] & value[0];
    mem[1] = mem[1] & value[1];
    mem[2] = mem[2] & value[2];

    if (pp_cim_outletgroup_set_membership(group, enable, mem) == 0)
        return PP_ERR;

    return PP_SUC;
}
/*
 * Create outletgroup under CIM_PowerStrip
 *
 */
static int outletgroup_clp_create(pp_cim_instance_t *instance, vector_t *args,
    pp_cim_data_t *result)
{
    int id = -1;
    pp_cim_instance_t *outletgroup;

    // check the name of the instance
    if (strcmp(instance->cim_class->cim_name, "CIM_PowerStrip")) {
        //printf("%s\n", instance->cim_class->cim_name);
        result->null = 0;
        result->types.reference = NULL; // return error
        return PP_ERR;
    }

    if (vector_size(args) > 0) {
        pp_clp_property_value_t *pv = vector_get(args, 0);
        if (!strcasecmp(pv->property, "instance_id")) {
            id = atoi(pv->value) - 1;
        }
    }

    // create a new instance
    outletgroup = pp_cim_outletgroup_findnew(id);

    if (outletgroup != NULL) {
        //backup instance_id;
        id = outletgroup->instance_id;
        pp_cim_add_instance(outletgroup, instance);
        outletgroup->instance_id = id;
        pp_cim_component_new("CIM_SystemDevice", instance, outletgroup);
        result->null = 0;
        result->types.reference = outletgroup; // return succeed
        return PP_SUC;
    } else {
        result->null = 0;
        result->types.reference = NULL;
        return PP_ERR;
    }
}

/*
 * Delete outletgroup under CIM_PowerStrip
 *
 */
static int outletgroup_clp_delete(pp_cim_instance_t *instance, vector_t *args UNUSED,
    pp_cim_data_t *result UNUSED)
{
    int group_id = instance->instance_id - 1;
    //instance->provider->deinit(instance);	//useless, the function is empty

    // remove SystemDeivce
    pp_cim_class_t *sd = pp_cim_class_lookup("CIM_SystemDevice");
    vector_t *v = pp_cim_get_instances(sd);
    //vector_t *v = sd->instances;
    pp_cim_instance_t *p;
    pp_cim_propval_t *pv;
    unsigned int i = 0;

    for (i = 0; i < vector_size(v); i++) {
        p = vector_get(v, i);
        pv = provider_common_get_property(p, "PartComponent");
        if (pv) {
            if (pv->data.types.reference == instance) {
                //pp_cim_instance_dump(p);
                //vector_remove(v, i);
                pp_cim_instance_unregister(p);
            }
        }
    }
    vector_delete(v);
    pp_cim_class_release(sd);

    printf("remove memberofcollection\n");
    // remove CIM_MemberOfCollection
    sd = pp_cim_class_lookup("CIM_MemberOfCollection");
    v = pp_cim_get_instances(sd);
    //v = sd->instances;

    for (i = 0; i < vector_size(v); i++) {
        p = vector_get(v, i);
        pv = provider_common_get_property(p, "Member");
        if (pv) {
            if (pv->data.types.reference == instance) {
                //pp_cim_instance_dump(p);
                //vector_remove(v, i);
                pp_cim_instance_unregister(p);
            }
        }
    }
    vector_delete(v);
    pp_cim_class_release(sd);

    printf("remove from tree\n");
    // remove from tree
    pp_cim_instance_remove_from_list(instance);

    printf("unregister instance\n");
    // unregister instance
    pp_cim_instance_unregister(instance);

    printf("remove from IPMI\n");
    // remove from IPMI
    if (pp_cim_outletgroup_set_membership(group_id, 0, NULL) == 0)
        return PP_ERR;

    return PP_SUC;
}
