#include <pp/cim.h>
#include <pp/cimTypes.h>
#include <pp/cim_provider.h>
#include "cim_common.h"
#include "instance.h"
#include "namespace.h"
#include "provider_common.h"
#include "provider_recordlog.h"
#include "provider_logrecord.h"
#include <pp/ipmi.h>

// forward declarations
static char *recordlog_clp_help_instance(pp_cim_instance_t *instance,
    int verbose);
static char *recordlog_clp_help_class(pp_cim_class_t *cim_class, int verbose);
static void recordlog_clp_update(pp_cim_class_t *cim_class);

static pp_cim_method_call_t *recordlog_map_reset(pp_cim_instance_t *instance);

static int recordlog_clearlog(pp_cim_instance_t *instance,
    vector_t *args, pp_cim_data_t *result);

// provider API functions
static pp_cim_provider_t provider =
{
    .deinit = provider_common_deinit,
    .update = provider_common_update,
    .commit = provider_common_commit,
    .authorize = provider_common_authorize,
    .get_property = provider_common_get_property,
    .set_property = provider_common_set_property,
    .get_properties = provider_common_get_properties,
    .set_properties = provider_common_set_properties,
    .get_method = provider_common_get_method,
    .call_method = provider_common_call_method,
    .clp_map_reset = recordlog_map_reset,
    .clp_map_set = NULL, // 'set' command not supported
    .clp_map_start = NULL, // 'start' command not supported
    .clp_map_stop = NULL, // 'stop' command not supported
    .clp_help = recordlog_clp_help_instance
};
// class properties
// { <name>, <type>, <array>, <valmap>, <key>, <priv>, <required>, <writable>, { <null>, { <defvalue> } } }
static pp_cim_property_t properties[] =
{
    {"InstanceID", PP_CIM_STRING_CONST, 0, NULL, 1, 0, 1, 0, {1, {0}}},
    {.name = NULL}
};

// class methods
// { <name>, <result>, <num_args>, <args>, <priv>, <defptr> }
static pp_cim_method_t methods[] =
{
    {"ClearLog", PP_CIM_UNSIGNED, 0, NULL, 0, recordlog_clearlog},
    {.name = NULL}
};

// class description
pp_cim_class_desc_t pp_cim_recordlog_desc =
{
    .cim_name = "CIM_RecordLog",
    .ufct = "log",
    .dispname = "CIM Record Log",
    .superclass = "CIM_Log",
    .assoc = 0,
    .properties = properties,
    .methods = methods,
    .update = NULL,
    .clp_update = recordlog_clp_update,
    .clp_help = recordlog_clp_help_class
};

// Create a new CIM_RecordLog instance
pp_cim_instance_t *pp_cim_recordlog_new()
{
    pp_cim_data_t d;
    pp_ipmi_return_t ipmi_ret;
    int err;
    d.null = 0;

    pp_cim_instance_t *i = pp_cim_instance_new("CIM_RecordLog", &provider);
    d.types.string_const = "IPMI: Host SEL Log";
    provider_common_set_property(i, "InstanceID", d, 1);
    d.types.unsigned_int = 2;
    provider_common_set_property(i, "EnabledState", d, 1);
    d.types.string_const = "IPMI SEL Log";
    provider_common_set_property(i, "Name", d, 1);
    d.types.string_const = "IPMI SEL";
    provider_common_set_property(i, "Caption", d, 1);
    provider_common_set_property(i, "Description", d, 1);
    provider_common_set_property(i, "ElementName", d, 1);

    // Get maximum number of records
    pp_ipmi_send_command(PP_IPMI_CMD_SEL, PP_IPMI_SEL_SUBCMD_INFO,
        NULL, &ipmi_ret, &err, NULL);
    if (ipmi_ret.data.sel_info.get_alloc_info_supported) {
        d.types.unsigned_int =
            ipmi_ret.data.sel_info.allocation_info.alloc_units.units;
        provider_common_set_property(i, "MaxNumberOfRecords", d, 1);
    }
    pp_ipmi_cleanup_ret(&ipmi_ret);

    return i;
}

static const char clp_help_message[] =
    "The CIM_RecordLog class is used to store information about a message log.\r\n"
    "A new CIM_LogRecord instance is generated for each log entry and associated\r\n"
    "with the log via a CIM_RecordInLog association instance.\r\n"
    "The CIM_RecordLog supports the following CLP operations:\r\n"
    "  - Clear the log:   RESET\r\n";

static char *recordlog_clp_help_instance(pp_cim_instance_t *instance UNUSED,
    int verbose UNUSED)
{
    return strdup(clp_help_message);
}

static char *recordlog_clp_help_class(pp_cim_class_t *cim_class UNUSED,
    int verbose UNUSED)
{
    return strdup(clp_help_message);
}

static void recordlog_clp_update(pp_cim_class_t *cim_class UNUSED)
{
    pp_cim_class_t *record = pp_cim_class_lookup("CIM_LogRecord");
    pp_cim_class_update(record);
    pp_cim_class_release(record);
}

static pp_cim_method_call_t *recordlog_map_reset(pp_cim_instance_t *instance)
{
    pp_cim_methodptr_t *methodptr;
    pp_cim_method_call_t *call;

    methodptr = instance->provider->get_method(instance, "ClearLog");
    call = pp_cim_method_call_new(methodptr, instance);

    return call;
}

static int recordlog_clearlog(pp_cim_instance_t *instance UNUSED,
    vector_t *args UNUSED, pp_cim_data_t *result)
{
    int ret, err;

    pp_cim_debug("CIM_RecordLog::ClearLog()\n");
    if (result) {
        result->null = 0; // "Completed with no error"
        result->types.unsigned_int = 0;
    }

    ret = pp_ipmi_send_command(PP_IPMI_CMD_SEL,
        PP_IPMI_SEL_SUBCMD_CLEAR, NULL, NULL,
        &err, NULL);
    if (ret) {
        printf("IPMI failure: %s.\n", pp_ipmi_get_error_string(err));
        if (result)
            result->types.unsigned_int = 4; // "Failed"
    }

    return PP_SUC;
}

