#include "clpParser.h"
#include "clp_common.h"
#include <ctype.h>


const char *defaultTarget[] = {"/", NULL};

clp_verb_name_t clp_verb_names[] =
{
	{CLP_VERB_CD, "cd"},
	{CLP_VERB_CREATE, "create"},
	{CLP_VERB_DELETE, "delete"},
	{CLP_VERB_DUMP, "dump"},
	{CLP_VERB_EXIT, "exit"},
	{CLP_VERB_HELP, "help"},
	{CLP_VERB_LOAD, "load"},
	{CLP_VERB_RESET, "reset"},
	{CLP_VERB_SET, "set"},
	{CLP_VERB_SHOW, "show"},
	{CLP_VERB_START, "start"},
	{CLP_VERB_STOP, "stop"},
	{CLP_VERB_VERSION, "version"},
	{-1, NULL}
};

clp_option_name_t clp_option_names[] =
{
	{CLP_OPTION_ALL, "-all", "-a"},
	{CLP_OPTION_DEFAULT, "-default", NULL},
	{CLP_OPTION_DESTINATION, "-destination", NULL},
	{CLP_OPTION_DISPLAY, "-display", "-d"},
	{CLP_OPTION_EXAMINE, "-examine", "-x"},
	{CLP_OPTION_FORCE, "-force", "-f"},
	{CLP_OPTION_HELP, "-help", "-h"},
	{CLP_OPTION_KEEP, "-keep", "-k"},
	{CLP_OPTION_LEVEL, "-level", "-l"},
	{CLP_OPTION_OUTPUT, "-output", "-o"},
	{CLP_OPTION_RESETSTATE, "-resetstate", NULL},
	{CLP_OPTION_SOURCE, "-source", NULL},
	{CLP_OPTION_STATE, "-state", NULL},
	{CLP_OPTION_VERSION, "-version", "-v"},
	{CLP_OPTION_WAIT, "-wait", "-w"},
	{-1, NULL, NULL}
};

clp_output_option_name_t clp_output_option_names[] =
{
	{OUT_FORMAT, "format"},
	{OUT_ERROR, "error"},
	{OUT_TERSE, "terse"},
	{OUT_VERBOSE, "verbose"},
	{OUT_LANGUAGE, "language"},
	{OUT_BEGIN, "begin"},
	{OUT_END, "end"},
	{OUT_ORDER, "order"},
	{OUT_COUNT, "count"},
	{OUT_NUMBER, "number"}
};

/*
 * Based on the table 16 (page 187) and the table 15 (page 185) in
 * "Server Management Command Line Protocol Specification (SM CLP)"
 * Version 1.0.a.9e
 *
 * clp_option_value_matrix is a bit-masked value specifying which
 * CLP options require an additional argument.
 *
 * clp_verb_option_matrix is an array of bit-masked values specifying
 * which CLP options apply to a particular verb. E.g., the CD verb
 * must implement the -default option, so the CLP_OPTION_DEFAULT bit
 * in clp_verb_option_matrix[CLP_VERB_CD] is enabled.
 */

uint16_t clp_verb_option_matrix[CLP_NUM_VERBS];

uint16_t clp_verb_target_matrix[CLP_NUM_VERBS];

uint16_t clp_option_value_matrix
	= CLP_OPTION_DESTINATION | CLP_OPTION_DISPLAY | CLP_OPTION_KEEP
	| CLP_OPTION_LEVEL | CLP_OPTION_OUTPUT | CLP_OPTION_RESETSTATE
	| CLP_OPTION_RESETSTATE | CLP_OPTION_SOURCE;

void clp_matrix_init()
{
	clp_verb_option_matrix[CLP_VERB_CD]
		= CLP_OPTION_DEFAULT | CLP_OPTION_EXAMINE | CLP_OPTION_HELP
		| CLP_OPTION_KEEP | CLP_OPTION_OUTPUT | CLP_OPTION_VERSION
		| CLP_OPTION_WAIT;
	clp_verb_target_matrix[CLP_VERB_CD]
		= CLP_TARGET_INSTANCE;

	clp_verb_option_matrix[CLP_VERB_CREATE]
		= CLP_OPTION_DEFAULT | CLP_OPTION_EXAMINE | CLP_OPTION_HELP
		| CLP_OPTION_KEEP | CLP_OPTION_OUTPUT | CLP_OPTION_VERSION
		| CLP_OPTION_WAIT;
	clp_verb_target_matrix[CLP_VERB_CREATE]
		//= CLP_TARGET_INSTANCE | CLP_TARGET_SET;	// BrianWang: should be a class
		= CLP_TARGET_CLASS | CLP_TARGET_ASSOCIATION | CLP_TARGET_INSTANCE | CLP_TARGET_SET;

	clp_verb_option_matrix[CLP_VERB_DELETE]
		= CLP_OPTION_EXAMINE | CLP_OPTION_FORCE | CLP_OPTION_HELP
		| CLP_OPTION_KEEP | CLP_OPTION_OUTPUT | CLP_OPTION_VERSION
		| CLP_OPTION_WAIT;
	clp_verb_target_matrix[CLP_VERB_DELETE]
		// = CLP_TARGET_INSTANCE | CLP_TARGET_SET;	// BrianWang: add association
		= CLP_TARGET_INSTANCE | CLP_TARGET_SET | CLP_TARGET_ASSOCIATION;

	clp_verb_option_matrix[CLP_VERB_DUMP]
		= CLP_OPTION_DESTINATION | CLP_OPTION_EXAMINE | CLP_OPTION_FORCE
		| CLP_OPTION_HELP | CLP_OPTION_KEEP | CLP_OPTION_OUTPUT
		| CLP_OPTION_VERSION | CLP_OPTION_WAIT;
	clp_verb_target_matrix[CLP_VERB_DUMP]
		= CLP_TARGET_INSTANCE;

	clp_verb_option_matrix[CLP_VERB_EXIT]
		= CLP_OPTION_EXAMINE | CLP_OPTION_HELP | CLP_OPTION_KEEP
		| CLP_OPTION_OUTPUT | CLP_OPTION_VERSION;
	clp_verb_target_matrix[CLP_VERB_EXIT] = 0; // default target only

	clp_verb_option_matrix[CLP_VERB_HELP]
		= CLP_OPTION_EXAMINE | CLP_OPTION_HELP | CLP_OPTION_KEEP
		| CLP_OPTION_OUTPUT | CLP_OPTION_VERSION | CLP_OPTION_WAIT;
	clp_verb_target_matrix[CLP_VERB_HELP]
		= CLP_TARGET_INSTANCE | CLP_TARGET_SET
		| CLP_TARGET_ASSOCIATION | CLP_TARGET_CLASS | CLP_TARGET_VERB;

	clp_verb_option_matrix[CLP_VERB_LOAD]
		= CLP_OPTION_EXAMINE | CLP_OPTION_FORCE | CLP_OPTION_HELP
		| CLP_OPTION_KEEP | CLP_OPTION_OUTPUT | CLP_OPTION_SOURCE
		| CLP_OPTION_VERSION | CLP_OPTION_WAIT;
	clp_verb_target_matrix[CLP_VERB_LOAD]
		= CLP_TARGET_INSTANCE;

	clp_verb_option_matrix[CLP_VERB_RESET]
		= CLP_OPTION_EXAMINE | CLP_OPTION_FORCE | CLP_OPTION_HELP
		| CLP_OPTION_KEEP | CLP_OPTION_OUTPUT | CLP_OPTION_RESETSTATE
		| CLP_OPTION_STATE | CLP_OPTION_VERSION | CLP_OPTION_WAIT;
	clp_verb_target_matrix[CLP_VERB_RESET]
		= CLP_TARGET_INSTANCE;

	clp_verb_option_matrix[CLP_VERB_SET]
		= CLP_OPTION_DEFAULT | CLP_OPTION_EXAMINE | CLP_OPTION_FORCE
		| CLP_OPTION_HELP | CLP_OPTION_KEEP | CLP_OPTION_OUTPUT
		| CLP_OPTION_VERSION | CLP_OPTION_WAIT;
	clp_verb_target_matrix[CLP_VERB_SET]
		= CLP_TARGET_INSTANCE | CLP_TARGET_ASSOCIATION;

	clp_verb_option_matrix[CLP_VERB_SHOW]
		= CLP_OPTION_ALL | CLP_OPTION_DEFAULT | CLP_OPTION_DISPLAY
		| CLP_OPTION_EXAMINE | CLP_OPTION_FORCE | CLP_OPTION_HELP
		| CLP_OPTION_KEEP | CLP_OPTION_LEVEL | CLP_OPTION_OUTPUT
		| CLP_OPTION_VERSION | CLP_OPTION_WAIT;
	clp_verb_target_matrix[CLP_VERB_SHOW]
		= CLP_TARGET_INSTANCE | CLP_TARGET_SET
		| CLP_TARGET_ASSOCIATION;

	clp_verb_option_matrix[CLP_VERB_START]
		= CLP_OPTION_EXAMINE | CLP_OPTION_FORCE | CLP_OPTION_HELP
		| CLP_OPTION_KEEP | CLP_OPTION_OUTPUT | CLP_OPTION_STATE
		| CLP_OPTION_VERSION | CLP_OPTION_WAIT;
	clp_verb_target_matrix[CLP_VERB_START]
		= CLP_TARGET_INSTANCE;

	clp_verb_option_matrix[CLP_VERB_STOP]
		= CLP_OPTION_EXAMINE | CLP_OPTION_FORCE | CLP_OPTION_HELP
		| CLP_OPTION_KEEP | CLP_OPTION_OUTPUT | CLP_OPTION_STATE
		| CLP_OPTION_VERSION | CLP_OPTION_WAIT;
	clp_verb_target_matrix[CLP_VERB_STOP]
		= CLP_TARGET_INSTANCE;

	clp_verb_option_matrix[CLP_VERB_VERSION]
		= CLP_OPTION_EXAMINE | CLP_OPTION_HELP | CLP_OPTION_KEEP
		| CLP_OPTION_OUTPUT | CLP_OPTION_VERSION | CLP_OPTION_WAIT;
	clp_verb_target_matrix[CLP_VERB_VERSION] = 0; // default target only
}

/* forward declarations */
static int option_needs_value(pp_clp_option_t option);
static int verb_supports_target(pp_clp_verb_t verb, pp_clp_target_type_t target);
static int parse_verb(pp_clp_session_t *session,
	char *token, pp_clp_cmd_t *cmd);
static int parse_display_options(pp_clp_session_t *session, vector_t *options,
	pp_clp_cmd_t *cmd);
static int check_exclusive(pp_clp_session_t *session, int used[],
	clp_output_option_t arg1, clp_output_option_t arg2);
static int check_output_args(pp_clp_session_t *session, int used[]);
static int parse_output_options(pp_clp_session_t *session, vector_t *options,
	pp_clp_cmd_t *cmd);
static int parse_options(pp_clp_session_t *session, vector_t *tokens,
	unsigned int *position, pp_clp_cmd_t *cmd);
static int is_property(char *token);
static int parse_target(pp_clp_session_t *session,
	char *token, pp_clp_cmd_t *cmd);
static int parse_properties(pp_clp_session_t *session, vector_t *tokens,
	unsigned int *position, pp_clp_cmd_t *cmd);

/* check if the option needs a value */
static int option_needs_value(pp_clp_option_t option)
{
	return (clp_option_value_matrix & option) > 0;
}

/* check if the verb supports an option */
int pp_clp_verb_supports_option(pp_clp_verb_t verb, pp_clp_option_t option)
{
	return (clp_verb_option_matrix[verb] & option) > 0;
}

/* check if the verb supports a given target type */
static int verb_supports_target(pp_clp_verb_t verb, pp_clp_target_type_t target)
{
	if (target == CLP_TARGET_DEFAULT)
		return 1;
	//printf("verb %d [verb] %d target %d\n", verb, clp_verb_target_matrix[verb], target);
	return (clp_verb_target_matrix[verb] & target) > 0;
}

/* tokenizer - tokens are stored in a vector */
int clp_parser_tokenizer(pp_clp_session_t *session, char *buf, vector_t *tokens)
{
	char *c = buf;
	char *new_token;

	session->cont = 0;

	while (*c != '\0') {
		switch (session->tok_state) {
			case PP_CLP_TOK_NORMAL:
				switch (*c) {
					case ' ':
					case '\t':
						break; // skip whitespaces
					case '`': // escape character
						if (c[1] == '\0') {
							session->cont = 1; // multi-line command
						} else {
							// start a new token
							session->tpos = 0;
							c++;
							session->token[session->tpos++] = *c;
							session->tok_state = PP_CLP_TOK_TOKEN;
						}
						break;
					case '"': // quote
						session->tpos = 0;
						session->tok_state = PP_CLP_TOK_QUOTED;
						break;
					default: // regular character
						session->tpos = 0;
						session->token[session->tpos++] = *c;
						session->tok_state = PP_CLP_TOK_TOKEN;
				}
				break;

			case PP_CLP_TOK_TOKEN:
				switch (*c) {
					case ' ':
					case '\t': // whitespace == end of token
						session->token[session->tpos] = '\0';
						new_token = strdup(session->token);
						vector_add(tokens, new_token);
						session->tok_state = PP_CLP_TOK_NORMAL;
						break;
					case '`': // escape character
						if (c[1] == '\0') {
							session->cont = 1; // multi-line command
						} else {
							// skip escape, append character to token
							c++;
							session->token[session->tpos++] = *c;
						}
						break;
					case '"': // quote
						session->tok_state = PP_CLP_TOK_QUOTED;
						break;
					default: // regular character
						session->token[session->tpos++] = *c;
				}
				break;

			case PP_CLP_TOK_QUOTED:
				switch (*c) {
					case '`': // escape character
						if (c[1] == '\0') {
							session->cont = 1; // multi-line command
						} else {
							// skip escape, append character to token
							c++;
							session->token[session->tpos++] = *c;
						}
						break;
					case '"': // quote
						session->tok_state = PP_CLP_TOK_TOKEN;
						break;
					default: // regular character
						session->token[session->tpos++] = *c;
				}
				break;

			default:
				eric_term_printf(session->term, "Illegal parser state.\r\n");
				return PP_ERR;
		}
		if (session->tpos > MAX_TOKENSIZE) {
			eric_term_printf(session->term, "Token too long.\r\n");
			return PP_ERR;
		}
		c++;
	}

	if (!session->cont) {
		switch (session->tok_state) {
			case PP_CLP_TOK_NORMAL:
				break; // nothing to do;
			case PP_CLP_TOK_TOKEN:
				// end of token;
				session->token[session->tpos] = '\0';
				new_token = strdup(session->token);
				vector_add(tokens, new_token);
				session->tok_state = PP_CLP_TOK_NORMAL;
				break;
			case PP_CLP_TOK_QUOTED:
				eric_term_printf(session->term, "Unmatched quote.\r\n");
				return PP_ERR;
			default:
				eric_term_printf(session->term, "Illegal parser state.\r\n");
				return PP_ERR;
		}
	}

	return PP_SUC;
}

/*
 * Perform CLP command line syntax check; extract verb, options,
 * target and properties.
 *
 * @param session:  CLP session handle
 * @param tokens:   Tokenized command line
 *
 * @return Newly allocated command structure, NULL in case of error.
 */

pp_clp_cmd_t *clp_token_check(pp_clp_session_t *session, vector_t *tokens)
{
	unsigned int pos;
	pp_clp_cmd_t *cmd;
	char *token;

	assert(tokens != NULL);

	cmd = pp_clp_cmd_new();

	pos = 0;
	if (pos >= vector_size(tokens)) {
		eric_term_printf(session->term, "No tokens found.\r\n");
		goto clp_token_check_err;
	}

	/* the first token must be the verb */
	token = vector_get(tokens, pos++);
	if (PP_FAILED(parse_verb(session, token, cmd))) {
		goto clp_token_check_err;
	}

	/* parse options (if any) */
	if (PP_FAILED(parse_options(session, tokens, &pos, cmd))) {
		goto clp_token_check_err;
	}

	/* next token might be either the target or the first property */
	if (pos < vector_size(tokens)) {
		token = vector_get(tokens, pos);
		if (!is_property(token)) {
			pos++;
			if (PP_FAILED(parse_target(session, token, cmd))) {
				goto clp_token_check_err;
			}
			if (!verb_supports_target(cmd->verb, cmd->target->type)) {
				eric_term_printf(session->term, "Invalid target type for command: '%s'.\r\n",
					token);
				goto clp_token_check_err;
			}
		}
	}

	/* any remaining tokens must be properties */
	if (PP_FAILED(parse_properties(session, tokens, &pos, cmd))) {
		goto clp_token_check_err;
	}

	return cmd;

clp_token_check_err:
	pp_clp_debug("clp_token_check: Syntax error\n");
	pp_clp_cmd_delete(cmd);
	return NULL;
}

/*
 * Parse a CLP verb token
 *
 * @param session:  CLP session handle
 * @param token:    The token to handle
 * @param cmd:      The command structure
 *
 * @return PP_SUC:  OK
 * @return PP_ERR:  unknown verb
 */

static int parse_verb(pp_clp_session_t *session, char *token, pp_clp_cmd_t *cmd)
{
	int i = 0;
	while (i < CLP_NUM_VERBS) {
		if (strcasecmp(token, clp_verb_names[i].verb_name) == 0)
			break;
		i++;
	}
	if (clp_verb_names[i].verb_name) {
		cmd->verb = clp_verb_names[i].verb_id;
		return PP_SUC;
	}
	eric_term_printf(session->term, "Unknown command: '%s'\r\n", token);
	return PP_ERR;
}

/*
 * Parse display parameters (argument of -display option)
 *
 * @param session:  CLP session handle.
 * @param options:  Tokenized display argument.
 * @param cmd:      The command structure.
 *
 * @return PP_SUC:  OK
 * @return PP_ERR:  Invalid argument
 */

static int parse_display_options(pp_clp_session_t *session, vector_t *options,
	pp_clp_cmd_t *cmd)
{
	unsigned int i;

	cmd->display.associations = 0;
	cmd->display.properties = 0;
	cmd->display.targets = 0;
	cmd->display.verbs = 0;

	for (i = 0; i < vector_size(options); i++) {
		int len;
		char *token = vector_get(options, i);
		char *tok2 = strchr(token, '=');
		if (tok2) {
			len = tok2 - token;
			tok2++;
			if (tok2[0] == '(')
				tok2++;
		} else {
			len = strlen(token);
		}

		if (len == 3 && strncasecmp(token, "all", len) == 0) {
			if (tok2) {
				eric_term_printf(session->term, "No value allowed for display argument 'all'.\r\n");
				return PP_ERR;
			}
			cmd->display.associations = 1;
			cmd->display.properties = 1;
			cmd->display.targets = 1;
			cmd->display.verbs = 1;
		} else if (len == 12 && strncasecmp(token, "associations", len) == 0) {
			cmd->display.associations = 1;
			if (tok2) {
				char *delim, *c;
				do {
					delim = strpbrk(tok2, ",)");
					if (delim) {
						len = delim - tok2;
						c = malloc(len + 1);
						memcpy(c, tok2, len);
						c[len] = '\0';
						if (*delim == ')')
							delim = NULL;
					} else {
						c = strdup(tok2);
					}
					vector_add(&cmd->display.association_names, c);
					if (delim) {
						tok2 = delim + 1;
					}
				} while (delim);
			}
		} else if (len == 10 && strncasecmp(token, "properties", len) == 0) {
			if (!tok2) {
				cmd->display.properties = 1;
			} else {
				char *delim, *c;
				do {
					char *c2;
					delim = strpbrk(tok2, ",)");
					if (delim) {
						len = delim - tok2;
						c = malloc(len + 1);
						memcpy(c, tok2, len);
						c[len] = '\0';
						if (*delim == ')')
							delim = NULL;
					} else {
						c = strdup(tok2);
					}
					c2 = strstr(c, "==");
					if (c2) {
						if (cmd->display.filter_property) {
							free(c);
							eric_term_printf(session->term, "Filtering is only supported for a single property.\r\n");
							return PP_ERR;
						}
						cmd->display.filter_property = malloc(c2 - c + 1);
						memcpy(cmd->display.filter_property, c, c2 - c);
						cmd->display.filter_property[c2 - c] = '\0';
						cmd->display.filter_value = strdup(c2 + 2);
						free(c);
					} else {
						cmd->display.properties = 1;
						vector_add(&cmd->display.property_names, c);
					}
					if (delim)
						tok2 = delim + 1;
				} while (delim);
			}
		} else if (len == 7 && strncasecmp(token, "targets", len) == 0) {
			if (!tok2) {
				cmd->display.targets = 1;
			} else {
				char *delim, *c;
				do {
					delim = strpbrk(tok2, ",)");
					if (delim) {
						len = delim - tok2;
						c = malloc(len + 1);
						memcpy(c, tok2, len);
						c[len] = '\0';
						if (*delim == ')')
							delim = NULL;
					} else {
						c = strdup(tok2);
					}
					vector_add(&cmd->display.target_names, c);
					if (delim)
						tok2 = delim + 1;
				} while (delim);
			}
		} else if (strncasecmp(token, "verbs", len) == 0) {
			if (tok2) {
				eric_term_printf(session->term, "No value allowed for display argument 'verbs'.\r\n");
				return PP_ERR;
			}
			cmd->display.verbs = 1;
		} else {
			eric_term_printf(session->term, "Invalid argument for -display: '%s'.\r\n", token);
			return PP_ERR;
		}
	}

	return PP_SUC;
}

/*
 * Check whether two mutally exclusive output parameters are specified
 * at the same time.
 *
 * @param session:  CLP session handle.
 * @param used:     Boolean array (NUM_OUT_OPTIONS elements), non-zero
 *                  means parameter has been specified.
 * @param arg1:     First parameter code.
 * @param arg2:     Second parameter code.
 *
 * @return PP_SUC:  OK
 * @return PP_ERR:  Both parameters specified
 */

static int check_exclusive(pp_clp_session_t *session, int used[],
	clp_output_option_t arg1, clp_output_option_t arg2)
{
	if (used[arg1] && used[arg2]) {
		eric_term_printf(session->term, "Output arguments '%s' and '%s' are mutually exclusive.\r\n",
			clp_output_option_names[arg1].option_name,
			clp_output_option_names[arg2].option_name);
		return PP_ERR;
	}
	return PP_SUC;
}

/*
 * Check all mutally exclusive parameter combinations
 *
 * @param session:  CLP session handle.
 * @param used:     Boolean array (NUM_OUT_OPTIONS elements), non-zero
 *                  means parameter has been specified.
 *
 * @return PP_SUC:  OK
 * @return PP_ERR:  Illegal combination
 */

static int check_output_args(pp_clp_session_t *session, int used[])
{
	if (PP_FAILED(check_exclusive(session, used, OUT_ERROR, OUT_TERSE))
		|| PP_FAILED(check_exclusive(session, used, OUT_ERROR, OUT_VERBOSE))
		|| PP_FAILED(check_exclusive(session, used, OUT_TERSE, OUT_VERBOSE))
		|| PP_FAILED(check_exclusive(session, used, OUT_BEGIN, OUT_END))
		|| PP_FAILED(check_exclusive(session, used, OUT_BEGIN, OUT_NUMBER))
		|| PP_FAILED(check_exclusive(session, used, OUT_END, OUT_NUMBER))
		) {
		return PP_ERR;
	}
	return PP_SUC;
}

/*
 * Parse output format parameters (argument of -output option)
 *
 * @param session:  CLP session handle.
 * @param options:  Tokenized option argument.
 * @param cmd:      The command structure.
 *
 * @return PP_SUC:  OK
 * @return PP_ERR:  Invalid argument
 */

static int parse_output_options(pp_clp_session_t *session, vector_t *options,
	pp_clp_cmd_t *cmd)
{
	unsigned int i;
	int option_used[NUM_OUT_OPTIONS];

	memset(option_used, 0, sizeof(option_used));
	for (i = 0; i < vector_size(options); i++) {
		int len, opt;
		char *token = vector_get(options, i);
		char *tok2 = strchr(token, '=');
		if (tok2) {
			len = tok2 - token;
			tok2++;
		} else {
			len = strlen(token);
		}
		for (opt = 0; opt < NUM_OUT_OPTIONS; opt++) {
			if ((int)strlen(clp_output_option_names[opt].option_name) == len &&
				strncasecmp(token, clp_output_option_names[opt].option_name,
				len) == 0)
				break;
		}
		if (opt == NUM_OUT_OPTIONS) {
			eric_term_printf(session->term, "Invalid argument for -output: '%s'.\r\n", token);
			return PP_ERR;
		}
		option_used[opt] = 1;
		if ((opt == OUT_ERROR || opt == OUT_TERSE || opt == OUT_VERBOSE
			|| opt == OUT_BEGIN || opt == OUT_END) && tok2) {
			eric_term_printf(session->term, "Output option '%s' must not have an argument.\r\n",
				clp_output_option_names[opt].option_name);
			return PP_ERR;
		}
		switch (opt) {
			case OUT_FORMAT:
				if (!tok2) {
					eric_term_printf(session->term, "Missing 'format' argument.\r\n");
					return PP_ERR;
				} else if (strcasecmp(tok2, "text") == 0) {
					cmd->output.format = CLP_OUTPUT_TEXT;
				} else if (strcasecmp(tok2, "clpcsv") == 0) {
					cmd->output.format = CLP_OUTPUT_CLPCSV;
				} else if (strcasecmp(tok2, "keyword") == 0) {
					cmd->output.format = CLP_OUTPUT_KEYWORD;
				} else if (strcasecmp(tok2, "clpxml") == 0) {
					cmd->output.format = CLP_OUTPUT_CLPXML;
				} else {
					eric_term_printf(session->term, "Invalid 'format' argument: '%s'.\r\n", tok2);
					return PP_ERR;
				}
				break;
			case OUT_ERROR:
				cmd->output.level = CLP_OUTPUT_ERROR;
				break;
			case OUT_TERSE:
				cmd->output.level = CLP_OUTPUT_TERSE;
				break;
			case OUT_VERBOSE:
				cmd->output.level = CLP_OUTPUT_VERBOSE;
				break;
			case OUT_LANGUAGE:
				if (!tok2) {
					eric_term_printf(session->term, "Missing 'language' argument.\r\n");
					return PP_ERR;
				} else if (strlen(tok2) > 3) {
					eric_term_printf(session->term, "Invalid 'language' argument: '%s'.\r\n", tok2);
					return PP_ERR;
				}
				strcpy(cmd->output.language, tok2);
				break;
			case OUT_BEGIN:
				cmd->output.end = 0;
				break;
			case OUT_END:
				cmd->output.end = 1;
				break;
			case OUT_ORDER:
				if (!tok2) {
					eric_term_printf(session->term, "Missing 'order' argument.\r\n");
					return PP_ERR;
				} else if (strcasecmp(tok2, "default") == 0) {
					cmd->output.reverse = 0;
				} else if (strcasecmp(tok2, "reverse") == 0) {
					cmd->output.reverse = 1;
				} else {
					eric_term_printf(session->term, "Invalid 'order' argument: '%s'.\r\n", tok2);
					return PP_ERR;
				}
				break;
			case OUT_COUNT:
				if (!tok2) {
					eric_term_printf(session->term, "Missing 'count' argument.\r\n");
					return PP_ERR;
				} else if (strcasecmp(tok2, "all") == 0) {
					cmd->output.count = -1;
				} else {
					char *c;
					cmd->output.count = strtol(tok2, &c, 10);
					if (*c != '\0') {
						eric_term_printf(session->term, "Invalid 'count' argument: '%s'.\r\n", tok2);
						return PP_ERR;
					}
				}
				break;
			case OUT_NUMBER:
				if (!tok2) {
					eric_term_printf(session->term, "Missing 'number' argument.\r\n");
					return PP_ERR;
				} else {
					char *c;
					int err = 0;
					cmd->output.range_min = strtol(tok2, &c, 10);
					if (c[0] != '-' || c[1] == '\0') {
						err = 1;
					} else {
						cmd->output.range_max = strtol(c + 1, &c, 10);
					}
					if (err || *c != '\0') {
						eric_term_printf(session->term, "Invalid 'number' argument: '%s'.\r\n", tok2);
						return PP_ERR;
					}
				}
				break;
			default:
				eric_term_printf(session->term, "Unhandled argument for -output: '%s'.\r\n", token);
				return PP_ERR;
		}
	}
	if (PP_FAILED(check_output_args(session, option_used))) {
		return PP_ERR;
	}
	return PP_SUC;
}

/*
 * Parse zero or more CLP options (including an argument, if required)
 *
 * @param session:  CLP session handle
 * @param tokens:   Vector of CLP tokens as returned by the tokenizer
 * @param position: Pointer to an integer indicating the next token to handle.
 *                  The position will be increased for each successfully
 *                  handled token.
 * @param cmd:      The command structure
 *
 * @return PP_SUC:  OK
 * @return PP_ERR:  Illegal option or missing option argument
 */

static int parse_options(pp_clp_session_t *session, vector_t *tokens,
	unsigned int *position, pp_clp_cmd_t *cmd)
{
	int i;
	char *token;
	pp_clp_option_t opt;
	pp_clp_option_value_t *optval;

	while (*position < vector_size(tokens)) {
		token = vector_get(tokens, *position);

		if (token[0] != '-')
			return PP_SUC;

		(*position)++;
		i = 0;
		while (clp_option_names[i].option_name) {
			if (strcasecmp(token, clp_option_names[i].option_name) == 0)
				break;
			if (clp_option_names[i].shortcut) {
				if (strcasecmp(token, clp_option_names[i].shortcut) == 0)
					break;
			}
			i++;
		}
		if (!clp_option_names[i].option_name) {
			eric_term_printf(session->term, "Unknown option: '%s'.\r\n", token);
			return PP_ERR;
		}

		opt = clp_option_names[i].option_id;
		if (!pp_clp_verb_supports_option(cmd->verb, opt)) {
			eric_term_printf(session->term, "Verb does not support option: '%s'.\r\n", token);
			return PP_ERR;
		}

		optval = pp_clp_optval_new(opt);
		vector_add(&cmd->options, optval);

		if (option_needs_value(opt)) {
			char *arg, *start, *delim;
			int parenth, len;
			if (*position >= vector_size(tokens)) {
				eric_term_printf(session->term, "Option requires an argument: '%s'.\r\n", token);
				return PP_ERR;
			}
			token = vector_get(tokens, (*position)++);
			parenth = 0;
			start = token;
			do {
				delim = strpbrk(token, ",(");
				if (delim && *delim == '(') {
					token = strchr(delim + 1, ')');
					if (!token) {
						eric_term_printf(session->term, "Unmatched '('.\r\n", token);
						return PP_ERR;
					}
					token++;
					continue;
				}
				if (!delim) {
					len = strlen(start);
				} else {
					len = delim - start;
				}
				arg = malloc(len + 1);
				strncpy(arg, start, len);
				arg[len] = '\0';
				vector_add(&optval->values, arg);
				start = token = delim + 1;
			} while (delim);
		}

		switch (opt) {
			case CLP_OPTION_ALL:
				cmd->opt_all = 1;
				break;
			case CLP_OPTION_DEFAULT:
				cmd->opt_default = 1;
				break;
			case CLP_OPTION_DISPLAY:
				if (PP_FAILED(parse_display_options(session,
					&optval->values, cmd))) {
					return PP_ERR;
				}
				break;
			case CLP_OPTION_EXAMINE:
				cmd->opt_examine = 1;
				break;
			case CLP_OPTION_LEVEL:
				token = vector_get(&optval->values, 0);
				if (strcasecmp(token, "all") == 0) {
					cmd->target->level = -1;
				} else {
					char *c;
					cmd->target->level = strtol(token, &c, 10);
					if (c[0] != '\0') {
						eric_term_printf(session->term, "Invalid argument for -level: '%s'.\r\n", token);
						return PP_ERR;
					}
				}
				break;
			case CLP_OPTION_HELP:
				cmd->opt_help = 1;
				break;
			case CLP_OPTION_VERSION:
				cmd->opt_version = 1;
				break;
			case CLP_OPTION_OUTPUT:
				if (PP_FAILED(parse_output_options(session,
					&optval->values, cmd))) {
					return PP_ERR;
				}
				break;
			default:
				;
		}
	}

	return PP_SUC;
}

/*
 * Check whether the given token qualifies as a target or a property.
 * Properties are recognized for containining a '=' character that is
 * not part of the association separator '=>'.
 *
 * Note: The SM CLP spec specifies different rules to distinguish
 *       between targets and properties, but unfortunately those
 *       wouldn't do the trick.
 */

static int is_property(char *token)
{
	char *c = strchr(token, '=');
	return (c && c[1] != '>');
}

/*
 * Parse a User Friendly Instance Path extracted from a target string.
 *
 * @param path      The string to analyze.
 * @param ufip      Result vector; element type: pp_clp_ufit_t
 * @param relative  Set to zero in case the UFiP is an absolute path.
 * @param instance  Set to zero in case the UFiP addresses a set of instances.
 * @param isclass	Set to zero in case the UFiP is no address to a class.
 *					If the pointer is NULL, return PP_ERR if there is a class inside.
 *
 * @return PP_SUC:  OK
 * @return PP_ERR:  Invalid UFiP
 */

int parse_ufip(const char *path, vector_t *ufip, int *relative, int *instance, int *isclass)
{
	enum
	{
		SEPARATOR,
		DOT,
		DOTDOT,
		TAG,
		SUFFIX,
		ASTERISK
	} state = SEPARATOR;
	const char *token = NULL;
	if (isclass != NULL)
		*isclass = 0;
	pp_clp_ufit_t *ufit = NULL;
	//printf("%s %s\n", __FUNCTION__, path);

	if (path[0] == '/' || path[0] == '\\') {
		*relative = 0;
		path++;
		if (path[0] == '\0') {
			*instance = 1;
			return PP_SUC;
		}
	} else {
		*relative = 1;
	}
	*instance = 1;

	ufit = pp_clp_ufit_new();
	vector_add(ufip, ufit);
	while (*path) {
		//printf("state %d\n", state);
		switch (state) {
			case SEPARATOR:
				if (*path == '.') {
					state = DOT;
				} else if (isalpha(*path)) {
					token = path;
					state = TAG;
				} else {
					return PP_ERR;
				}
				break;

			case DOT:
				if (*path == '.') {
					state = DOTDOT;
				} else if (*path == '/' || *path == '\\') {
					ufit->ufct = strdup(".");
					ufit = pp_clp_ufit_new();
					vector_add(ufip, ufit);
					state = SEPARATOR;
				} else {
					return PP_ERR;
				}
				break;

			case DOTDOT:
				if (*path == '/' || *path == '\\') {
					ufit->ufct = strdup("..");
					ufit = pp_clp_ufit_new();
					vector_add(ufip, ufit);
					state = SEPARATOR;
				} else {
					return PP_ERR;
				}
				break;

			case TAG:
				if (isalpha(*path)) {
					// no change
				} else if (isdigit(*path)) {
					ufit->ufct = malloc(path - token + 1);
					memcpy(ufit->ufct, token, path - token);
					ufit->ufct[path - token] = '\0';
					ufit->instance_id = *path - '0';
					state = SUFFIX;
				} else if (*path == '*') {
					ufit->ufct = malloc(path - token + 1);
					memcpy(ufit->ufct, token, path - token);
					ufit->ufct[path - token] = '\0';
					ufit->instance_id = -1;
					*instance = 0;
					state = ASTERISK;
				} else {
					return PP_ERR;
				}
				break;

			case SUFFIX:
				if (isdigit(*path)) {
					ufit->instance_id *= 10;
					ufit->instance_id += *path - '0';
				} else if (*path == '/' || *path == '\\') {
					ufit = pp_clp_ufit_new();
					vector_add(ufip, ufit);
					state = SEPARATOR;
				} else {
					return PP_ERR;
				}
				break;

			case ASTERISK:
				// the '*' must be the final character
				return PP_ERR;
		}
		path++;
	}
	switch (state) {
		case TAG:
			if (isclass == NULL)
				return PP_ERR;
			*isclass = 1;
			*instance = 0;
			ufit->ufct = malloc(path - token + 1);
			memcpy(ufit->ufct, token, path - token);
			ufit->ufct[path - token] = '\0';
			return PP_SUC;
		case SEPARATOR:
			return PP_ERR;
		case DOT:
			ufit->ufct = strdup(".");
			break;
		case DOTDOT:
			ufit->ufct = strdup("..");
			break;
		default:
			break;
	}
	return PP_SUC;
}

/*
 * Parse a CLP command target. The target string may contain up to two
 * '=>' character sequences, separating a user friendly instance path,
 * an optional association class name and a second UFiP.
 *
 * @param token:    The token to handle
 * @param cmd:      The command structure
 *
 * @return PP_SUC:  OK
 * @return PP_ERR:  Invalid target
 */

static int parse_target(pp_clp_session_t *session, char *token, pp_clp_cmd_t *cmd)
{
	char *sep;
	char *ufip;
	int relative, instance, isclass;

	if (token[0] == '\0') {
		eric_term_printf(session->term, "Empty target string.\r\n", token);
		return PP_ERR;
	}

	/* first part: UFiP or a class name */
	sep = strstr(token, "=>");
	if (!sep) {
		//char *c = token;
		int i; //, index_of_class;

		/* target might be a CLP verb */
		for (i = 0; i < CLP_NUM_VERBS; i++) {
			if (strcasecmp(token, clp_verb_names[i].verb_name) == 0) {
				cmd->target->type = CLP_TARGET_VERB;
				cmd->target->verb = clp_verb_names[i].verb_id;
				return PP_SUC;
			}
		}

		/*
		while (isalpha(*c))
			c++;
		if (*c == '\0')
		{
			cmd->target->type = CLP_TARGET_CLASS;
			cmd->target->ufct = strdup(token);
			return PP_SUC;
		}
		*/
		ufip = strdup(token);
	} else {
		ufip = malloc(sep - token + 1);
		strncpy(ufip, token, sep - token);
		ufip[sep - token] = '\0';
	}

	cmd->target->ufip1 = vector_new(NULL, 10,
		(vector_elem_del_func_simple)pp_clp_ufit_delete);
	if (PP_FAILED(parse_ufip(ufip, cmd->target->ufip1, &relative, &instance, &isclass))) {
		eric_term_printf(session->term, "Invalid target: '%s'.\r\n", token);
		free(ufip);
		return PP_ERR;
	}
	free(ufip);

	cmd->target->ufip1_relative = relative;
	if (!sep) {
		if (instance) {
			cmd->target->type = CLP_TARGET_INSTANCE;
		} else {
			unsigned int last = vector_size(cmd->target->ufip1) - 1;
			pp_clp_ufit_t *ufit = vector_get(cmd->target->ufip1, last);
			cmd->target->ufct = strdup(ufit->ufct);
			vector_remove(cmd->target->ufip1, last);

			if (isclass)
				cmd->target->type = CLP_TARGET_CLASS;
			else
				cmd->target->type = CLP_TARGET_SET;
		}
		return PP_SUC;
	} else if (!instance) {
		eric_term_printf(session->term, "Non-instance path as first part of association: '%s'.\r\n", token);
		return PP_ERR;
	}

	//printf("second part %s\n", sep);
	/* second part: association class name */
	cmd->target->type = CLP_TARGET_ASSOCIATION;
	token = sep + 2;
	sep = strstr(token, "=>");
	if (!sep) {
		cmd->target->ufct = strdup(token);
		return PP_SUC;
	}
	cmd->target->ufct = malloc(sep - token + 1);
	strncpy(cmd->target->ufct, token, sep - token);
	cmd->target->ufct[sep - token] = '\0';

	//printf("third part %s\n", sep);
	/* third part: one more UFiP */
	token = sep + 2;
	sep = strstr(token, "=>");
	if (sep) {
		eric_term_printf(session->term, "Too many association separators '=>' in target.\r\n");
		return PP_ERR;
	}
	cmd->target->ufip2 = vector_new(NULL, 10,
		(vector_elem_del_func_simple)pp_clp_ufit_delete);
	//printf("before parse_ufip\n");
	if (PP_FAILED(parse_ufip(token, cmd->target->ufip2, &relative, &instance, NULL))) {
		eric_term_printf(session->term, "Invalid target: '%s'.\r\n", token);
		return PP_ERR;
	} else if (!instance) {
		eric_term_printf(session->term, "Non-instance path as third part of association: '%s'.\r\n", token);
		return PP_ERR;
	}
	//printf("end of %s\n", __FUNCTION__);
	cmd->target->ufip2_relative = relative;
	return PP_SUC;
}

/*
 * Parse zero or more CLP property/value pairs
 *
 * @param session:  CLP session handle
 * @param tokens:   Vector of CLP tokens as returned by the tokenizer
 * @param position: Pointer to an integer indicating the next token to handle.
 *                  The position will be increased for each successfully
 *                  handled token.
 * @param cmd:      The command structure
 *
 * @return PP_SUC:  OK
 * @return PP_ERR:  Token is not a property/value pair
 */

static int parse_properties(pp_clp_session_t *session, vector_t *tokens,
	unsigned int *position, pp_clp_cmd_t *cmd)
{
	char *token, *delim;
	pp_clp_property_value_t *prop;

	while (*position < vector_size(tokens)) {
		token = vector_get(tokens, *position);
		delim = strchr(token, '=');
		if (!delim) {
			eric_term_printf(session->term, "Not a property: '%s'.\r\n", token);
			return PP_ERR;
		}
		(*position)++;
		prop = pp_clp_propval_new();
		prop->property = malloc(delim - token + 1);
		strncpy(prop->property, token, delim - token);
		prop->property[delim - token] = '\0';
		prop->value = strdup(delim + 1);
		vector_add(&cmd->properties, prop);
	}

	return PP_SUC;
}

