#include "clpParser.h"
#include "command.h"
#include <pp/cim.h>

void clp_help_global(term_cl_t *term, int verbose);
void clp_help_class(pp_clp_session_t *session, const char *ufct, int verbose);
void clp_help_instance(pp_clp_session_t *session, pp_clp_target_t *target, int verbose);

int clp_verb_handler_help(pp_clp_session_t *session, pp_clp_cmd_t *cmd)
{
	int verbose = (cmd->output.level == CLP_OUTPUT_VERBOSE);

	if (cmd->opt_help) {
		clp_help_verb(session->term, CLP_VERB_HELP, verbose);
	} else if (cmd->opt_version) {
		eric_term_printf(session->term, "HELP %s\r\n", VERSION);
	} else if (cmd->opt_examine) {
		eric_term_printf(session->term, "Command validity check successful.\r\n");
		eric_term_printf(session->term, "Run without -examine option to perform command.\r\n");
	} else {
		switch (cmd->target->type) {
			case CLP_TARGET_DEFAULT:
				clp_help_global(session->term, verbose);
				break;
			case CLP_TARGET_VERB:
				clp_help_verb(session->term, cmd->target->verb, verbose);
				break;
			case CLP_TARGET_CLASS:
				clp_help_class(session, cmd->target->ufct, verbose);
				break;
			case CLP_TARGET_INSTANCE:
				clp_help_instance(session, cmd->target, verbose);
				break;
			default:
				eric_term_printf(session->term, "No help available.\r\n");
		}
	}
	return PP_SUC;
}

void clp_help_global(term_cl_t *term, int verbose)
{
	clp_help_verb(term, CLP_VERB_HELP, verbose);
	eric_term_printf(term, "CLP commands (commands in parentheses are not yet supported):\r\n");
	eric_term_printf(term, "\r\n");
	eric_term_printf(term, "    cd       create    delete     (dump)  exit\r\n");
	eric_term_printf(term, "    help     (load)    reset      set     show\r\n");
	eric_term_printf(term, "    start    stop      version\r\n");
	eric_term_printf(term, "\r\n");
	eric_term_printf(term, "To get a list of verbs supported for a given target, use:\r\n");
	eric_term_printf(term, "    show -display verbs <target>\r\n");
	eric_term_printf(term, "\r\n");
}

void clp_help_verb(term_cl_t *term, pp_clp_verb_t verb, int verbose)
{
	clp_option_name_t *opt;

	switch (verb) {
		case CLP_VERB_CD:
			eric_term_printf(term, "The CD command is used to change or display the current CLP\r\n");
			eric_term_printf(term, "default target.\r\n");
			eric_term_printf(term, "\r\n");
			eric_term_printf(term, "Usage: CD [options] [target]\r\n");
			eric_term_printf(term, "\r\n");
			eric_term_printf(term, "The target may be any valid user friendly instance path, either\r\n");
			eric_term_printf(term, "absolute or relative. If no target is specified the current\r\n");
			eric_term_printf(term, "default target is displayed.\r\n");
			break;
		case CLP_VERB_EXIT:
			eric_term_printf(term, "The Exit command is used to leave the current CLP session.\r\n");
			eric_term_printf(term, "\r\n");
			eric_term_printf(term, "Usage: EXIT [options]\r\n");
			break;
		case CLP_VERB_HELP:
			eric_term_printf(term, "The Help command is used to view useful information about\r\n");
			eric_term_printf(term, "targets and verbs.\r\n");
			eric_term_printf(term, "\r\n");
			eric_term_printf(term, "Usage: HELP [options] [topic]\r\n");
			eric_term_printf(term, "\r\n");
			eric_term_printf(term, "The help topic may be a CLP command, a User Friendly Class Tag\r\n");
			eric_term_printf(term, "or any valid CLP instance, set or association target.\r\n");
			break;
		case CLP_VERB_RESET:
			eric_term_printf(term, "The Reset command performs the reset action on the specified target.\r\n");
			eric_term_printf(term, "\r\n");
			eric_term_printf(term, "Usage: RESET [options] [target]\r\n");
			break;
		case CLP_VERB_SHOW:
			eric_term_printf(term, "The Show command is used to display information about objects\r\n");
			eric_term_printf(term, "within the CLP namespace.\r\n");
			eric_term_printf(term, "\r\n");
			eric_term_printf(term, "Usage: SHOW [options] [target]\r\n");
			break;
		case CLP_VERB_SET:
			eric_term_printf(term, "The Set command is used to alter an arbitrary number of properties\r\n");
			eric_term_printf(term, "of a CLP instance.\r\n");
			eric_term_printf(term, "\r\n");
			eric_term_printf(term, "Usage: SET [options] [target] [property [property [...]]]\r\n");
			break;
		case CLP_VERB_START:
			eric_term_printf(term, "The Start command performs the start action on the specified target.\r\n");
			eric_term_printf(term, "\r\n");
			eric_term_printf(term, "Usage: START [options] [target]\r\n");
			break;
		case CLP_VERB_STOP:
			eric_term_printf(term, "The Stop command performs the stop action on the specified target.\r\n");
			eric_term_printf(term, "\r\n");
			eric_term_printf(term, "Usage: STOP [options] [target]\r\n");
			break;
		case CLP_VERB_VERSION:
			eric_term_printf(term, "The Version command is used to display information about the CLP protocol\r\n");
			eric_term_printf(term, "version supported by this implementation.\r\n");
			eric_term_printf(term, "\r\n");
			eric_term_printf(term, "Usage: VERSION [options]\r\n");
			break;
		default:
			eric_term_printf(term, "No help available.\r\n");
			return;
	}

	eric_term_printf(term, "\r\n");
	if (!verbose) {
		eric_term_printf(term, "Use \"-output verbose\" option for more detailed help.\r\n");
		eric_term_printf(term, "\r\n");
		return;
	}
	eric_term_printf(term, "Supported options:\r\n");

	for (opt = clp_option_names; opt->option_name; opt++) {
		if (!pp_clp_verb_supports_option(verb, opt->option_id))
			continue;
		switch (opt->option_id) {
			case CLP_OPTION_DISPLAY:
				eric_term_printf(term, "    -display <arg> (-d)    Select information to display.\r\n");
				break;
			case CLP_OPTION_EXAMINE:
				eric_term_printf(term, "    -examine       (-x)    Check syntax, don't execute command.\r\n");
				break;
			case CLP_OPTION_HELP:
				eric_term_printf(term, "    -help          (-h)    Display this help.\r\n");
				break;
			case CLP_OPTION_LEVEL:
				eric_term_printf(term, "    -level <n>     (-l)    Recurse n (or 'all') levels below target.\r\n");
				break;
			case CLP_OPTION_OUTPUT:
				eric_term_printf(term, "    -output <arg>  (-o)    Specify output format.\r\n");
				break;
			case CLP_OPTION_VERSION:
				eric_term_printf(term, "    -version       (-v)    Display version information.\r\n");
				break;
			default:
				;
		}
	}
	eric_term_printf(term, "\r\n");
}

void clp_help_class(pp_clp_session_t *session, const char *ufct, int verbose)
{
	char *message;
	pp_cim_class_t *cim_class = pp_cim_class_lookup_ufct(ufct);
	if (!cim_class) {
		eric_term_printf(session->term, "Unknown UFcT: '%s'.\r\n", ufct);
		return;
	}
	message = pp_cim_clp_help_class(cim_class, verbose);
	if (!message) {
		eric_term_printf(session->term, "No help available for class.\r\n");
	} else {
		eric_term_printf(session->term, message);
		free(message);
	}
	pp_cim_class_release(cim_class);
}

void clp_help_instance(pp_clp_session_t *session, pp_clp_target_t *target, int verbose)
{
	vector_t *instances = pp_cim_clp_lookup(target, &session->default_target);
	if (!instances || vector_size(instances) != 1) {
		eric_term_printf(session->term, "Target not found.\r\n");
	} else {
		pp_cim_instance_t *instance = vector_get(instances, 0);
		char *message = pp_cim_clp_help_instance(instance, verbose);
		if (!message) {
			eric_term_printf(session->term, "No help available for instance.\r\n");
		} else {
			eric_term_printf(session->term, message);
			free(message);
		}
	}
	if (instances)
		vector_delete(instances);
}

