#include <openssl/evp.h>
#include <pp/bio.h>

#include "crypto.h"
#include "disconf_intern.h"

/***************************************************************
 * Signature
 ***************************************************************/

pp_dc_signature_t*
pp_dc_signature_new()
{
    pp_dc_signature_t *ret=(pp_dc_signature_t*)malloc(sizeof(pp_dc_signature_t));
    ret->hash_algo=0;
    ret->pub_algo=0;
    ret->size=0;
    ret->data=NULL;
    return ret;
}

pp_dc_cnode_t*
pp_dc_signature2cnode(pp_dc_signature_t *x)
{
    return pp_dc_cnode_create(PP_DC_TAG_SIGNATURE,x);
}

pp_dc_signature_t*
pp_dc_cnode2signature(pp_dc_cnode_t *x)
{
    if(x->tag==PP_DC_TAG_SIGNATURE)
	return x->un.signature;
    else
	return NULL;
}

int
pp_dc_signature_parse(BIO *bio,pp_dc_signature_t **signature,size_t size)
{
    *signature=pp_dc_signature_new();
    (*signature)->size=size-10;
    return read_uint8(bio,&(*signature)->hash_algo)&&
	read_uint8(bio,&(*signature)->pub_algo)&&
	read_uint64(bio,(u_int64_t*)&(*signature)->keyid)&&
	read_data(bio,(char**)&(*signature)->data,(*signature)->size);
}

int
pp_dc_signature_build(BIO *bio,pp_dc_signature_t *signature)
{
    return write_uint8(bio,signature->hash_algo)&&
	write_uint8(bio,signature->pub_algo)&&
	write_data(bio,(char*)&signature->keyid,8)&&
	write_data(bio,(char*)signature->data,signature->size);
}

void
pp_dc_signature_free(pp_dc_signature_t *signature)
{
    free(signature->data);
    free(signature);
}

pp_dc_signature_t*
pp_dc_signature_sign(pp_dc_cnode_t *node,pp_dc_hash_algo_t algo,pp_dc_privkey_t *privkey)
{
    pp_dc_signature_t *ret=NULL;
    size_t siglen,mdlen;
    char md[EVP_MAX_MD_SIZE],*sig;
    BIO *bio;

    bio = BIO_new(BIO_s_null());
    bio = BIO_push(BIO_new(BIO_f_md()),bio);

    switch(algo) {
    case PP_DC_HASH_SHA1:
	pp_bio_set_md(bio, EVP_sha1());
	break;
    case PP_DC_HASH_MD5:
	pp_bio_set_md(bio, EVP_md5());
	break;
    default:
	goto bail;
    }


    pp_dc_cnode_build(bio,node);

    mdlen=BIO_gets(bio,md,EVP_MAX_MD_SIZE);

    if(!pp_dc_privkey_encrypt(md,mdlen,privkey,&sig,&siglen))
	goto bail;

    ret=pp_dc_signature_new();

    ret->hash_algo=algo;
    ret->keyid=privkey->keyid;
    ret->pub_algo=privkey->algo;

    ret->data=(u_int8_t*)sig;
    ret->size=siglen;

 bail:
    BIO_free_all(bio);
    return ret;
}

int
pp_dc_signature_verify(pp_dc_cnode_t *node,pp_dc_signature_t *signature,pp_dc_pubkey_t *pubkey)
{
    int ret=0;
    size_t mdlen,sigmdlen;
    char md[EVP_MAX_MD_SIZE],*sigmd=NULL;
    BIO *bio;

    if((memcmp(&signature->keyid,&pubkey->keyid,8)!=0)||
       (signature->pub_algo!=pubkey->algo)) {
	return 0;
    }

    bio = BIO_new(BIO_s_null());
    bio = BIO_push(BIO_new(BIO_f_md()),bio);

    switch(signature->hash_algo) {
    case PP_DC_HASH_SHA1:
	pp_bio_set_md(bio, EVP_sha1());
	break;
    case PP_DC_HASH_MD5:
	pp_bio_set_md(bio, EVP_md5());
	break;
    default:
	goto bail;
    }

    pp_dc_cnode_build(bio,node);
    mdlen=BIO_gets(bio,md,EVP_MAX_MD_SIZE);

    if(!pp_dc_pubkey_decrypt((char*)signature->data,signature->size,pubkey,&sigmd,&sigmdlen)) {
	goto bail;
    }

    if(mdlen!=sigmdlen) {
	goto bail;
    }

    if(memcmp(md,sigmd,mdlen)!=0) {
	goto bail;
    }

    ret=1;

 bail:
    BIO_free_all(bio);
    free(sigmd);
    return ret;
}
