/*
 * atomic write
 *
 * 18.8.2005
 * by Rolf Kampffmyer roka@peppercon.de
 *
 */

#include <pp/atomic_write.h>
#include <pp/base.h>
#include <pp/vector.h>
#include <sys/vfs.h>
//#include <pp/bmc/debug.h> // fuer test_cases()

#include <dirent.h>
#include <fcntl.h> // for O_CREAT
#include <string.h>
#include <stdio.h>


typedef struct atomic_file_s {
    char * name;
    char * tempname;
    int fd;
    int mode;
    int successful_so_far;
} atomic_file_t;

vector_t * open_files = NULL;
pthread_mutex_t open_files_mtx;
pthread_cond_t some_file_closed;

static void atomic_file_t_destroy(atomic_file_t*file);
//static void test_cases();

int pp_fa_at_init(void) {
    assert(PP_ERR == -1);
    open_files = vector_new(NULL, 0, 
			    (void (*) (void *)) atomic_file_t_destroy);
    MUTEX_CREATE_ERRORCHECKING(&open_files_mtx);

    // attributes for conditions are not supported in
    // linux-threads. Thus the thread who tries to open the same file
    // twice will hang.
    pthread_cond_init(&some_file_closed, NULL);

    //    test_cases();
    return PP_SUC;
}
void pp_fa_at_cleanup() {
    MUTEX_LOCK(&open_files_mtx);
    assert(open_files);
    vector_delete(open_files);
    open_files = NULL;
    MUTEX_UNLOCK(&open_files_mtx);
}


int pp_fa_at_regular_file_absent(char *name){
    int fd = open(name, O_RDONLY | O_EXCL);
    if ((fd == -1) && (errno == ENOENT) ) {
	return 1;
    }
    else if (fd != -1) {
	close(fd);
    }
    return 0;
}

static void atomic_file_t_destroy (atomic_file_t * file) {
    free(file->name);
    if (file->tempname) {
	free(file->tempname);
    }
    free(file);
}

static void remove_fd(int fd) {
    atomic_file_t * file;
    unsigned int i;

    for (i=0; i<vector_size(open_files); i++) {
	file = vector_get(open_files, i);
	if (file->fd == fd) {
	    vector_remove_dont_delete(open_files, i);
	    break;
	}
    }
    pthread_cond_broadcast(&some_file_closed);
}

static atomic_file_t * get_file_by_name(char * filename) {
    atomic_file_t * file;
    unsigned int i;

    for (i=0; i<vector_size(open_files); i++) {
	file = vector_get(open_files, i);
	if ( strcmp(file->name, filename)  == 0) {
	    return file;
	}
    }
    return NULL;
}

static atomic_file_t * get_file_by_fd(int fd) {
    atomic_file_t * file;
    unsigned int i;

    for (i=0; i<vector_size(open_files); i++) {
	file = vector_get(open_files, i);
	assert(file);
	if (file->fd  == fd) {
	    return file;
	}
    }
    return NULL;
}

int pp_fa_at_open(char *filename, int mode) {
    atomic_file_t * file_entry;
    assert(open_files);

    MUTEX_LOCK(&open_files_mtx);
    do {
	file_entry = get_file_by_name(filename);
	if (file_entry != NULL) {
	    pthread_cond_wait(&some_file_closed, &open_files_mtx);
	}
    }
    while (file_entry != NULL);
    MUTEX_UNLOCK(&open_files_mtx);


    file_entry = malloc(sizeof(atomic_file_t));
    file_entry->name = strdup(filename);
    file_entry->tempname = NULL;

    if (mode == PP_FA_AT_WRITE) {
	pp_strstream_t * tempfile;

	tempfile = pp_strstream_init(NULL);
	pp_strappend(tempfile, filename);
	pp_strappend(tempfile, ".tmp");

	file_entry->tempname = pp_strstream_buf_and_free(tempfile);
	file_entry->fd = open(file_entry->tempname, 
			      O_WRONLY | O_CREAT |O_TRUNC, 
			      S_IRUSR|S_IWUSR|S_IRGRP|S_IROTH);
    }
    else {
	assert (mode == PP_FA_AT_READ);
	file_entry->fd = open(filename, O_RDONLY);
    }

    if (file_entry->fd == -1) {
	atomic_file_t_destroy(file_entry);
	return -1;
    }
    file_entry->successful_so_far = 1;
    file_entry->mode = mode;

    MUTEX_LOCK(&open_files_mtx);
    vector_add(open_files, file_entry);
    MUTEX_UNLOCK(&open_files_mtx);

    return file_entry->fd;
}

int pp_fa_at_close(int fd){
    int close_ret;
    atomic_file_t *file_entry;

    MUTEX_LOCK(&open_files_mtx);
    assert(open_files);
    file_entry = get_file_by_fd(fd);
    if (file_entry) remove_fd(fd);
    MUTEX_UNLOCK(&open_files_mtx);

    if (file_entry == NULL) {
	pp_log("[FA] close(): file is NULL");
	return PP_ERR;
    }

    close_ret = close(file_entry->fd);
    if (close_ret == -1) {
	file_entry->successful_so_far = 0;
    }

    if (file_entry->mode == PP_FA_AT_WRITE) {
	int fail;
	if (!file_entry->successful_so_far) {
	    fail = unlink(file_entry->tempname);
	    if (fail) {
		pp_log_err("[FA] unlink() temporary file %s", 
			   file_entry->tempname);
	    }
	    atomic_file_t_destroy(file_entry);
	    return PP_ERR;
	}

	fail = rename(file_entry->tempname, file_entry->name);
	if (fail) {
	    pp_log_err("[FA] rename <%s> to <%s>", 
		       file_entry->tempname, file_entry->name);
	    atomic_file_t_destroy(file_entry);
	    return PP_ERR;
	}
    }
    else {
	assert(file_entry->mode == PP_FA_AT_READ);
    }

    atomic_file_t_destroy(file_entry);
    return PP_SUC;
}

size_t pp_fa_at_read(int fd, void * buf, int length){
    int num = read(fd, buf, length);
    if (num == -1) {
	atomic_file_t * file = get_file_by_fd(fd);
	if (file) file->successful_so_far = 0;
    }
    return num;
}
size_t pp_fa_at_write(int fd, void * buf, int length){
    int num = write(fd, buf, length);
    if (num == -1) {
	atomic_file_t * file = get_file_by_fd(fd);
	if (file) file->successful_so_far = 0;
    }
    return num;
}


void pp_fa_at_fail(int fd){
    atomic_file_t * file = get_file_by_fd(fd);
    if (file != NULL) {
	file->successful_so_far = 0;
    }
}


/* Das hier ist halbautomatisch. 

Folgende Dateien muessen mit dem angegebenen Inhalt existieren
bzw. nicht existieren:

exist.txt
doesnt_exist.txt

longfile:
"longfile-content, longfile-content

extend_longfile:
"longfile-content, longfile-content"

shorten_longfile:
"longfile-content, longfile-content"

open_close_longfile:
"longfile-content, longfile-content"

write_zero_longfile:
"longfile-content, longfile-content"

okfile:
"file is sane"

*/
/*
void test_thread() {
    int fd;
    int retval;

    pp_bmc_log_warn("\nTest thread started");
    sleep(2);
    pp_bmc_log_warn("concurrent open second PRAE");
    fd = pp_fa_at_open("concurrent_file", PP_FA_AT_WRITE);
    pp_bmc_log_warn("concurrent open second , fd %d", fd);
    retval = pp_fa_at_write(fd, &"Second content written here.  ", strlen("Second content written here.  "));
    pp_bmc_log_warn("retval %d", retval);
    pp_bmc_log_warn("closing second");
    pp_fa_at_close(fd);
    pp_bmc_log_warn("Quitting test thread");
}
pthread_t thread;

void test_cases() {
    pp_bmc_log_warn("Testing atomic write logic...");

    int fd;
    int retval;

    fd = pp_fa_at_open("open_close_longfile", PP_FA_AT_WRITE);
    pp_bmc_log_warn("\nopenclose test, fd %d", fd);
    pp_fa_at_close(fd);

    fd = pp_fa_at_open("write_zero_longfile", PP_FA_AT_WRITE);
    pp_bmc_log_warn("\nwrite zero test, fd %d", fd);
    retval = pp_fa_at_write(fd, &"",0);
    pp_bmc_log_warn("write() retval  zero longfile %d", retval);
    pp_fa_at_close(fd);
    
    fd = pp_fa_at_open("doesnt_exist", PP_FA_AT_READ);
    pp_bmc_log_warn("\ndoesn_t exist read , fd %d", fd);
    assert(fd == -1);

    fd = pp_fa_at_open("doesnt_exist_zero_write", PP_FA_AT_WRITE);
    pp_bmc_log_warn("\ndoenst exist zero write, fd %d", fd);
    retval = pp_fa_at_write(fd, &"", 0);
    pp_bmc_log_warn("write zero doesnt exist retval %d", retval);
    pp_fa_at_close(fd);

    fd = pp_fa_at_open("doesnt_exist", PP_FA_AT_WRITE);
    pp_bmc_log_warn("\ndoesnt_exist write!, fd %d", fd);
    retval = pp_fa_at_write(fd, &"doesnt_exist content", strlen("doesnt_exist content"));
    pp_bmc_log_warn("write something doesnt exist retval %d", retval);
    pp_fa_at_close(fd);

// Does the mechanism to not copy the .tmp file back if a write fails work?
// dd if=/dev/zero of=smalldisk bs=1000k count=70
// mke2fs smalldisk 70000
// mount -o loop smalldisk install_root_devel
//
// Or just mount the flash on KIM somewhere
//    fd = pp_fa_at_open("/flashdisk/doesnt_exist_write_a_lot", PP_FA_AT_WRITE);
//    pp_bmc_log_warn("\n/flashdisk/ flashdisk doenst exist write a lot, fd %d", fd);
//    int i;
//    long int stuff;
//    char rand_state[32];

//    assert(initstate(15, &rand_state, 32));
//    i=0;
//    do {
//	i++;
//	stuff = random(); // the flash filesystem compresses. so better save uncompressible stuff.
//	retval = pp_fa_at_write(fd, &stuff, sizeof stuff);
//	printf(".");
//   } while (retval != -1);
//    pp_bmc_log_warn("written a lot");
//    pp_bmc_log_warn("retval %d, i %d", retval, i);
//    pp_fa_at_close(fd);
//
//    fd = pp_fa_at_open("/flashdisk/exists", PP_FA_AT_WRITE);
//    pp_bmc_log_warn("\n/flashdisk/exists write a lot, fd %d", fd);
//    i=0;
//    do {
//	i++;
//	stuff = random();
//	retval = pp_fa_at_write(fd, &stuff, sizeof stuff);
//	printf(".");
//    } while (retval != -1);
//    pp_bmc_log_warn("written a lot. still exists as original?!");
//    pp_bmc_log_warn("retval %d, i %d", retval, i);
//    pp_fa_at_close(fd);


    fd = pp_fa_at_open("shorten_longfile", PP_FA_AT_WRITE);
    pp_bmc_log_warn("\nshorten_longfile, fd %d", fd);
    retval = pp_fa_at_write(fd, &"short_cont", strlen("short_cont"));
    pp_bmc_log_warn("retval %d", retval);
    pp_fa_at_close(fd);

    fd = pp_fa_at_open("extend_longfile", PP_FA_AT_WRITE);
    pp_bmc_log_warn("\nextend longfile , fd %d", fd);
    retval = pp_fa_at_write(fd, &"very long long long long long long long content", 
			    strlen("very long long long long long long long content"));
    pp_bmc_log_warn("retval %d", retval);
    pp_fa_at_close(fd);

    fd = pp_fa_at_open("no_exist.dir/no_exist", PP_FA_AT_WRITE);
    pp_bmc_log_warn("\nno exist dir write, fd %d", fd);
    assert(fd == -1);


    pthread_create(&thread, NULL,(void * (*)(void *)) test_thread, 
		   NULL);
    pp_bmc_log_warn("concurrent open first");
    fd = pp_fa_at_open("concurrent_file", PP_FA_AT_WRITE);
    pp_bmc_log_warn("concurrent file first write, fd %d", fd);
    retval = pp_fa_at_write(fd, &"content concurrent file first first first first first first thirst first content", strlen("content concurrent file first first first first first first thirst first content"));
    pp_bmc_log_warn("retval %d", retval);
    sleep(4);
    pp_bmc_log_warn("concurrent close first");
    pp_fa_at_close(fd);
    sleep(3);


    fd = pp_fa_at_open("no_exist_fail_file", PP_FA_AT_WRITE);
    pp_bmc_log_warn("\nno_exist_fail_file test, fd %d", fd);
    retval = pp_fa_at_write(fd, &"bla", strlen("bla"));
    pp_bmc_log_warn("retval %d", retval);
    pp_fa_at_fail(fd);
    pp_fa_at_close(fd);

    fd = pp_fa_at_open("okfile", PP_FA_AT_WRITE);
    pp_bmc_log_warn("\nokfile fail test, fd %d", fd);
    retval = pp_fa_at_write(fd, &"failed", strlen("failed"));
    pp_bmc_log_warn("retval %d", retval);
    pp_fa_at_fail(fd);
    pp_fa_at_close(fd);
// the condition doesn't assert() if accessed by the same thread a second time
//    int fd2;
//    fd = pp_fa_at_open("longfile", PP_FA_AT_WRITE);
//    pp_bmc_log_warn("\nlongfile write-open , fd %d", fd);
//    fd2 = pp_fa_at_open("longfile", PP_FA_AT_WRITE);
//    pp_bmc_log_warn("lonfile write-open after write-open , fd %d", fd2);
//
//    assert("The error checking mutex should've killed me by now (write)" == NULL);
//
//    fd = pp_fa_at_open("longfile", PP_FA_AT_READ);
//    pp_bmc_log_warn("\nlonfile read-open , fd %d", fd);
//    fd2 = pp_fa_at_open("longfile", PP_FA_AT_READ);
//    pp_bmc_log_warn("lonfile read-open after read-open , fd %d", fd2);
//
//    assert("The error checking mutex should've killed me by now (read)" == NULL);
//
    pp_bmc_log_warn("...done testing.");

}
*/
