#include <linux/version.h>
#include <inttypes.h>
#include <stdio.h>
#include <sys/types.h>
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
# include <linux/mtd/mtd.h>
#else
# include <mtd/mtd-user.h>
#endif
#include <fcntl.h>
#include <unistd.h>
#include <sys/ioctl.h>
#include <sys/sem.h>
#include <sys/types.h>
#include <sys/mman.h>
#include <sys/resource.h>
#include <sys/time.h>
#include <sys/wait.h>
#include <openssl/md5.h>

#include <pp/features.h>
#include <pp/base.h>
#include <pp/firmware.h>
#include <pp/sem_keys.h>
#include <pp/um.h>
#include <liberic_config.h>
#include <liberic_misc.h>
#include <liberic_notify.h>
#if defined(PP_FEAT_REMOTE_CONSOLE)
#include <pp/km.h>
#include <pp/rfb.h>
#endif /* PP_FEAT_REMOTE_CONSOLE */
#include <liberic_session.h>
#if defined(PP_FEAT_MASTERCONSOLE_FW_UPDATE)
# include <pp/kvm.h>
#endif
#ifdef PRODUCT_SMARTIPC
# include <pp/minicom.h>
#endif
#if defined(KIRA_RPC)
#include <pp/bmc/bmc.h>
#include <pp/bmc/host_sensors.h>
#include <pp/bmc/tp_rs485_chip.h>
#endif
/*#include <lara.h>*/

#include "debug.h"
#include "erla.h"

#define PP_FW_FLASHDISK_SIZE_CMD      "tar -sz"
#define PP_FW_UPDSYS_FLDISK_PATH      "/flashdisk"
#define PP_FW_UPDSYS_CMD              "fwupdsys.sh"
#define PP_FW_UPDATE_CMD              "fwupdate.sh"
#define PP_FW_FLDIR_UPDATE_CMD        "fldirupdate.sh"


#if PP_HWID_INT >= 0x0E
# define ATMEL_FUSE_COMMAND		"/bin/atmel -w 0xD9AF"
#else
# define ATMEL_FUSE_COMMAND		"/bin/atmel -w 0xD9EF"
#endif

typedef struct {
    const char* name;
    u_char id;
    long int max_size;
} flash_dir_info_t;

static flash_dir_info_t supported_flash_dirs[] = {
    { "/",      FW_PART_TYPE_FLASH_DIR_NUM_ROOTFS,  (12 * 1024 * 1024) },
    { "oem",	FW_PART_TYPE_FLASH_DIR_NUM_OEM,	    (3 * 1024 * 1024)  },
    { "i18n",	FW_PART_TYPE_FLASH_DIR_NUM_I18N,    (1 * 1024 * 1024)  },
    { NULL,	0, 0 } /* keep this */
};

typedef struct {
    fw_part_hdr_t* next_part;
    void* fwdata;
    size_t fwdata_len;
} fw_part_iter_t;

typedef struct {
    fw_part_hdr_t hdr;
    void* data;
    size_t len;
} fw_part_info_t;

/* this structure contains the firmware version compiled in the code  *
 * It will be patched in binary form by mkfirmware.pl with the actual *
 * firmware version, buildnumber and tag                              */
typedef struct {
    /* magic number for retrieving the position of this struct in binary form*/
    char magic[32];
    /* version str, null-terminated  */
    char version[PP_FIRMWARE_ERLA_VERSION_LEN+1];
    /* build_nr str, null-terminated */
    char build_nr_str[PP_FIRMWARE_ERLA_BUILD_NR_LEN+1];
    /* minimum allowed version for downgrade, null-terminated */
    char min_downgrade_version[PP_FIRMWARE_ERLA_VERSION_LEN+1];
    /* tag str, null-terminated      */
    char build_tag[PP_FIRMWARE_ERLA_TAG_LEN+1]; 
} __attribute__ ((packed)) pp_firmware_erla_internal_t;

pp_firmware_erla_internal_t firmware_erla_internal = {
    .magic = "pp_fw_ver_hdr251105v10_fzsaiudl\0",
    /* this is the data that is changed later on */
    .version = "040202",
    .build_nr_str = " 0123",
    .min_downgrade_version = ERIC_FW_MIN_DOWNGRADE_VER_STR,
    .build_tag = "Devel\0                                                          ",
};
// Note: currently the FW_MIN_DOWNGRADE_VER_STR is still compiled in,
// this will be replaced by time ...

/* public pointers to the internal firmware-version structure above */
char* pp_firmware_erla_version = firmware_erla_internal.version;
u_char pp_firmware_erla_version_major = 0;
u_char pp_firmware_erla_version_minor = 0;
u_char pp_firmware_erla_version_subminor = 0;
char* pp_firmware_erla_build_nr_str = firmware_erla_internal.build_nr_str;
u_int pp_firmware_erla_build_nr = 0;
char* pp_firmware_erla_tag = firmware_erla_internal.build_tag;

/* internal vars */
static int fw_update_sem_id;
static int fw_update_lock_held = 0;

static void * fw_data = NULL;
static size_t fw_data_len = 0;
static pp_firmware_ctx_t fw_data_ctx_curr = 0;
static pp_firmware_ctx_t fw_data_ctx_next = 0;
static pthread_mutex_t fw_data_mtx = PTHREAD_RECURSIVE_MUTEX_INITIALIZER_NP;

/* iterate over firmware partition headers */
static void fw_part_iter_init(fw_part_iter_t* iter, void* fwdata,
			      size_t fwdata_len);
static int fw_part_iter_has_next(fw_part_iter_t* iter, fw_part_info_t* part);

/* prepare the raw header data before accessing it (endianess swap)
   caller has to free the returned header after using it */
static void fw_prepare_hdr(const fw_hdr_t* hdr_raw, fw_hdr_t* hdr);
static void fw_prepare_part_hdr(const fw_part_hdr_t *hdr_raw,
				fw_part_hdr_t* hdr);

static int firmware_erla_update_internal(pp_firmware_ctx_t ctx);
static int jffs2_validate(void* fwdata, size_t fwdata_len);
#ifdef PP_FEAT_JFFS2_ROOTFS
static int flash_dir_all_update(void* data, size_t len);
static int flash_dir_rootfs_update(const char * data, size_t len);
#else /* !PP_FEAT_JFFS2_ROOTFS */
static int jffs2_conv_requested(void* fwdata, size_t fwdata_len);
static void flash_dir_clear(u_char id);
#endif /* !PP_FEAT_JFFS2_ROOTFS */
static int flash_dir_all_check_size(void* data, size_t len);
//static void flash_dir_clear(u_char id);
static int flash_dir_update(u_char id, const char* updcmd, const char * data,
			    size_t len);
static const flash_dir_info_t* flash_dir_find(u_char id);
static long int flash_dir_size(const char * data, size_t len);

static int fw_flash(const char * filename, const char * data, size_t len);
#if !defined(PRODUCT_MSIDC)&& !defined(PRODUCT_SMIDC)&& !defined(PRODUCT_AMDDC)&& !defined(PRODUCT_ASMIDC)
static int atmel_update(const char * data, size_t len, int type);
#endif
#if defined(LARA_KIMSMI) || defined (PRODUCT_ASMIDC)
static int maxii_update(const char * data, size_t len);
#endif /* LARA_KIMSMI */

#if defined(KIRA_RPC)
static int cy8c26_update_all(const char* data, size_t len);
#endif /* KIRA_RPC */
    
static int finalize_update(void);

int
firmware_erla_init(void)
{
    char buf[3]; buf[2] = 0;

    /* variables are already initialized, *
     * only make sure that the strings are 0-terminated */
    firmware_erla_internal.version[PP_FIRMWARE_ERLA_VERSION_LEN] = 0;
    firmware_erla_internal.build_nr_str[PP_FIRMWARE_ERLA_BUILD_NR_LEN] = 0;
    firmware_erla_internal.min_downgrade_version[PP_FIRMWARE_ERLA_VERSION_LEN] = 0;
    firmware_erla_internal.build_tag[PP_FIRMWARE_ERLA_TAG_LEN] = 0;
    /* parse build_nr and version numbers */
    strncpy(buf, &firmware_erla_internal.version[0], 2);
    pp_firmware_erla_version_major = pp_strtoul_10(buf, 0, NULL);
    strncpy(buf, &firmware_erla_internal.version[2], 2);
    pp_firmware_erla_version_minor = pp_strtoul_10(buf, 0, NULL);
    strncpy(buf, &firmware_erla_internal.version[4], 2);
    pp_firmware_erla_version_subminor = pp_strtoul_10(buf, 0, NULL);
    pp_firmware_erla_build_nr = pp_strtoul_10(pp_firmware_erla_build_nr_str,
					      0, NULL);
    
    //printf("version = %s\n    %s\n    %s\n    %s\n",
    //       firmware_erla_internal.version,
    //       firmware_erla_internal.build_nr_str,
    //       firmware_erla_internal.min_downgrade_version,
    //       firmware_erla_internal.build_tag);

    if ((fw_update_sem_id = pp_sem_get(FW_UPDATE_SEM_KEY)) == -1) {
        errno = EBUSY;
        return PP_ERR;
    }

    return PP_SUC;
}

pp_firmware_ctx_t
pp_firmware_erla_ctx_new(size_t data_len, int preempt)
{
    pp_firmware_ctx_t ctx = 0;

    MUTEX_LOCK(&fw_data_mtx);
    if (!fw_update_lock_held && (fw_data_ctx_curr == 0 || preempt)) {
	pp_mmap_free(fw_data);
	if ((fw_data = pp_mmap_malloc(data_len)) != NULL) {
	    fw_data_len = data_len;
	    /*wrap-around case not handled, should not occur in this universe*/
	    fw_data_ctx_curr = ++fw_data_ctx_next;
	    ctx = fw_data_ctx_curr;
	} else {
	    fw_data_len = 0;
	    fw_data_ctx_curr = 0;
	}
    }
    MUTEX_UNLOCK(&fw_data_mtx);

    return ctx;
}

void
pp_firmware_erla_ctx_free(pp_firmware_ctx_t ctx)
{
    MUTEX_LOCK(&fw_data_mtx);
    if (ctx != 0 && ctx == fw_data_ctx_curr) {
	pp_mmap_free(fw_data);
	fw_data = NULL;
	fw_data_len = 0;
	fw_data_ctx_curr = 0;
    }
    MUTEX_UNLOCK(&fw_data_mtx);
}

int 
pp_firmware_erla_data_set_len(pp_firmware_ctx_t ctx, size_t data_len)
{
    int ret = PP_ERR;

    MUTEX_LOCK(&fw_data_mtx);
    if (ctx != 0 && ctx == fw_data_ctx_curr) {
	if (data_len <= fw_data_len) {
	    fw_data_len = data_len;
	    ret = PP_SUC;
	} else {
	    errno = PP_FIRMWARE_ERR_SIZE_EXCEEDED;
	}
    } else {
	errno = PP_FIRMWARE_ERR_BAD_CONTEXT;
    }
    MUTEX_UNLOCK(&fw_data_mtx);

    return ret;
}

ssize_t
pp_firmware_erla_data_get_len(pp_firmware_ctx_t ctx)
{
    ssize_t size = -1;

    MUTEX_LOCK(&fw_data_mtx);
    if (ctx != 0 && ctx == fw_data_ctx_curr) size = fw_data_len;
    MUTEX_UNLOCK(&fw_data_mtx);

    return size;
}

int 
pp_firmware_erla_data_set(pp_firmware_ctx_t ctx, const void * data,
			  size_t offset, size_t data_len)
{
    int ret = PP_ERR;

    MUTEX_LOCK(&fw_data_mtx);
    if (ctx != 0 && ctx == fw_data_ctx_curr) {
	if (offset <= fw_data_len && (offset + data_len) <= fw_data_len) {
	    memcpy(fw_data + offset, data, data_len);
	    ret = PP_SUC;
	} else {
	    errno = PP_FIRMWARE_ERR_SIZE_EXCEEDED;
	}
    } else {
	errno = PP_FIRMWARE_ERR_BAD_CONTEXT;
    }
    MUTEX_UNLOCK(&fw_data_mtx);

    return ret;
}

int 
pp_firmware_erla_data_get(pp_firmware_ctx_t ctx, void * data, size_t offset,
			  size_t data_len)
{
    int ret = PP_ERR;

    MUTEX_LOCK(&fw_data_mtx);
    if (ctx != 0 && ctx == fw_data_ctx_curr) {
	if (offset <= fw_data_len && (offset + data_len) <= fw_data_len) {
	    memcpy(data, fw_data + offset, data_len);
	    ret = PP_SUC;
	} else {
	    errno = PP_FIRMWARE_ERR_SIZE_EXCEEDED;
	}
    } else {
	errno = PP_FIRMWARE_ERR_BAD_CONTEXT;
    }
    MUTEX_UNLOCK(&fw_data_mtx);

    return ret;
}

int
pp_firmware_erla_get_md5_hash(pp_firmware_ctx_t ctx, u_char * hash)
{
    int ret = PP_ERR;

    MUTEX_LOCK(&fw_data_mtx);
    if (ctx != 0 && ctx == fw_data_ctx_curr) {
	if (fw_data && fw_data_len) {
	    MD5(fw_data, fw_data_len, hash);
	    ret = PP_SUC;
	} else {
	    errno = PP_FIRMWARE_ERR_CORRUPTED;
	}
    } else {
	errno = PP_FIRMWARE_ERR_BAD_CONTEXT;
    }
    MUTEX_UNLOCK(&fw_data_mtx);

    return ret;
}

int
pp_firmware_erla_is_update_locked_no_ctx(void)
{
    int ret;

    MUTEX_LOCK(&fw_data_mtx);
    ret = fw_update_lock_held;
    MUTEX_UNLOCK(&fw_data_mtx);

    return ret;
}


int
pp_firmware_erla_test_and_set_update_lock_no_ctx(void)
{
    int ret = PP_ERR;

    MUTEX_LOCK(&fw_data_mtx);
    if (pp_sem_lock_nb(fw_update_sem_id) == 0) {
	fw_update_lock_held = 1;
	ret = PP_SUC;
    }
    MUTEX_UNLOCK(&fw_data_mtx);

    return ret;
}


int
pp_firmware_erla_test_and_set_update_lock(pp_firmware_ctx_t ctx)
{
    int ret = PP_ERR;

    MUTEX_LOCK(&fw_data_mtx);
    if (ctx != 0 && ctx == fw_data_ctx_curr) {
	ret = pp_firmware_erla_test_and_set_update_lock_no_ctx();
    } else {
	errno = PP_FIRMWARE_ERR_BAD_CONTEXT;
    }
    MUTEX_UNLOCK(&fw_data_mtx);

    return ret;
}

int
pp_firmware_erla_release_update_lock_no_ctx(void)
{
    int ret = PP_ERR;

    MUTEX_LOCK(&fw_data_mtx);
    if (pp_sem_unlock(fw_update_sem_id) == 0) {
	fw_update_lock_held = 0;
	ret = PP_SUC;
    } else {
	errno = PP_FIRMWARE_ERR_INTERNAL;
    }
    MUTEX_UNLOCK(&fw_data_mtx);

    return ret;
}

int
pp_firmware_erla_release_update_lock(pp_firmware_ctx_t ctx)
{
    int ret = PP_ERR;

    MUTEX_LOCK(&fw_data_mtx);
    if (ctx != 0 && ctx == fw_data_ctx_curr) {
	ret = pp_firmware_erla_release_update_lock_no_ctx();
    } else {
	errno = PP_FIRMWARE_ERR_BAD_CONTEXT;
    }
    MUTEX_UNLOCK(&fw_data_mtx);

    return ret;
}

int
pp_firmware_erla_get_info(pp_firmware_ctx_t ctx, char** ver, u_int* buildnr,
			  char** tag, char** min_req_ver, char** min_dwngrd_ver)
{
    int ret = PP_ERR;
    fw_hdr_t hdr;

    MUTEX_LOCK(&fw_data_mtx);
    if (ctx != 0 && ctx == fw_data_ctx_curr && fw_data != NULL) {
	fw_prepare_hdr((fw_hdr_t *)fw_data, &hdr);
	if (ver != NULL) *ver = strndup(hdr.ver, sizeof(hdr.ver));
	if (buildnr != NULL) *buildnr = hdr.build_nr;
	if (tag != NULL) *tag = strndup(hdr.tag, sizeof(hdr.tag));
	if (min_req_ver != NULL) *min_req_ver = strndup(hdr.min_req_ver, sizeof(hdr.min_req_ver));
	if (min_dwngrd_ver != NULL) *min_dwngrd_ver = strdup(firmware_erla_internal.min_downgrade_version);
	ret = PP_SUC;
    } else {
	errno = PP_FIRMWARE_ERR_BAD_CONTEXT;
    }
    MUTEX_UNLOCK(&fw_data_mtx);

    return ret;
}

/*
 * with respect of jffs2 rootfs, the following check is implemented
 * PP_FEAT_JFFS2_ROOTFS is set (device runs already on jffs-rootfs)
 *  - accept boot code part of 256K
 *  - accept flashdisk part of type OEM and of type ROOTFS (TODO: size check)
 *  - accept kme2 part of type 4 or 5
 *  * any other part will result in an error
 * PP_FEAT_JFFS2_ROOTFS is not set (device may convert to jffs-rootfs)
 *  - if boot-code part size is 128K - old style firmware, as is
 *  - if boot-code part size is 256K - jffs2 conversion requested
 *    - take rules above
 *    - initiate conversion
 */
int
pp_firmware_erla_validate(pp_firmware_ctx_t ctx, int cross_oem, int cross_hwid)
{
    fw_hdr_t hdr;
    u_char md5_hash[MD5_DIGEST_LENGTH];
    char oem_tag[sizeof(hdr.oem) + 1];
    char product_str[sizeof(hdr.product) + 1];
    char min_req_ver_str[sizeof(hdr.min_req_ver) + 1];
    char new_ver_str[sizeof(hdr.ver) + 1];
    char * endptr, *endptr2;
    char* curr_downgrade_ver_str =
	firmware_erla_internal.min_downgrade_version;
    u_int curr_ver = 0;
    u_int curr_downgrade_ver = 0;
    u_int min_req_ver;
    u_int new_ver;
    int curr_version_unknown = 0;

    MUTEX_LOCK(&fw_data_mtx);

    if (ctx == 0 || ctx != fw_data_ctx_curr) {
	pp_log("%s(): uploaded firmware was deleted by task with higher "
	       "priority\n", ___F);
	errno = PP_FIRMWARE_ERR_BAD_CONTEXT;
	goto fail;
    }

    if (fw_data == NULL || fw_data_len < FW_HDR_SIZE) goto fail_invalid;
    
    fw_prepare_hdr((fw_hdr_t *)fw_data, &hdr);
    
    MD5(fw_data + sizeof(hdr.md5_hash), fw_data_len - sizeof(hdr.md5_hash),
	md5_hash);
    if (memcmp(md5_hash, fw_data, sizeof(hdr.md5_hash)) != 0
	|| memcmp(hdr.magic, FW_MAGIC, sizeof(hdr.magic)) != 0
	|| hdr.hdr_ver > FW_HDR_VER) {
	errno = PP_FIRMWARE_ERR_BAD_CRC;
	goto fail_invalid;	
    }

    /* parse version, min_downgrade_version string */
    if (   pp_firmware_erla_version[0] == '\0'
	   || (curr_ver = strtoul(pp_firmware_erla_version, &endptr, 10)) == 0
	   || *endptr != '\0'
	   || curr_downgrade_ver_str == '\0'
	   || (curr_downgrade_ver = strtoul(curr_downgrade_ver_str, &endptr2, 10))
	   == 0
	   || *endptr2 != '\0' ) {
	curr_version_unknown = 1;
    }

    memcpy(new_ver_str, hdr.ver, 6);
    new_ver_str[6] = '\0';
    new_ver = strtoul(new_ver_str, &endptr, 10);

    if (*endptr != '\0') {
	errno = PP_FIRMWARE_ERR_CORRUPTED;
	goto fail_invalid;
    }

    if (!cross_oem && !curr_version_unknown && new_ver < curr_downgrade_ver) {
	pp_log("%s(): can't downgrade to uploaded version (minimum version is %c%c.%c%c.%c%c)\n",
		 ___F, curr_downgrade_ver_str[0], curr_downgrade_ver_str[1],
		 curr_downgrade_ver_str[2], curr_downgrade_ver_str[3],
		 curr_downgrade_ver_str[4], curr_downgrade_ver_str[5]);
	errno = PP_FIRMWARE_ERR_NEW_VER_TOO_OLD;
	goto fail;
    }

    memcpy(min_req_ver_str, hdr.min_req_ver, 6);
    min_req_ver_str[6] = '\0';
    min_req_ver = strtoul(min_req_ver_str, &endptr, 10);

    if (*endptr != '\0') {
	errno = PP_FIRMWARE_ERR_CORRUPTED;
	goto fail_invalid;
    }

    if (!cross_oem && !curr_version_unknown && curr_ver < min_req_ver) {
	pp_log("%s(): current firmware too old (minimum expected version is %c%c.%c%c.%c%c)\n",
		 ___F, min_req_ver_str[0], min_req_ver_str[1], min_req_ver_str[2],
		 min_req_ver_str[3], min_req_ver_str[4], min_req_ver_str[5]);
	errno = PP_FIRMWARE_ERR_CUR_VER_TOO_OLD;
	goto fail;
    }

    /* check OEM version */
    if (hdr.hdr_ver >= 2) {
	memcpy(oem_tag, hdr.oem, sizeof(hdr.oem));
	oem_tag[sizeof(hdr.oem)] = '\0';
	pp_log("OEM-Version %s, we are %s\n", oem_tag,
	       PP_STRINGIFY_EXPANDED(PP_OEM));
	if (!cross_oem && strcmp(oem_tag, PP_STRINGIFY_EXPANDED(PP_OEM))) {
	    pp_log("%s(): OEM version not compatible", ___F);
	    errno = PP_FIRMWARE_ERR_BAD_OEM;
	    goto fail_incompatible;
	}
    }

    /* check product type */
    if (hdr.hdr_ver >= 2) {
    	memcpy(product_str, hdr.product, sizeof(hdr.product));
	product_str[sizeof(hdr.product)] = '\0';
	pp_log("Product-Type %s, we are %s\n", product_str,
	       PP_STRINGIFY_EXPANDED(PP_PRODUCT));
	if (!cross_oem && strcmp(product_str,
				 PP_STRINGIFY_EXPANDED(PP_PRODUCT))) {
	    pp_log("%s(): product type not compatible", ___F);
	    errno = PP_FIRMWARE_ERR_BAD_PRODUCT;
	    goto fail_incompatible;
	}
    }

    /* check if firmware fits to this hardware (id) */   
    if (hdr.hdr_ver >= 2) {
	u_char hw_id = eric_misc_get_hardware_rev_int();
	int i = 0, found_in = 0, found_ex = 0;

	if (hdr.include_hwids[0]) {
	    pp_log("Compatible HW-IDs:   ");
	    for (i = 0; i < 16 && hdr.include_hwids[i]; ++i) {
		printf("%d ", hdr.include_hwids[i]);
		found_in |= (hw_id == hdr.include_hwids[i]) ? 1 : 0;
	    }
	    printf("\n");
	} else {
	    found_in = 1;
	}

	if (hdr.exclude_hwids[0]) {
	    pp_log("Incompatible HW-IDs: ");
	    for (i = 0; i < 16 && hdr.exclude_hwids[i]; ++i) {
		printf("%d ", hdr.exclude_hwids[i]);
		found_ex |= (hw_id == hdr.exclude_hwids[i]) ? 1 : 0;
	    }
	    printf("\n");
	}
	

	pp_log("Our HW-ID: %d\n", hw_id);
	
	if (!cross_hwid && (!found_in || found_ex)) {
	    pp_log("%s(): h/w ID not compatible", ___F);
	    errno = PP_FIRMWARE_ERR_BAD_HWID;
	    goto fail;
	}    
    }
    
#if defined(PP_FEAT_JFFS2_ROOTFS)
    if (jffs2_validate(fw_data, fw_data_len) == PP_ERR) {
	goto fail_invalid;
    }
#else
    fw_part_iter_t piter;
    fw_part_info_t part;
    fw_part_iter_init(&piter, fw_data, fw_data_len);
    while (fw_part_iter_has_next(&piter, &part)) {
	if (memcmp(part.hdr.magic, FW_PART_MAGIC, sizeof(part.hdr.magic))
	    || part.hdr.hdr_ver > FW_PART_HDR_VER) {
	    errno = PP_FIRMWARE_ERR_CORRUPTED;
	    goto fail_invalid;
	}
	pp_log("firmware part %d type %d found\n", part.hdr.num,
	       part.hdr.type);
    }

    if (jffs2_conv_requested(fw_data, fw_data_len)) {
	pp_log("fw-validate: jffs2 conversion requested!\n");
	if (jffs2_validate(fw_data, fw_data_len) == PP_ERR) {
	    goto fail_invalid;
	}
	pp_log("fw_validate: jffs2 conversion not implemented, yet!\n");
	errno = PP_FIRMWARE_ERR_JFFS2_NOT_SUPPORTED;
	goto fail_invalid;
    }
#endif /* PP_FEAT_JFFS2_ROOTFS */

    MUTEX_UNLOCK(&fw_data_mtx);
    return PP_SUC;

 fail_invalid:
    MUTEX_UNLOCK(&fw_data_mtx);
    pp_log_err("%s(): firmware file invalid", ___F);
    return PP_ERR;

 fail_incompatible:
    MUTEX_UNLOCK(&fw_data_mtx);
    pp_log_err("%s(): firmware not compatible", ___F);
    return PP_ERR;

 fail:
    MUTEX_UNLOCK(&fw_data_mtx);
    pp_log_err("%s(): firmware failed", ___F);
    return PP_ERR;
}

/*
 * jffs2_validate
 *  - accept boot code part of 256K
 *  - accept flashdisk part of type OEM and of type ROOTFS
 *  - accept kme2 part of type 4 or 5
 *  - accept rpc part
 *  * any other part will result in an error
 * (presumes that fw_data_mtx has been locked)
 * returns PP_SUC if ok, or PP_ERR if something is wrong
 */
static int
jffs2_validate(void* fwdata, size_t fwdata_len)
{
    int ret = PP_SUC;
    fw_part_iter_t piter;
    fw_part_info_t part;
    u_int32_t p_type, p_num, p_len;

    // TODO: check anticipated size this must be done for
    //       all jffs partitions at once
    fw_part_iter_init(&piter, fwdata, fwdata_len);
    while (fw_part_iter_has_next(&piter, &part)) {
	if (memcmp(part.hdr.magic, FW_PART_MAGIC, sizeof(part.hdr.magic))) {
	    pp_log("%s(): unknown firmware\n", ___F);
	    errno = PP_FIRMWARE_ERR_CORRUPTED;
	    return PP_ERR;
	}
	if (part.hdr.hdr_ver > FW_PART_HDR_VER) {
	    pp_log("%s(): unsupported version\n", ___F);
	    errno = PP_FIRMWARE_ERR_CUR_VER_TOO_OLD;
	    return PP_ERR;
	}
	p_type = part.hdr.type;
	p_num = part.hdr.num;
	p_len = part.len;
	pp_log("jffs2_validate: fw-part type=%d, num=%d, len=%d\n",
	       p_type, p_num, p_len);
	
	if (p_type == FW_PART_TYPE_FLASH) {
	    if (p_num == FW_PART_TYPE_FLASH_NUM_UBOOT_JFFS2) {
		if (p_len != FW_PART_SIZE_UBOOT_JFFS2) {
		    pp_log("%s(): uboot partition (type=%d, num=%d) has wrong "
			   "size (%d bytes)!\n", ___F, p_type, p_num, p_len);
		    errno = PP_FIRMWARE_ERR_BAD_UBOOT_PART;
		    ret = PP_ERR;
		}
	    } else {
		pp_log("%s(): unexpected flash-partition (type=%d, num=%d)!\n",
		       ___F, p_type, p_num);
		errno = PP_FIRMWARE_ERR_UNEXP_FLASH_PART;
		ret = PP_ERR;
	    }
	} else if (p_type == FW_PART_TYPE_FLASH_DIR) {
	    if (NULL == flash_dir_find(p_num)) {
		pp_log("%s(): unexpected jffs-file (type=%d, num=%d)!\n",
		       ___F, p_type, p_num);
		errno = PP_FIRMWARE_ERR_UNEXP_JFFS_PART;
		ret = PP_ERR;
	    }
	} else if (p_type == FW_PART_TYPE_KME_NEW) {
	    if (p_num != 4 && p_num != 5) {
		pp_log("%s(): unexpected kme2-partition (type=%d, num=%d)!\n",
		       ___F, p_type, p_num);
		errno = PP_FIRMWARE_ERR_UNEXP_KME2_PART;
		ret = PP_ERR;
	    }
	} else if (p_type == FW_PART_TYPE_KME) {
	    pp_log("%s(): unexpected kme partition (type=%d, num=%d)!\n",
		   ___F, p_type, p_num);
	    errno = PP_FIRMWARE_ERR_UNEXP_KME_PART;
	    ret = PP_ERR;
	} else if (p_type == FW_PART_TYPE_RPC_CY8_CTRL) {
	    if (p_num != FW_PART_TYPE_RPC_CY8_C26) {
		pp_log("%s(): unexpected RPC partition (type=%d, num=%d)!\n",
		       ___F, p_type, p_num);
		errno = PP_FIRMWARE_ERR_UNEXP_RPC_PART;
		ret = PP_ERR;
	    }
	}
    }
    return ret;
}

int
pp_firmware_erla_update(pp_firmware_ctx_t ctx)
{
    int ret;
    const char *lockfile = "/var/lock/erla_fw_update.lock";
    int fd;

    /* prevent kimtstmaster from killing eric during firmware update */
    fd = open(lockfile, O_CREAT);
    close(fd);

    ret = firmware_erla_update_internal(ctx);

    unlink(lockfile);

    return ret;
}

int
pp_firmware_erla_reset_device(eric_session_int_id_t session)
{
    return eric_misc_trigger_board_reset(session);
}

/*
 * Remarks for error handling:
 * we'll try to update as many parts as possible, even if an error
 * occures. This is due to possible dependencies between the parts
 * and an error condition during update indicates some serious
 * problem anyway that is difficult to recover from.
 * Error messages will be concatenated together.
 *
 * @return PP_SUC                 if all ok
 *         PP_FIRMWARE_ERR_UNCRIT if some preconditions failed,
 *                                flashing was not started yet
 *         PP_FIRMWARE_ERR_CRIT   one or more firmware part failed
 *                                to flash
 */
static int
firmware_erla_update_internal(pp_firmware_ctx_t ctx)
{
    fw_hdr_t hdr;
    fw_part_iter_t piter;
    fw_part_info_t part;
    char logmsg[PP_NOTIFY_MAX_MSG_LEN+1];
    int r, ret;

    MUTEX_LOCK(&fw_data_mtx);
    ret = PP_FIRMWARE_ERR_NONCRIT;
    if (ctx == 0 || ctx != fw_data_ctx_curr || fw_data == NULL) {
	pp_log("%s(): uploaded firmware was deleted\n", ___F);
	errno = PP_FIRMWARE_ERR_BAD_CONTEXT;
	goto bail;
    }
    if ((r = pp_firmware_erla_test_and_set_update_lock_no_ctx()) > 0) {
	pp_log("%s(): firmware update is already running or the device is being reset!\n",
	       ___F);
	errno = EBUSY;
	goto bail;
    } else if (r < 0) {
	pp_log("%s(): internal error\n", ___F);
	errno = PP_FIRMWARE_ERR_INTERNAL;
	goto bail;
    }
    fw_prepare_hdr((fw_hdr_t *)fw_data, &hdr);
    MUTEX_UNLOCK(&fw_data_mtx);

    /* disconnect all rfb clients */
    // TODO: terminate listeners (bmc, rfb, etc.)!
#if defined(PP_FEAT_REMOTE_CONSOLE)
    pp_rfb_disconnect_all = 1;
    sleep(1);
#endif
    
    /* check flash dir sizes */
    if (flash_dir_all_check_size(fw_data, fw_data_len) != PP_SUC) {
	ret = PP_FIRMWARE_ERR_NONCRIT;
	goto bail;
    }
    
    MUTEX_LOCK(&eric_config_flash_write_mtx);
    ret = PP_SUC;

#if !defined(PP_FEAT_JFFS2_ROOTFS)
    /* Mount /flashdisk if necessary, clear the LightOEM data */
    pp_system("mount_flashdisk.sh");
    flash_dir_clear(FW_PART_TYPE_FLASH_DIR_NUM_OEM);
    flash_dir_clear(FW_PART_TYPE_FLASH_DIR_NUM_I18N);
#endif

    fw_part_iter_init(&piter, fw_data, fw_data_len);
    while (fw_part_iter_has_next(&piter, &part)) {
	char filename[32];
	
	if (part.hdr.type == FW_PART_TYPE_FLASH_DIR) {
#if !defined(PP_FEAT_JFFS2_ROOTFS) // flashed below, all together
	    if (flash_dir_update(part.hdr.num, PP_FW_FLDIR_UPDATE_CMD,
				 part.data, part.len)) {
		pp_log("%s(): part [%d,%d] failed: device may not work anymore"
		       " if reset!\n", ___F, part.hdr.type, part.hdr.num);
		errno = PP_FIRMWARE_ERR_FLASH_FLDIR_FAILED;
		ret = PP_FIRMWARE_ERR_CRIT;
	    }
#endif /* !PP_FEAT_JFFS2_ROOTFS */
	} else if (part.hdr.type == FW_PART_TYPE_FLASH) {
	    snprintf(filename, sizeof(filename), "/dev/mtd%d", part.hdr.num);
	    pp_log("fw_flash(%s, %p, %#zx)\n", filename, part.data, part.len);
	    if (fw_flash(filename, part.data, part.len) == -1) {
		pp_log("%s(): part [%d,%d] failed: device may not work anymore"
		       " if reset!\n", ___F, part.hdr.type, part.hdr.num);
		errno = PP_FIRMWARE_ERR_FLASH_PART_FAILED;
		ret = PP_FIRMWARE_ERR_CRIT;
	    }
	}
	
#if !defined(PRODUCT_MSIDC)&& !defined(PRODUCT_SMIDC)&& !defined(PRODUCT_AMDDC)&& !defined(PRODUCT_ASMIDC)
	else if (part.hdr.type == FW_PART_TYPE_KME
		 || part.hdr.type == FW_PART_TYPE_KME_NEW) {
	    if (atmel_update(part.data, part.len, part.hdr.type)) {
		pp_log("%s(): part [%d] failed: keyb/mouse may not work "
		       "anymore\n", ___F, part.hdr.type);
		errno = PP_FIRMWARE_ERR_FLASH_KME2_FAILED;
		ret = PP_FIRMWARE_ERR_CRIT;
	    }
	}
#endif /* !PRODUCT_MSIDC && !PRODUCT_SMIDC && !PRODUCT_AMDDC */
	
#if defined(PRODUCT_ERICXP) || defined(PRODUCT_ERICG4)
	else if (part.hdr.type == FW_PART_TYPE_ADC_ATMEL) {
	    if (atmel_update(part.data, part.len, part.hdr.type)) {
		pp_log("%s(): part [%d] failed: voltage monitor may not work "
		       "anymore\n", ___F, part.hdr.type);
		errno = PP_FIRMWARE_ERR_FLASH_ATMEL_FAILED;
		ret = PP_FIRMWARE_ERR_CRIT;
	    }
	}
#endif /* PRODUCT_ERICXP/G4 */
	
#ifdef PRODUCT_SMARTIPC
	else if (part.hdr.type == FW_PART_TYPE_SMARTIPC) {
	    // this is per definition the firemware part
	    // for the minicom cat5 smartipc switch
	    // ok, we have an update for the cat5 smart switch
	    r = pp_minicom_fw_update(part.data, part.len, part.hdr.num);
	    if (r < 0) {
		pp_log("upload rv: %d\n", r);
		pp_log("%s(): part [%d] failed: KVM switching may not work "
		       "anymore\n", ___F, part.hdr.type);
		errno = PP_FIRMWARE_ERR_FLASH_SMARTIPC_FAILED;
		ret = PP_FIRMWARE_ERR_CRIT;
	    } else {
		if (r == PP_MINICOM_FW_UPLOAD_ASK_OSD_UPDATE) {
		    websSetVar(wp, "_fw_smartipc_osd_update", "yes");
		}
		pp_log("upload ok: %d\n", r);
	    }
	}
#endif /* PRODUCT_SMARTIPC */
	
#if defined(PP_FEAT_MASTERCONSOLE_FW_UPDATE)
	else if (part.hdr.type == FW_PART_TYPE_MASTERCONSOLE) {
	    if (pp_kvm_update_masterconsole_ip_firmware(part.data, part.len)) {
		pp_log("%s(): part [%d] failed: device may not work anymore "
		       "if reset!\n", ___F, part.hdr.type);
		errno = PP_FIRMWARE_ERR_FLASH_MASTCONIP_FAILED;
		ret = PP_FIRMWARE_ERR_CRIT;
	    }
	}
#endif /* PP_FEAT_MASTERCONSOLE_FW_UPDATE */
	
#if defined(LARA_KIMSMI) || defined(PRODUCT_ASMIDC)
    	else if (part.hdr.type == FW_PART_TYPE_MAX_II_JAM) {
	    if (maxii_update(part.data, part.len)) {
		pp_log("%s(): part [%d] failed: Video Redirection may not "
		       "work anymore\n", ___F, part.hdr.type);
		errno = PP_FIRMWARE_ERR_FLASH_MAX2_FAILED;
		ret = PP_FIRMWARE_ERR_CRIT;
	    }
	}
#endif /* LARA_KIMSMI */

#if defined (KIRA_RPC)
	else if (part.hdr.type == FW_PART_TYPE_RPC_CY8_CTRL &&
		 part.hdr.num == FW_PART_TYPE_RPC_CY8_C26) {
	    if (cy8c26_update_all(part.data, part.len)) {
		pp_log("%s(): part [%d, %d] failed: Relay boards may not work "
		       "anymore\n", ___F, part.hdr.type, part.hdr.num);
		errno = PP_FIRMWARE_ERR_FLASH_CY8_FAILED;
		ret = PP_FIRMWARE_ERR_CRIT;
	    }
	}
#endif /* KIRA_RPC */	    
	
	else {
	    pp_log("%s(): Part [%d, %d] not allowed for this hardware!\n",
		   ___F, part.hdr.type, part.hdr.num);
	}
    }

#ifdef PP_FEAT_JFFS2_ROOTFS // flash jffs2 dirs together
    if (flash_dir_all_update(fw_data, fw_data_len) != PP_SUC) {
	ret = PP_FIRMWARE_ERR_CRIT;
    }
#endif /* !PP_FEAT_JFFS2_ROOTFS */

/* FIXME: adjust this for jffs2 update */
    // possibly do some final adjustments
    finalize_update();
    
    /* Note (geo): version, build_no, tag are now persistent in FW.
     * Values are not written to config sys any more */
    
    MUTEX_UNLOCK(&eric_config_flash_write_mtx);

    snprintf(logmsg, sizeof(logmsg),
	     "Firmware updated to %c%c.%c%c.%c%c (Build %u).%s",
	     hdr.ver[0], hdr.ver[1], hdr.ver[2], hdr.ver[3], hdr.ver[4],
	     hdr.ver[5], hdr.build_nr,
	     (ret != PP_SUC ? " There were errors!" : ""));
    eric_notify_post_event(logmsg, "device", PP_NOTIFY_EVENT_GENERIC);

    // make sure the event list gets flushed before we reboot
    eric_config_flush_in_foreground();

#if defined(PP_FEAT_REMOTE_CONSOLE)
    pp_rfb_disconnect_all = 0;
#endif

    pp_firmware_erla_release_update_lock(ctx); /* ignore errors */

 bail:
    return ret;
}

#if !defined(PRODUCT_MSIDC)&& !defined(PRODUCT_SMIDC)&& !defined(PRODUCT_AMDDC)&& !defined(PRODUCT_ASMIDC)
static int
atmel_update(const char * data, size_t len, int type)
{
    int ret = PP_ERR, retval, atmelfile = -1;
    char cmd[128];

    /* save new ATMEL firmware */
    if ((atmelfile = open("/tmp/atmel.bin",
			  O_CREAT | O_RDWR | O_TRUNC | O_SYNC, 0644)) == -1) {
	pp_log("%s(): Cannot open kme temp file.\n", ___F);
	goto failed;
    }

    if (write(atmelfile, data, len) == -1) {
	pp_log("%s(): Cannot write kme temp file.\n", ___F);
	goto failed;
    }

    /*
     * FIXME: there is a problem when flashing the KME-atmel right
     * after setting the fuse bits, the mouse may not work anymore
     * because the host doesn't react on our BAT. Seems the MDE<>Host
     * gets interrupted during initial setup sequence and doesn't
     * like it. (#610)
     * Workaround: Do not program fuse bits on KME-atmel. The programming
     *             is done during production process.
     */
    if (type == FW_PART_TYPE_ADC_ATMEL) {
	/* set ATMEL fuse bits */
	if ((retval = pp_system(ATMEL_FUSE_COMMAND)) == -1) {
	    pp_log("%s(): Executing atmel tool failed.\n", ___F);
	    goto failed;
	}
	
	if (WEXITSTATUS(retval) != 0) {
	    pp_log("%s(): Atmel tool returned error.\n", ___F);
	    goto failed;
	}
    }

    snprintf(cmd, sizeof(cmd), "/bin/atmel -a %u -f /tmp/atmel.bin",
	     type == FW_PART_TYPE_ADC_ATMEL ? 1 : 0);

    /* flash ATMEL firmware */
    if ((retval = pp_system(cmd)) == -1) {
	pp_log("%s(): Executing atmel tool failed.\n", ___F);
	goto failed;
    }

    if (WEXITSTATUS(retval) != 0) {
	pp_log("%s(): Atmel tool returned error.\n", ___F);
	goto failed;
    }

    ret = PP_SUC;
    
 failed:
    if (atmelfile >= 0) close(atmelfile);
    return ret;
}
#endif /* !PRODUCT_MSIDC && !PRODUCT_SMIDC && !PRODUCT_AMDDC */

#if defined (LARA_KIMSMI) || defined (PRODUCT_ASMIDC)
static int
maxii_update(const char * data, size_t len)
{
    int ret = PP_ERR, retval, maxfile = -1;
    char cmd[128];

    if ((maxfile = open("/tmp/dvo.jbc",
			O_CREAT | O_RDWR | O_TRUNC | O_SYNC, 0644)) == -1) {
	pp_log("%s(): Cannot open MAX II temp file.\n", ___F);
	goto failed;
    }

    if (write(maxfile, data, len) == -1) {
	pp_log("%s(): Cannot write MAX II temp file.\n", ___F);
	goto failed;
    }

    snprintf(cmd, sizeof(cmd), "/bin/jam_player -aPROGRAM /tmp/dvo.jbc");

    if ((retval = pp_system(cmd)) == -1) {
	pp_log("%s(): Executing jam_player tool failed.\n", ___F);
	goto failed;
    }

    if (WEXITSTATUS(retval) != 0) {
	pp_log("%s(): jam_player tool returned error.\n", ___F);
	goto failed;
    }

    ret = PP_SUC;

    /* TODO: set video.epldprogrammed to true in CFG! */
    
 failed:
    if (maxfile >= 0) close(maxfile);
    return ret;
}
#endif // LARA_KIMSMI

#if defined (KIRA_RPC)
static int cy8c26_update_all(const char* data, size_t len) {
    const char* msg = "cy8c26_update_all: updateing %d.CY8C26 (%s) %s\n";
    int i, ret, errors = 0;
    pp_tp_rs485_chip_t* chip;
    
    /* stop bmc server before fw-update */
    pp_bmc_stop();

    for (i = 0; i < PP_BMC_RS485_CHIP_PDU_MAX; ++i) {
	if ((chip = pp_bmc_host_get_rs485_chip(PP_BMC_RS485_CHIP_RPC0 + i))
	    != NULL) {
	    pp_log(msg, i, pp_tp_rs485_chip_to_string(chip), "");
	    if (PP_ERR == pp_tp_rs485_chip_fwupdate(chip, data, len)) {
		pp_log(msg, i, pp_tp_rs485_chip_to_string(chip), "FAILED");
		++errors;
	    } else {
		pp_log(msg, i, pp_tp_rs485_chip_to_string(chip), "DONE");
	    }
	}
    }
    if (errors > 0) {
	pp_log("%s: ERROR: %d PSoC failed to update!\n", ___F, errors);
	ret = PP_ERR;
    } else {
	ret = PP_SUC;
    }
    /* bmc will remain stopped, i.e. no matter whether error or not  *
     * there is supposed to be a reboot happening                    */
    return ret;
}    
#endif /* KIRA_RPC */

static int
flash_dir_all_check_size(void* data, size_t len)
{
    const flash_dir_info_t* fldir;
    fw_part_iter_t piter;
    fw_part_info_t part;
    long int size;

/* TODO: Check overall size with precalculated jffs2 size in header */
    
    fw_part_iter_init(&piter, data, len);
    while (fw_part_iter_has_next(&piter, &part)) {
	if (part.hdr.type != FW_PART_TYPE_FLASH_DIR) continue;
	if ((fldir = flash_dir_find(part.hdr.num)) == NULL) {
	    pp_log("%s(): part [%d,%d]: invalid Flash-FS file\n",
		   ___F, part.hdr.type, part.hdr.num);
	    errno = PP_FIRMWARE_ERR_BAD_FLASHFS;
	    return PP_ERR;
	}
	pp_log("Checking size for '%s'\n", fldir->name);
	if ((size = flash_dir_size(part.data, part.len)) < 0) {
	    pp_log("%s(): part [%d,%d]: Flash-FS '%s' has unknown size\n",
		   ___F, part.hdr.type, part.hdr.num, fldir->name);
	    errno = PP_FIRMWARE_ERR_BAD_FLASHFS;
	    return PP_ERR;
	}
	if (size == 0) {
            /* debug error only, should never occur in final! */
	    pp_log("%s(): part [%d,%d]: Flash-FS '%s' is empty\n",
		   ___F, part.hdr.type, part.hdr.num, fldir->name);
	    errno = PP_FIRMWARE_ERR_BAD_FLASHFS;
	    return PP_ERR;
	}
	if (size > fldir->max_size) {
	    pp_log("%s(): part [%d,%d]: Flash-FS '%s' is too big "
		   "(%ld KByte / max is %ld KByte)\n",
		   ___F, part.hdr.type, part.hdr.num, fldir->name,
		   size / 1024, fldir->max_size / 1024);
	    errno = PP_FIRMWARE_ERR_SIZE_EXCEEDED;
	    return PP_ERR;
	}
	pp_log("%ld Byte, ok!\n", size);
    }
    return PP_SUC;
}

#if defined (PP_FEAT_JFFS2_ROOTFS)

/*
 * # [eric: fwupsys.sh] create tiny fw-update sys in tmpfs /tmp
 *   * busybox + app-links
 *   * uClibc
 *   * ld-uClibc
 *   * update result page (simple, no images)
 *   * mount flashdisk
 *   * all in all about 600K
 * 
 * # [eric] pivot_root + chroot to tmp
 *
 * # [eric: fwupdate.sh]
 *   * rm all, except /flashdisk/config
 *   * untar fw from mem to /flashdisk
 *   * return mem-full code, if yes: unrecoverable error!!
 * 
 * # [eric: fldirupdate.sh]
 *   * untar oem from mem to /flashdisk/oem
 *   * return mem-full code, if yes: error msg
 *
 * # [eric] deliver page and reboot
 */
static int
flash_dir_all_update(void* data, size_t len)
{
    fw_part_iter_t piter;
    fw_part_info_t part;
    int ret = PP_SUC;

    /* rootfs first */
    fw_part_iter_init(&piter, data, len);
    while (fw_part_iter_has_next(&piter, &part)) {
	if (part.hdr.type == FW_PART_TYPE_FLASH_DIR &&
	    part.hdr.num == FW_PART_TYPE_FLASH_DIR_NUM_ROOTFS) {
	    if (flash_dir_rootfs_update(part.data, part.len) != PP_SUC) {
		pp_log("%s(): part [%d,%d]: Core system update failed!  The device may not work anymore!\n",
		       ___F, part.hdr.type, part.hdr.num);
		errno = PP_FIRMWARE_ERR_CORE_UPDATE_FAILED;
		return PP_ERR;
	    }
	    /* we are in the chroot fw-update-system now */
	    if (0 > chdir(PP_FW_UPDSYS_FLDISK_PATH)) {
		pp_log("%s(): chdir to %s failed!\n",
		       ___F, PP_FW_UPDSYS_FLDISK_PATH);
		pp_log("%s(): can't update OEM dirs - device may not work correctly!\n",
		       ___F);
		errno = PP_FIRMWARE_ERR_OEM_UPDATE_FAILED;
		return PP_ERR;
	    }
	    break;
	}
    }

    /* flash dirs second */
    fw_part_iter_init(&piter, data, len);
    while (fw_part_iter_has_next(&piter, &part)) {
	if (part.hdr.type == FW_PART_TYPE_FLASH_DIR &&
	    part.hdr.num != FW_PART_TYPE_FLASH_DIR_NUM_ROOTFS) {
	    if (flash_dir_update(part.hdr.num, PP_FW_FLDIR_UPDATE_CMD,
				 part.data, part.len)
		!= PP_SUC) {
		pp_log("%s(): part [%d,%d]: Flash directory update failed - "
		       "device may not work anymore!\n",
		       ___F, part.hdr.type, part.hdr.num);
		errno = PP_FIRMWARE_ERR_FLDIR_UPDATE_FAILED;
		ret = PP_ERR;
	    }
	}
    }
    return ret;
}

static int
flash_dir_rootfs_update(const char * data, size_t len)
{
    int status, ret = PP_ERR;

    pp_log("Check 1\n");
    
    /* create fw-update-sys in tmpfs */
    if ((status = pp_system(PP_FW_UPDSYS_CMD)) != 0) {
	pp_log("%s(): can't create update system: %d\n", ___F, status);
	goto bail;
    }

    /* chroot into update sys, just in case cwd ist wrong */
    if (chroot("/") != 0) {
	pp_log_err("%s(): chroot to / failed", ___F);
	goto bail;
    }

    /* copy new fw into flashdisk */
    ret = flash_dir_update(FW_PART_TYPE_FLASH_DIR_NUM_ROOTFS, PP_FW_UPDATE_CMD,
			   data, len);
 bail:
    return ret;
}

#else /* !PP_FEAT_JFFS2_ROOTFS */

/* should return true if old to jffs2 convertion is requested */
static int
jffs2_conv_requested(void* fwdata, size_t fwdata_len)
{
    fw_part_iter_t piter;
    fw_part_info_t part;

    // TODO: also check for FW_HDR_VERSION
    fw_part_iter_init(&piter, fwdata, fwdata_len);
    while (fw_part_iter_has_next(&piter, &part)) {
	if (part.hdr.type == FW_PART_TYPE_FLASH &&
	    part.hdr.num == FW_PART_TYPE_FLASH_NUM_UBOOT_JFFS2 &&
	    part.len == (256 * 1024)) {
	    return 1;
	}
    }
    return 0;
}

static void
flash_dir_clear(u_char id)
{
    char command[256], path[64];
    const flash_dir_info_t *flash_dir;
    int retval;

    flash_dir = flash_dir_find(id);
    if (flash_dir) {
	snprintf(path, sizeof(path), "%s/%s", PP_FW_UPDSYS_FLDISK_PATH, flash_dir->name);
	snprintf(command, sizeof(command), "rm -r %s; mkdir %s", path, path);
	retval = pp_system(command);
    }
}

#endif /* !PP_FEAT_JFFS2_ROOTFS */

static int
flash_dir_update(u_char id, const char* updcmd, const char * data, size_t len)
{
    const flash_dir_info_t* fldir;
    pp_strstream_t cmd = PP_STRSTREAM_INITIALIZER;
    int status, ret = PP_ERR;
    FILE* cin_str;

    if ((fldir = flash_dir_find(id)) == NULL) {
	pp_log("%s(): Unknown flash dir type %d\n", ___F, id);
	goto bail;
    }

    /* copy new dir into flashdisk */
    pp_strappendf(&cmd, "%s %s", updcmd, fldir->name);
    if (NULL == (cin_str = popen(pp_strstream_buf(&cmd), "w"))) {
	pp_log_err("%s(): popen %s failed", ___F, PP_FW_UPDATE_CMD);
	goto bail;
    }
    if (0 == fwrite(data, len, 1, cin_str)) {
	pp_log_err("%s(): fwrite to %s failed", ___F, PP_FW_UPDATE_CMD);
	pclose(cin_str);
	goto bail;
    }
    
    status = pclose(cin_str);
    if (status < 0) {
	pp_log_err("%s(): pclose failed", ___F);
	goto bail;
    } else if (status > 0) {
	pp_log_err("%s(): %s failed: %d, firmware maybe broken!",
		   ___F, PP_FW_UPDATE_CMD, status);
	goto bail;
    }
    pp_log("Successfully extracted flash dir '%s' (type %d)\n",
	   fldir->name, id);

    ret = PP_SUC;
 bail:
    pp_strstream_free(&cmd);
    return ret;
}

static const flash_dir_info_t*
flash_dir_find(u_char id)
{
    const flash_dir_info_t *fldir;

    for (fldir = supported_flash_dirs; fldir->name != NULL; ++fldir) {
	if (fldir->id == id) {
	    return fldir;
	}
    }
    return NULL;
}

static long
flash_dir_size(const char * data, size_t len)
{
    long int size = -1;
    pp_popen_rw_t pp_popen_data;
    char outbuf[32];
    
    if (pp_popen_rw(PP_FW_FLASHDISK_SIZE_CMD, &pp_popen_data) != 0) {
	return size;
    }
    if (fwrite(data, len, 1, pp_popen_data.stdout) != 1) {
	perror("flash_dir_size: fwrite");
	goto bail;
    }
    fflush(pp_popen_data.stdout);
    if (fread(outbuf, sizeof(outbuf), 1, pp_popen_data.stdin) != 1 &&
	ferror(pp_popen_data.stdin)) {
	perror("flash_dir_size: fread");
	goto bail;
    }
    outbuf[sizeof(outbuf) - 1] = '\0';
    size = strtol(outbuf, NULL, 10);
 bail:
    pp_pclose_rw(&pp_popen_data);
    return size;
}

static int
fw_flash(const char * filename, const char * data, size_t len)
{
    struct mtd_info_user mtd_info;
    struct erase_info_user mtd_lock_info;
    erase_info_t erase;
    int fd = -1;
    int ret = -1;

    if ((fd = open(filename, O_RDWR)) == -1 ||
	ioctl(fd, MEMGETINFO, &mtd_info) == -1) {
	goto fail;
    }

    /* unlock */

    mtd_lock_info.start = 0;
    mtd_lock_info.length = mtd_info.size;
    if (ioctl(fd, MEMUNLOCK, &mtd_lock_info) == -1) {
	goto fail;
    }

    /* erase all */

    erase.length = mtd_info.erasesize;
    for (erase.start = 0; erase.start < mtd_info.size;
	 erase.start += mtd_info.erasesize) {
       
	if (ioctl(fd, MEMERASE, &erase) == -1) {
	    goto fail;
	}
    }

    /* write data */

    if (write(fd, data, len) != (ssize_t)len) {
	goto fail;
    }

    /* lock */

#if !defined(PP_FEAT_JFFS2_ROOTFS) // nothing logged on JFFS2
    /* HACK: do *NOT* lock config, ppcboot and flashdisk partitions */ 
    if (strcmp(filename, "/dev/mtd2") &&
	strcmp(filename, "/dev/mtd3") &&
	strcmp(filename, "/dev/mtd4") &&
	strcmp(filename, "/dev/mtd5")) {
	mtd_lock_info.start = 0;
	mtd_lock_info.length = mtd_info.size;
	if (ioctl(fd, MEMLOCK, &mtd_lock_info) == -1) {
	    goto fail;
	}
    }
#endif

    ret = 0;

 fail:
    if (fd != -1) {
	close(fd);
    }
    return ret;
}

/**
 * do some final preparations after each firmware update
 * shouldn't be here
 */
static int
finalize_update(void)
{
#if defined(PP_FEAT_OPMA_HW) || defined(LARA_KIMMSI)
    int ret = -1;
    /* clean persistent sdr file so new data from our topo
       file is used again */
    if (unlink("/flashdisk/bmc/sdr") != 0) {
	pp_log_err("%s: could not clean sdr", ___F);
	goto bail;
    }
    ret = 0;
 bail:
    return ret;
#else
    return 0;
#endif

}

static void fw_prepare_hdr(const fw_hdr_t* hdr_raw, fw_hdr_t* hdr) {
    assert(hdr_raw != NULL);
    memcpy(hdr, hdr_raw, sizeof(fw_hdr_t));

    /* some raw header data is big endian, swap it */
    hdr->build_nr = be32_to_cpu(hdr->build_nr);
}

static void fw_prepare_part_hdr(const fw_part_hdr_t *hdr_raw,
				fw_part_hdr_t* hdr) {
    assert(hdr_raw != NULL);
    memcpy(hdr, hdr_raw, sizeof(fw_part_hdr_t));

    /* some raw part header data is big endian, swap it */
    hdr->num = be32_to_cpu(hdr->num);
    hdr->next_part_off = be32_to_cpu(hdr->next_part_off);
    hdr->type = be32_to_cpu(hdr->type);
}

static void fw_part_iter_init(fw_part_iter_t* iter, void* fwdata,
			      size_t fwdata_len) {
    iter->next_part = (fw_part_hdr_t *)(fwdata + FW_HDR_SIZE);
    iter->fwdata = fwdata;
    iter->fwdata_len = fwdata_len;
}
    
static int fw_part_iter_has_next(fw_part_iter_t* iter, fw_part_info_t* part) {
    int ret;
    fw_part_hdr_t* rawptr = iter->next_part;
    if ((void*)rawptr < (iter->fwdata + iter->fwdata_len)) {
	fw_prepare_part_hdr(rawptr, &part->hdr);
	part->data = (void*)rawptr + FW_PART_HDR_SIZE;
	part->len =  iter->fwdata + part->hdr.next_part_off - part->data;
	iter->next_part = (fw_part_hdr_t *)
	    (iter->fwdata + part->hdr.next_part_off);
	ret = 1;
    } else {
	ret = 0;
    }
    return ret;
}
