#ifndef _GRAB_DATA_H
#define _GRAB_DATA_H

#include <sys/types.h>

#include <lara.h>
#include <pp/base.h>
#include <pp/grab.h>
#include "mempool.h"

/**
 * states used when VSC library detected a video mode change,
 * tells grabber to stop while changing the video mode
 * and resumes grabber when ready
 */
typedef enum {
    CHANGE_NONE,
    CHANGE_ANNOUNCED,
    CHANGE_WAITING
} fb_change_t;

/**
 * states used during externally triggered suspend
 * of the grabber for any reason
 */
typedef enum {
    SUSPEND_SUSPEND,
    SUSPEND_SUSPENDED,
    SUSPEND_RESUME,
    SUSPEND_RESUMED
} fb_suspend_t;

/**
 * states used for hardware encoded clients
 * when waiting for their transfer request to be fulfilled
 */
typedef enum {
    CLIENT_STATE_IDLE,
    CLIENT_STATE_WAITING,
    CLIENT_STATE_TRANSFERRING
} client_state_t;

/**
 * states for the special synchronous transfers
 * (suspending the grabber)
 * used by direct transfer mode to avoid tearing,
 * used for the screenshot
 */
typedef enum {
    REQUEST_SYNC_NONE,
    REQUEST_SYNC_REQUEST,
    REQUEST_SYNC_IN_PROGRESS
} request_sync_t;

/**
 * state of the buffer memory for hardware encoding
 * used to stop a client when reorganizing the buffer memory
 * in case another client is added or removed
 */
typedef enum {
    BUF_MEM_STATE_IDLE,
    BUF_MEM_STATE_REQUEST
} buf_mem_state_t;

typedef enum {
    FB_SYNC_UNLOCKED,
    FB_SYNC_LOCKED,
    FB_SYNC_UNLOCKING,
} fb_sync_lock_state;
    
/**
 * internally used data structure per client,
 * tracks memory, encoding data/control
 */
struct grab_client_int_s {
    struct list_head listnode;		/* list handle */
    pp_grab_client_t *ext;		/* link to externally available data */

    mem_desc_t *mem_desc;		/* buffer memory descriptor */
    vsc_fetch_descriptor_t fetch_desc;	/* determines encoding for all requests */
    u_short max_tile_size;		/* worst case size for a single tile */

    volatile client_state_t state;	/* client request fulfillment state */    
    pthread_cond_t cond;		/* waiting cond during requests */
    pthread_mutex_t mtx;		/* lock state info and cond */

    buf_mem_state_t buf_mem_state;	/* buffer used by client or not */
    pthread_cond_t buf_mem_cond;	/* condition to wait/signal buffer release */
    pthread_mutex_t buf_mem_mtx;
    
    u_char fb_sync_lock;		/* lock fb (diff) until release */

    u_char ignore_diff;			/* full request, don't use mod_reg */
    u_char refresh;			/* refresh pending, don't use mod_reg */
    u_char dont_split;			/* don't split request region */
    u_char dont_touch;			/* dont touch our requested region at all,
					   use it for transfer as-is.
					   implies dont_split and ignore_diff */
    
    RegionRec *req_reg;			/* ptr to requested region */
    RegionRec *ack_reg;			/* ptr to acknowledged region to transfer */
    RegionRec *rem_reg;			/* ptr to remaining region not processed */
    RegionRec mod_reg;			/* per-client modified region (diffmap) */
    RegionRec client_map;		/* temporary working client diffmap */
    RegionRec global_map;		/* temporary working global diffmap */
    RegionRec temp_map;			/* temporary working diffmap */
};
typedef struct grab_client_int_s grab_client_int_t;

/**
 * combination of functions defining the grabber behavior
 * atm there are only 2 methods available:
 * - direct transfer with framebuffer (VSC 1.0)
 * - hardware encoding per client (VSC 2.0)
 */
typedef struct {
    u_char direct;
    fb_sync_lock_state fb_sync_lock;
    pthread_mutex_t fb_sync_lock_mtx;
   
    pp_grab_client_t* (*new_client)(pp_grab_mem_desc_id_t id, u_char video_link);
    void (*remove_client)(pp_grab_client_t *client);
    int (*set_encoding)(pp_grab_client_t *client, vsc_encoding_desc_t enc, u_int32_t encoding_tag);
    
    int (*request_region)(pp_grab_client_t *client, pp_grab_req_flags_t flags,
			  RegionRec *req, RegionRec *ack, RegionRec *remain);
    int (*request_rect)(pp_grab_client_t *client, pp_grab_req_flags_t flags,
			u_short x, u_short y, u_short w, u_short h);
    void (*release_buffer)(pp_grab_client_t *client);
    void (*sync_fb)(pp_grab_sync_lock_t action, u_char video_link);
    
    int (*request_screen_sync)(pp_grab_client_t *client, u_char hw_downscale, u_char *scale);
    int (*release_screen_sync)(pp_grab_client_t *client);
    
    void (*cancel_requests) (u_char reason, u_char video_link);
    void (*transfer_regions) (RegionRec *diff_reg, u_char refresh_pending, u_char video_link);

    u_char debug_use_diffmap;
} grab_driver_t;

/* commonly used (read: grabber-global) variables */
extern pthread_mutexattr_t errcheck_mtx_attr;


typedef struct{
    unsigned char suspended;
    unsigned char grabber_available;
    volatile unsigned char nr_clients;

    fb_format_info_t* current_fb;
}pp_grab_fb_info_t;

extern pp_grab_fb_info_t *pp_grab_fb_info;

extern int fake_picture_error;

extern pthread_mutex_t clients_mtx;
extern struct list_head clients_list;

extern pp_grab_encoder_desc_t* grab_encoder_hook;

extern int current_num_video_links;

#endif /* _GRAB_DATA_H */
