#include <sys/ioctl.h>
#include <stdlib.h>
#include <unistd.h>

#include <vsc_regs.h>
#include "pp/grab.h"
#include "iipptr_bitmap.h"
#include "mempool.h"
#include "debug.h"

#define LRLE_HEADER_SIZE 14

vsc_encoding_desc_t enc_null = {
    algo:			VSC_ALGO_NULL,
    lrle_r_margin_rb:		0,
    lrle_r_margin_g:		0,
    lrle_grey_green:		0,
    lrle_g_margin:		0,
    lrle_runlimit:		0,
    lrle_linecopy:		0,
    lrle_grey_force:		0,
    lrle_grey_disable:		0,
    lrle_c_margin_rb:		0,
    lrle_c_margin_g:		0,
    lrle_color:			0,
    lrle_runlimit_reacc:	0,
    down_mode:                  0,
};

#if defined(DEBUG_DIFFMAP)
void
print_buffer(u_char *buf, int size, int brk)
{
    int i;
    printf("0x%p\n", buf);
    for (i = 0; i < size; i++) {        
        if ((i % brk) == 0) printf("\n");
        printf("%02x ", buf[i]);
    }
    printf("\n");
}
#endif

static u_long dbg_calc_ccr(vsc_encoding_desc_t *enc);
static u_long dbg_fetch(vsc_encoding_desc_t *enc, BoxRec box, u_int32_t ofs, u_char video_link);
static void dbg_dump_bmp(char* name, u_char* buf, u_short w, u_short h, fb_color_info_t *fb_c_info);
static void dbg_dump_buf(vsc_encoding_desc_t *enc, char* name, u_char *buf, u_long size, u_char to_stdout,
			 u_char lrle_hdr, u_short w, u_short h);
static void dbg_get_color_data(vsc_encoding_desc_t *enc, char* colorchar, u_char *depth);

void
dbg_dump_reg(u_char *buf, u_long buf_size UNUSED, RegionRec *reg, vsc_encoding_desc_t enc, fb_color_info_t *fb_c_info, u_char video_link)
{
    static int dump_count = 0;
    mem_desc_t *mem_desc = mempool_get_desc(GRAB_FIX_MEM_DESC_DEBUG1, video_link);
    int i;
    BoxRec box;

    DCH(D_BLABLA, "Debug dump run %d\n", dump_count);
    for (i = 0; i < REGION_NUM_RECTS(reg); i++) {
	u_long null_size;
	char namebuf[128];
	u_short w, h;
	
	box.x1 = REGION_RECTS(reg)[i].x1;
	box.y1 = REGION_RECTS(reg)[i].y1;
	box.x2 = REGION_RECTS(reg)[i].x2;
	box.y2 = REGION_RECTS(reg)[i].y2;

	w = box.x2 - box.x1;
	h = box.y2 - box.y1;

	w = (w % PP_FB_TILE_WIDTH) ? (w/PP_FB_TILE_WIDTH + 1) * PP_FB_TILE_WIDTH : w;
	h = (h % PP_FB_TILE_HEIGHT) ? (h/PP_FB_TILE_HEIGHT + 1) * PP_FB_TILE_HEIGHT : h;

	null_size = dbg_fetch(&enc_null, box, mem_desc->offset, video_link);

	snprintf(namebuf, sizeof(namebuf), "null_dump_d%02d_r%02d_x%03dy%03dw%03dh%03d.bmp", dump_count, i,
		 box.x1, box.y1, w, h);

	
	snprintf(namebuf, sizeof(namebuf), "null_dump_d%02d_r%02d_x%03dy%03dw%03dh%03d.txt", dump_count, i,
		 box.x1, box.y1, w, h);

	{
	    vsc_update_rect_hdr_t * hdr = (vsc_update_rect_hdr_t*)buf;
	    //	    print_buffer(buf, 32, 16);
	    //printf("call decoder with %d bytes, byte1 = %x\n", hdr->size, buf[16]);
	    
	    //	    if (!decodeRLEFrame(buf+16, hdr->size, w, h, enc.lrle_color)) {
	    //		decodeRLEFrame(buf+16, hdr->size, w, h, enc.lrle_color);
		//		extern int pretty;
		dbg_dump_buf(&enc, namebuf, buf+16, hdr->size, 0, 1, w, h);
		snprintf(namebuf, sizeof(namebuf), "null_dump_d%02d_r%02d_x%03dy%03dw%03dh%03d.bmp", dump_count, i,
			 box.x1, box.y1, w, h);
		dbg_dump_bmp(namebuf, mem_pool + mem_desc->offset, w, h, fb_c_info);
		//	    }
	    buf += 16;
	    buf += hdr->size;
	    if (hdr->size % 4) buf += 4 - (hdr->size % 4);
	}

    }
    DCH(D_BLABLA, "Debug dump run %d done\n", dump_count);
    dump_count++;

    mempool_release_desc(mem_desc);
}


/* check the encoding of single rects
   1) fetch NULL data
   2) encode NULL data using software encoder
   3) fetch the LRLE encoded data from VSC
   -> compare and dump if different

   ATTENTION, uses mem_pool and destroys its contents
*/

int
dbg_check_reg(RegionRec *reg, vsc_encoding_desc_t enc, fb_color_info_t *fb_c_info, u_char video_link)
{
    static int dump_count = 0;
    static const char* fn = ___F;
    int i, ret = 0;
    BoxRec box;
    mem_desc_t *mem_desc_null = mempool_get_desc(GRAB_FIX_MEM_DESC_DEBUG1, video_link);
    mem_desc_t *mem_desc_lrle = mempool_get_desc(GRAB_FIX_MEM_DESC_DEBUG2, video_link);
    u_long soft_whole_size = FIX_MEM_DESC_DEBUG1_SIZE;
    u_char *soft_buf = malloc(soft_whole_size);
    
    for (i = 0; i < REGION_NUM_RECTS(reg); i++) {
	u_char command[512];
	u_long  null_size = 0, lrle_size = 0, soft_size = 0;
	u_short w, h;
	u_long ccr;
	pp_popen_rw_t popen_data;
	int fd;
	u_char *null_buf = mem_desc_null->ptr, *lrle_buf = mem_desc_lrle->ptr, *soft_read = soft_buf;
	
	box.x1 = REGION_RECTS(reg)[i].x1;
	box.y1 = REGION_RECTS(reg)[i].y1;
	box.x2 = REGION_RECTS(reg)[i].x2;
	box.y2 = REGION_RECTS(reg)[i].y2;

	w = box.x2 - box.x1;
	h = box.y2 - box.y1;

	w = (w % PP_FB_TILE_WIDTH) ? (w/PP_FB_TILE_WIDTH + 1) * PP_FB_TILE_WIDTH : w;
	h = (h % PP_FB_TILE_HEIGHT) ? (h/PP_FB_TILE_HEIGHT + 1) * PP_FB_TILE_HEIGHT : h;

	null_size = dbg_fetch(&enc_null, box, mem_desc_null->offset, video_link);
	lrle_size = dbg_fetch(&enc, box, mem_desc_lrle->offset, video_link);

	ccr = dbg_calc_ccr(&enc);
	DCH(D_ALWAYS, "Comparing (%4d,%4d)-(%4d,%4d) with Soft-LRLE\n", box.x1, box.y1, w, h);
	
	snprintf(command, sizeof(command),
		 "cellsplit -r -w %d -h %d | lrle_engine -c 0x%08lx | "
		 "rle_greycomp -c 0x%08lx  | rle_reduce -c 0x%08lx  | "
		 "rle_runaccum -c 0x%08lx  | rle_format -c 0x%08lx",
		 w, h, ccr, ccr, ccr, ccr, ccr);

	if (pp_popen_rw(command, &popen_data) != 0) {
	    DCH(D_ERROR, "%s: popen failed\n", fn);
	    goto bail;
	}
		
	if (fwrite(null_buf, null_size, 1, popen_data.stdout) != 1) {
	    DCH(D_ERROR, "%s: fwrite failed\n", fn);
	    goto bail;
	}

	fclose(popen_data.stdout);
	
	fd = fileno(popen_data.stdin);
	while (1) {
	    struct timeval to = { 10, 0 };
	    fd_set set;
	    int r;
	    
	    FD_ZERO(&set);
	    FD_SET(fd, &set);

	    if ((r = select(fd+1, &set, NULL, NULL, &to)) < 0) {
		DCH(D_ERROR, "%s: select error\n", fn);
		goto bail;
	    }

	    if (FD_ISSET(fd, &set)) {
		int got = read(fd, soft_read, soft_whole_size - soft_size);

		if (got < 0) {
		    DCH(D_ERROR, "%s: reading failed\n", fn);
		    goto bail;
		} else if (got == 0) {
		    break;
		}
		soft_size += got;
		soft_read += got;
	    } else {
		DCH(D_ERROR, "%s: select timeout\n", fn);
		goto bail;
	    }
	}

	if (pp_pclose_rw(&popen_data) != 0) {
	    DCH(D_ERROR, "%s: pclose failed\n", fn);
	    goto bail;
	}

	if ((soft_size - LRLE_HEADER_SIZE) != lrle_size ||
	    memcmp(lrle_buf, &soft_buf[LRLE_HEADER_SIZE], lrle_size) != 0) {

	    char namebuf[128];

	    DCH(D_ERROR, "!!! Found difference !!! (Files with %05d)\n", dump_count);

	    // bad encoded data, now write debug info
	    snprintf(namebuf, sizeof(namebuf), "null_dump_d%03d_x%03dy%03dw%03dh%03dccr%08lx.bmp", dump_count,
		     box.x1, box.y1, w, h, ccr);
	    dbg_dump_bmp(namebuf, null_buf, w, h, fb_c_info);

	    snprintf(namebuf, sizeof(namebuf), "lrle_dump_d%03d_x%03dy%03dw%03dh%03dccr%08lx.txt", dump_count,
		     box.x1, box.y1, w, h, ccr);   
	    dbg_dump_buf(&enc, namebuf, lrle_buf, lrle_size, 0, 1, w, h);

	    snprintf(namebuf, sizeof(namebuf), "soft_dump_d%03d_x%03dy%03dw%03dh%03dccr%08lx.txt", dump_count,
		     box.x1, box.y1, w, h, ccr);
	    dbg_dump_buf(&enc, namebuf, soft_buf, soft_size, 0, 0, w, h);

	    dump_count++;

	    ret = -1;
	}	
    }
    
 bail:
    mempool_release_desc(mem_desc_lrle);
    mempool_release_desc(mem_desc_null);
    free(soft_buf);
    return ret;
}

static u_long
dbg_fetch(vsc_encoding_desc_t *enc, BoxRec box, u_int32_t ofs, u_char video_link)
{
    static const char* fn = ___F;
    int grab_fd = -1;

    RegionRec reg; 
    vsc_fetch_descriptor_t fetch_desc = {
	mem_offset	: ofs,
	ctrl		: 0,
	enc		: *enc,
	enc_tag	        : 0,
	size		: 0,
	error		: 0,
	reg             : NULL,
    };
    u_long ret = 0;

    REGION_INIT(&reg, &box, 0);

    fetch_desc.reg = &reg;

    grab_fd = pp_grab_fd(video_link);
    if (ioctl(grab_fd, PPIOCVSCFETCHTILES, &fetch_desc) == -1) {
	DCH(D_ERROR, "%s: error during PPIOCVSCFETCHTILES ioctl\n", fn);
	goto bail;
    }
    if (fetch_desc.error != 0) {
	DCH(D_ERROR, "%s: fetch returned error %d\n", fn, fetch_desc.error);
	goto bail;
    }
    
    ret = fetch_desc.size;
    
 bail:
    REGION_UNINIT(&reg);
    return ret;
}

static void
dbg_dump_bmp(char* name, u_char* buf, u_short w, u_short h, fb_color_info_t *fb_c_info)
{
    t_bitmap bmp;
    int val;
    u_int16_t * i_16_ptr;        
    int offset_bmp=0;
    u_short x, y;

    x = 0;
    y = 0;
    w = (w % PP_FB_TILE_WIDTH) ? (w/PP_FB_TILE_WIDTH + 1) * PP_FB_TILE_WIDTH : w;
    h = (h % PP_FB_TILE_HEIGHT) ? (h/PP_FB_TILE_HEIGHT + 1) * PP_FB_TILE_HEIGHT : h;

    bmp.width = w;
    bmp.height= h;
    bmp.rgb = (t_rgb *)malloc(w * h * sizeof(t_rgb));
    
    // add returned offset to the fb start -> aligned rect start
    i_16_ptr = (u_int16_t *)buf;
    
    {
	int bol_skip  = x % PP_FB_TILE_WIDTH;
	int eol_skip  = ((bol_skip + w) % PP_FB_TILE_WIDTH) ? (bol_skip+w) - ((bol_skip+w) % PP_FB_TILE_WIDTH) : 0;
	int line_skip = -(w * PP_FB_TILE_HEIGHT - PP_FB_TILE_WIDTH);
	int row_skip  = (w - PP_FB_TILE_WIDTH) * PP_FB_TILE_HEIGHT;
	int w_single, line=0;

	D(D_BLABLA, "bol_skip=%d, eol_skip=%d, line_skip=%d, row_skip=%d\n",
	  bol_skip, eol_skip, line_skip, row_skip);
	
	// goto start pixel, still in the same first hextile
	i_16_ptr += (y % PP_FB_TILE_HEIGHT) * PP_FB_TILE_WIDTH + (x % PP_FB_TILE_WIDTH);
	
	while (line++ < h) {
	    w_single = w;
	    
	    while (w_single-- > 0) {
		val = *i_16_ptr++;
		bmp.rgb[offset_bmp].r = ((val >> (fb_c_info->blue_bits + fb_c_info->green_bits)) << (8 - fb_c_info->red_bits)) & 0xFF;
		bmp.rgb[offset_bmp].g = ((val >> fb_c_info->blue_bits) << (8 - fb_c_info->green_bits)) & 0xFF;
		bmp.rgb[offset_bmp].b = (val << (8 - fb_c_info->blue_bits)) & 0xFF;
		offset_bmp++;

		if (w_single % PP_FB_TILE_WIDTH == 0)
		    i_16_ptr += (PP_FB_TILE_WIDTH * PP_FB_TILE_HEIGHT - PP_FB_TILE_WIDTH);
	    }
	    
	    i_16_ptr += eol_skip;  // skip partial rect at end of line
	    i_16_ptr += line_skip; // start next pixel line
	    
	    if (line % PP_FB_TILE_HEIGHT == 0) i_16_ptr += row_skip;
	    
	    i_16_ptr += bol_skip;  // skip partial rect at beginning of line
	}
    }

    write_bitmap(&bmp, name);

    free(bmp.rgb);
}

static void
dbg_dump_buf(vsc_encoding_desc_t *enc, char* name, u_char *buf,
	     u_long size, u_char to_stdout, u_char lrle_hdr,
	     u_short w, u_short h)
{
    FILE *dump = fopen(name, "w");
    u_long i;
    u_char depth;
    char colorchar;

    dbg_get_color_data(enc, &colorchar, &depth);
    if (lrle_hdr) {
	fprintf(dump, "%-4d %-4d %c%02d\n", w, h, colorchar, depth);
    }
    for (i = 0; i < size; i++) {
	if (to_stdout) {
	    if ((i % 16) == 0) printf("\n");
	    printf("%02x ", buf[i]);
	}
	putc(buf[i], dump);
    }
    if (to_stdout) {
	printf("\n");
    }

    fclose(dump);      
}

static u_long
dbg_calc_ccr(vsc_encoding_desc_t *enc)
{
    return (CCR_VSC_ALGO(enc->algo)				|
	    CCR_LRLE_RMARGIN_RB(enc->lrle_r_margin_rb)		|
	    CCR_LRLE_RMARGIN_G(enc->lrle_r_margin_g)		|
	    CCR_LRLE_GREY_GREEN(enc->lrle_grey_green)		|
	    CCR_LRLE_GMARGIN(enc->lrle_g_margin)		|
	    CCR_LRLE_RUNLIMIT(enc->lrle_runlimit)		|
	    CCR_LRLE_LINECOPY(enc->lrle_linecopy)		|
	    CCR_LRLE_GREY_FORCE(enc->lrle_grey_force)		|
	    CCR_LRLE_GREY_DISABLE(enc->lrle_grey_disable)	|
	    CCR_LRLE_CMARGIN_RB(enc->lrle_c_margin_rb)		|
	    CCR_LRLE_CMARGIN_G(enc->lrle_c_margin_g)		|
	    CCR_LRLE_COLOR(enc->lrle_color)			|
	    CCR_LRLE_REACC_RUNLIMIT(enc->lrle_runlimit_reacc));
}

static void
dbg_get_color_data(vsc_encoding_desc_t *enc, char* colorchar, u_char *depth)
{
    assert(enc && depth);
    
    switch (enc->lrle_color) {
      case LRLE_COLOR_15BIT_DIRECT:
	  *colorchar = 'c';
	  *depth = 15;
	  break;
      case LRLE_COLOR_7BIT_DIRECT:
	  *colorchar = 'c';
	  *depth = 7;
	  break;
      case LRLE_COLOR_4BIT_PALETTE:
	  *colorchar = 'C';
	  *depth = 4;
	  break;
      case LRLE_COLOR_4BIT_GREY:
	  *colorchar = 'G';
	  *depth = 4;
	  break;
      case LRLE_COLOR_3BIT_GREY:
	  *colorchar = 'G';
	  *depth = 3;
	  break;
      case LRLE_COLOR_2BIT_GREY:
	  *colorchar = 'G';
	  *depth = 2;
	  break;
      case LRLE_COLOR_1BIT_GREY:
	  *colorchar = 'G';
	  *depth = 1;
	  break;
      default:
	  D(D_ERROR, "%s: color mode %d not supported\n", ___F, enc->lrle_color);
	  break;
    }
}

#ifdef DEBUG_PROFILE
/*
 * profiling functions
 */

typedef struct {
    int used;
    char *name;
    u_int64_t last_time;
    u_int64_t whole_time;
    u_int32_t no_calls;
} profiler_t;

static profiler_t profilers[MAX_NO_PROFILERS];

void profiler_init(void) {
    memset(profilers, 0, sizeof(profilers));
}

void profiler_add(unsigned int id, char *name) {
    profilers[id].used = 1;
    profilers[id].name = name;
    profiler_reset(id);
}

void profiler_reset(unsigned int id) {
    profilers[id].no_calls = 0;
    profilers[id].whole_time = 0;
}

void profiler_reset_all() {
    int i;
    
    for (i = 0; i < MAX_NO_PROFILERS; i++) {
        profiler_reset(i);
    }
}

void profiler_start(unsigned int id) {
    profilers[id].last_time = pp_hrtime_us();
}

void profiler_done(unsigned int id) {
    profilers[id].whole_time += pp_hrtime_us() - profilers[id].last_time;
    profilers[id].no_calls++;
}

void profiler_print(int dbg_level, unsigned int id) {
    u_int32_t time;
    u_int32_t calls;
    
    time = profilers[id].whole_time / 1000;
    calls = profilers[id].no_calls;

    D(dbg_level, "%s\t: %4d calls, %8d ms (%8d ms each).\n",
        profilers[id].name, calls, time,
        calls ? (time / calls) : 0);
}

void profiler_print_all(int dbg_level) {
    int i;
    
    D(dbg_level, "\x1b[H\x1b[2J----------------------------------------\n");
    D(dbg_level, "    profiler:\n");
    for (i = 0; i < MAX_NO_PROFILERS; i++) {
        if (profilers[i].used) {
            profiler_print(dbg_level, i);
        }
    }
    D(dbg_level, "----------------------------------------\n");
}
#endif

