// rle_decode
//		This decodes RLE input (all formats) and generates a BMP file for
//		output.
//
//		One option is provided:  -p turns on "pretty" output, which means that
//		instead of a BMP file, a human-readable description of the RLE is
//		printed to standard out.  Handy for debugging!!!

// Includes
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <ctype.h>
#include <unistd.h>

#include <vsc_regs.h>
#include "debug.h"

// Defines
#define MAXHEIGHT	1200
#define MAXWIDTH	1600
#define MIN(a,b)	((a)<(b)?(a):(b))
#define MAX(a,b)	((a)>(b)?(a):(b))
#define CWIDTH 16
#define CHEIGHT 16

#define RED     2
#define GREEN   1
#define BLUE    0


// Globals
#define DEBUG_DUMP_REGION
#ifdef DEBUG_DUMP_REGION
#ifdef REALWRITE
/* FIXME: attention, big array */
unsigned char	pixelMap[MAXHEIGHT][MAXWIDTH][3];
unsigned short	pixelArray[MAXHEIGHT][MAXWIDTH];

#endif


static int pretty = 0;

// settable args

static int depth = 0;
static int isGrey = 0;
static int isCompact = 0;

// Functions
static void update_mode(u_int32_t enc);
static void transColor(int code, unsigned char *color);
static void genGreyTable(int coldepth);
static void genColorTable(int coldepth);
static u_char *decodeRLECell(u_char *p, int cx, int cy, int lastX, int lastY);
static u_char *decodeRLEMap(u_char *p, int cx, int cy, int lastX, int lastY, int gcoldepth);

static u_int32_t encoding = 0;

unsigned char *lastp;

static void
update_mode(u_int32_t enc) {
    switch (enc) {
      case LRLE_COLOR_15BIT_DIRECT:
	  isGrey = 0;
	  isCompact = 0;
	  depth = 15;
	  break;
      case LRLE_COLOR_7BIT_DIRECT:
	  isGrey = 0;
	  isCompact = 0;
	  depth = 7;
	  break;
      case LRLE_COLOR_4BIT_PALETTE:
	  isGrey = 0;
	  isCompact = 1;
	  depth = 4;
	  break;
      case LRLE_COLOR_4BIT_GREY:
	  isGrey = 1;
	  isCompact = 1;
	  depth = 4;
	  break;
      case LRLE_COLOR_3BIT_GREY:
	  isGrey = 1;
	  isCompact = 1;
	  depth = 3;
	  break;
      case LRLE_COLOR_2BIT_GREY:
	  isGrey = 1;
	  isCompact = 1;
	  depth = 2;
	  break;
      case LRLE_COLOR_1BIT_GREY:
	  isGrey = 1;
	  isCompact = 1;
	  depth = 1;
	  break;
      default:
	  D(D_ERROR, "%s: color mode %d not supported\n", ___F, enc);
	  break;
    }

    // Create 24-bit pixelMap
	if (isGrey)
		genGreyTable(depth);
	else {
		genColorTable(depth);
		genGreyTable(depth == 7 ? 4 : 6);
	  }

	if (depth != 1 && depth != 2) {
	    //	    genPixelMap();
	}

  }

int
decodeRLEFrame(char* buf, int size, int w, int h, uint32_t enc) {
	int cx, cy, lastX, lastY;
	unsigned char *rlep;

	int width= w;
	int height = h;
	unsigned char*	rleCode;
	int rleSize = 0;

	rleCode = buf;
	rlep = rleCode;
	rleSize = size;
	
	lastp = &rleCode[rleSize];

	//	if (encoding != enc) {
	encoding = enc;
	update_mode(encoding);
	    //	}
	
	if (pretty) 		// Header + start of first cell
		printf("RLESIZE = %d\n", rleSize);
	cx = cy = 0;
	
	while (rlep != lastp) {
	// calculate bounds of cell
		lastX = MIN(cx + CWIDTH, width);
		lastY = MIN(cy + CHEIGHT, height);

	// decode cell
		if (pretty)
			printf("[%-6d]CELL AT [%d,%d]\n  LINE 0:\n", (rlep-rleCode),cx,cy);

		if (depth == 1 || depth == 2) {
		    rlep = decodeRLEMap(rlep, cx, cy, lastX, lastY, depth);
		} else {
		    rlep = decodeRLECell(rlep, cx, cy, lastX, lastY);
		}
		if (rlep == NULL) {
		    printf("error, decode_rlecell creturned NULL\n");
		    return 0;
		}
		if (pretty) {
			printf(">>END OF CELL\n");
		}

	// determine position of next cell
		cx = lastX;
		if (cx == width) {
			cx = 0;
			cy = lastY;
			if (cy == height)
				break;
		  }
	  }

	if (rlep != lastp) {
	    D(D_ERROR, "lastp-rlep: %d\n", lastp-rlep);
	    return 0;
	}
	return 1;
  }

static u_char
getByte(u_char **p, int* error)
{
    FILE *out;
    static u_char last_code = 0x0;

    *error = 0;
    out=pretty?stdout:stderr;
    if ((*p) > lastp) {
	*error = 1;
	printf("readbyte tried to read more bytes than available - last_code: %x\n", last_code);
	return 0;
    }  /*      *p++         */
    last_code = *(*p);
    return *(*p)++;
}

static u_char *
decodeRLEMap(u_char *p, int cx, int cy, int lastX, int lastY, int gdepth)
{
    int w = lastX - cx;
    int steps = 8 / depth;
    int lineBytes = w / steps;
    int lineRest  = w % steps;
    int segment;
    int k, code;
#ifdef REALWRITE
    int mask  = (1 << depth) - 1;
#endif
    int error;
    while (cy < lastY) {
	for (segment = 0; segment < lineBytes; segment++) {
	    code = getByte(&p, &error);
	    if (error) {
		return NULL;
	    }
	    for (k = steps - 1; k >= 0; k--) {
#ifdef REALWRITE
		transColor((code & mask) | 0x8000,
			   pixelMap[cy][cx + (segment * steps) + k]);
#endif
		code >>= gdepth;
	    }
	}

	if (lineRest != 0) {
	    code = getByte(&p, &error);
	    if (error) {
		return NULL;
	    }
	    for (k = lineRest - 1; k >= 0; k--) {
#ifdef REALWRITE
		transColor((code & mask) | 0x8000,
			   pixelMap[cy][cx + (lineBytes * steps) + k]);
#endif
		code >>= gdepth;
	    }    
	}
	cy++;
    }
    
    return p;
}

static u_char *
decodeRLECell(u_char *p, int cx, int cy, int lastX, int lastY) {
	int j;
	unsigned char color[3];
	int code;
	int	dx, dy;
	int c, run, copy, repeat;
	unsigned short grey;
#ifdef REALWRITE
	unsigned short prevLine[64];
	unsigned short *pa;
#endif
	int error;
	
	dx = dy = 0;
	c = -1;
	run = copy = repeat = 0;
	grey = (isGrey ? 0x8000 : 0);
	
#ifdef REALWRITE
	pa = &pixelArray[cy][cx];
#endif
	
	while (cy + dy < lastY) {
	    code = getByte(&p, &error);
	    if (error) {
		printf("error 1 dx: %d, dy: %d\n", dx, dy);
		return NULL;
	    }
	// look at code, set up for execution
		repeat = 0;
		copy = ((code & 0xE0) == 0xE0);		// Line copy always the same format!
		if (copy) {					
		    if (code == 0xFF) {
			    code = (code << 8) + (run = getByte(&p, &error));
			    if (error) {
				printf("error 2 dx: %d, dy: %d, \n", dx, dy);
				return NULL;
			    }
		    }
			else
				run = (code & 0x1f);
		  }
		else if (isCompact) {				// Decode compact 3/4 bit format
			if (depth <= 3) {					// 3-bit mode
				c = (code & 7);
				run = code >> 3;
			  }
			else {								// 4-bit mode
				c = (code & 15);
				run = code >> 4;
			  }
		  }
		else if (code & 0x80) {				// Grey compression or repeat
			if (code & 0x40) {					// repeat
				run = (code & 0x1f);
				repeat = 1;
			  }
			else {								// grey pixel
				c = (code & 0x3f);
				run = 0;
				grey = 0x8000;
			  }
		  }
//XXX Experimental compact mode, used for 5-bit mode
		else if (depth < 7) {
			c = (code & 0x7C)>>(7-depth);
			run = (code & ((1<<(7-depth))-1));
			if (run == (1<<(7-depth))-1) {
			    run = getByte(&p, &error);
			    if (error) {
				printf("error 7bit mode\n");
				return NULL;
			    }
			}
		  }
		else {
		    if (depth == 15) {
			code = (code << 8) + getByte(&p, &error);
			if (error) {
			    printf("error 3 dx: %d, dy: %d\n", dx, dy);
			    return NULL;
			}
			
		    }
			c = code;
			run = 0;
			grey = 0;
		  }

	// Do pretty printing
		if (pretty) {
			printf("\tX=%-2d ", dx);
			if (copy)
				printf("   LINE COPY 0x%02x (%d)\n", code, run+1);
			else if (repeat)
				printf("   RUN 0x%02x (%d)\n", code, run+1);
			else if (isCompact)
				printf("PIX RUN 0x%02x; [%2d] (%d)\n", code, c, run+1);
			else if (grey)
				printf("GREY 0x%02x; [%3d]\n", code, (c-grey));
			else {
				transColor(c, color);
				if (depth == 15)
					printf("PIX 0x%04x; [%3d,%3d,%3d]\n", 
						code, color[RED], color[GREEN], color[BLUE]);
				else 
					printf("PIX 0x%02x; [%3d,%3d,%3d]\n", 
						code, color[RED], color[GREEN], color[BLUE]);
			  }
		  }

	// Execute the run
		c |= grey;
		for (j = 0; j <= run; j++) {
#ifdef REALWRITE
		    if (!copy)
				prevLine[dx] = pa[dx] = c;
			else
				pa[dx] = prevLine[dx];
#endif
			dx++;
			if (cx+dx == lastX) {
				dx = 0;
				dy++;
#ifdef REALWRITE
				pa = &pixelArray[cy+dy][cx];
#endif
				if (pretty)
					printf("  LINE %d:\n", dy);
			  }
			if (((dx > 0)&(dy >= 16)) | (dy > 16)) {
			    printf("decoded more lines than in a hextile - dx: %d, dy: %d\n", dx, dy);
			    return NULL;
			}
		  }
	  }
	return(p);
  }

unsigned char greyTable[64];

static void
genGreyTable(int coldepth) {
	int i;

	for (i = 0; i < (1 << coldepth); i++)
		switch(coldepth) {
			case 1:		greyTable[i] = (i*255);			break;
			case 2:		greyTable[i] = (i* 85);			break;
			case 3:		greyTable[i] = (i* 73) / 2;		break;
			case 4:		greyTable[i] = (i* 17);			break;
			case 5:		greyTable[i] = (i* 33) / 4;		break;
			case 6:		greyTable[i] = (i* 65) / 16;	break;
			default:	greyTable[i] = 0;				break;
		  }
  }

unsigned char	c16table[16][3] = {
	{0,  0,  0  },	// black
	{0,  0,  127},	// dark red
	{0,  127,0  },	// dark green
	{0,  127,127},	// dark yellow
	{127,0,  0  },	// dark blue
	{127,0,  127},	// dark magenta
	{127,127,0  },	// dark cyan
	{127,127,127},	// dark grey
	{192,192,192},	// light grey
	{0,  0,  255},	// light red
	{0,  255,0  },	// light green
	{0,  255,255},	// light yellow
	{255,0,  0  },	// light blue
	{255,0,  255}, 	// light magenta
	{255,255,0  },	// light cyan
	{255,255,255},	// white
  };

unsigned char	c32table[32][3] = {
	{0,  0,  0  },  {127,0,  0  },  {255,0,  0  },
	{0,  127,0  },  {127,127,0  },  {255,127,0  },
	{0,  255,0  },  {127,255,0  },  {255,255,0  },
	{0,  0,  127},  {127,0,  127},  {255,0,  127},
	{0,  127,127},  {127,127,127},  {255,127,127},
	{0,  255,127},  {127,255,127},  {255,255,127},
	{0,  0,  255},  {127,0,  255},  {255,0,  255},
	{0,  127,255},  {127,127,255},  {255,127,255},
	{0,  255,255},  {127,255,255},  {255,255,255},

	{0,  0,  0  },  { 64, 64, 64},  {128,128,128},
	{192,192,192},  {255,255,255}
  };

unsigned char	colorTable[1<<15][3];

static void
genColorTable(int coldepth) {
	int i;
	static unsigned char decode7_5[5] = { 0, 64, 128, 192, 255 };

	for (i = 0; i < (1 << coldepth); i++) {
		switch(coldepth) {
			case 4:
				colorTable[i][RED]		= c16table[i][RED];
				colorTable[i][GREEN]	= c16table[i][GREEN];
				colorTable[i][BLUE]		= c16table[i][BLUE];
			  break;
			case 5:
				colorTable[i][RED]		= c32table[i][RED];
				colorTable[i][GREEN]	= c32table[i][GREEN];
				colorTable[i][BLUE]		= c32table[i][BLUE];
			  break;
			case 7:
/*XXX DECODER FOR OLD 7-BIT MODE
				colorTable[i][RED]		= ((i & 0x60) << 1) + ((i & 0x60) >> 1);
				colorTable[i][GREEN] 	= ((i & 0x1C) << 3) + ((i & 0x18)     );
				colorTable[i][BLUE]		= ((i & 0x03) << 6) + ((i & 0x03) << 4);
*/
				if (i >= 125)	// shouldn't ever be true
					colorTable[i][RED] = colorTable[i][GREEN] = 
						colorTable[i][BLUE] = 0xff;
				else {
					colorTable[i][RED]		= decode7_5[i / 25];
					colorTable[i][GREEN]	= decode7_5[(i / 5) % 5];
					colorTable[i][BLUE]		= decode7_5[i % 5];
				  }
			  break;
			case 15:
				colorTable[i][RED]		= (((i & 0x7C00) >> 10) * 33) / 4;
				colorTable[i][GREEN]	= (((i & 0x03E0) >> 5 ) * 33) / 4;
				colorTable[i][BLUE]		= (((i & 0x001F)      ) * 33) / 4;
			  break;
			default:
				fprintf(stderr, "ERROR: color depth '%d' unsupported\n", coldepth);
				exit(-1);
			  break;
		  }
	  }
  }

static void
transColor(int code, unsigned char *color) {
    if (code & 0x8000) { 		// grey
	color[RED] = color[GREEN] = color[BLUE] = greyTable[code & 0x7fff];
    } else {
		color[RED]		= colorTable[code][RED];
		color[GREEN]	= colorTable[code][GREEN];
		color[BLUE]		= colorTable[code][BLUE];
	  }
  }

#endif /* DEBUG_DUMP_REGION */
