#include <sys/ioctl.h>

#include <pp/rfb.h>
#include <pp/vsc.h>

#include "debug.h"
#include "direct.h"

/* single static fetch descriptor for direct/VSC1 transfer fb */
static vsc_fetch_descriptor_t fetch_desc_direct = {
    ctrl:	VSC_FETCH_CTRL_DIRECT_FB,
    mem_offset:	0,
    enc:	{
	algo:			VSC_ALGO_NULL,
	lrle_r_margin_rb:	0,
	lrle_r_margin_g:	0,
	lrle_grey_green:	0,
	lrle_g_margin:		0,
	lrle_runlimit:		0,
	lrle_linecopy:		0,
	lrle_grey_force:	0,
	lrle_grey_disable:	0,
	lrle_c_margin_rb:	0,
	lrle_c_margin_g:	0,
	lrle_color:		0,
	lrle_runlimit_reacc:	0,
	down_mode:		0
    },
    reg: NULL,
    enc_tag: 0,
    size: 0,
    error: 0,
};

grab_driver_t grab_driver_direct = {
    direct:		1,
    fb_sync_lock:       FB_SYNC_UNLOCKED,
    fb_sync_lock_mtx:	PTHREAD_ERRORCHECK_MUTEX_INITIALIZER_NP,
    
    new_client:		new_client_direct,
    remove_client:	remove_client_direct,
    set_encoding:	NULL,
    
    request_region:	request_region_direct,
    request_rect:	request_rect_direct,
    release_buffer:	NULL,
    sync_fb:		sync_fb_direct,
    
    request_screen_sync:request_screen_sync_direct,
    release_screen_sync:release_screen_sync_direct,
    
    transfer_regions:	transfer_direct,
    cancel_requests:	cancel_requests_direct,

    debug_use_diffmap: 1
};

static pthread_mutex_t fb_sync_mtx		= PTHREAD_ERRORCHECK_MUTEX_INITIALIZER_NP;
static pthread_mutex_t request_rect_sync_mtx	= PTHREAD_ERRORCHECK_MUTEX_INITIALIZER_NP;

static pthread_cond_t  request_sync_cond	= PTHREAD_COND_INITIALIZER;
static pthread_mutex_t request_sync_cond_mtx	= PTHREAD_ERRORCHECK_MUTEX_INITIALIZER_NP;

static request_sync_t  request_sync = REQUEST_SYNC_NONE;
static pp_grab_client_t *request_sync_client = NULL;

static void create_fake_picture(RegionRec *reg, u_char video_link);

/*********************************************************************
 * functions for the old raw grabber
 *********************************************************************/

void
transfer_direct(RegionRec *diff_reg, u_char refresh_pending UNUSED, u_char video_link)
{
    static const char* fn = ___F;
    struct list_head *node, *tmp;
    pp_grab_encoder_desc_t* act;
    
    fetch_desc_direct.reg = diff_reg;
      
//TODO: Do we need separate mutex for each grabber?
    MUTEX_LOCK(&fb_sync_mtx);

    if (REGION_NOTEMPTY(diff_reg)) {
	if (ioctl(pp_grab_fd(video_link), PPIOCVSCFETCHTILES, &fetch_desc_direct) == -1) {
	    if (errno != EAGAIN) {
		DCH(D_ERROR, "%s: ioctl(PPIOCVSCFETCHTILES)\n", fn);
	    }
	    goto err_out;
	}
    }

    if (fake_picture_error) {
	create_fake_picture(diff_reg, video_link);
    }

    MUTEX_LOCK(&clients_mtx);
    
    list_for_each_safe(node, tmp, &clients_list) {
	grab_client_int_t *cl_int = list_entry(node, grab_client_int_t, listnode);
	// omit clients with a different video_link
	// if (cl_int->video_link != video_link) continue;
	MUTEX_LOCK(&cl_int->mtx);
	REGION_UNION(&cl_int->mod_reg, &cl_int->mod_reg, diff_reg);
	MUTEX_UNLOCK(&cl_int->mtx);
    }
    
    MUTEX_UNLOCK(&clients_mtx);

    act = grab_encoder_hook;
    while (act) {
        if (act->new_data_available) {
            act->new_data_available(video_link);
        }
        act = act->next;
    }

    /* handle synchronous request */
    MUTEX_LOCK(&request_sync_cond_mtx);
    if (request_sync == REQUEST_SYNC_REQUEST) {
	D(D_VERBOSE, "%s: REQUEST_SYNC_REQUEST\n", fn);
	request_sync = REQUEST_SYNC_IN_PROGRESS;
	pthread_cond_signal(&request_sync_cond);
	while (request_sync != REQUEST_SYNC_NONE) {
	    pthread_cond_wait(&request_sync_cond, &request_sync_cond_mtx);
	}
	D(D_VERBOSE, "%s: REQUEST_SYNC_NONE\n", fn);
    }
    MUTEX_UNLOCK(&request_sync_cond_mtx);
    
 err_out:    
    MUTEX_UNLOCK(&fb_sync_mtx);
}

void
cancel_requests_direct(u_char reason, u_char video_link)
{
    static const char* fn = ___F;

    pp_grab_fb_info[video_link].grabber_available = 0;
    
    MUTEX_LOCK(&request_sync_cond_mtx);
    if (request_sync == REQUEST_SYNC_REQUEST) {
	D(D_VERBOSE, "%s: REQUEST_SYNC_REQUEST - cancel %d\n", fn, reason);
	switch (reason) {
	  case VSC_FETCH_ERROR_NO_SIGNAL:
	      request_sync_client->req_err = GRAB_REQ_ERR_NO_SIGNAL;
	      break;
	  case VSC_FETCH_ERROR_NOT_AVAILABLE:
	      request_sync_client->req_err = GRAB_REQ_ERR_NOT_AVAILABLE;
	      break;
	  case VSC_FETCH_ERROR_UNKNOWN_MODE:
	      request_sync_client->req_err = GRAB_REQ_ERR_UNKNOWN_MODE;
	      break;
	  default:
	      request_sync_client->req_err = GRAB_REQ_ERR_INTERNAL_ERROR;
	      break;
	}
	request_sync = REQUEST_SYNC_IN_PROGRESS;
	pthread_cond_signal(&request_sync_cond);
	while (request_sync != REQUEST_SYNC_NONE) {
	    pthread_cond_wait(&request_sync_cond, &request_sync_cond_mtx);
	}
	D(D_VERBOSE, "%s: REQUEST_SYNC_NONE\n", fn);
    }
    MUTEX_UNLOCK(&request_sync_cond_mtx);
}

pp_grab_client_t*
new_client_direct(pp_grab_mem_desc_id_t id UNUSED, u_char video_link)
{
    pp_grab_client_t *ret = NULL;
    grab_client_int_t *cl_i = NULL;

    D(D_VERBOSE, "Request for new client\n");    
    
    ret = malloc(sizeof(pp_grab_client_t));
    cl_i = malloc(sizeof(grab_client_int_t));

    MUTEX_LOCK(&clients_mtx);
    
    ret->intern = cl_i;
    ret->buf = mem_pool;

    cl_i->ext = ret;
    REGION_INIT(&cl_i->mod_reg, NullBox, 0);
    pthread_mutex_init(&cl_i->mtx, &errcheck_mtx_attr);

    list_add(&cl_i->listnode, &clients_list);
    pp_grab_fb_info[video_link].nr_clients++;
    ret->video_link = video_link;

    MUTEX_UNLOCK(&clients_mtx);

    /* FIXME: wait until we actually grabbed the first frame */

    D(D_VERBOSE, "Added client %p\n", ret);
    return ret;
}

void
remove_client_direct(pp_grab_client_t *client)
{
    grab_client_int_t *cl_i = client->intern;
    u_char video_link = client->video_link;

    MUTEX_LOCK(&clients_mtx);

    list_del(&cl_i->listnode);
    pp_grab_fb_info[video_link].nr_clients--;

    D(D_VERBOSE, "Removed client %p\n", client);

    pthread_mutex_destroy(&cl_i->mtx);
    REGION_UNINIT(&cl_i->mod_reg);

    MUTEX_UNLOCK(&clients_mtx);
    
    free(client->intern);
    free(client);
}

int
request_region_direct(pp_grab_client_t *client, pp_grab_req_flags_t flags UNUSED, RegionRec *req,
		      RegionRec *ack, RegionRec *remain UNUSED)
{
    grab_client_int_t *cl_int = client->intern;

    assert(req && ack);

    if (pp_vsc_has_signal(client->video_link) != VIDEO_SIGNAL_ON) {
	REGION_EMPTY(ack);
	return 0;
    }

    if (flags & GRAB_REQ_FLAGS_IGNORE_DIFFMAP) {
	REGION_COPY(ack, req);
    } else {	
	/* we just return the modified region here */
	MUTEX_LOCK(&cl_int->mtx);
	
	REGION_INTERSECT(ack, req, &cl_int->mod_reg);
	REGION_SUBTRACT(&cl_int->mod_reg, &cl_int->mod_reg, ack);

	MUTEX_UNLOCK(&cl_int->mtx);
    }
    
    return 0;
}

int
request_rect_direct(pp_grab_client_t *client, pp_grab_req_flags_t flags UNUSED,
		    u_short x, u_short y, u_short w UNUSED, u_short h UNUSED)
{
    video_signal_state_t vsc_signal;
    vsc_output_state_t  vsc_output;
    int ret = -1;

    u_char suspended = pp_grab_fb_info[client->video_link].suspended;    
    fb_format_info_t *current_fb = pp_grab_fb_info[client->video_link].current_fb;

    /* error handling, situations where grab is not possible */
    vsc_signal = pp_vsc_has_signal(client->video_link);
    vsc_output = pp_vsc_output_state(client->video_link);

    if (suspended || vsc_signal != VIDEO_SIGNAL_ON || vsc_output != OUTPUT_VALID) {
	if (vsc_signal != VIDEO_SIGNAL_ON) {
	    client->req_err = GRAB_REQ_ERR_NO_SIGNAL;
	} else if (vsc_output != OUTPUT_VALID) {
	    client->req_err = GRAB_REQ_ERR_UNKNOWN_MODE;
	} else if (suspended) {
	    client->req_err = GRAB_REQ_ERR_NOT_AVAILABLE;
	}	
	goto bail;
    }

    /* client might access the fb directly,
       we just return start of the rect and pitch */
   
    client->rect_offset = y * PP_FB_TILE_WIDTH * current_fb->tiles_w + x * PP_FB_TILE_HEIGHT;
    client->rect_pitch  = current_fb->g_w_pd;

    ret = 0;
    
 bail:
    if (ret != 0) errno = pp_grab_errno_base() + client->req_err;
    return ret;
}

int
request_screen_sync_direct(pp_grab_client_t *client, u_char hw_downscale UNUSED, u_char *scale)
{
    static const char* fn = ___F;
    int ret;

    assert(scale);
    *scale = 0;
    
    /* only one sync request at the same time */
    MUTEX_LOCK(&request_rect_sync_mtx);

    MUTEX_LOCK(&request_sync_cond_mtx);
    request_sync = REQUEST_SYNC_REQUEST;
    request_sync_client = client;
    D(D_VERBOSE, "%s: REQUEST_SYNC_REQUEST\n", fn);
    while (request_sync != REQUEST_SYNC_IN_PROGRESS) {
	pthread_cond_wait(&request_sync_cond, &request_sync_cond_mtx);
    }
    D(D_VERBOSE, "%s: REQUEST_SYNC_IN_PROGRESS\n", fn);
    MUTEX_UNLOCK(&request_sync_cond_mtx);

    client->fb = *pp_grab_fb_info[client->video_link].current_fb;
    
    ret = request_rect_direct(client, GRAB_REQ_FLAGS_NONE, 0, 0, client->fb.g_w_pd, client->fb.g_h);
    
    return ret;
}

int
release_screen_sync_direct(pp_grab_client_t UNUSED *client)
{
    static const char* fn = ___F;

    D(D_VERBOSE, "%s: REQUEST_SYNC_NONE\n", fn);
    request_sync = REQUEST_SYNC_NONE;
    pthread_cond_signal(&request_sync_cond);
    
    MUTEX_UNLOCK(&request_rect_sync_mtx);

    return 0;
}

void
sync_fb_direct(pp_grab_sync_lock_t action, u_char video_link UNUSED)
{
//TODO: Do we need separate mutex for each grabber?
    if (action == GRAB_SYNC_LOCK) {
	MUTEX_LOCK(&fb_sync_mtx);
    } else if (action == GRAB_SYNC_UNLOCK) {
	MUTEX_UNLOCK(&fb_sync_mtx);
    } else {
	// unknown pp_grab_sync_lock_t, must never happen
	assert(0);
    }    
}

static void
create_fake_picture(RegionRec *reg, u_char video_link)
{
    unsigned int i, j, k;
    unsigned short pixel = 0xFFFF & B_MSK;
    BoxRec tileBox;
    RegionRec tileReg;
    
    fb_format_info_t *current_fb = pp_grab_fb_info[video_link].current_fb;

    for (i=0; i < current_fb->tiles_w; i++) {
	for (j=0; j < current_fb->tiles_h / 2; j++) {
	    unsigned int tile_start = (j * current_fb->tiles_w + i) * PP_FB_TILE_SIZE;	    

	    for (k=0; k<PP_FB_TILE_SIZE; k++) {
		unsigned short *addr = (unsigned short*) &mem_pool[(tile_start + k) * VSC_PIXEL_SIZE];
		*addr = pixel;
	    }

	    tileBox.x1 = i * PP_FB_TILE_WIDTH;
	    tileBox.x2 = i * PP_FB_TILE_WIDTH + PP_FB_TILE_WIDTH;
	    tileBox.y1 = j * PP_FB_TILE_HEIGHT;
	    tileBox.y2 = j * PP_FB_TILE_HEIGHT + PP_FB_TILE_HEIGHT;
	    
	    if ((u_int) tileBox.x2 > current_fb->g_w) tileBox.x2 = current_fb->g_w;
	    if ((u_int) tileBox.y2 > current_fb->g_h) tileBox.y2 = current_fb->g_h;
	    
	    REGION_INIT(&tileReg, &tileBox, 0);
	    REGION_UNION(reg, reg, &tileReg);
	    REGION_UNINIT(&tileReg);
	}
    }
}
