#include <stdio.h>
#include <sys/types.h>
#include <jpeglib.h>

#include <pp/grab.h>
#include <pp/vsc.h>
#include <pp/rfb.h>
#include <liberic_session.h>

#include "grab_jpg.h"
#include "debug.h"
#include "common.h"

#define SCREENSHOT_TARGET_WIDTH		320

static pthread_mutex_t create_jpg_mtx		= PTHREAD_ERRORCHECK_MUTEX_INITIALIZER_NP;

static int translate_fb(pp_grab_client_t *client, struct jpeg_compress_struct *cinfo, u_int scale);
static int translate_fb_scaled(pp_grab_client_t *client, struct jpeg_compress_struct *cinfo, u_char scale);
static void fb_get_rows(fb_format_info_t *fb_f_info, u_char *fb, u_short *buf,
			int x, int y, int width, int height);
static int get_error_screenshot(pp_grab_request_error_t err);
static int make_jpg(pp_grab_client_t *client, u_char scale);

static unsigned short screenshot_width, screenshot_height;

/**
 * Triggers the creation of a jpeg from current frame buffer content
 */
int
pp_grab_create_snapshot(unsigned char video_link, const char *webs_user, u_short *width, u_short *height)
{
    pp_grab_client_t *client = NULL;
    u_char scale, scale_hw = 0;
    const char *excl_user;
    eric_session_int_id_t excl_sid;    
    pp_grab_encoder_desc_t* act;
    int ret = PP_GRAB_SCREENSHOT_ERROR;

    // prevent a second jpg creation process
    MUTEX_LOCK(&create_jpg_mtx);

    excl_sid = 0;
    /* check exclusive access (find any exclusive sid) */
    act = grab_encoder_hook;
    while (act && (excl_sid == 0)) {
        excl_sid = act->get_excl_session(video_link);
        act = act->next;
    }

    if (excl_sid != 0) {
	excl_user = eric_session_get_user(excl_sid);
	if (strncmp(webs_user, excl_user, ERIC_MAX_USER_LENGTH)) {
	    ret = SCREENSHOT_EXCL_ACCESS;
	    goto bail;
	}
    }

    D(D_VERBOSE, "%s(): creating screenshot\n", ___F);
    client = pp_grab_new_client(video_link, GRAB_FIX_MEM_DESC_SCREENSHOT);
       
    if (client == NULL) {
	D(D_ERROR, "%s(): no grab client available\n", ___F);
	goto bail;
    }

#ifdef PP_FEAT_VSC_HW_DOWNSCALING
    scale_hw = 1;
#endif
    
    if (grab_request_screen_sync(client, scale_hw, &scale) != 0) {
	D(D_VERBOSE,"%s(): request_screen_sync failed %d\n", ___F, client->req_err);
	ret = get_error_screenshot(client->req_err);
	goto bail;
    }

    if (make_jpg(client, scale) != 0) {
	ret = get_error_screenshot(client->req_err);
	goto bail;
    }
    
    ret = PP_SUC;
    
 bail:
    if (client) grab_release_screen_sync(client);
    if (width)  *width = screenshot_width;
    if (height) *height= screenshot_height;
    if (client) pp_grab_remove_client(client);
    
    D(D_VERBOSE, "%s(): screenshot width=%d, height=%d\n", ___F, screenshot_width, screenshot_height);
    MUTEX_UNLOCK(&create_jpg_mtx);    
    
    return ret;
}

static int
get_error_screenshot(pp_grab_request_error_t err)
{
    screenshot_width = 320;
    screenshot_height = 240;
    switch (err) {
      case GRAB_REQ_ERR_NO_SIGNAL:
	  return PP_GRAB_SCREENSHOT_NO_SIGNAL;
      case GRAB_REQ_ERR_NOT_AVAILABLE:
	  return PP_GRAB_SCREENSHOT_NOT_POSSIBLE;
      case GRAB_REQ_ERR_UNKNOWN_MODE:
	  return PP_GRAB_SCREENSHOT_UNKNOWN_MODE;
      default:
	  return PP_GRAB_SCREENSHOT_ERROR;
    }
}

static int
make_jpg(pp_grab_client_t *client, u_char scale)
{ 
    struct jpeg_compress_struct cinfo;
    struct jpeg_error_mgr jerr;
    unsigned int w, h, scale_self = (scale == 0) ? 1 : 0;
    FILE * outfile;
    int ret = -1;

    cinfo.err = jpeg_std_error(&jerr);
    
    jpeg_create_compress(&cinfo);

    if ((outfile = fopen("/lib/webpages/screenshot.jpg", "wb")) == NULL) {
        pp_log_err("%s(): fopen() failed", ___F);
	ret = -1;
	goto bail;
    }    
    
    jpeg_stdio_dest(&cinfo, outfile);
    
    if (scale_self) {
	scale = client->fb.g_w > 1024 ? 4 : 2;
    }
   
    w = (client->fb.g_w / scale)+0.5;
    h = (client->fb.g_h / scale)+0.5;    
    
    cinfo.image_width = w;
    cinfo.image_height = h;
    cinfo.input_components = 3;		       
    cinfo.in_color_space = JCS_RGB; 	       
    
    jpeg_set_defaults(&cinfo);

    jpeg_set_quality(&cinfo, 90, TRUE);
    
    jpeg_start_compress(&cinfo, TRUE);

    if (scale_self) {
	ret = translate_fb(client, &cinfo, scale);
    } else {
	ret = translate_fb_scaled(client, &cinfo, scale);
    }

    if (ret != 0) {
	D(D_ERROR, "%s(): Error during framebuffer translation\n", ___F);
	goto bail;
    }
    
    jpeg_finish_compress(&cinfo);
    
    fclose(outfile);
    
    jpeg_destroy_compress(&cinfo);

    // update screenshot size info
    screenshot_width = SCREENSHOT_TARGET_WIDTH;
    screenshot_height = h * screenshot_width / w;

    ret = 0;
    
 bail:
    return ret;
}

static int
translate_fb(pp_grab_client_t *client, struct jpeg_compress_struct *cinfo,
	     unsigned int scale)
{
    u_short temp_buf[client->fb.g_w * PP_FB_TILE_HEIGHT];
    JSAMPLE data[cinfo->image_width * 3];
    JSAMPROW row_pointer[1];
    unsigned int i, y_ofs=0, pcnt = scale*scale;
#if defined(PP_FEAT_VSC_HW_ENCODING) && !defined(PP_FEAT_VSC_HW_DOWNSCALING)
    fb_format_info_t old_fb_format;
#endif
    
    while (cinfo->next_scanline < cinfo->image_height) {	
	int y = cinfo->next_scanline * scale;

	if (y % PP_FB_TILE_HEIGHT == 0) {
#if defined(PP_FEAT_VSC_HW_ENCODING) && !defined(PP_FEAT_VSC_HW_DOWNSCALING)
	    int req_ret = pp_grab_request_rect(client, GRAB_REQ_FLAGS_LOCK_DIFF | GRAB_REQ_FLAGS_IGNORE_DIFFMAP,
					       0, y, client->fb.g_w, PP_FB_TILE_HEIGHT);
	    if (req_ret != 0) {
		D(D_ERROR, "%s(): error while translating fb\n", ___F);
		return -1;
	    }    
	    /* compare fb format with last one, if it changed in-between bail out */
	    if (y != 0 &&
		client->fb.g_w != old_fb_format.g_w &&
		client->fb.g_h != old_fb_format.g_h) {
		
		D(D_ERROR, "%s(): framebuffer format changed\n", ___F);
		return -1;
	    }
	    old_fb_format = client->fb;
	    fb_get_rows(&client->fb, client->buf, temp_buf, 0, 0, client->fb.g_w, PP_FB_TILE_HEIGHT);
#else
	    fb_get_rows(&client->fb, client->buf, temp_buf, 0, y, client->fb.g_w, PP_FB_TILE_HEIGHT);
#endif
	    y_ofs = 0;
	}
	
	for (i = 0; i < cinfo->image_width; i++) {
	    int x = i * scale;
	    u_char ix, iy;
	    u_int r=0,g=0,b=0;

	    for (iy=0; iy<scale;iy++) {
		for (ix=0; ix<scale;ix++) {
		    u_short pxl = temp_buf[(y_ofs + iy) * client->fb.g_w + x + ix];
		    r += (pxl & R_MSK) >> 8;
		    g += (pxl & G_MSK) >> 3;
		    b += (pxl & B_MSK) << 3;
		}
	    }

	    data[i*3+0] = r / pcnt;                
	    data[i*3+1] = g / pcnt;
	    data[i*3+2] = b / pcnt;
	}
	y_ofs +=scale;

	row_pointer[0] = data;
	(void) jpeg_write_scanlines(cinfo, row_pointer, 1);
    }

    return 0;
}

static void
fb_get_rows(fb_format_info_t *fb_f_info, u_char* fb, u_short *buf, int x, int y,
	    int width, int height)
{
    u_short *ip = (u_short*)(fb + (y * PP_FB_TILE_WIDTH * fb_f_info->tiles_w + x * PP_FB_TILE_HEIGHT) * 2);
	
    int eol_skip  = (width % PP_FB_TILE_WIDTH) ? width - (width % PP_FB_TILE_WIDTH) : 0;
    int line_skip = -(width * PP_FB_TILE_HEIGHT - PP_FB_TILE_WIDTH);
    int row_skip  = (fb_f_info->g_w_pd - PP_FB_TILE_WIDTH) * PP_FB_TILE_HEIGHT;
    int w_single, line=0;

    while (line++ < height) {
	w_single = width;
	while (w_single-- > 0) {
	    *(buf++) = *(ip++);
	    if (w_single % PP_FB_TILE_WIDTH == 0) ip += (PP_FB_TILE_WIDTH * PP_FB_TILE_HEIGHT - PP_FB_TILE_WIDTH);
	}
	ip += eol_skip;   
	ip += line_skip;
	if (line % PP_FB_TILE_HEIGHT == 0) ip += row_skip;
    }
}

static int
translate_fb_scaled(pp_grab_client_t *client,
		    struct jpeg_compress_struct *cinfo, u_char scale)
{
    JSAMPLE data[cinfo->image_width * 3];
    JSAMPROW row_pointer[1];
    u_char tw = PP_FB_TILE_WIDTH / scale;
    u_char th = PP_FB_TILE_HEIGHT / scale;
    u_short *ip = (u_short*) client->buf;
    int line_skip = -(cinfo->image_width * th - tw);
    int row_skip  = (client->rect_pitch - tw) * th;
    int w_single;
    u_char* buf = data;

    D(D_VERBOSE, "%s(): w,h=%d,%d, scale=%d, pitch=%d\n", ___F,
      cinfo->image_width, cinfo->image_height, scale, client->rect_pitch);

    while (cinfo->next_scanline < cinfo->image_height) {
	w_single = cinfo->image_width;
	while (w_single-- > 0) {
	    *buf++ = (*ip & R_MSK) >> 8;
	    *buf++ = (*ip & G_MSK) >> 3;
	    *buf++ = (*ip & B_MSK) << 3;

	    ip++;
	    if (w_single % tw == 0) ip += (tw * th - tw);
	}

	ip += line_skip;

	row_pointer[0] = data;
	(void) jpeg_write_scanlines(cinfo, row_pointer, 1);

	if (cinfo->next_scanline % th == 0) ip += row_skip;
		
	buf = data;
    }

    return 0;
}
