#include <errno.h>
#include <assert.h>
#include <fcntl.h>
#include <signal.h>
#include <stdio.h>
#include <unistd.h>
#include <pthread.h>
#include <sys/ioctl.h>
#include <sys/stat.h>

#include <pp/base.h>
#include <pp/error.h>
#include <pp/hal_common.h>
#include <liberic_config.h>
#include <liberic_misc.h>
#include <pp/rfb.h>
#include <liberic_pthread.h>
#include <pp/vsc.h>
#if defined (PP_FEAT_SIMPLERFB_SUPPORT)
# include <pp/simplerfb.h>
#endif

#include "grab_jpg.h"
#include "direct.h"
#include "hwenc.h"
#include "debug.h"

#define MAX_HEXTILE_DATA_SIZE		(512 + sizeof(vsc_update_rect_hdr_t))

#define FIX_MEM_DESC_MOUSESYNC_SIZE	(128 * 1024)
#define FIX_MEM_DESC_SCREENSHOT_SIZE	(384 * 1024)
#define FIX_MEM_DESC_OFFSET_SIZE	(512 * (VSC_MAX_RES_X / PP_FB_TILE_WIDTH))
#if defined (PP_FEAT_SIMPLERFB_SUPPORT)
# define FIX_MEM_DESC_SIMPLERFB_SIZE	(SIMPLERFB_WINDOW_SIZE * MAX_HEXTILE_DATA_SIZE)
#endif


typedef struct{
    u_char		    video_link;			    /* video link, i.e. frame grabber i.e. channel */
    pthread_t		    thread;			    /* grabber thread */
    char		    thread_run;			    /* grabber thread is running */
    fb_change_t		    fb_change;			    /* state to handle fb change */
    fb_suspend_t	    fb_suspend;			    /* state of grab suspending */
    u_int32_t		    diffmap[VSC_DIFFMAP_SIZE/4];    /* current vsc difference bitmap */
    RegionRec		    diff_reg;			    /* region representation of the diffmap */
    unsigned char	    fb_cleared;			    /* in case of unknown fb format, use dummy */
    unsigned char	    fb_need_reconfigure;	    /* force new fb format notify (during startup) */
    volatile unsigned char  skip_diff;			    /* mark whole fb as changed (refresh) */

    fb_format_info_t	    fb1;
    fb_format_info_t	    fb2;
    fb_format_info_t*	    fb_new;
    fb_format_info_t*	    fb_old;

    int			    grab_vsc_fd;		    /* frame grabber file descriptor */

    pthread_mutex_t	    info_mtx;
    pthread_cond_t	    grab_suspend_cond;
    pthread_mutex_t	    grab_suspend_cond_mtx;
    pthread_mutex_t	    grab_suspend_mtx;
    pthread_cond_t	    fb_change_cond;
    pthread_mutex_t	    fb_change_cond_mtx;
}grab_t;

grab_t *p_all_grabs = NULL;

/* fb color information is static for all VSC
   devices currently */
static fb_color_info_t fb_color_info = {
    bpp:	VSC_PIXEL_BITS,
    depth:	VSC_PIXEL_BITS,
    red_bits:	5,
    green_bits:	6,
    blue_bits:	5
};

vsc_encoding_desc_t pp_grab_null_encoding_desc = {
    algo:			VSC_ALGO_NULL,
    lrle_r_margin_rb:		0,
    lrle_r_margin_g:		0,
    lrle_grey_green:		0,
    lrle_g_margin:		0,
    lrle_runlimit:		0,
    lrle_linecopy:		0,
    lrle_grey_force:		0,
    lrle_grey_disable:		0,
    lrle_c_margin_rb:		0,
    lrle_c_margin_g:		0,
    lrle_color:			0,
    lrle_runlimit_reacc:	0,
    down_mode:			0
};

static int initialized  = 0;

#ifdef DEBUG_PROFILE
static unsigned int profile_print_count = 0;
#endif

static const char * pp_grab_errors[] = {
    /* GRAB_REQ_ERR_NO_ERROR */			"No error.",
    /* GRAB_REQ_ERR_NO_SIGNAL */		"No signal.",
    /* GRAB_REQ_ERR_NOT_AVAILABLE */		"Grabber not available (suspended, e.g. autp adjust in progress).",
    /* GRAB_REQ_ERR_MODE_CHANGE */		"Framebuffer format change occured.",
    /* GRAB_REQ_ERR_UNKNOWN_MODE */		"Unknown video mode.",
    /* GRAB_REQ_ERR_INTERNAL_ERROR */		"Internal grabber error.",
    /* GRAB_REQ_ERR_DEBUGGING_ERROR */		"Debugging error, should not occur.",
    /* GRAB_REQ_ERR_REALLOC_IN_PROGRESS */	"Reallocation from memory pool in progress.",
};

static int grab_errno_base = 0; // triggers assertion failed if not registered

static void set_fb_format_to_vsc(fb_format_info_t *fb_f_info, grab_t *p_grab);
static void dump_and_diff_vga_graphics(grab_t *);
static void grabber_signal_available(grab_t *);

static void* grab_thread_func(void * arg);
static int handle_fb_change(grab_t *);
static const char * get_error_string(const int error);
static int init_grabbers_structs(void);
static void cleanup_grabbers_structs(void);

#ifdef PP_FEAT_VSC_HW_ENCODING
static grab_driver_t *driver = &grab_driver_hwenc;
#else
static grab_driver_t *driver = &grab_driver_direct;
#endif

#if defined(PP_FEAT_CAT) || defined(PP_FEAT_KITTY_CAT)
static pp_grab_diffmap_consumer_t *dmc = NULL;
#endif /* PP_FEAT_CAT || PP_FEAT_KITTY_CAT */

static
int
init_grabbers_structs()
{
    static const char * fn = ___F;
    char devname[65];
    int i;

    grab_t *p_grab = NULL;

    assert( p_all_grabs == NULL );
    p_all_grabs = (grab_t *)malloc( sizeof(grab_t)*current_num_video_links);

    assert( pp_grab_fb_info == NULL );
    pp_grab_fb_info = (pp_grab_fb_info_t *)malloc( sizeof(pp_grab_fb_info_t)*current_num_video_links);

    memset( pp_grab_fb_info, 0, sizeof(pp_grab_fb_info_t)*current_num_video_links);
    memset( p_all_grabs, 0, sizeof(grab_t)*current_num_video_links);

    pthread_mutexattr_t mattr;
    pthread_mutexattr_init(&mattr);
    pthread_mutexattr_settype(&mattr, PTHREAD_MUTEX_ERRORCHECK_NP);

    for( i=0; i<current_num_video_links; i++ ){
	p_grab = &p_all_grabs[i];

	//initialize mutexes and condition vars
	pthread_mutex_init( &p_grab->info_mtx, &mattr);
	pthread_mutex_init( &p_grab->grab_suspend_cond_mtx, &mattr);
	pthread_mutex_init( &p_grab->grab_suspend_mtx, &mattr);
	pthread_mutex_init( &p_grab->fb_change_cond_mtx, &mattr);

	pthread_cond_init( &p_grab->grab_suspend_cond, NULL );
	pthread_cond_init( &p_grab->fb_change_cond, NULL );

	memset(&p_grab->fb1, 0, sizeof(fb_format_info_t));
	memset(&p_grab->fb2, 0, sizeof(fb_format_info_t));

	p_grab->video_link = i;
	p_grab->fb_cleared = 0;
	p_grab->fb_need_reconfigure = 1;
	p_grab->fb_change = CHANGE_NONE;
	p_grab->fb_suspend = SUSPEND_RESUMED;
	pp_grab_fb_info[i].suspended = 1;

	snprintf(devname, 64, "%s%d", DEV_VSC_PRFX, i);
	if ((p_grab->grab_vsc_fd = open(devname, O_RDWR)) == -1) {
	    D(D_ERROR, "%s: Opening %s failed, video_link=%d, errno=%s\n", fn, devname, i, pp_error_string(errno));
	    return -1;
	}    
	REGION_INIT(&p_grab->diff_reg, NullBox, VSC_MAX_RES_X / PP_FB_TILE_WIDTH * VSC_MAX_RES_Y / PP_FB_TILE_HEIGHT);
    }
    pthread_mutexattr_destroy(&mattr);
    return 0;
}

static
void
cleanup_grabbers_structs()
{
    int i;
    grab_t *p_grab= NULL;

    assert( p_all_grabs != NULL );
    assert( pp_grab_fb_info != NULL );

    for( i=0; i<current_num_video_links; i++ ){

	p_grab= &p_all_grabs[i];

        if (p_grab->grab_vsc_fd != -1) {
            close(p_grab->grab_vsc_fd);
	    p_grab->grab_vsc_fd = -1;
        }


	pthread_mutex_destroy( &p_grab->info_mtx );
	pthread_mutex_destroy( &p_grab->grab_suspend_cond_mtx );
	pthread_mutex_destroy( &p_grab->grab_suspend_mtx );
	pthread_mutex_destroy( &p_grab->fb_change_cond_mtx );

	pthread_cond_destroy( &p_grab->grab_suspend_cond );
	pthread_cond_destroy( &p_grab->fb_change_cond );
    }

    free( pp_grab_fb_info );
    free( p_all_grabs );
}



int
pp_grab_init(void)
{
    static const char * fn = ___F;
    int i;

    if (initialized) {
	return 0;
    }

    current_num_video_links = pp_hal_common_get_vsc_cnt();
    assert( current_num_video_links > 0 );

    assert(grab_errno_base == 0);

    grab_errno_base = pp_register_errnos(sizeof(pp_grab_errors) /
					 sizeof(*pp_grab_errors),
					 get_error_string);

    if( init_grabbers_structs() == -1 ){
	D(D_ERROR, "Failed to initialize pp_grab library\n");
	return -1;
    }

    INIT_LIST_HEAD(&clients_list);

    pthread_mutexattr_init(&errcheck_mtx_attr);
    pthread_mutexattr_settype(&errcheck_mtx_attr, PTHREAD_MUTEX_ERRORCHECK);

    if (mempool_init() != 0) {	
	D(D_ERROR, "%s: mempool init failed\n", fn);
	return -1;
    }

    for( i=0; i<current_num_video_links; i++ ){
#ifdef PP_FEAT_VSC_HW_ENCODING
	if (mempool_reserve_fix_desc(GRAB_FIX_MEM_DESC_MOUSESYNC, FIX_MEM_DESC_MOUSESYNC_SIZE, i) != 0
	    || mempool_reserve_fix_desc(GRAB_FIX_MEM_DESC_SCREENSHOT, FIX_MEM_DESC_SCREENSHOT_SIZE, i) != 0
	    || mempool_reserve_fix_desc(GRAB_FIX_MEM_DESC_OFFSET, FIX_MEM_DESC_OFFSET_SIZE, i) != 0
#if defined (PP_FEAT_SIMPLERFB_SUPPORT)
	    || mempool_reserve_fix_desc(GRAB_FIX_MEM_DESC_SIMPLERFB, FIX_MEM_DESC_SIMPLERFB_SIZE, i) != 0
#endif
#if defined(DEBUG_DUMP_REGION) || defined(DEBUG_CHECK_REGION)
	    || mempool_reserve_fix_desc(GRAB_FIX_MEM_DESC_DEBUG1, FIX_MEM_DESC_DEBUG1_SIZE, i) != 0
	    || mempool_reserve_fix_desc(GRAB_FIX_MEM_DESC_DEBUG2, FIX_MEM_DESC_DEBUG2_SIZE, i) != 0
#endif
	    ) {
	    D(D_ERROR, "%s: reserving fixed blocks from mempool failed, video_link=%d\n", fn, i);
	    return -1;
	}
#endif
    }
    
#ifdef DEBUG_PROFILE
    profiler_init();
    profiler_add(PROFILE_FETCH_DIFFMAP, "FETCH_DIFFMAP");
    profiler_add(PROFILE_FETCH_TILES, "FETCH_REGION");
    profiler_add(PROFILE_SPLIT_REGION, "SPLIT_REGION");
    profiler_add(PROFILE_CALC_DIFFMAP, "CALC_DIFFMAP");    
    profiler_add(PROFILE_CALC_DIFFMAP_REGIONS, "CALC_DM_REGIONS");
    profiler_add(PROFILE_TRANSFER_ALL, "TRANSFER_ALL");
    profiler_add(PROFILE_FETCH_ADD_MOD, "FETCH_ADD_MOD");
    profiler_add(PROFILE_MISC, "MISC");
#endif

    if (!pp_hal_common_is_flash_id_matching()) {
	fake_picture_error = 1;
    }

    initialized = 1;
    return 0;
}

void
pp_grab_cleanup(void)
{
    int i;
    
    for( i=0; i<current_num_video_links; i++ ){
	p_all_grabs[i].thread_run = 0;
	pthread_join(p_all_grabs[i].thread, NULL);
    }

    pthread_mutexattr_destroy(&errcheck_mtx_attr);

    for( i=0; i<current_num_video_links; i++ ){
	REGION_UNINIT(&p_all_grabs[i].diff_reg );
    }    
    
    mempool_cleanup();

    if (grab_errno_base > 0) {
	pp_unregister_errnos(grab_errno_base);
	grab_errno_base = 0;
    }
    
    cleanup_grabbers_structs();

    initialized = 0;
}

int pp_grab_connect_encoder(pp_grab_encoder_desc_t* encoder) {
    pp_grab_encoder_desc_t* act;
    assert(encoder != NULL);
    
    encoder->next = NULL;
    
    // register
    if (grab_encoder_hook == NULL) {
        grab_encoder_hook = encoder;
    } else {
        act = grab_encoder_hook;
        while (act->next != NULL) {
            act = act->next;
        }
        act->next = encoder;
    }
    
    return PP_SUC;
}


void
pp_grab_start(void)
{

    int i;

    grab_t *p_grab = NULL;
    for( i=0; i<current_num_video_links; i++ ){
	p_grab = &p_all_grabs[i];
	/* initialize rfb server framebuffer format */
	p_grab->fb_new = &p_grab->fb1;
	p_grab->fb_old = &p_grab->fb2;

	pp_grab_fb_info[i].current_fb = p_grab->fb_old;
    
	pp_grab_encoder_desc_t* act = grab_encoder_hook;
	while (act != NULL) {
	    act->set_color_translation(0, &fb_color_info);
	    act = act->next;
	}
    }
    mempool_start_var_alloc();
    
    for( i=0; i<current_num_video_links; i++ ){
	p_grab= &p_all_grabs[i];

	MUTEX_LOCK(&p_grab->grab_suspend_mtx);

	p_grab->thread_run = 1;
	if (eric_pthread_create(&p_grab->thread, 0, 64 * 1024,
				grab_thread_func, p_grab)) {
	    pp_log_err("Cannot create grabber thread.\n");
	    MUTEX_UNLOCK(&p_grab->grab_suspend_mtx);
	    return;
	}

	MUTEX_UNLOCK(&p_grab->grab_suspend_mtx);

	pp_grab_suspend(i, GRAB_RESUME);
    }
}

inline int
pp_grab_errno_base()
{
    assert(grab_errno_base > 0);
    return grab_errno_base;
}

static const char *
get_error_string(const int error)
{
    return pp_grab_errors[error - grab_errno_base];
}

static void*
grab_thread_func(void * arg)
{
    video_signal_state_t vsc_signal;
    vsc_output_state_t  vsc_output;
    u_char video_link;
    pp_grab_fb_info_t *p_grab_fb;
    grab_t *p_grab = (grab_t *)arg;
    assert( p_grab != NULL );

    video_link= p_grab->video_link;

    p_grab_fb = &pp_grab_fb_info[video_link];

    DCH(D_NOTICE, "%s: PID %d\n", ___F, getpid());
    
    while (p_grab->thread_run) {

	/* handle fb changes */
	MUTEX_LOCK(&p_grab->fb_change_cond_mtx);
	if (p_grab->fb_change == CHANGE_ANNOUNCED) {
	    p_grab->fb_change = CHANGE_WAITING;
	    pthread_cond_signal(&p_grab->fb_change_cond);
	    while (p_grab->fb_change != CHANGE_NONE) {
		pthread_cond_wait(&p_grab->fb_change_cond, &p_grab->fb_change_cond_mtx);
	    }
	    p_grab->skip_diff++;
	}
	MUTEX_UNLOCK(&p_grab->fb_change_cond_mtx);

	/* handle grab suspending/resuming */
	MUTEX_LOCK(&p_grab->grab_suspend_cond_mtx);
	switch (p_grab->fb_suspend) {
	  case SUSPEND_SUSPEND:
	      if (!p_grab_fb->suspended) {
		  p_grab->skip_diff++;
	      }
	      p_grab_fb->suspended++;
	      p_grab->fb_suspend = SUSPEND_SUSPENDED;
	      pthread_cond_signal(&p_grab->grab_suspend_cond);
	      break;
	  case SUSPEND_RESUME:
	      if (p_grab_fb->suspended) p_grab_fb->suspended--;
	      p_grab->fb_suspend = SUSPEND_RESUMED;
	      pthread_cond_signal(&p_grab->grab_suspend_cond);
	  default:
	      break;
	}
	MUTEX_UNLOCK(&p_grab->grab_suspend_cond_mtx);

	/* check for new framebuffer format */
	if (handle_fb_change( p_grab) != 0) {
	    if( p_grab->fb_new == &p_grab->fb1 ){
		p_grab->fb_new = &p_grab->fb2;
		p_grab->fb_old = &p_grab->fb1;
	    }
	    else {
		p_grab->fb_new = &p_grab->fb1;
		p_grab->fb_old = &p_grab->fb2;
	    }
	    if (driver->cancel_requests) driver->cancel_requests(VSC_FETCH_ERROR_NO_SIGNAL, video_link);
	    usleep(100000);
	    continue;
	}

#ifdef DEBUG_PROFILE
	if (++profile_print_count == PROFILE_PRINT_COUNT) {
	    profiler_print_all(D_ALWAYS);
	    profile_print_count = 0;
	}
#endif
	
	vsc_signal = pp_vsc_has_signal(video_link);
	vsc_output = pp_vsc_output_state(video_link);

        if (p_grab_fb->suspended || p_grab_fb->nr_clients == 0) {

	    if (p_grab_fb->suspended) {
		if (driver->cancel_requests) driver->cancel_requests(VSC_FETCH_ERROR_NOT_AVAILABLE, video_link);
	    }
	    
	    /*
	     * If we have no video signal then we should skip the diff when
	     * the grabber starts running again.
	     */
	    if (pp_vsc_has_signal(video_link) != VIDEO_SIGNAL_ON) {
		p_grab->skip_diff++;
	    }
	    
	    usleep(200000);
	    if( p_grab->fb_new == &p_grab->fb1 ){
		p_grab->fb_new = &p_grab->fb2;
		p_grab->fb_old = &p_grab->fb1;
	    }
	    else{
		p_grab->fb_new = &p_grab->fb1;
		p_grab->fb_old = &p_grab->fb2;
	    }
	    continue;
	}

	if (vsc_signal == VIDEO_SIGNAL_ON && vsc_output == OUTPUT_VALID) {
	    if (p_grab_fb->grabber_available == 0) {
		grabber_signal_available( p_grab);
	    }
	    dump_and_diff_vga_graphics(p_grab);
	} else {
	    if (vsc_signal != VIDEO_SIGNAL_ON) {
		if (driver->cancel_requests) driver->cancel_requests(VSC_FETCH_ERROR_NO_SIGNAL, video_link);
	    } else if (vsc_output != OUTPUT_VALID) {
		if (driver->cancel_requests) driver->cancel_requests(VSC_FETCH_ERROR_UNKNOWN_MODE, video_link);
	    }
	    usleep(200000);
	    continue;
	}

	if( p_grab->fb_new == &p_grab->fb1 ){
	    p_grab->fb_new = &p_grab->fb2;
	    p_grab->fb_old = &p_grab->fb1;
	}
	else{
	    p_grab->fb_new = &p_grab->fb1;
	    p_grab->fb_old = &p_grab->fb2;
	}

    }

    return NULL;
}

static int
handle_fb_change( grab_t *p_grab )
{

    int r;
    pp_grab_encoder_desc_t* act;
    u_char video_link = p_grab->video_link;

    pp_grab_fb_info_t *p_grab_fb = &pp_grab_fb_info[video_link];

    fb_format_info_t* fb_new = p_grab->fb_new;
    fb_format_info_t* fb_old = p_grab->fb_old;

    MUTEX_LOCK(&p_grab->info_mtx);
    r = pp_vsc_get_fb_format(video_link, fb_new);
    MUTEX_UNLOCK(&p_grab->info_mtx);

    if (r == -1 || fb_new->g_w < 320 || fb_new->g_w > 1600 ||
	fb_new->g_h < 200 || fb_new->g_h > 1200) {
	if (!p_grab->fb_cleared) {
	    BoxRec box;

	    MUTEX_LOCK(&p_grab->info_mtx);
	    fb_old->is_unsupported = 0;
	    fb_old->g_w = 640;
	    fb_old->g_h = 400;
	    fb_old->g_wb = 640;
	    fb_old->g_w_pd = 640;
	    fb_old->g_h_pd = 400;
	    fb_old->g_wb_pd = 640;
	    fb_old->tiles_w = 640 / PP_FB_TILE_WIDTH;
	    fb_old->tiles_h =400 / PP_FB_TILE_HEIGHT;
	    box.x1 = 0;
	    box.y1 = 0;
	    box.x2 = fb_old->g_w;
	    box.y2 = fb_old->g_h;
	    MUTEX_UNLOCK(&p_grab->info_mtx);
	    REGION_RESET(&p_grab->diff_reg, &box);
	    set_fb_format_to_vsc(fb_old, p_grab);
            act = grab_encoder_hook;
            while (act) {
                act->set_framebuffer_format(video_link, fb_new);
                if (act->new_data_available) {
                    act->new_data_available(video_link);
                }
                act = act->next;
            }
            
	    p_grab->fb_cleared = 1;
	    p_grab->skip_diff++;
	}
	return -1;
    }
	
    p_grab->fb_cleared = 0;
	
    MUTEX_LOCK(&p_grab->info_mtx);
    p_grab_fb->current_fb = fb_new;
    MUTEX_UNLOCK(&p_grab->info_mtx);

    if (p_grab->fb_need_reconfigure ||
	fb_new->is_unsupported != fb_old->is_unsupported ||
	fb_new->g_w != fb_old->g_w ||
	fb_new->g_h != fb_old->g_h) {

	DCH(D_NOTICE, "new fb format (%d,%d)\n", fb_new->g_w, fb_new->g_h);
	
	set_fb_format_to_vsc(fb_new, p_grab);

        act = grab_encoder_hook;
        while (act) {
	    act->set_framebuffer_format(video_link, fb_new);
            act = act->next;
        }
	if (driver->cancel_requests) driver->cancel_requests(VSC_FETCH_ERROR_MODE_CHANGE, video_link);
	
	p_grab->fb_need_reconfigure = 0;
	p_grab->skip_diff++;
    }

    return 0;
}

static void
dump_and_diff_vga_graphics( grab_t *p_grab )
{
    BoxRec tileBox;
    RegionRec tileReg;
    unsigned int i, j, in_new_map;
    u_int32_t x, *dmap;
    Bool overlap;
    u_char refresh_pending = 0;
    pp_grab_encoder_desc_t* act;
    fb_format_info_t* curr_fb = NULL;

    u_char video_link = p_grab->video_link;

    pp_grab_fb_info_t *p_grab_fb = &pp_grab_fb_info[video_link];

    if (p_grab_fb->suspended) goto bail;

    REGION_INIT(&tileReg, &tileBox, 0);
    REGION_EMPTY(&p_grab->diff_reg);
    
    /* if user wants a refresh, add the whole screen to mod region */
    if (p_grab->skip_diff) {
	if (ioctl(p_grab->grab_vsc_fd, PPIOCVSCSAMPLE) == -1) {
	    if (errno != EAGAIN) {
		DCH(D_ERROR, "%s(): ioctl(PPIOCVSCSAMPPLE)\n", ___F);
	    }
	    goto bail_uninit;
	}

	tileBox.x1 = 0; tileBox.y1 = 0;
	curr_fb = pp_grab_fb_info[video_link].current_fb;
	tileBox.x2 = curr_fb->g_w_pd; tileBox.y2 = curr_fb->g_h_pd;
	REGION_RESET(&tileReg, &tileBox);
	REGION_APPEND(&p_grab->diff_reg, &tileReg);

	p_grab->skip_diff = 0;
	refresh_pending = 1;
    } else {
	if (driver->fb_sync_lock == FB_SYNC_UNLOCKED ||
	    driver->fb_sync_lock == FB_SYNC_UNLOCKING) {

	    if (driver->debug_use_diffmap != 0) {
	        profiler_start(PROFILE_FETCH_DIFFMAP);
	        if (ioctl(p_grab->grab_vsc_fd, PPIOCVSCSAMPLEANDDIFF, &p_grab->diffmap) == -1) {
		    if (errno != EAGAIN) {
		        D(D_ERROR, "%s(): ioctl(PPIOCVSCSAMPLEANDDIFF)\n", ___F);
		    }
		    goto bail_uninit;
	        }
	        profiler_done(PROFILE_FETCH_DIFFMAP);
	    }

	    if (driver->debug_use_diffmap == 0) {
		memset(p_grab->diffmap, 0, VSC_DIFFMAP_SIZE);
	    } else if (driver->debug_use_diffmap == 2) {
		memset(p_grab->diffmap, 0xFF, VSC_DIFFMAP_SIZE);
	    }
	    
            act = grab_encoder_hook;
            while (act) {
                if (act->new_diffmap_available) {
                    act->new_diffmap_available(video_link, p_grab->diffmap);
                }
                act = act->next;
            }

#if defined(PP_FEAT_CAT) || defined(PP_FEAT_KITTY_CAT)
            if(dmc) {
                dmc->dmc_cb(dmc->target, p_grab->diffmap);
            }
#endif /* PP_FEAT_CAT || PP_FEAT_KITTY_CAT */
            
	    if (driver->fb_sync_lock == FB_SYNC_UNLOCKING) {
		driver->fb_sync_lock = FB_SYNC_UNLOCKED;
	    }
	}
	
#ifdef DEBUG_DIFFMAP
	{
	    char input[16];
	    
	    printf("\x1b[H\n");
	    print_buffer((u_char*)p_grab->diffmap, 24*4*4, 16);
	    //fgets((char*)&input, 16, stdin);
	}
#endif
	profiler_start(PROFILE_CALC_DIFFMAP);
	for (j = 0; j < p_grab->fb_new->tiles_h; j++) {
	    x = 0x00000001;
	    dmap = &p_grab->diffmap[j*PP_FB_TILES_PER_LINE/32];

	    i = 0;
	    while (i < p_grab->fb_new->tiles_w) {	
		in_new_map = (*dmap & x);
		
		if (in_new_map) {
		    tileBox.x1 = i * PP_FB_TILE_WIDTH;
		    tileBox.x2 = i * PP_FB_TILE_WIDTH + PP_FB_TILE_WIDTH;
		    tileBox.y1 = j * PP_FB_TILE_HEIGHT;
		    tileBox.y2 = j * PP_FB_TILE_HEIGHT + PP_FB_TILE_HEIGHT;

		    profiler_start(PROFILE_CALC_DIFFMAP_REGIONS);
		    REGION_RESET(&tileReg, &tileBox);
		    REGION_APPEND(&p_grab->diff_reg, &tileReg);
		    profiler_done(PROFILE_CALC_DIFFMAP_REGIONS);
		}
		
		x <<= 1;
		i++;
		if ((i % 32) == 0) {
		    x = 0x00000001;
		    dmap++;
		    if (*dmap == 0) {
			dmap++;
			i += 32;
		    }
		}
	    }
	}
	REGION_VALIDATE(&p_grab->diff_reg, &overlap);
	profiler_done(PROFILE_CALC_DIFFMAP);
    }
    
    profiler_start(PROFILE_TRANSFER_ALL);
    if (driver->transfer_regions) {
	driver->transfer_regions(&p_grab->diff_reg, refresh_pending, video_link);
    }
    profiler_done(PROFILE_TRANSFER_ALL);
       
 bail_uninit:
    REGION_UNINIT(&tileReg);
 bail:
    return;
}

static void
grabber_signal_available( grab_t *p_grab )
{
    u_char video_link;
    pp_grab_encoder_desc_t* act;

    video_link = p_grab->video_link;
    DCH(D_VERBOSE, "Grabber available again, notifying\n");

    pp_grab_fb_info[video_link].grabber_available = 1;
    act = grab_encoder_hook;
    while (act) {
        act->grabber_available( video_link );
        act = act->next;
    }
}


static void
set_fb_format_to_vsc(fb_format_info_t *fb_f_info, grab_t *p_grab)
{
    assert(fb_f_info);
    
    if (ioctl(p_grab->grab_vsc_fd, PPIOCSETFBFORMATINFO, fb_f_info) == -1) {

	if (errno != EAGAIN) {
	    D(D_ERROR, "%s(): ioctl(PPIOCSETFBFORMATINFO), video_link=%d\n", ___F, p_grab->video_link);
	}		
    }

    pp_propchange_enqueue(PP_PROP_VIDEO_MODE_CHANGED, 0);
}

/*********************************************************************
 * functions called from the "outside" to trigger different things
 * in the grabber
 *********************************************************************/

/*
  used by vsc thread to announce pending
  mode change, we have to stop grabbing during that
  (only works with 1 vsc talking to 1 grabber thread)
*/

void
pp_grab_fbchange_start( u_char video_link )
{
    grab_t *p_grab = &p_all_grabs[video_link];

    DCH(D_BLABLA, "FB change REQ\n");
    MUTEX_LOCK(&p_all_grabs[video_link].fb_change_cond_mtx);
    p_grab->fb_change = CHANGE_ANNOUNCED;
    while (p_all_grabs[video_link].fb_change != CHANGE_WAITING) {
	pthread_cond_wait(&p_all_grabs[video_link].fb_change_cond, &p_all_grabs[video_link].fb_change_cond_mtx);
    }
    MUTEX_UNLOCK(&p_all_grabs[video_link].fb_change_cond_mtx);
    DCH(D_BLABLA, "FB change ACK\n");
}



int 
pp_grab_fd( unsigned char video_link )
{
    assert( p_all_grabs != NULL );
    return p_all_grabs[video_link].grab_vsc_fd;
}



void
pp_grab_fbchange_ready( u_char video_link )
{

    DCH(D_BLABLA, "FB change ready REQ\n");
    MUTEX_LOCK(&p_all_grabs[video_link].fb_change_cond_mtx);
    p_all_grabs[video_link].fb_change = CHANGE_NONE;
    pthread_cond_signal(&p_all_grabs[video_link].fb_change_cond);
    MUTEX_UNLOCK(&p_all_grabs[video_link].fb_change_cond_mtx);
    DCH(D_BLABLA, "FB change ready ACK\n");
}




/*
  used to stop the grabber for the following things:
  -VSC:	 claim grabber during auto phase adjustment
	 and while resetting the VSC
*/

void
pp_grab_suspend(unsigned char video_link, pp_grab_suspend_action_t action)
{
    grab_t *p_grab = &p_all_grabs[video_link];

    /* only one thread should mess around with suspend at a time */
    MUTEX_LOCK(&p_grab->grab_suspend_mtx);
    if (!p_grab->thread_run) {
	switch (action) {
	  case GRAB_SUSPEND:
	      pp_grab_fb_info[video_link].suspended++;
	      break;
	  case GRAB_RESUME:
	      if (pp_grab_fb_info[video_link].suspended) pp_grab_fb_info[video_link].suspended--;
	      break;
	  default:
	      break;
	}
	goto bail;
    }

    MUTEX_LOCK(&p_grab->grab_suspend_cond_mtx);
    switch (action) {
      case GRAB_SUSPEND:
	  /* stop grabber */
	  DCH(D_BLABLA, "SUSPEND REQ\n");
	  p_grab->fb_suspend = SUSPEND_SUSPEND;	      
	  while(p_grab->fb_suspend != SUSPEND_SUSPENDED) {
	      pthread_cond_wait(&p_grab->grab_suspend_cond,
				&p_grab->grab_suspend_cond_mtx);
	  }
	  DCH(D_BLABLA, "SUSPEND ACK\n");
	  break;

      case GRAB_RESUME:
	  DCH(D_BLABLA, "RESUME REQ\n");
	  p_grab->fb_suspend = SUSPEND_RESUME;
	  while(p_grab->fb_suspend != SUSPEND_RESUMED) {
	      pthread_cond_wait(&p_grab->grab_suspend_cond,
				&p_grab->grab_suspend_cond_mtx);
	  }
	  DCH(D_BLABLA, "RESUME ACKanenl\n");
	  break;
      default:
	  break;
    }
    MUTEX_UNLOCK(&p_grab->grab_suspend_cond_mtx);

 bail:
    MUTEX_UNLOCK(&p_grab->grab_suspend_mtx);
}


void
pp_grab_refresh(unsigned char video_link)
{
    p_all_grabs[video_link].skip_diff++;
}


pp_grab_client_t*
pp_grab_new_client(unsigned char video_link, pp_grab_mem_desc_id_t id)
{
    assert(driver && driver->new_client);

    assert( video_link < current_num_video_links );
    return driver->new_client(id, video_link);
}


void
pp_grab_remove_client(pp_grab_client_t *client)
{
    assert(driver && driver->remove_client);

    return driver->remove_client(client);
}

int
pp_grab_set_encoding(pp_grab_client_t *client, vsc_encoding_desc_t enc,
		     u_int32_t encoding_tag)
{
    int ret = 0;
    assert (driver);

    if (driver->set_encoding) ret = driver->set_encoding(client, enc, encoding_tag);

    return ret;
}

int
pp_grab_request_region(pp_grab_client_t *client, pp_grab_req_flags_t flags,
		       RegionRec* req, RegionRec *ack, RegionRec *remain)
{
    assert(driver && driver->request_region);

    return driver->request_region(client, flags, req, ack, remain);
}

int
pp_grab_request_rect(pp_grab_client_t *client, pp_grab_req_flags_t flags,
		     u_short x, u_short y, u_short w, u_short h)
{
    assert(driver && driver->request_rect);
    
    assert(x % PP_FB_TILE_WIDTH == 0 && y % PP_FB_TILE_HEIGHT == 0 &&
	   w % PP_FB_TILE_WIDTH == 0 && h % PP_FB_TILE_HEIGHT == 0);

    return driver->request_rect(client, flags, x, y, w, h);
}

void
pp_grab_release_buffer(pp_grab_client_t *client)
{
    assert(driver);

    if (driver->release_buffer) driver->release_buffer(client);
}

int
pp_grab_get_color_info(unsigned char video_link UNUSED, fb_color_info_t *fb_c_info)
{
    assert(fb_c_info);

    memcpy(fb_c_info, &fb_color_info, sizeof(fb_color_info_t));

    return 0;
}

int
pp_grab_get_format_info(unsigned char video_link, fb_format_info_t *fb_f_info)
{
    grab_t *p_grab = &p_all_grabs[video_link];

    assert(fb_f_info);

    if (pp_grab_fb_info[video_link].current_fb == NULL) {
	DCH(D_NOTICE, "No framebuffer info yet\n");
	return -1;
    }
   
    MUTEX_LOCK(&p_grab->info_mtx);
    fb_f_info->g_w = pp_grab_fb_info[video_link].current_fb->g_w;
    fb_f_info->g_w_pd = pp_grab_fb_info[video_link].current_fb->g_w_pd;
    fb_f_info->tiles_w = pp_grab_fb_info[video_link].current_fb->tiles_w;
    
    fb_f_info->g_h = pp_grab_fb_info[video_link].current_fb->g_h;
    fb_f_info->g_h_pd = pp_grab_fb_info[video_link].current_fb->g_h_pd;
    fb_f_info->tiles_h = pp_grab_fb_info[video_link].current_fb->tiles_h;
    
    fb_f_info->bpp = pp_grab_fb_info[video_link].current_fb->bpp;
    fb_f_info->is_unsupported = pp_grab_fb_info[video_link].current_fb->is_unsupported;    
    MUTEX_UNLOCK(&p_grab->info_mtx);

    return 0;
}


void
pp_grab_sync_frame_buffer(u_char video_link, pp_grab_sync_lock_t action)
{
    if (driver && driver->sync_fb) driver->sync_fb(action, video_link);
}


void
pp_grab_debug_use_diffmap(unsigned char use_diffmap)
{
    driver->debug_use_diffmap = use_diffmap;
}

/*
 * only used internal in the grab lib
 */

int
grab_request_screen_sync(pp_grab_client_t *client, u_char hw_downscale, u_char *scale)
{
    assert(driver && driver->request_screen_sync);

//TODO: MULTICHANNEL SUPPORT: Need to extend request_screen_sync with video_link!!!
    return driver->request_screen_sync(client, hw_downscale, scale);
}

int
grab_release_screen_sync(pp_grab_client_t *client)
{
    assert(driver && driver->release_screen_sync);

//TODO: MULTICHANNEL SUPPORT: Need to extend  release_screen_sync with video_link
    return driver->release_screen_sync(client);
}

#if defined(PP_FEAT_CAT) || defined(PP_FEAT_KITTY_CAT)
/**
 * diffmap consumer
 * allows to register callback to process diffmap information
 * currently only needed to enqueue diffmaps in cursor action tracking CAT
 */

//TODO: MULTICHANNEL SUPPORT: later we will need to store video_link
void pp_grab_register_diffmap_consumer(pp_grab_diffmap_consumer_t _dmc) {
    if(!dmc) {
        dmc = (pp_grab_diffmap_consumer_t*)
              malloc(sizeof(pp_grab_diffmap_consumer_t));
    }
    dmc->target = _dmc.target;
    dmc->dmc_cb = _dmc.dmc_cb;
}

void pp_grab_unregister_diffmap_consumer() {
    free(dmc);
    dmc = NULL;
}
#endif /* PP_FEAT_CAT || PP_FEAT_KITTY_CAT */

