#include <sys/ioctl.h>

#include <pp/rfb.h>

#include "debug.h"
#include "hwenc.h"

grab_driver_t grab_driver_hwenc = {
    direct:		0,
    fb_sync_lock:       FB_SYNC_UNLOCKED,
    fb_sync_lock_mtx:	PTHREAD_ERRORCHECK_MUTEX_INITIALIZER_NP,
    
    new_client:		new_client_hwenc,
    remove_client:	remove_client_hwenc,
    set_encoding:	set_encoding_hwenc,
    
    request_region:	request_region_hwenc,
    request_rect:	request_rect_hwenc,
    release_buffer:	release_buffer_hwenc,
    sync_fb:	        NULL,
    
    request_screen_sync:request_screen_sync_hwenc,
    release_screen_sync:release_screen_sync_hwenc,
    
    transfer_regions:	transfer_hwenc,
    cancel_requests:	cancel_requests_hwenc,

    debug_use_diffmap: 1
};

/* default encoding (raw) */
vsc_encoding_desc_t default_enc = {
    algo:			VSC_ALGO_NULL,
    lrle_r_margin_rb:		0,
    lrle_r_margin_g:		0,
    lrle_grey_green:		0,
    lrle_g_margin:		0,
    lrle_runlimit:		0,
    lrle_linecopy:		0,
    lrle_grey_force:		0,
    lrle_grey_disable:		0,
    lrle_c_margin_rb:		0,
    lrle_c_margin_g:		0,
    lrle_color:			0,
    lrle_runlimit_reacc:	0,
    down_mode:			0
};

/* encoding descriptor template for downscaling */
vsc_encoding_desc_t enc_desc_down = {
    algo:		VSC_ALGO_DOWNSAMPLE,
    lrle_r_margin_rb:	0,
    lrle_r_margin_g:	0,
    lrle_c_margin_rb:	0,
    lrle_c_margin_g:	0,
    lrle_g_margin:	0,
    lrle_runlimit:	0,
    lrle_linecopy:	0,
    lrle_grey_force:	0,
    lrle_grey_disable:	0,
    lrle_grey_green:	0,
    lrle_color:		0,
    lrle_runlimit_reacc:0,
    down_mode:		DOWN_MODE_1X1,
};

static volatile unsigned char realloc_in_progress	= 0;

static void wait_for_all_buffers_idle(void);

static void transfer_handle_client(grab_client_int_t *cl_int, RegionRec *diff_reg, u_char video_link);
static inline void create_fake_picture(RegionRec *reg, u_char video_link);

static void spread_diffmap(grab_client_int_t *target_cl_int, RegionRec *map);
static void split_region(RegionRec *in, RegionRec *out1, RegionRec *out2,
			 u_short tile_size, u_long max_size);

#if !defined(PP_BUILD_TYPE_FINAL)
static int check_region(RegionRec *reg, u_short tile_size, u_long max_size);
#endif

static void
split_region(RegionRec *in, RegionRec *out1, RegionRec *out2,
	     u_short tile_size, u_long max_size)
{
    static const char* fn = ___F;
    RegionRec tmp;
    BoxRec box;
    Bool overlap;
    u_long size=0, free_size, tline_size, rect_size;
    u_char tlines, fill_remaining = 0;
    int i;
   
    profiler_start(PROFILE_SPLIT_REGION);
    D(D_BLABLA, "%s: tile_size=%d, max_size=%ld\n", fn, tile_size, max_size);    
    for (i = 0; i < REGION_NUM_RECTS(in); i++) {
	box.x1 = REGION_RECTS(in)[i].x1;
	box.y1 = REGION_RECTS(in)[i].y1;
	box.x2 = REGION_RECTS(in)[i].x2;
	box.y2 = REGION_RECTS(in)[i].y2;

	D(D_BLABLA, "%s: box (%4d,%4d)-(%4d,%4d)\n", fn, box.x1, box.y1,
	  box.x2, box.y2);

	/* adding the poor rest to the remaining region */
	if (fill_remaining) {

	    REGION_INIT(&tmp, &box, 0);
	    REGION_APPEND(out2, &tmp);
	    REGION_UNINIT(&tmp);
	    continue;
	}
	
	rect_size = (box.x2 - box.x1) / PP_FB_TILE_WIDTH * (box.y2 - box.y1) / PP_FB_TILE_HEIGHT * tile_size;
	/* rect fits in the buffer, good and easy */
	if (size + rect_size <= max_size) {
	    D(D_BLABLA, "%s: rect fits, size=%ld\n", fn, rect_size);
   
	    REGION_INIT(&tmp, &box, 0);
	    REGION_APPEND(out1, &tmp);
	    REGION_UNINIT(&tmp);
	    
	    size += rect_size;
	    continue;
	}  

	/* split the rect (vertical, tile-line based) */
	free_size = max_size - size;
	tline_size= (box.x2 - box.x1) / PP_FB_TILE_WIDTH * tile_size;
	D(D_BLABLA, "%s: free_size=%ld, tline_size=%ld\n", fn, free_size, tline_size);
	
	if ((tlines = free_size / tline_size) > 0) {
	    BoxRec new_box = box;
	    u_short ys = new_box.y1 + tlines * PP_FB_TILE_HEIGHT;

	    new_box.y2 = ys;

	    D(D_BLABLA, "%s: split box 1 (%4d,%4d)-(%4d,%4d)\n", fn, new_box.x1,
	      new_box.y1, new_box.x2, new_box.y2);
	    
	    REGION_INIT(&tmp, &new_box, 0);	    
	    REGION_APPEND(out1, &tmp);
	    REGION_UNINIT(&tmp);

	    new_box = box;
	    new_box.y1 = ys;

	    D(D_BLABLA, "%s: split box 2 (%4d,%4d)-(%4d,%4d)\n", fn, new_box.x1,
             new_box.y1, new_box.x2, new_box.y2);
	    	    
	    REGION_INIT(&tmp, &new_box, 0);
	    REGION_APPEND(out2, &tmp);	    
	    REGION_UNINIT(&tmp);

	    size += tlines * tline_size;
	} else {
	    /* even a single tile-line is too large,
	       reject the whole rect and all following */
	    REGION_INIT(&tmp, &box, 0);
	    REGION_APPEND(out2, &tmp);
	    REGION_UNINIT(&tmp);

	    fill_remaining = 1;
	}
    }

    REGION_VALIDATE(out1, &overlap);
    REGION_VALIDATE(out2, &overlap);
    
    /* must be able to hold at least one tile-line of a rect
       if its the only one in this update */
    assert(REGION_NUM_RECTS(out1) != 0);

    profiler_done(PROFILE_SPLIT_REGION);
}

#if !defined(PP_BUILD_TYPE_FINAL)
static int
check_region(RegionRec *reg, u_short tile_size, u_long max_size)
{
    BoxRec box;
    u_long size = 0;
    int i, ret = -1;

    for (i = 0; i < REGION_NUM_RECTS(reg); i++) {
	box.x1 = REGION_RECTS(reg)[i].x1;
	box.y1 = REGION_RECTS(reg)[i].y1;
	box.x2 = REGION_RECTS(reg)[i].x2;
	box.y2 = REGION_RECTS(reg)[i].y2;

	size += (box.x2 - box.x1) / PP_FB_TILE_WIDTH * (box.y2 - box.y1) / PP_FB_TILE_HEIGHT * tile_size;

	if (size > max_size) {
	    goto bail;
	}  
    }

    ret = 0;
    
 bail:
    return ret;
}
#endif

static u_short
calc_max_tile_size(vsc_encoding_desc_t *enc)
{
    static const char * fn = ___F;
    u_short size = 0;

    switch (enc->algo) {
      case VSC_ALGO_NULL:
	  size = PP_FB_TILE_SIZE * VSC_PIXEL_SIZE;
	  break;
      case VSC_ALGO_LRLE:
	  D(D_VERBOSE, "%s() lrle_color=%d\n", fn, enc->lrle_color);
	  /* worst case assuming chess pattern, 1 Pixel per run, no line copies,
	     no byte compression */
	  switch (enc->lrle_color) {
	    case LRLE_COLOR_15BIT_DIRECT:
		size = PP_FB_TILE_SIZE * VSC_PIXEL_SIZE;
		break;
	    case LRLE_COLOR_7BIT_DIRECT:
		size = PP_FB_TILE_SIZE; 
		break;
	    case LRLE_COLOR_4BIT_GREY:
	    case LRLE_COLOR_4BIT_PALETTE:
	    case LRLE_COLOR_3BIT_GREY:
		size = PP_FB_TILE_SIZE;
		break;
	    case LRLE_COLOR_2BIT_GREY:
		size = PP_FB_TILE_SIZE / 4;
		break;
	    case LRLE_COLOR_1BIT_GREY:
		size = PP_FB_TILE_SIZE / 8;
		break;
	    default:
		D(D_ERROR, "%s() unknown LRLE color\n", fn);
		break;
	  }
	  break;
      case VSC_ALGO_DOWNSAMPLE:
	  D(D_VERBOSE, "%s() down_mode=%d\n", fn, enc->down_mode);
	  switch (enc->down_mode) {
	    case DOWN_MODE_2X2:
		size = PP_FB_TILE_SIZE / (2 * 2) * VSC_PIXEL_SIZE;
		break;
	    case DOWN_MODE_4X4:
		size = PP_FB_TILE_SIZE / (4 * 4) * VSC_PIXEL_SIZE;
		break;
	    case DOWN_MODE_8X8:
		size = PP_FB_TILE_SIZE / (8 * 8) * VSC_PIXEL_SIZE;
		break;		
	    default:
		size = PP_FB_TILE_SIZE * VSC_PIXEL_SIZE;
	      break;
	  }
	  break;
      case VSC_ALGO_DCT:
	  D(D_ERROR, "%s() DCT not yet implemented\n", fn);
	  assert(0);
	  break;
      default:
	  D(D_ERROR, "%s() unknown algo %d\n",fn, enc->algo);
	  break;
    }
    
    D(D_VERBOSE, "%s() maximum size %d for algo %d\n", fn, size, enc->algo);
    return size;
}

static void
spread_diffmap(grab_client_int_t *cl_int, RegionRec *map)
{
    struct list_head *node, *tmp;

    /* we already hold the clients mutex here */
    list_for_each_safe(node, tmp, &clients_list) {
	grab_client_int_t *target_cl_int = list_entry(node, grab_client_int_t, listnode);
	// omit clients with a different video_link
	if (cl_int->ext->video_link != target_cl_int->ext->video_link) continue;
	// omit the client itself
	if (cl_int == target_cl_int) continue;
	
	MUTEX_LOCK(&target_cl_int->mtx);
	REGION_UNION(&target_cl_int->mod_reg, &target_cl_int->mod_reg, map);
	MUTEX_UNLOCK(&target_cl_int->mtx);
    }
}

static void
transfer_handle_client(grab_client_int_t *cl_int, RegionRec *diff_reg, u_char video_link)
{
    RegionRec *map;
    u_short max_tile_size = cl_int->max_tile_size;
    
    int grab_vsc_fd = pp_grab_fd( video_link );

    DCH(D_BLABLA, "%s client %p (ignore_diff=%d, dont_split=%d)\n",
      ___F, cl_int->ext, cl_int->ignore_diff, cl_int->dont_split);

    /* create region we have to transfer */
    if (cl_int->ignore_diff) {
	map = cl_int->req_reg;
    } else {
	REGION_INTERSECT(&cl_int->client_map, &cl_int->mod_reg, cl_int->req_reg);
	REGION_INTERSECT(&cl_int->global_map, diff_reg, cl_int->req_reg);
	REGION_UNION(&cl_int->temp_map, &cl_int->client_map, &cl_int->global_map);

	map = &cl_int->temp_map;
    }

    DCH(D_BLABLA, "%s: client_map=%ld, global_map=%ld -> map=%ld\n", ___F,
      REGION_NUM_RECTS(&cl_int->client_map), REGION_NUM_RECTS(&cl_int->global_map),
      REGION_NUM_RECTS(map));

    /* nothing to do anymore, our transfer map is empty */
    if (!REGION_NOTEMPTY(map)) {
	cl_int->fetch_desc.error = VSC_FETCH_ERROR_EMPTY_REQUEST;
	REGION_EMPTY(cl_int->ack_reg);
	return;
    }
    
    /* truncate region to not exceed maximum (worst case) size,
       depending on the current encoding */
    if (cl_int->dont_split) {
#if !defined(PP_BUILD_TYPE_FINAL)
	assert(check_region(map, max_tile_size, cl_int->mem_desc->size) == 0);
#endif
	REGION_COPY(cl_int->ack_reg, map);
    } else {
    	split_region(map, cl_int->ack_reg, cl_int->rem_reg, max_tile_size, cl_int->mem_desc->size);
    }

    /* if caller wishes to get the transfer exactly as requested (i.e. simplerfb)
       validate its region for our internal diffmap handling because it may be not
       a valid one originally */
    if (cl_int->dont_touch) {
	int overlap;

	REGION_VALIDATE(cl_int->ack_reg, &overlap);
    }
    
    DCH(D_BLABLA, "%s: ack: (%3d,%3d)-(%3d,%3d)\n", ___F, 
      cl_int->ack_reg->extents.x1, cl_int->ack_reg->extents.y1,
      cl_int->ack_reg->extents.x2, cl_int->ack_reg->extents.y2);
    
    DCH(D_BLABLA, "%s after split: req=%ld, ack=%ld, remain=%ld\n", ___F,
      REGION_NUM_RECTS(map), REGION_NUM_RECTS(cl_int->ack_reg), REGION_NUM_RECTS(cl_int->rem_reg));

    /* add global part of the acknowledged region to other clients diffmaps */
    REGION_INTERSECT(&cl_int->temp_map, diff_reg, cl_int->ack_reg);
    spread_diffmap(cl_int, &cl_int->temp_map);

    /* remove acknowledged region from our own diffmap and the global one */
    REGION_SUBTRACT(&cl_int->mod_reg, &cl_int->mod_reg, cl_int->ack_reg);
    REGION_SUBTRACT(diff_reg, diff_reg, cl_int->ack_reg);

    /* transfer the data */
    cl_int->ext->fb = *pp_grab_fb_info[video_link].current_fb;
    cl_int->fetch_desc.reg = cl_int->dont_touch ? map : cl_int->ack_reg;

    if (cl_int->fetch_desc.ctrl & VSC_FETCH_CTRL_ADD_HDR) {
	max_tile_size += sizeof(vsc_update_rect_hdr_t);
    }
   
#ifdef DEBUG_CHECK_REGION
    if (dbg_check_reg(cl_int->fetch_desc.reg, cl_int->fetch_desc.enc,
		     &fb_color_info) != 0) {
	
	cl_int->fetch_desc.error = VSC_FETCH_ERROR_DEBUGGING_ERROR;
	return;
    }
#endif   

    cl_int->fetch_desc.mem_offset = cl_int->mem_desc->offset;
    profiler_start(PROFILE_FETCH_TILES);
    if (ioctl(grab_vsc_fd, PPIOCVSCFETCHTILES, &cl_int->fetch_desc) == -1) {
	if (errno != EAGAIN) {
	    DCH(D_ERROR, "%s(): ioctl(PPIOCVSCFETCHTILES) for %p\n", ___F, cl_int->ext);
	    cl_int->fetch_desc.error = VSC_FETCH_ERROR_TIMEOUT;
        } else cl_int->fetch_desc.error = VSC_FETCH_ERROR_RETRY;
    }
    profiler_done(PROFILE_FETCH_TILES);

    MUTEX_LOCK(&grab_driver_hwenc.fb_sync_lock_mtx);
    if (cl_int->fb_sync_lock) {
	grab_driver_hwenc.fb_sync_lock = FB_SYNC_LOCKED;
    }
    MUTEX_UNLOCK(&grab_driver_hwenc.fb_sync_lock_mtx);
    
#ifdef DEBUG_DUMP_REGION
    dbg_dump_reg(cl_int->mem_desc->ptr, cl_int->fetch_desc.size, cl_int->fetch_desc.reg,
		 cl_int->fetch_desc.enc, &fb_color_info);
#endif
}

static inline void
create_fake_picture(RegionRec *reg, u_char video_link)
{
    fb_format_info_t *current_fb = pp_grab_fb_info[video_link].current_fb;

    BoxRec box = { 0, 0, current_fb->g_w_pd, current_fb->g_h_pd / 2};
    RegionRec cut;
    
    REGION_INIT(&cut, &box, 0);
    REGION_INTERSECT(reg, reg, &cut);
    REGION_UNINIT(&cut);   
}

void
transfer_hwenc(RegionRec *diff_reg, u_char refresh_pending, u_char video_link)
{
    struct list_head *node, *tmp;
    pp_grab_encoder_desc_t* act;

    fb_format_info_t* current_fb = pp_grab_fb_info[video_link].current_fb;

    /* cut the diff region in the middle of the screen */
    if (fake_picture_error) {
	create_fake_picture(diff_reg, video_link);
    }

    if (grab_driver_hwenc.fb_sync_lock == FB_SYNC_UNLOCKED && REGION_NOTEMPTY(diff_reg)) {
        act = grab_encoder_hook;
        while (act) {
            if (act->new_data_available) {
                act->new_data_available(video_link);
            }
            act = act->next;
        }
    }
    
    MUTEX_LOCK(&clients_mtx);
    list_for_each_safe(node, tmp, &clients_list) {
	grab_client_int_t *cl_int = list_entry(node, grab_client_int_t, listnode);
	// omit clients with a different video_link
	if (cl_int->ext->video_link != video_link) continue;

	MUTEX_LOCK(&cl_int->mtx);
	/* fill modified region for our clients */
	if (refresh_pending) {
	    BoxRec box;
		
	    box.x1 = box.y1 = 0;
	    box.x2 = current_fb->g_w_pd;
	    box.y2 = current_fb->g_h_pd;

	    DCH(D_VERBOSE, "Refreshing (%d,%d)\n", current_fb->g_w_pd, current_fb->g_h_pd);

	    REGION_RESET(&cl_int->mod_reg, &box);
	}

	if (cl_int->state == CLIENT_STATE_WAITING &&
	    !(cl_int->fb_sync_lock && grab_driver_hwenc.fb_sync_lock == FB_SYNC_UNLOCKING)) {
    
	    transfer_handle_client(cl_int, diff_reg, video_link);
	    cl_int->state = CLIENT_STATE_IDLE;
	    pthread_cond_signal(&cl_int->cond);
	}
	MUTEX_UNLOCK(&cl_int->mtx);
    }

    MUTEX_UNLOCK(&clients_mtx);
}

void
cancel_requests_hwenc(u_char reason, u_char video_link)
{
    static const char* fn = ___F;
    struct list_head *node, *tmp;    

    pp_grab_fb_info[video_link].grabber_available = 0;

    MUTEX_LOCK(&clients_mtx);

    list_for_each_safe(node, tmp, &clients_list) {
	grab_client_int_t *cl_int = list_entry(node, grab_client_int_t, listnode);
	// omit clients with a different video_link
	if (cl_int->ext->video_link != video_link) continue;

	MUTEX_LOCK(&cl_int->mtx);
	if (cl_int->state == CLIENT_STATE_WAITING) {
	    DCH(D_VERBOSE, "%s: canceling client %p for reason %d\n", fn, cl_int->ext, reason);
	    cl_int->fetch_desc.error = reason;
	    cl_int->state = CLIENT_STATE_IDLE;
	    pthread_cond_signal(&cl_int->cond);
	}
	MUTEX_UNLOCK(&cl_int->mtx);
    }
    
    MUTEX_UNLOCK(&clients_mtx);   
}

pp_grab_client_t*
new_client_hwenc(pp_grab_mem_desc_id_t id, u_char video_link)
{
    pp_grab_client_t *ret = NULL;
    grab_client_int_t *cl_i = NULL;
    mem_desc_t *mem_desc;
    pp_grab_encoder_desc_t* act;

    DCH(D_VERBOSE, "%s: Request for new client\n", ___F);
    
    realloc_in_progress = 1;
    cancel_requests_hwenc(VSC_FETCH_ERROR_REALLOC_IN_PROGRESS, video_link);

    MUTEX_LOCK(&clients_mtx);

    wait_for_all_buffers_idle();
    
    ret = malloc(sizeof(pp_grab_client_t));
    cl_i = malloc(sizeof(grab_client_int_t));

    ret->video_link= video_link;
    ret->intern = cl_i;
    
    if ((mem_desc = mempool_get_desc(id, video_link)) == NULL) {
	DCH(D_ERROR, "Could not reserve a memory block\n");
	goto err_out;
    }
    cl_i->mem_desc   = mem_desc;  
    DCH(D_VERBOSE, "%s: client: %p got memory block @ offset %08x, size %d\n", ___F, ret, cl_i->mem_desc->offset,
      cl_i->mem_desc->size);

    /* temporary working diffmaps */
    REGION_INIT(&cl_i->mod_reg, NullBox, 0);
    REGION_INIT(&cl_i->client_map, NullBox, 0);
    REGION_INIT(&cl_i->global_map, NullBox, 0);
    REGION_INIT(&cl_i->temp_map, NullBox, 0);
    
    cl_i->ext	= ret;
    cl_i->state	= CLIENT_STATE_IDLE;
    cl_i->buf_mem_state = BUF_MEM_STATE_IDLE;

    /* initialize fetch descriptor */
    cl_i->fetch_desc.ctrl = 0;

    /* use the standard RAW encoding per default */
    cl_i->fetch_desc.enc   = default_enc;
    cl_i->fetch_desc.error = VSC_FETCH_ERROR_NO_ERROR;
    cl_i->fetch_desc.size  = 0;
    
    cl_i->fb_sync_lock = FB_SYNC_UNLOCKED;
    cl_i->max_tile_size = calc_max_tile_size(&default_enc);

    /* initialize some flags */
    cl_i->ignore_diff = cl_i->refresh = cl_i->dont_split = 0;
    
    pthread_mutex_init(&cl_i->mtx, &errcheck_mtx_attr);
    pthread_cond_init(&cl_i->cond, NULL);

    pthread_mutex_init(&cl_i->buf_mem_mtx, &errcheck_mtx_attr);
    pthread_cond_init(&cl_i->buf_mem_cond, NULL);

    list_add(&cl_i->listnode, &clients_list);

    pp_grab_fb_info[video_link].nr_clients++;    
    DCH(D_VERBOSE, "Added client %p\n", ret);

    goto bail;
	
 err_out:
    free(cl_i);
    free(ret); ret = NULL;    
    REGION_UNINIT(&cl_i->temp_map);
    REGION_UNINIT(&cl_i->global_map);
    REGION_UNINIT(&cl_i->client_map);
    REGION_UNINIT(&cl_i->mod_reg);
    
 bail:
    realloc_in_progress = 0;
    act = grab_encoder_hook;
    while (act) {
        act->grabber_available(video_link); 
        act = act->next;
    }
    MUTEX_UNLOCK(&clients_mtx);
    return ret;
}

void
wait_for_all_buffers_idle(void)
{
    struct list_head *node, *tmp;
	
    list_for_each_safe(node, tmp, &clients_list) {
	// wait for really all clients
	grab_client_int_t *cl_int = list_entry(node, grab_client_int_t, listnode);

	MUTEX_LOCK(&cl_int->buf_mem_mtx);
	while (cl_int->buf_mem_state != BUF_MEM_STATE_IDLE) {
	    pthread_cond_wait(&cl_int->buf_mem_cond, &cl_int->buf_mem_mtx);
	}
	MUTEX_UNLOCK(&cl_int->buf_mem_mtx);
    }
    
}

void
remove_client_hwenc(pp_grab_client_t *client)
{
    grab_client_int_t *cl_i = client->intern;
    pp_grab_encoder_desc_t* act;

    D(D_VERBOSE, "Request to remove client %p, video_link=%d\n", client, client->video_link);
    
    realloc_in_progress = 1;
    cancel_requests_hwenc(VSC_FETCH_ERROR_REALLOC_IN_PROGRESS, client->video_link);

    MUTEX_LOCK(&clients_mtx);

    wait_for_all_buffers_idle();
    
    assert(cl_i->state != CLIENT_STATE_TRANSFERRING);
    
    list_del(&cl_i->listnode);

    mempool_release_desc(cl_i->mem_desc);
    
    pp_grab_fb_info[client->video_link].nr_clients--;
    pthread_cond_destroy(&cl_i->cond);
    pthread_mutex_destroy(&cl_i->mtx);
    pthread_cond_destroy(&cl_i->buf_mem_cond);
    pthread_mutex_destroy(&cl_i->buf_mem_mtx);

    REGION_UNINIT(&cl_i->temp_map);
    REGION_UNINIT(&cl_i->global_map);
    REGION_UNINIT(&cl_i->client_map);
    REGION_UNINIT(&cl_i->mod_reg);
    
    free(client->intern);
    free(client);

    D(D_VERBOSE, "Removed client %p, video_link=%d\n", client, client->video_link);

    realloc_in_progress = 0;
    act = grab_encoder_hook;
    while (act) {
        act->grabber_available(client->video_link); 
        act = act->next;
    }

    MUTEX_UNLOCK(&clients_mtx);
}

int
set_encoding_hwenc(pp_grab_client_t *client, vsc_encoding_desc_t enc, u_int32_t encoding_tag)
{
    static const char* fn = ___F;
    grab_client_int_t *cl_int = client->intern;
    
    MUTEX_LOCK(&cl_int->mtx);

    assert(cl_int->state == CLIENT_STATE_IDLE);
  
    cl_int->fetch_desc.enc = enc;
    cl_int->fetch_desc.enc_tag = encoding_tag;
    
    cl_int->max_tile_size = calc_max_tile_size(&enc);
   
    D(D_BLABLA, "%s: algo=%d, tag=%08x,\n", fn, enc.algo, encoding_tag);
    if (enc.algo == VSC_ALGO_LRLE) {
	D(D_BLABLA, "  r_margin_rb=%d, r_margin_g=%d, c_margin_rb=%d, c_margin_g=%d,\n",
	  enc.lrle_r_margin_rb, enc.lrle_r_margin_g, enc.lrle_c_margin_rb, enc.lrle_c_margin_g);
	D(D_BLABLA, "  runlimit=%d, runlimit_reacc=%d, linecopy=%d, color=%d\n",
	  enc.lrle_runlimit, enc.lrle_runlimit_reacc, enc.lrle_linecopy, enc.lrle_color);
	D(D_BLABLA, "  g_margin=%d, grey_green=%d, grey_force=%d, grey_disable=%d\n",
	  enc.lrle_g_margin, enc.lrle_grey_green, enc.lrle_grey_force, enc.lrle_grey_disable);
    }
	
    MUTEX_UNLOCK(&cl_int->mtx);
    return 0;
}

int
request_region_hwenc(pp_grab_client_t *client, pp_grab_req_flags_t flags,
		     RegionRec *req, RegionRec *ack, RegionRec *remain)
{
    static const char * fn = ___F;
    grab_client_int_t *cl_int = client->intern;
    int ret = -1;

    assert(req && ack && remain);

    client->req_err = GRAB_REQ_ERR_NO_ERROR;

    /* don't allow new requests once realloc started */
    if (realloc_in_progress) {
	client->req_err = GRAB_REQ_ERR_REALLOC_IN_PROGRESS;
	goto bail;
    }
        
    if (flags & GRAB_REQ_FLAGS_LOCK_DIFF) {
	cl_int->fb_sync_lock = 1;
    }

    D(D_BLABLA, "%s: client %p req (%3d,%3d)-(%3d,%3d), video_link=%d\n", fn,
      client, req->extents.x1, req->extents.y1, req->extents.x2, req->extents.y2, client->video_link);
   
    MUTEX_LOCK(&cl_int->mtx);
        
    assert(cl_int->state == CLIENT_STATE_IDLE);

    cl_int->fetch_desc.ctrl &= ~VSC_FETCH_CTRL_ADD_HDR;
    if (flags & GRAB_REQ_FLAGS_ADD_HDR) {
	cl_int->fetch_desc.ctrl |= VSC_FETCH_CTRL_ADD_HDR;
    }
    
    cl_int->fetch_desc.error = VSC_FETCH_ERROR_NO_ERROR;
    cl_int->req_reg = req;
    cl_int->ack_reg = ack;
    cl_int->rem_reg = remain;
    cl_int->ignore_diff = (flags & GRAB_REQ_FLAGS_IGNORE_DIFFMAP) ? 1 : 0;
    cl_int->dont_split  = (flags & GRAB_REQ_FLAGS_DONT_SPLIT_REGION) ? 1 : 0;

    if (flags & GRAB_REQ_FLAGS_DONT_TOUCH_REGION) {
	cl_int->ignore_diff = cl_int->dont_split = cl_int->dont_touch = 1;
    } else {
	cl_int->dont_touch = 0;
    }
    
    cl_int->state = CLIENT_STATE_WAITING;
    while (cl_int->state != CLIENT_STATE_IDLE) {
	pthread_cond_wait(&cl_int->cond, &cl_int->mtx);
    }
    
    MUTEX_UNLOCK(&cl_int->mtx);

    if (cl_int->fetch_desc.error == VSC_FETCH_ERROR_EMPTY_REQUEST) {
	/* there was nothing to do, not a real error for the client */
	D(D_BLABLA, "%s: client %p, got NOTHING, video_link=%d\n", fn, client, client->video_link);
	cl_int->buf_mem_state = BUF_MEM_STATE_REQUEST;
	ret = 0;
	goto bail;
    } else if (cl_int->fetch_desc.error != VSC_FETCH_ERROR_NO_ERROR) {
	D(D_VERBOSE, "%s got fetch error %d; video_link=%d\n", fn, cl_int->fetch_desc.error, client->video_link);
	switch (cl_int->fetch_desc.error) {
	  case VSC_FETCH_ERROR_NO_SIGNAL:
	      client->req_err = GRAB_REQ_ERR_NO_SIGNAL;
	      break;
	  case VSC_FETCH_ERROR_NOT_AVAILABLE:
	      client->req_err = GRAB_REQ_ERR_NOT_AVAILABLE;
	      break;
	  case VSC_FETCH_ERROR_UNKNOWN_MODE:
	      client->req_err = GRAB_REQ_ERR_UNKNOWN_MODE;
	      break;
	  case VSC_FETCH_ERROR_DEBUGGING_ERROR:
	      client->req_err = GRAB_REQ_ERR_DEBUGGING_ERROR;
              break;
	  case VSC_FETCH_ERROR_REALLOC_IN_PROGRESS:
	      client->req_err = GRAB_REQ_ERR_REALLOC_IN_PROGRESS;
              break;
	  case VSC_FETCH_ERROR_MODE_CHANGE:
	      client->req_err = GRAB_REQ_ERR_MODE_CHANGE;
              break;
          case VSC_FETCH_ERROR_RETRY:
              client->req_err = GRAB_REQ_ERR_RETRY;
              break;
	  default:
	      client->req_err = GRAB_REQ_ERR_INTERNAL_ERROR;
	      break;
	}

	errno = pp_grab_errno_base() + client->req_err;
	D(D_ERROR, "%s failed ('%s'), video_link=%d\n", fn, pp_error_string(pp_grab_errno_base() + client->req_err), client->video_link);
	goto bail;
    }

    client->buf	= cl_int->mem_desc->ptr;
    client->buf_size = cl_int->fetch_desc.size;
    D(D_BLABLA, "%s: client %p, got %ld, video_link=%d\n", fn, client, client->buf_size, client->video_link);
    cl_int->buf_mem_state = BUF_MEM_STATE_REQUEST;
    ret = 0;
    
 bail:
    return ret;
}

int
request_rect_hwenc(pp_grab_client_t *client, pp_grab_req_flags_t flags,
		   u_short x, u_short y, u_short w, u_short h)
{
    BoxRec rect;
    RegionRec req, ack, remain;
    int ret = -1;
       
    rect.x1 = x;
    rect.y1 = y;
    rect.x2 = x + w;
    rect.y2 = y + h;

    REGION_INIT(&req, &rect, 0);
    REGION_INIT(&ack, NULL, 0);
    REGION_INIT(&remain, NULL, 0);
  
    if (request_region_hwenc(client, flags, &req, &ack, &remain) != 0) {
	goto bail;
    }

    /* everything must fit into the buffer for a single rect */
    assert(!REGION_NOTEMPTY(&remain));

    /* rect located right at the buffer start, no space between lines */
    client->rect_offset = 0;
    client->rect_pitch  = w;

    ret = 0;
    
 bail:
    REGION_UNINIT(&remain);
    REGION_UNINIT(&ack);
    REGION_UNINIT(&req);
    
    return ret;
}

void
release_buffer_hwenc(pp_grab_client_t *client)
{
    grab_client_int_t *cl_int = client->intern;
    
    MUTEX_LOCK(&cl_int->buf_mem_mtx);
    cl_int->buf_mem_state = BUF_MEM_STATE_IDLE;
    pthread_cond_signal(&cl_int->buf_mem_cond);
    MUTEX_UNLOCK(&cl_int->buf_mem_mtx);

    MUTEX_LOCK(&grab_driver_hwenc.fb_sync_lock_mtx);
    if (cl_int->fb_sync_lock == FB_SYNC_LOCKED) {
	grab_driver_hwenc.fb_sync_lock = FB_SYNC_UNLOCKING;
	cl_int->fb_sync_lock = 0;
    }
    MUTEX_UNLOCK(&grab_driver_hwenc.fb_sync_lock_mtx);
}

int
request_screen_sync_hwenc(pp_grab_client_t *client, u_char hw_downscale, u_char *scale)
{
    fb_format_info_t* current_fb = pp_grab_fb_info[client->video_link].current_fb;
    u_char scale_fact = (current_fb->g_w > 1024) ? 4 : 2;
    static const char* fn = ___F;
    int ret = -1;
    vsc_encoding_desc_t enc_desc;
    
    if (hw_downscale) {
	enc_desc = enc_desc_down;

	switch (scale_fact) {
	  case   2: enc_desc_down.down_mode = DOWN_MODE_2X2; break;
	  case   4: enc_desc_down.down_mode = DOWN_MODE_4X4; break;
	  case   8: enc_desc_down.down_mode = DOWN_MODE_8X8; break;
	  default : enc_desc_down.down_mode = DOWN_MODE_1X1; break;
	}
    } else {
	/* default fetch descriptor for raw transfer */
	enc_desc = default_enc;  
    }
    
    assert(scale);
       
    if (pp_grab_set_encoding(client, enc_desc_down, 0) != 0) {
	goto bail;
    }

    if (hw_downscale) {	
	/* request already downscaled rect */
	*scale = scale_fact;
	ret = request_rect_hwenc(client, GRAB_REQ_FLAGS_IGNORE_DIFFMAP, 0, 0, current_fb->g_w, current_fb->g_h);
	
	client->rect_pitch = client->rect_pitch / scale_fact;   
	D(D_VERBOSE, "%s: hw downscaled, factor=%d -> pitch=%d\n", fn, scale_fact, client->rect_pitch);
    } else {
	/* requestor has to do scaling itself */
        *scale = 0; 
	client->fb = *current_fb;
	ret = 0;
	D(D_VERBOSE, "%s: raw\n", fn);
    }
    
 bail:
    return ret;
}

int
release_screen_sync_hwenc(pp_grab_client_t *client)
{
    release_buffer_hwenc(client);
    return 0;
}
