/**
 * \file grab.h
 *
 * This is the interface to the video grabber engine.
 * The library is responsible for controlling the sampling,
 * reading the difference information and the actual framebuffer
 * content and delivering them to the encoder.
 *
 * \verbinclude framebuffer.txt
 *
 * (c) 2004 Peppercon AG
 */

#ifndef _LIBPP_GRAB_H
#define _LIBPP_GRAB_H

#ifdef __cplusplus
extern "C"
{
#endif

#include <pthread.h>
#include <sys/types.h>
#include <pp/base.h>
#include <lara.h>
#include <liberic_session.h>

#define DEV_VSC_PRFX	"/dev/vsc"


#define R_MSK   0xF800
#define G_MSK   0x07E0
#define B_MSK   0x001F

typedef enum {
    GRAB_SUSPEND_NO_ACTION,
    GRAB_SUSPEND,
    GRAB_SUSPEND_AND_UNMAP,
    GRAB_RESUME,
    GRAB_RESUME_AND_MAP
} pp_grab_suspend_action_t;

typedef enum {
    GRAB_SYNC_LOCK,
    GRAB_SYNC_UNLOCK
} pp_grab_sync_lock_t;

typedef enum {
    GRAB_REQ_FLAGS_NONE			= 0x00,
    GRAB_REQ_FLAGS_ADD_HDR		= 0x01,
    GRAB_REQ_FLAGS_DONT_SPLIT_REGION	= 0x02,
    GRAB_REQ_FLAGS_LOCK_DIFF        	= 0x04,
    GRAB_REQ_FLAGS_IGNORE_DIFFMAP	= 0x08,
    GRAB_REQ_FLAGS_DONT_TOUCH_REGION	= 0x10,
} pp_grab_req_flags_t;

typedef enum {
    GRAB_REQ_ERR_NO_ERROR=0,
    GRAB_REQ_ERR_NO_SIGNAL,
    GRAB_REQ_ERR_NOT_AVAILABLE,
    GRAB_REQ_ERR_MODE_CHANGE,
    GRAB_REQ_ERR_UNKNOWN_MODE,
    GRAB_REQ_ERR_INTERNAL_ERROR,
    GRAB_REQ_ERR_DEBUGGING_ERROR,
    GRAB_REQ_ERR_REALLOC_IN_PROGRESS,
    GRAB_REQ_ERR_RETRY,
} pp_grab_request_error_t;

typedef enum {
    GRAB_VARIABLE_MEM_DESC	 = 0,
    GRAB_FIX_MEM_DESC_MOUSESYNC	 = 1,
    GRAB_FIX_MEM_DESC_SCREENSHOT = 2,
    GRAB_FIX_MEM_DESC_OFFSET	 = 3,
    GRAB_FIX_MEM_DESC_SIMPLERFB  = 4,
    GRAB_FIX_MEM_DESC_DEBUG1	 = 5,
    GRAB_FIX_MEM_DESC_DEBUG2	 = 6,
} pp_grab_mem_desc_id_t;

/* vsc encoding descriptor. Do not mix this up with the  *
 * grab_encoder_descriptor for connecting (rfb)encoders */
extern vsc_encoding_desc_t pp_grab_null_encoding_desc;


/**
 * grab client descriptor. This structure holds individual grabber data
 * like settings for each connected client. Several clients can exist for
 * one video link. Clients usually correspond to (rfb)encoder sessions.
 */
typedef struct grab_client_s {
    struct grab_client_int_s *intern;	/* internal client data */
    u_char *buf;			/* buffer for encoded video data,
					   in direct (VSC1) mode a pointer to fb */
    u_long buf_size;			/* size of the current data in buf */

    pp_grab_request_error_t req_err;	/* error during grab request */
    fb_format_info_t fb;		/* fb format when request was processed */
    u_int32_t rect_offset;		/* offset for single rect requests (in pixels) */
    u_short rect_pitch;			/* fb pitch for single rect requests (in pixels) */
    u_char video_link;			/* video_link i.e.  channel i.e. frame grabber */
} pp_grab_client_t;


/**
 * encoder descriptor. Used to connect an encoder and set
 * some callbacks that the grabber will use during the grab
 * cycle. Callbacks must not be NULL unless otherwise noted.
 */
typedef struct pp_grab_encoder_desc_s {
    /**
     * Notify the encoder that a new raw diffmap is available. Called after the
     * framebuffer contains the new parts. Usually this information is only
     * used if the client handles the diffmaps himself and requests new regions
     * with the GRAB_REQ_FLAGS_IGNORE_DIFFMAP flag. Encoders that let the
     * grabber do the work can savely ignore this call.
     * 
     * Note: this callback can be set to NULL.
     * 
     * FIXME: is this callback completely implemented ? (only in grab_vsc.c)
     * 
     * @param video_link frame grabber that this notification is for. In case
     *                  its a single ported hardware, video_link is always 0
     * @param diffmap	raw diffmap from VSC, see framebuffer.txt for a description
     */
    void (*new_diffmap_available)(unsigned char video_link, u_int32_t diffmap[VSC_DIFFMAP_SIZE]);

    /**
     * Notify the encoder that new data is available. Usually the encoder will
     * call request_region to retrieve the new data. This function can be seen
     * as a ligthweight version of new_diffmap_available().
     * 
     * Note: original function name was schedule_update() but 'schedule' is
     * semantically misleading as we have no queue. The new data is available
     * immediately and its up to the client to handle that.
     * 
     * Note: this callback can be set to NULL.
     * 
     * @param video_link frame grabber that this notification is for. In case
     *                  its a single ported hardware, video_link is always 0
     */
    void (*new_data_available)(unsigned char video_link);

    /**
     * Notify the encoder that the grabber is now available again. In contrast 
     * to new_diffmap_available() and new_data_available() this function is
     * called only once to indicate that the grabber as a whole is available
     * again. This may for example happen after a suspend.
     *
     * @param video_link frame grabber that this notification is for. In case
     *                  its a single ported hardware, video_link is always 0
     */
    void (*grabber_available)(unsigned char video_link);

    /**
     * Notify the encoder of a new color translation setting of the grabber
     * hardware. This does not affect the color reduction/compression in the
     * encoder.
     * 
     * @param video_link frame grabber that this setting is for. In case
     *                   its a single ported hardware, video_link is always 0
     * @param fb_c_info  the struct with the new color information
     */
    void (*set_color_translation)(unsigned char video_link, fb_color_info_t *fb_c_info);
    
    /**
     * Notify the encoder of a new framebuffer format of the grabber. The
     * encoder has to change his visualization accordingly.
     * 
     * @param video_link frame grabber that this setting is for. In case
     *                 its a single ported hardware, video_link is always 0
     */
    void (*set_framebuffer_format)(unsigned char video_link, fb_format_info_t *fb_f_info);
    
    /**
     * Get the current exclusive session from the encoder. If a session
     * is retrieved, only the user with the exclusive session will get
     * a valid preview window.
     * 
     * @param video_link frame grabber that this query is for. In case
     *                 its a single ported hardware, video_link is always 0
     */
    eric_session_int_id_t (*get_excl_session)(unsigned char video_link);

    /**
     * Internal use only. Do not change this.
     */
    struct pp_grab_encoder_desc_s* next;
} pp_grab_encoder_desc_t;




/**
 * Initialize the grab library and internal structures,
 * must be done before vsc initialization.
 *
 * @return     0, if initialization succeeded,
 *            -1, if initialization failed 
 */
int  pp_grab_init(void);

/**
 * Cleanup any references the grabber has
 * into the lower level layers.
 */
void pp_grab_cleanup(void);

/**
 * Starts independend grabber threads and return,
 * called after all initializations are done
 */
void pp_grab_start(void);

/**
 * Connect an encoder implementation to the grabber. This
 * function may only be called once to initially add encoder
 * callbacks. Several encoders can be registered in parallel
 * by calling this function multiple times. Encoders cannot
 * be disconnected.
 * 
 * @param encoder  encoder decription structure containing
 *                 the information and callbacks of the actual encoder
 * @return         PP_SUC if successful
 *                 PP_ERR on error
 */
int pp_grab_connect_encoder(pp_grab_encoder_desc_t* encoder);


/**
 * Connect a client to the grabber. This client can have his own
 * settings and individual encoded framebuffer(vsc2 only).
 * 
 * @param video_link frame grabber that this client will operate on.
 * @param id       grabber type
 * @return         a client structure that identifies this client
 */
pp_grab_client_t* pp_grab_new_client(unsigned char video_link, pp_grab_mem_desc_id_t id);

/**
 * Remove a client from the grabber.
 * 
 * @param client  the client structure that identifies the client.
 */
void pp_grab_remove_client(pp_grab_client_t *client);

/**
 * Set the encoding type for the specified client.
 * 
 * @param client        client for which to set the encoding
 * @param encoding      encoding type to set
 * @param encoding_tag  encoding tag to set
 * @return              0 on success
 */
int pp_grab_set_encoding(pp_grab_client_t *client, vsc_encoding_desc_t encoding, u_int32_t encoding_tag);

/**
 * Request data for a certain region from the grabber. The data can
 * be accessed through the client structure after this call returns.
 * 
 * @param client  client that requests the data
 * @param flags   flags to apply when gathering the grabber data.
 * @param req     Regions that are requested. If GRAB_REQ_FLAGS_IGNORE_DIFFMAP
 *                is not set, the request will be automatically minimized using
 *                the current diffmap for this client.
 * @param ack     Regions that were successfully grabbed.
 * @param remain  Remaining regions that were not grabbed
 * 
 * @return        0 on success
 *                -1 on error
 */
int pp_grab_request_region(pp_grab_client_t *client, pp_grab_req_flags_t flags,
			   RegionRec *req, RegionRec *ack, RegionRec *remain);

/**
 * Request data for a certain region from the grabber. Alternative 
 * function with different signature.
 * 
 * @param client  client that requests the data
 * @param flags   flags to apply when gathering the grabber data.
 * @param x, y    region start coordinates
 * @param w, h    region width and heigth
 * 
 * @return        0 on success
 *                -1 on error
 */
int pp_grab_request_rect(pp_grab_client_t *client, pp_grab_req_flags_t flags,
			 u_short x, u_short y, u_short w, u_short h);

/**
 * Release the framegrabber buffer for the specified client.
 * 
 * @param client  the client with the framebuffer to release
 */
void pp_grab_release_buffer(pp_grab_client_t *client);


/**
 * Get the color info for a certain video_link.
 * 
 * @param video_link frame grabber to get the color info for 
 * @param fb_c_info  user supplied memory to store the color info
 * @return           0 for success
 */
int pp_grab_get_color_info(unsigned char video_link, fb_color_info_t *fb_c_info);

/**
 * Get the format info for a certain video_link.
 * 
 * @param video_link frame grabber to get the format info for 
 * @param fb_f_info  user supplied memory to store the format info
 * @return           0 on success
 *                   -1 on error
 */
int pp_grab_get_format_info(unsigned char video_link, fb_format_info_t *fb_f_info);

/**
 * Synchronize the frame buffer with the encoder. This is done via simple mutexes
 * and it prevents reading from frame buffer while content is about to change.
 * This results in an enhanced picture qualitiy.
 *
 * @param video_link frame grabber to use, in case its a single
 *                 ported hardware, video_link is always 0
 * @param action   locks the frame buffer if GRAB_SYNC_LOCK and unlocks it if GRAB_SYNC_UNLOCK
 */
void pp_grab_sync_frame_buffer(unsigned char video_link, pp_grab_sync_lock_t action);


/**
 * Refresh the picture, assume the whole screen has changed
 *
 * @param video_link frame grabber to use, in case its a single
 *                 ported hardware, video_link is always 0
 */
void pp_grab_refresh(unsigned char video_link);

/**
 * used to stop the grabber for the following things:
 * VSC: claim grabber during auto phase adjustment
 *      and while resetting the VSC
 * 
 * @param video_link frame grabber to suspend or resume
 * @param action   action to perform on the video_link 
 */
void pp_grab_suspend(unsigned char video_link, pp_grab_suspend_action_t action);

/**
 * Create a screenshot for preview of the framebuffer,
 * after the routine returns the screenshot is located in
 * /lib/webpages/screenshot.jpg
 *
 * @param video_link frame grabber to use, in case its a single
 *                  ported hardware, video_link is always 0
 * @param webs_user the webs user that this snapshot is created for
 *                  (dont show exclusive sessions for non-session user)
 * @param width     returns the width of the created snapshot image
 * @param height    returns the height of the created snapshot image
 *
 * @return PP_SUC if successful
 *         or one of the special codes below
 */
int pp_grab_create_snapshot(unsigned char video_link, const char *webs_user, u_short *width, u_short *height);

#define PP_GRAB_SCREENSHOT_NO_SIGNAL    1
#define PP_GRAB_SCREENSHOT_NOT_POSSIBLE 2
#define PP_GRAB_SCREENSHOT_UNKNOWN_MODE 3
#define PP_GRAB_SCREENSHOT_EXCL_ACCESS  4
#define PP_GRAB_SCREENSHOT_ERROR        5

// some non-encoder functions, used for vsc, debug and cat

/**
 * used by vsc thread to announce pending
 * mode change, we have to stop grabbing during that
 */
void pp_grab_fbchange_start( u_char video_link);

/**
 * used by vsc thread to announce pending
 * mode change, we have to stop grabbing during that
 */
void pp_grab_fbchange_ready( u_char video_link);

int pp_grab_fd( u_char video_link);


void pp_grab_debug_use_diffmap(unsigned char use_diffmap);
inline int pp_grab_errno_base(void);



#if defined(PP_FEAT_CAT) || defined(PP_FEAT_KITTY_CAT)
/**
 * diffmap consumer
 * allows to register callback to process diffmap information
 * currently only needed to enqueue diffmaps in cursor action tracking CAT
 */
typedef void (*pp_grab_diffmap_consumer_cb_t)(const void *target, 
                                              u_int32_t *diffmap);
 
typedef struct {
    void *                              target;
    pp_grab_diffmap_consumer_cb_t       dmc_cb;
} pp_grab_diffmap_consumer_t;

void pp_grab_register_diffmap_consumer(pp_grab_diffmap_consumer_t dmc);
void pp_grab_unregister_diffmap_consumer(void);
#endif /* PP_FEAT_CAT || PP_FEAT_KITTY_CAT */




/**
 * region functions, orginally from RFB lib
 * temporarily parked here until we merge with the adapter branch
 */

#ifndef TRUE
#define TRUE 1
#define FALSE 0
#endif

typedef int Bool;

#define NullBox ((BoxPtr)0)

/* Return values from RectIn() */
#define rgnOUT 0
#define rgnIN  1
#define rgnPART 2

#define NullRegion ((RegionPtr)0)

extern BoxRec miEmptyBox;
extern RegDataRec miEmptyData;

#define REGION_NIL(reg) ((reg)->data && !(reg)->data->numRects)
#define REGION_NUM_RECTS(reg) ((reg)->data ? (reg)->data->numRects : 1)
#define REGION_SIZE(reg) ((reg)->data ? (reg)->data->size : 0)
#define REGION_RECTS(reg) ((reg)->data ? (BoxPtr)((reg)->data + 1) \
		                                       : &(reg)->extents)
#define REGION_BOXPTR(reg) ((BoxPtr)((reg)->data + 1))
#define REGION_BOX(reg,i) (&REGION_BOXPTR(reg)[i])
#define REGION_TOP(reg) REGION_BOX(reg, (reg)->data->numRects)
#define REGION_END(reg) REGION_BOX(reg, (reg)->data->numRects - 1)
#define REGION_SZOF(n) (sizeof(RegDataRec) + ((n) * sizeof(BoxRec)))

#define REGION_COPY(dst, src) pp_grab_miRegionCopy(dst, src)
#define REGION_INTERSECT(newReg, reg1, reg2) pp_grab_miIntersect(newReg, reg1, reg2)
#define REGION_UNION(newReg, reg1, reg2) pp_grab_miUnion(newReg, reg1, reg2)
#define REGION_SUBTRACT(newReg, reg1, reg2) pp_grab_miSubtract(newReg, reg1, reg2)
#define REGION_INVERSE(newReg, reg1, invRect) pp_grab_miInverse(newReg, reg1, invRect)
#define REGION_TRANSLATE(pReg, x, y) pp_grab_miTranslateRegion(pReg, x, y)
#define RECT_IN_REGION(pReg, prect) pp_grab_miRectIn(pReg, prect)
#define POINT_IN_REGION(pReg, x, y, prect) pp_grab_miPointInRegion(pReg, x, y, prect)
#define REGION_APPEND(dstrgn, rgn) pp_grab_miRegionAppend(dstrgn, rgn)
#define REGION_INIT(pReg, rect, size) pp_grab_miRegionInit(pReg, rect, size)
#define REGION_UNINIT(pReg) pp_grab_miRegionUninit(pReg)
#define REGION_RESET(pReg, pBox) pp_grab_miRegionReset(pReg, pBox)
#define REGION_NOTEMPTY(pReg) pp_grab_miRegionNotEmpty(pReg)
#define REGION_EMPTY(pReg) pp_grab_miRegionEmpty(pReg)
#define REGION_EXTENTS(pReg) pp_grab_miRegionExtents(pReg)
#define REGION_VALIDATE(badreg, pOverlap) pp_grab_miRegionValidate(badreg, pOverlap)

extern void pp_grab_miRegionInit(RegionPtr pReg, BoxPtr rect, int size);
extern void pp_grab_miRegionUninit(RegionPtr pReg);
extern Bool pp_grab_miRegionCopy(RegionPtr dst, RegionPtr src);
extern Bool pp_grab_miIntersect(RegionPtr newReg, RegionPtr reg1, RegionPtr reg2);
extern Bool pp_grab_miUnion(RegionPtr newReg, RegionPtr reg1, RegionPtr reg2);
extern Bool pp_grab_miRegionAppend(RegionPtr dstrgn, RegionPtr rgn);
extern Bool pp_grab_miSubtract(RegionPtr regD, RegionPtr regM, RegionPtr regS);
extern Bool pp_grab_miInverse(RegionPtr newReg, RegionPtr reg1, BoxPtr invRect);
extern int pp_grab_miRectIn(RegionPtr region, BoxPtr prect);
extern void pp_grab_miTranslateRegion(RegionPtr pReg, int x, int y);
extern void pp_grab_miRegionReset(RegionPtr pReg, BoxPtr pBox);
extern Bool pp_grab_miPointInRegion(RegionPtr pReg, int x, int y, BoxPtr box);
extern Bool pp_grab_miRegionNotEmpty(RegionPtr pReg);
extern void pp_grab_miRegionEmpty(RegionPtr pReg);
extern BoxPtr pp_grab_miRegionExtents(RegionPtr pReg);
extern Bool pp_grab_miRegionValidate(RegionPtr badreg, Bool *pOverlap);
extern void pp_grab_miRegionDump(RegionPtr pReg);


#ifdef __cplusplus
}
#endif

#endif /* _LIBPP_GRAB_H */
