#include <sys/types.h>
#include <pp/i2c.h>
#include <pp/hal_kx2.h>
#include <pp/base.h>
#include "kx2.h"

static u_char i2c_handle;
static u_char user_leds = KX2_DUAL_POWER_MASK;

static pthread_mutex_t user_leds_and_pcf_mtx = PTHREAD_ERRORCHECK_MUTEX_INITIALIZER_NP;

int pcf8574_init(void) {
    int error;
    i2c_handle = pp_i2c_open("ibm-0", &error);
    if (error != PP_I2C_NO_ERROR) return PP_ERR;

    
    pp_hal_kx2_set_user_led(PP_HAL_KX2_USER_LED_OFF,
			    PP_HAL_KX2_USER_LED_OFF,
			    PP_HAL_KX2_USER_LED_OFF,
			    PP_HAL_KX2_USER_LED_OFF);

    return PP_SUC;
}

int pp_hal_kx2_set_power_led(int blue, int red) {
    int ret = PP_ERR;
    int error;
    MUTEX_LOCK(&user_leds_and_pcf_mtx);
    if (blue) user_leds |= KX2_POWER_LED_BLUE; else user_leds &= ~KX2_POWER_LED_BLUE;
    if (red) user_leds |= KX2_POWER_LED_RED; else user_leds &= ~KX2_POWER_LED_RED;
    pp_i2c_tx_byte(i2c_handle, KX2_PCF8574_I2C_ADDR, user_leds, &error);
    if (error == PP_I2C_NO_ERROR) ret = PP_SUC;
    MUTEX_UNLOCK(&user_leds_and_pcf_mtx);
    return ret;
}

int pp_hal_kx2_get_power_status(void) {
    int error;
    int ret = 0;

    MUTEX_LOCK(&user_leds_and_pcf_mtx);
    u_char flags = pp_i2c_rx_byte(i2c_handle, KX2_PCF8574_I2C_ADDR, &error);
    MUTEX_UNLOCK(&user_leds_and_pcf_mtx);

    if (error != PP_I2C_NO_ERROR) {
	ret = -1;
	goto bail;
    }
    if (flags & KX2_DUAL_POWER_1) ret |= PP_HAL_KX2_DUAL_POWER_1;
    if (flags & KX2_DUAL_POWER_2) ret |= PP_HAL_KX2_DUAL_POWER_2;
    
bail:
    return ret;
}

// this function will change the global user_leds (atomic)
// user_leds_and_pcf_mtx must be hold when calling this function
static void set_user_led(u_char status, u_char led_bit);

static void set_user_led(u_char status, u_char led_bit) {
    switch (status) {
	case PP_HAL_KX2_USER_LED_OFF:
	    user_leds |= led_bit;
	    break;
	case PP_HAL_KX2_USER_LED_ON:
	    user_leds &= ~led_bit;
	    break;
	case PP_HAL_KX2_USER_LED_OLD_VALUE: /* fall through */
	default:
	    break;
    }
}

int pp_hal_kx2_set_user_led(u_char user1, u_char user2, u_char user3, u_char user4) {
    int ret = PP_ERR;
    int error;
   
    MUTEX_LOCK(&user_leds_and_pcf_mtx);
    set_user_led(user1, KX2_USER_LED_1); 
    set_user_led(user2, KX2_USER_LED_2); 
    set_user_led(user3, KX2_USER_LED_3); 
    set_user_led(user4, KX2_USER_LED_4); 
    
    pp_i2c_tx_byte(i2c_handle, KX2_PCF8574_I2C_ADDR, user_leds, &error);
    if (error == PP_I2C_NO_ERROR) ret = PP_SUC;
    
    MUTEX_UNLOCK(&user_leds_and_pcf_mtx);
    return ret;
}

int pp_hal_kx2_set_user_led_by_video_link(int videolink /* starts from 0 */, u_char status)
{
    int nRet = 0 ;
    switch ( videolink ) {
	case 0 :
	    nRet = pp_hal_kx2_set_user_led( status,PP_HAL_KX2_USER_LED_OLD_VALUE, PP_HAL_KX2_USER_LED_OLD_VALUE, PP_HAL_KX2_USER_LED_OLD_VALUE);
	    break;
	case 1 :
	    nRet = pp_hal_kx2_set_user_led( PP_HAL_KX2_USER_LED_OLD_VALUE,status, PP_HAL_KX2_USER_LED_OLD_VALUE, PP_HAL_KX2_USER_LED_OLD_VALUE);
                  break;
	case 2 :
	    nRet = pp_hal_kx2_set_user_led( PP_HAL_KX2_USER_LED_OLD_VALUE, PP_HAL_KX2_USER_LED_OLD_VALUE,status,PP_HAL_KX2_USER_LED_OLD_VALUE);
	    break;
	case 3 :
	    nRet = pp_hal_kx2_set_user_led( PP_HAL_KX2_USER_LED_OLD_VALUE, PP_HAL_KX2_USER_LED_OLD_VALUE, PP_HAL_KX2_USER_LED_OLD_VALUE, status);
                  break;
	default:
	    break;    
    	}
    return nRet;
}
void pcf8574_cleanup(void) {
    pp_i2c_close(i2c_handle);
}

#if 0
// test code
    // test
    printf("dual_power status: %x\n", pp_hal_kx2_get_power_status());

    printf("set blue\n");
    pp_hal_kx2_set_power_led(1,0);
    sleep(1);
    printf("set red\n");
    pp_hal_kx2_set_power_led(0,1);
    sleep(1);
    printf("set both\n");
    pp_hal_kx2_set_power_led(1,1);
    sleep(1);
     pp_hal_kx2_set_user_led(PP_HAL_KX2_USER_LED_OFF,
			    PP_HAL_KX2_USER_LED_ON,
			    PP_HAL_KX2_USER_LED_OFF,
			    PP_HAL_KX2_USER_LED_OFF);
    sleep(1);
#if 1
      pp_hal_kx2_set_user_led(PP_HAL_KX2_USER_LED_OLD_VALUE,
			    PP_HAL_KX2_USER_LED_OLD_VALUE,
			    PP_HAL_KX2_USER_LED_OFF,
			    PP_HAL_KX2_USER_LED_ON);
#endif
    printf("dual_power status: %x\n", pp_hal_kx2_get_power_status());

    pp_hal_kx2_buzzer(1);
    sleep(1);
    pp_hal_kx2_buzzer(0);
 
    pp_hal_kx2_gbit_set_interface(PP_HAL_KX2_GBIT_LAN2);
    sleep(10);
    pp_hal_kx2_gbit_set_interface(PP_HAL_KX2_GBIT_LAN1);
    sleep(10);

#endif	


