/**
 * rpc.c
 * 
 * RPC specific hardware settings
 *
 * (c) 2006 Peppercon AG, tbr@peppercon.de tweb@raritan.com
 */

/* system includes */

/* firmware includes */
#include <pp/base.h>
#include <pp/cfg.h>
#include <pp/hal_rpc.h>

/* local includes */
#include "rpc.h"

/* every model will have circuit breaker configured as *
 * circuit breaker sensor is a 'virtual' sensor only   */
#define RPC_4x3  "rpc_4x3"   /*  4 relays per board x3 */
#define RPC_4x5  "rpc_4x5"   /*  4 relays per board x5 */
#define RPC_8x1  "rpc_8x1"   /*  8 relays per board x1 */
#define RPC_8x2  "rpc_8x2"   /*  8 relays per board x2 */
#define RPC_10x2 "rpc_10x2"  /* 10 relays per board x2 */

#define CURR_MAX_0U_BOARD 160
#define CURR_MAX_xU_BOARD 200

pp_hal_rpc_model_t pp_hal_rpc_models[] = {
    { .model = "PCR8-15",
      .no_outlets=8,
      .no_rows = 1,
      .no_outlets_per_board = 8,
      .controller = RELAIS_BOARD_CONTROLLER_5_3,
      .curr_max = 150,
      .curr_max_board = CURR_MAX_xU_BOARD,
      .topo=RPC_8x1 },
    { .model = "PCR8A-16",
      .no_outlets=8,
      .no_rows = 1,
      .no_outlets_per_board = 8,
      .controller = RELAIS_BOARD_CONTROLLER_5_3,
      .curr_max = 160,
      .curr_max_board = CURR_MAX_xU_BOARD,
      .topo=RPC_8x1 },
    { .model = "PCR8A-20L6",
      .no_outlets=8,
      .no_rows = 1,
      .no_outlets_per_board = 8,
      .controller = RELAIS_BOARD_CONTROLLER_5_3,
      .curr_max = 200,
      .curr_max_board = CURR_MAX_xU_BOARD,
      .topo=RPC_8x1 },
    { .model = "PCS12-20",
      .no_outlets=12,
      .no_rows = 12,
      .no_outlets_per_board = 4,
      .controller = RELAIS_BOARD_CONTROLLER_4,
      .curr_max = 200,
      .curr_max_board = CURR_MAX_0U_BOARD,
      .topo=RPC_4x3 },
    { .model = "PCS12-20L",
      .no_outlets=12,
      .no_rows = 12,
      .no_outlets_per_board = 4,
      .controller = RELAIS_BOARD_CONTROLLER_4,
      .curr_max = 200,
      .curr_max_board = CURR_MAX_0U_BOARD,
      .topo=RPC_4x3 },
    { .model = "PCS12A-16",
      .no_outlets=12,
      .no_rows = 12,
      .no_outlets_per_board = 4,
      .controller = RELAIS_BOARD_CONTROLLER_4,
      .curr_max = 160,
      .curr_max_board = CURR_MAX_0U_BOARD,
      .topo=RPC_4x3 },
    { .model = "PCS12-30L",
      .no_outlets=12,
      .no_rows = 12,
      .no_outlets_per_board = 4,
      .controller = RELAIS_BOARD_CONTROLLER_4,
      .curr_max = 300,
      .curr_max_board = CURR_MAX_0U_BOARD,
      .topo=RPC_4x3 },
    { .model = "PCR2X8-15",
      .no_outlets=16,
      .no_rows = 2,
      .no_outlets_per_board = 8,
      .controller = RELAIS_BOARD_CONTROLLER_5_3,
      .curr_max = 150,
      .curr_max_board = CURR_MAX_xU_BOARD,
      .topo=RPC_8x2 },
    { .model = "PCR2X8-16",
      .no_outlets=16,
      .no_rows = 2,
      .no_outlets_per_board = 8,
      .controller = RELAIS_BOARD_CONTROLLER_5_3,
      .curr_max = 160,
      .curr_max_board = CURR_MAX_xU_BOARD,
      .topo=RPC_8x2 },
    { .model = "PCRA2X8",
      .no_outlets=16,
      .no_rows = 2,
      .no_outlets_per_board = 8,
      .controller = RELAIS_BOARD_CONTROLLER_5_3,
      .curr_max = 200,
      .curr_max_board = CURR_MAX_xU_BOARD,
      .topo=RPC_8x2 },
    { .model = "PCS20-20",
      .no_outlets=20,
      .no_rows = 20,
      .no_outlets_per_board = 4,
      .controller = RELAIS_BOARD_CONTROLLER_4,
      .curr_max = 200,
      .curr_max_board = CURR_MAX_0U_BOARD,
      .topo=RPC_4x5 },
    { .model = "PCS20-20L",
      .no_outlets=20,
      .no_rows = 20,
      .no_outlets_per_board = 4,
      .controller = RELAIS_BOARD_CONTROLLER_4,
      .curr_max = 200,
      .curr_max_board = CURR_MAX_0U_BOARD,
      .topo=RPC_4x5 },
    { .model = "PCS20-30L",
      .no_outlets=20,
      .no_rows = 20,
      .no_outlets_per_board = 4,
      .controller = RELAIS_BOARD_CONTROLLER_4,
      .curr_max = 300,
      .curr_max_board = CURR_MAX_0U_BOARD,
      .topo=RPC_4x5 },
    { .model = "PCS20A-32",
      .no_outlets=20,
      .no_rows = 20,
      .no_outlets_per_board = 4,
      .controller = RELAIS_BOARD_CONTROLLER_4,
      .curr_max = 320,
      .curr_max_board = CURR_MAX_0U_BOARD,
      .topo=RPC_4x5 },
    { .model = "PCS20A-30L6",
      .no_outlets=20,
      .no_rows = 20,
      .no_outlets_per_board = 4,
      .controller = RELAIS_BOARD_CONTROLLER_4,
      .curr_max = 300,
      .curr_max_board = CURR_MAX_0U_BOARD,
      .topo=RPC_4x5 },
    { .model = "PCS20A-16",
      .no_outlets=20,
      .no_rows = 20,
      .no_outlets_per_board = 4,
      .controller = RELAIS_BOARD_CONTROLLER_4,
      .curr_max = 160,
      .curr_max_board = CURR_MAX_0U_BOARD,
      .topo=RPC_4x5 },
    { .model = "PCS20A-20L6",
      .no_outlets=20,
      .no_rows = 20,
      .no_outlets_per_board = 4,
      .controller = RELAIS_BOARD_CONTROLLER_4,
      .curr_max = 200,
      .curr_max_board = CURR_MAX_0U_BOARD,
      .topo=RPC_4x5 },
    { .model = "PCR20-20",
      .no_outlets=20,
      .no_rows = 2,
      .no_outlets_per_board = 10,
      .controller = RELAIS_BOARD_CONTROLLER_5,
      .curr_max = 200,
      .curr_max_board = CURR_MAX_xU_BOARD,
      .topo=RPC_10x2 },
    { .model = "PCR20-20L",
      .no_outlets=20,
      .no_rows = 2,
      .no_outlets_per_board = 10,
      .controller = RELAIS_BOARD_CONTROLLER_5,
      .curr_max = 200,
      .curr_max_board = CURR_MAX_xU_BOARD,
      .topo=RPC_10x2 },  
    { .model = "PCR20-30L",
      .no_outlets=20,
      .no_rows = 2,
      .no_outlets_per_board = 10,
      .controller = RELAIS_BOARD_CONTROLLER_5,
      .curr_max = 300,
      .curr_max_board = CURR_MAX_xU_BOARD,
      .topo=RPC_10x2 },
    { .model = "PCR20A-16",
      .no_outlets=20,
      .no_rows = 2,
      .no_outlets_per_board = 10,
      .controller = RELAIS_BOARD_CONTROLLER_5,
      .curr_max = 160,
      .curr_max_board = CURR_MAX_xU_BOARD,
      .topo=RPC_10x2 },
    { .model = "PCR20A-32",
      .no_outlets=20,
      .no_rows = 2,
      .no_outlets_per_board = 10,
      .controller = RELAIS_BOARD_CONTROLLER_5,
      .curr_max = 320,
      .curr_max_board = CURR_MAX_xU_BOARD,
      .topo=RPC_10x2 },
};

int pp_hal_rpc_models_sz = sizeof(pp_hal_rpc_models) /
                           sizeof(pp_hal_rpc_model_t);
                               
static pp_hal_rpc_model_t *current_model = NULL;

int rpc_init(void) {
    char *model = NULL;
    int ret, idx;
    
    if(PP_ERR != (ret = pp_cfg_get_nodflt(&model, "device.model")) &&
       PP_ERR != (idx = pp_hal_rpc_model_idx_by_name(model))) {
        current_model = &pp_hal_rpc_models[idx];
    }
    
    free(model);
    
    return ret;
}

int pp_hal_rpc_model_idx_by_name(const char* mname) {
    int i;
    for (i = 0; i < pp_hal_rpc_models_sz; ++i) {
	if (!strcmp(pp_hal_rpc_models[i].model, mname))
	    return i;
    }
    return -1;
}

inline const char* pp_hal_rpc_get_model(void) {
    if(!current_model) {
        /* no model initialized */
        return NULL;
    }
    
    return current_model->model;
}

inline int pp_hal_rpc_get_no_outlets(void) {
    if(!current_model) {
        /* no model initialized */
        return 0;
    }
    
    return current_model->no_outlets;
}

inline int pp_hal_rpc_get_no_rows(void) {
    if(!current_model) {
        /* no model initialized */
        return 0;
    }
    
    return current_model->no_rows;
}

inline int pp_hal_rpc_get_no_outlets_per_board(void) {
    if(!current_model) {
        /* no model initialized */
        return 0;
    }
    
    return current_model->no_outlets_per_board;
}

inline pp_hal_rpc_relais_board_controller_t pp_hal_rpc_get_controller(void) {
    if(!current_model) {
        /* no model initialized */
        return RELAIS_BOARD_CONTROLLER_UNDEF;
    }
    
    return current_model->controller;
}

inline int pp_hal_rpc_get_curr_max(void) {
    if(!current_model) {
        /* no model initialized */
        return 0;
    }
    
    return current_model->curr_max;
}

inline int pp_hal_rpc_get_curr_max_board(void) {
    if(!current_model) {
        /* no model initialized */
        return 0;
    }
    
    return current_model->curr_max_board;
}

inline const char* pp_hal_rpc_get_topo(void) {
    if(!current_model) {
        /* no model initialized */
        return NULL;
    }
    
    return current_model->topo;
}

