/*
 * Copyright (c) 2003 Sun Microsystems, Inc.  All Rights Reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 
 * Redistribution of source code must retain the above copyright
 * notice, this list of conditions and the following disclaimer.
 * 
 * Redistribution in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the distribution.
 * 
 * Neither the name of Sun Microsystems, Inc. or the names of
 * contributors may be used to endorse or promote products derived
 * from this software without specific prior written permission.
 * 
 * This software is provided "AS IS," without a warranty of any kind.
 * ALL EXPRESS OR IMPLIED CONDITIONS, REPRESENTATIONS AND WARRANTIES,
 * INCLUDING ANY IMPLIED WARRANTY OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE OR NON-INFRINGEMENT, ARE HEREBY EXCLUDED.
 * SUN MICROSYSTEMS, INC. ("SUN") AND ITS LICENSORS SHALL NOT BE LIABLE
 * FOR ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING
 * OR DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.  IN NO EVENT WILL
 * SUN OR ITS LICENSORS BE LIABLE FOR ANY LOST REVENUE, PROFIT OR DATA,
 * OR FOR DIRECT, INDIRECT, SPECIAL, CONSEQUENTIAL, INCIDENTAL OR
 * PUNITIVE DAMAGES, HOWEVER CAUSED AND REGARDLESS OF THE THEORY OF
 * LIABILITY, ARISING OUT OF THE USE OF OR INABILITY TO USE THIS SOFTWARE,
 * EVEN IF SUN HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGES.
 * 
 * You acknowledge that this software is not designed or intended for use
 * in the design, construction, operation or maintenance of any nuclear
 * facility.
 */

#ifndef IPMI_INTF_H
#define IPMI_INTF_H

#include <ipmitool/ipmi.h>
#include <ipmitool/ipmi_constants.h>

#include <sys/types.h>
#ifdef WIN32
#include <winsock2.h>
#include <windows.h>
#else
#include <arpa/inet.h>
#include <sys/socket.h>
#include <netinet/in.h>
#endif

#include <pp/ipmi.h>

/*
 * An enumeration that describes every possible session state for
 * an IPMIv2 / RMCP+ session.
 */
enum LANPLUS_SESSION_STATE {
	LANPLUS_STATE_PRESESSION = 0,
	LANPLUS_STATE_OPEN_SESSION_SENT,
	LANPLUS_STATE_OPEN_SESSION_RECEIEVED,
	LANPLUS_STATE_RAKP_1_SENT,
	LANPLUS_STATE_RAKP_2_RECEIVED,
	LANPLUS_STATE_RAKP_3_SENT,
	LANPLUS_STATE_ACTIVE,
	LANPLUS_STATE_CLOSE_SENT,
};


#define IPMI_AUTHCODE_BUFFER_SIZE 20
#define IPMI_SIK_BUFFER_SIZE      20
#define IPMI_KG_BUFFER_SIZE       21 /* key plus null byte */

struct ipmi_session {
	uint8_t hostname[64];
	uint8_t username[16];
	uint8_t authcode[IPMI_AUTHCODE_BUFFER_SIZE + 1];
	char device[64];
	char ppscsi_product[64];
	uint8_t challenge[16];
	uint8_t authtype;
	uint8_t authtype_set;
#define IPMI_AUTHSTATUS_PER_MSG_DISABLED	0x10
#define IPMI_AUTHSTATUS_PER_USER_DISABLED	0x08
#define IPMI_AUTHSTATUS_NONNULL_USERS_ENABLED	0x04
#define IPMI_AUTHSTATUS_NULL_USERS_ENABLED	0x02
#define IPMI_AUTHSTATUS_ANONYMOUS_USERS_ENABLED	0x01
	uint8_t authstatus;
	uint8_t authextra;
	uint8_t privlvl;
	uint8_t cipher_suite_id;
	int password;
	int port;
	int active;
	int retry;

	uint32_t session_id;
	uint32_t in_seq;
	uint32_t out_seq;
	uint32_t timeout;

	struct sockaddr_in addr;
	socklen_t addrlen;

	/*
	 * This struct holds state data specific to IMPI v2 / RMCP+ sessions
	 */
	struct {
		enum LANPLUS_SESSION_STATE session_state;

		/* These are the algorithms agreed upon for the session */
		uint8_t requested_auth_alg;
		uint8_t requested_integrity_alg;
		uint8_t requested_crypt_alg;
		uint8_t auth_alg;
		uint8_t integrity_alg;
		uint8_t crypt_alg;
		uint8_t max_priv_level;

		uint32_t console_id;
		uint32_t bmc_id;

		/*
		 * Values required for RAKP mesages
		 */

		/* Random number generated byt the console */
		uint8_t console_rand[16]; 
		/* Random number generated by the BMC */
		uint8_t bmc_rand[16];

		uint8_t bmc_guid[16];
		uint8_t requested_role;   /* As sent in the RAKP 1 message */
		uint8_t rakp2_return_code;

		uint8_t sik[IPMI_SIK_BUFFER_SIZE]; /* Session integrity key */
		uint8_t kg[IPMI_KG_BUFFER_SIZE];   /* BMC key */
		uint8_t k1[20];   /* Used for Integrity checking? */
		uint8_t k2[20];   /* First 16 bytes used for AES  */
	} v2_data;


	/*
	 * This data is specific to the Serial Over Lan session
	 */
	struct {
		uint16_t max_inbound_payload_size;
		uint16_t max_outbound_payload_size;
		uint16_t port;
		uint8_t sequence_number;

		/*  This data describes the last SOL packet */
		uint8_t last_received_sequence_number;
		uint8_t last_received_byte_count;
		void (*sol_input_handler)(struct ipmi_rs * rsp);
	} sol_data;

    	// serial settings
    	struct {
		int port;
		int baud_rate;
		char parity[2];
		int bits;
		int stop2;
		int hwf;
		int swf;
	} serial_settings;

};


struct ipmi_intf {
	char name[16];
	char desc[128];
	int fd;
	int opened;
	int abort;
	int noanswer;
	int cancel;
	void *intf_ctx; /* interface-specific context data */
   
        /* special values for loopi */
        int cmd_wfd;
        int rsp_rfd;
        int loopi_chan;

	struct ipmi_session * session;
	struct ipmi_oem_handle * oem;
	unsigned int my_addr;
	unsigned int bmc_addr;
	unsigned int target_addr;

	int (*setup)(struct ipmi_intf * intf, int * error);
	int (*open)(struct ipmi_intf * intf, int * error);
	void (*close)(struct ipmi_intf * intf, int force, int * error);
	void (*do_abort)(struct ipmi_intf * intf, int * error);
	struct ipmi_rs *(*sendrecv)(struct ipmi_intf * intf, struct ipmi_rq * req, int * error);
	int (*sendrsp)(struct ipmi_intf * intf, struct ipmi_rs * rsp, int * error);
	struct ipmi_rs *(*recv_sol)(struct ipmi_intf * intf, int * error);
	struct ipmi_rs *(*send_sol)(struct ipmi_intf * intf, struct ipmi_v2_payload * payload, int * error);
	int (*keepalive)(struct ipmi_intf * intf, int * error);
};

struct ipmi_intf * ipmi_intf_load(const char * name, int * error);
void ipmi_intf_print(void);

void ipmi_intf_session_set_hostname(struct ipmi_intf * intf, const char * hostname);
void ipmi_intf_session_set_device(struct ipmi_intf * intf, const char * device);
void ipmi_intf_session_set_ppscsi_product(struct ipmi_intf * intf, const char * ppscsi_product);
void ipmi_intf_session_set_username(struct ipmi_intf * intf, const char * username);
void ipmi_intf_session_set_password(struct ipmi_intf * intf, const char * password);
void ipmi_intf_session_set_privlvl(struct ipmi_intf * intf, uint8_t privlvl);
void ipmi_intf_session_set_cipher_suite_id(struct ipmi_intf * intf, uint8_t cipher_suite_id);
void ipmi_intf_session_set_port(struct ipmi_intf * intf, int port);
void ipmi_intf_session_set_authtype(struct ipmi_intf * intf, uint8_t authtype);
void ipmi_intf_session_set_timeout(struct ipmi_intf * intf, uint32_t timeout);
void ipmi_intf_session_set_retry(struct ipmi_intf * intf, int retry);
void ipmi_intf_session_set_serial(struct ipmi_intf * intf, int port, int baud_rate, char *parity, int bits, int stop2, int hwf, int swf);
void ipmi_cleanup(struct ipmi_intf * intf);

#endif /* IPMI_INTF_H */

