/*
 * Copyright (c) 2003 Sun Microsystems, Inc.  All Rights Reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 
 * Redistribution of source code must retain the above copyright
 * notice, this list of conditions and the following disclaimer.
 * 
 * Redistribution in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the distribution.
 * 
 * Neither the name of Sun Microsystems, Inc. or the names of
 * contributors may be used to endorse or promote products derived
 * from this software without specific prior written permission.
 * 
 * This software is provided "AS IS," without a warranty of any kind.
 * ALL EXPRESS OR IMPLIED CONDITIONS, REPRESENTATIONS AND WARRANTIES,
 * INCLUDING ANY IMPLIED WARRANTY OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE OR NON-INFRINGEMENT, ARE HEREBY EXCLUDED.
 * SUN MICROSYSTEMS, INC. ("SUN") AND ITS LICENSORS SHALL NOT BE LIABLE
 * FOR ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING
 * OR DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.  IN NO EVENT WILL
 * SUN OR ITS LICENSORS BE LIABLE FOR ANY LOST REVENUE, PROFIT OR DATA,
 * OR FOR DIRECT, INDIRECT, SPECIAL, CONSEQUENTIAL, INCIDENTAL OR
 * PUNITIVE DAMAGES, HOWEVER CAUSED AND REGARDLESS OF THE THEORY OF
 * LIABILITY, ARISING OUT OF THE USE OF OR INABILITY TO USE THIS SOFTWARE,
 * EVEN IF SUN HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGES.
 * 
 * You acknowledge that this software is not designed or intended for use
 * in the design, construction, operation or maintenance of any nuclear
 * facility.
 */

#include <stdlib.h>
#include <stdio.h>

#ifndef _WIN32
#include <inttypes.h>
#include <unistd.h>
#endif

#include <signal.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <errno.h>

#include <zlib.h>

#include <pp/ipmi.h>

#include <ipmitool/ipmi.h>
#include <ipmitool/helper.h>
#include <ipmitool/ipmi_print.h>

#ifdef WIN32
#define snprintf _snprintf
#define strncasecmp _strnicmp
#endif

uint32_t buf2long(uint8_t * buf)
{
	return (uint32_t)(buf[3] << 24 | buf[2] << 16 | buf[1] << 8 | buf[0]);
}

uint16_t buf2short(uint8_t * buf)
{
	return (uint16_t)(buf[1] << 8 | buf[0]);
}

const char * buf2str(uint8_t * buf, int len)
{
	static char str[1024];
	int i;

	if (len <= 0 || len > 1024)
		return NULL;

	memset(str, 0, 1024);

	for (i=0; i<len; i++)
		sprintf(str+i+i, "%2.2x", buf[i]);

	str[len*2] = '\0';

	return (const char *)str;
}

void printbuf(const uint8_t * buf, int len, const char * desc)
{
	int i;

	if (len <= 0)
		return;

	if (verbose < 1)
  		return;

	ipmi_printf("%s (%d bytes)\n", desc, len);
	for (i=0; i<len; i++) {
		if (((i%16) == 0) && (i != 0))
			ipmi_printf("\n");
		ipmi_printf(" %2.2x", buf[i]);
	}
	ipmi_printf("\n");
}

const char * val2str(uint16_t val, const struct valstr *vs)
{
	static char un_str[16];
	int i = 0;

	while (vs[i].str != NULL) {
		if (vs[i].val == val)
			return _(vs[i].str);
		i++;
	}

	memset(un_str, 0, sizeof(un_str));
	snprintf(un_str, sizeof(un_str), _("Unknown (0x%02x)"), val);

	return un_str;
}

uint16_t str2val(const char *str, const struct valstr *vs)
{
	int i = 0;

	while (vs[i].str != NULL) {
		if (strncasecmp(vs[i].str, str, __maxlen(str, vs[i].str)) == 0)
			return vs[i].val;
		i++;
	}

	return vs[i].val;
}

int get_cmdline_ipaddr(const char * arg, uint8_t * buf)
{
	uint32_t ip1, ip2, ip3, ip4;
	if (sscanf(arg, "%d.%d.%d.%d", &ip1, &ip2, &ip3, &ip4) != 4) {
		ipmi_printf("Invalid IP address: %s\n", arg);
		return -1;
	}
	buf[0] = (uint8_t)ip1;
	buf[1] = (uint8_t)ip2;
	buf[2] = (uint8_t)ip3;
	buf[3] = (uint8_t)ip4;
	return 0;
}

int get_cmdline_macaddr(const char * arg, uint8_t * buf)
{
	uint32_t m1, m2, m3, m4, m5, m6;
	if (sscanf(arg, "%02x:%02x:%02x:%02x:%02x:%02x",
		   &m1, &m2, &m3, &m4, &m5, &m6) != 6) {
		ipmi_printf("Invalid MAC address: %s\n", arg);
		return -1;
	}
	buf[0] = (uint8_t)m1;
	buf[1] = (uint8_t)m2;
	buf[2] = (uint8_t)m3;
	buf[3] = (uint8_t)m4;
	buf[4] = (uint8_t)m5;
	buf[5] = (uint8_t)m6;
	return 0;
}


/* ipmi_csum  -  calculate an ipmi checksum
 *
 * @d:		buffer to check
 * @s:		position in buffer to start checksum from
 */
uint8_t
ipmi_csum(uint8_t * d, int s)
{
	uint8_t c = 0;
	for (; s > 0; s--, d++)
		c += *d;
	return -c;
}

/* ipmi_open_file  -  safely open a file for reading or writing
 *
 * @file:	filename
 * @rw:		read-write flag, 1=write
 *
 * returns pointer to file handler on success
 * returns NULL on error
 */

#ifdef WIN32
# define lstat _stat
typedef struct _stat s_lstat;
typedef struct stat s_fstat;
# define S_ISREG(m) ((m) & _S_IFREG)
#else
typedef struct stat s_lstat;
typedef struct stat s_fstat;
#endif

FILE *
ipmi_open_file(const char * file, int rw, int binary)
{
	s_lstat st1;
	s_fstat st2;
	FILE * fp;

	/* verify existance */
	if (lstat(file, &st1) < 0) {
		if (rw) {
			/* does not exist, ok to create */
		    fp = fopen(file, binary ? "wb" : "w");
			if (fp == NULL) {
				ipmi_printf("Unable to open file %s "
					"for write\n", file);
				return NULL;
			}
			/* created ok, now return the descriptor */
			return fp;
		} else {
			ipmi_printf("File %s does not exist\n", file);
			return NULL;
		}
	}

#ifndef ENABLE_FILE_SECURITY
	if (!rw) {
		/* on read skip the extra checks */
	    fp = fopen(file, binary ? "rb" : "r");
		if (fp == NULL) {
			ipmi_printf("Unable to open file %s\n", file);
			return NULL;
		}
		return fp;
	}
#endif

	/* it exists - only regular files, not links */
	if (S_ISREG(st1.st_mode) == 0) {
		ipmi_printf("File %s has invalid mode: %d\n",
			file, st1.st_mode);
		return NULL;
	}

	/* allow only files with 1 link (itself) */
	if (st1.st_nlink != 1) {
		ipmi_printf("File %s has invalid link count: %d != 1\n",
		       file, (int)st1.st_nlink);
		return NULL;
	}

	fp = fopen(file, rw ? (binary ? "w+b" : "w+") : (binary ? "rb" : "r"));
	if (fp == NULL) {
		ipmi_printf("Unable to open file %s\n", file);
		return NULL;
	}

	/* stat again */
	if (fstat(fileno(fp), &st2) < 0) {
		ipmi_printf("Unable to stat file %s\n", file);
		fclose(fp);
		return NULL;
	}

	/* verify inode */
	if (st1.st_ino != st2.st_ino) {
		ipmi_printf("File %s has invalid inode: %d != %d\n",
			file, st1.st_ino, st2.st_ino);
		fclose(fp);
		return NULL;
	}

	/* verify owner */
	if (st1.st_uid != st2.st_uid) {
		ipmi_printf("File %s has invalid user id: %d != %d\n",
			file, st1.st_uid, st2.st_uid);
		fclose(fp);
		return NULL;
	}

	/* verify inode */
	if (st2.st_nlink != 1) {
		ipmi_printf("File %s has invalid link count: %d != 1\n",
			file, st2.st_nlink);
		fclose(fp);
		return NULL;
	}

	return fp;
}

int
ipmi_crc32_file(const char *file, unsigned int *crc)
{
    int ret = -1;
    uLong result = 0;
    size_t total = 0;

    FILE *fp = ipmi_open_file(file, 0, 1);
    if (!fp) goto bail;

    for (;;) {
	unsigned char buffer[512];
	ssize_t len = fread(buffer, 1, 512, fp);
	if (len > 0) {
	    total += len;
	    result = crc32(result, buffer, len);
	}
	if (len != 512) break;
    }

    if (!ferror(fp)) {
	*crc = result;
	ret = 0;
    }

bail:
    if (fp) fclose(fp);
    return ret;
}

