/*
 * Copyright (c) 2003 Sun Microsystems, Inc.  All Rights Reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 
 * Redistribution of source code must retain the above copyright
 * notice, this list of conditions and the following disclaimer.
 * 
 * Redistribution in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the distribution.
 * 
 * Neither the name of Sun Microsystems, Inc. or the names of
 * contributors may be used to endorse or promote products derived
 * from this software without specific prior written permission.
 * 
 * This software is provided "AS IS," without a warranty of any kind.
 * ALL EXPRESS OR IMPLIED CONDITIONS, REPRESENTATIONS AND WARRANTIES,
 * INCLUDING ANY IMPLIED WARRANTY OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE OR NON-INFRINGEMENT, ARE HEREBY EXCLUDED.
 * SUN MICROSYSTEMS, INC. ("SUN") AND ITS LICENSORS SHALL NOT BE LIABLE
 * FOR ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING
 * OR DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.  IN NO EVENT WILL
 * SUN OR ITS LICENSORS BE LIABLE FOR ANY LOST REVENUE, PROFIT OR DATA,
 * OR FOR DIRECT, INDIRECT, SPECIAL, CONSEQUENTIAL, INCIDENTAL OR
 * PUNITIVE DAMAGES, HOWEVER CAUSED AND REGARDLESS OF THE THEORY OF
 * LIABILITY, ARISING OUT OF THE USE OF OR INABILITY TO USE THIS SOFTWARE,
 * EVEN IF SUN HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGES.
 * 
 * You acknowledge that this software is not designed or intended for use
 * in the design, construction, operation or maintenance of any nuclear
 * facility.
 */

#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <time.h>

#ifdef WIN32
#include <pp/strstream.h>
#else
#include <pp/base.h>
#endif
#include <pp/vector.h>

#include <config.h>

#include <ipmitool/helper.h>
#include <ipmitool/bswap.h>
#include <ipmitool/ipmi.h>
#include <ipmitool/ipmi_intf.h>
#include <ipmitool/ipmi_bmc.h>
#include <ipmitool/ipmi_strings.h>
#include <ipmitool/ipmi_print.h>
#include <ipmitool/ipmi_error.h>
#include <ipmi_return.h>
#include <ipmi_params.h>

/* ipmi_bmc_reset  -  attempt to reset an MC
 *
 * @intf:	ipmi interface
 * @cmd:	reset command to send
 *              BMC_WARM_RESET or
 *              BMC_COLD_RESET
 *
 * returns 0 on success
 * returns -1 on error
 */
static int
ipmi_bmc_reset(struct ipmi_intf * intf, int cmd, int * error)
{
	struct ipmi_rs * rsp;
	struct ipmi_rq req;

	intf->open(intf, error);

	memset(&req, 0, sizeof(req));
	req.msg.netfn = IPMI_NETFN_APP;
	req.msg.cmd = cmd;
	req.msg.data_len = 0;

	if (cmd == BMC_COLD_RESET)
		intf->noanswer = 1;

	rsp = intf->sendrecv(intf, &req, error);

	if (cmd == BMC_COLD_RESET)
		intf->abort = 1;

	ipmi_printf("Sent %s reset command to MC\n",
	       (cmd == BMC_WARM_RESET) ? "warm" : "cold");
  
  	return 0;
}

/* ipmi_bmc_get_enables  -  print out MC enables
 *
 * @intf:	ipmi inteface
 *
 * returns 0 on success
 * returns -1 on error
 */
static int
ipmi_bmc_get_enables(struct ipmi_intf * intf, pp_ipmi_return_t *ipmi_ret, int * error)
{
	struct ipmi_rs * rsp;
	struct ipmi_rq req;

	if (ipmi_init_bmc_get_enables(ipmi_ret)) {
		ipmi_printf("Error: Could not allocate return structure.\n");
		return -1;
	}

	memset(&req, 0, sizeof(req));
	req.msg.netfn = IPMI_NETFN_APP;
	req.msg.cmd = BMC_GET_GLOBAL_ENABLES;

	rsp = intf->sendrecv(intf, &req, error);
	if (rsp == NULL) {
		ipmi_printf("Get Global Enables command failed\n");
		return -1;
	}
	if (rsp->ccode > 0) {
		ipmi_printf("BMC Get Global Enables command failed: %s\n",
		       val2str(rsp->ccode, completion_code_vals));
		return -1;
	}

	ipmi_ret->data.bmc_enables.receive_msg_int_enabled	= (rsp->data[0] & (1 << 0)) ? 1 : 0;
	ipmi_ret->data.bmc_enables.event_msg_buf_int_enabled	= (rsp->data[0] & (1 << 1)) ? 1 : 0;
	ipmi_ret->data.bmc_enables.event_msg_buf_enabled	= (rsp->data[0] & (1 << 2)) ? 1 : 0;
	ipmi_ret->data.bmc_enables.sel_enabled			= (rsp->data[0] & (1 << 3)) ? 1 : 0;
	ipmi_ret->data.bmc_enables.oem0_enabled			= (rsp->data[0] & (1 << 5)) ? 1 : 0;
	ipmi_ret->data.bmc_enables.oem1_enabled			= (rsp->data[0] & (1 << 6)) ? 1 : 0;
	ipmi_ret->data.bmc_enables.oem2_enabled			= (rsp->data[0] & (1 << 7)) ? 1 : 0;

	return 0;
}

/* ipmi_bmc_set_enables  -  set MC enable flags
 *
 * @intf:	ipmi inteface
 *
 * returns 0 on success
 * returns -1 on error
 */
static int
ipmi_bmc_set_enables(struct ipmi_intf * intf, pp_ipmi_bmc_set_enables_param_t *enables, pp_ipmi_return_t *ipmi_ret, int * error)
{
	struct ipmi_rs * rsp;
	struct ipmi_rq req;
	uint8_t en;

	memset(&req, 0, sizeof(req));
	req.msg.netfn = IPMI_NETFN_APP;
	req.msg.cmd = BMC_GET_GLOBAL_ENABLES;

	rsp = intf->sendrecv(intf, &req, error);
	if (rsp == NULL) {
		ipmi_printf("Get Global Enables command failed\n");
		return -1;
	}
	if (rsp->ccode) {
		ipmi_printf("Get Global Enables command failed: %s\n",
		       val2str(rsp->ccode, completion_code_vals));
		return -1;
	}

	en = rsp->data[0];

	if (enables->receive_msg_int_enabled) {
		en |= (1 << 0);
	} else {
		en &= ~(1 << 0);
	}

	if (enables->event_msg_buf_int_enabled) {
		en |= (1 << 1);
	} else {
		en &= ~(1 << 1);
	}

	if (enables->event_msg_buf_enabled) {
		en |= (1 << 2);
	} else {
		en &= ~(1 << 2);
	}

	if (enables->sel_enabled) {
		en |= (1 << 3);
	} else {
		en &= ~(1 << 3);
	}

	if (enables->oem0_enabled) {
		en |= (1 << 5);
	} else {
		en &= ~(1 << 5);
	}

	if (enables->oem1_enabled) {
		en |= (1 << 6);
	} else {
		en &= ~(1 << 6);
	}

	if (enables->oem2_enabled) {
		en |= (1 << 7);
	} else {
		en &= ~(1 << 7);
	}

	if (en == rsp->data[0]) {
		if (verbose) {
			ipmi_printf("Nothing to change...\n");
		}
		return ipmi_bmc_get_enables(intf, ipmi_ret, error);
	}

	req.msg.cmd = BMC_SET_GLOBAL_ENABLES;
	req.msg.data = &en;
	req.msg.data_len = 1;

	rsp = intf->sendrecv(intf, &req, error);
	if (rsp == NULL) {
		ipmi_printf("Set Global Enables command failed\n");
		return -1;
	}
	else if (rsp->ccode > 0) {
		ipmi_printf("Set Global Enables command failed: %s\n",
		       val2str(rsp->ccode, completion_code_vals));
		return -1;
	}
	if (verbose) {
		ipmi_printf("Verifying...\n");
	}
	return ipmi_bmc_get_enables(intf, ipmi_ret, error);
}

/* IPM Device, Get Device ID Command - Additional Device Support */
const char *ipm_dev_adtl_dev_support[8] = {
        N_("Sensor Device"),         /* bit 0 */
        N_("SDR Repository Device"), /* bit 1 */
        N_("SEL Device"),            /* bit 2 */
        N_("FRU Inventory Device"),  /*  ...  */
        N_("IPMB Event Receiver"),
        N_("IPMB Event Generator"),
        N_("Bridge"),
        N_("Chassis Device")         /* bit 7 */
};

/* ipmi_bmc_get_deviceid  -  print information about this MC
 *
 * @intf:	ipmi interface
 *
 * returns 0 on success
 * returns -1 on error
 */
static int
ipmi_bmc_get_deviceid(struct ipmi_intf * intf, pp_ipmi_return_t *ret, int * error)
{
	struct ipmi_rs * rsp;
	struct ipmi_rq req;
	struct ipm_devid_rsp *devid;
	int i;
	pp_ipmi_bmc_info_t *info = &ret->data.bmc_info;

	if (ipmi_init_bmc_info(ret)) {
		ipmi_printf("Could not initialize return structure.\n");
		return -1;
	}

	memset(&req, 0, sizeof(req));
	req.msg.netfn = IPMI_NETFN_APP;
	req.msg.cmd = BMC_GET_DEVICE_ID;
	req.msg.data_len = 0;

	rsp = intf->sendrecv(intf, &req, error);
	if (rsp == NULL) {
		ipmi_printf("Get Device ID command failed\n");
		return -1;
	}
	if (rsp->ccode > 0) {
		ipmi_printf("Get Device ID command failed: %s\n",
			val2str(rsp->ccode, completion_code_vals));
		return -1;
	}

	devid = (struct ipm_devid_rsp *) rsp->data;
	
	info->device_id = devid->device_id;
	info->device_revision = devid->device_revision & IPM_DEV_DEVICE_ID_REV_MASK;
	pp_strappendf(&info->firmware_revision, "%u.%x",
		devid->fw_rev1 & IPM_DEV_FWREV1_MAJOR_MASK,
		devid->fw_rev2);
	pp_strappendf(&info->ipmi_version, "%x.%x",
		IPM_DEV_IPMI_VERSION_MAJOR(devid->ipmi_version),
		IPM_DEV_IPMI_VERSION_MINOR(devid->ipmi_version));
	info->manufacturer_id = (unsigned long)IPM_DEV_MANUFACTURER_ID(devid->manufacturer_id);
	info->product_id = buf2short((uint8_t *)(devid->product_id));
	info->device_available =
		(devid->fw_rev1 & IPM_DEV_FWREV1_AVAIL_MASK) ? 
		0 : 1;
	info->provides_sdrs =
		(devid->device_revision & IPM_DEV_DEVICE_ID_SDR_MASK) ?
		1 : 0;
	for (i = 0; i < IPM_DEV_ADTL_SUPPORT_BITS; i++) {
		if (devid->adtl_device_support & (1 << i)) {
			vector_add_const(info->additional_device_support, ipm_dev_adtl_dev_support[i]);
		}
	}
	/* These values could be looked-up by vendor if documented,
	 * so we put them on individual lines for better treatment later
	 */
	for (i = 0; i < 4; i++) {
		info->aux_fw_info[i] = devid->aux_fw_rev[i];
	}
	return 0;
}

struct ipmi_guid {
	uint32_t  time_low;	/* timestamp low field */
	uint16_t  time_mid;	/* timestamp middle field */
	uint16_t  time_hi_and_version; /* timestamp high field and version number */
	uint8_t   clock_seq_hi_variant;/* clock sequence high field and variant */
	uint8_t   clock_seq_low; /* clock sequence low field */
	uint8_t   node[6];	/* node */
} PACKED_STRUCT;

/* ipmi_bmc_get_device_guid  -  print this MC GUID
 *
 * @intf:	ipmi interface
 *
 * returns 0 on success
 * returns -1 on error
 */
static int
ipmi_bmc_get_device_guid(struct ipmi_intf * intf, pp_ipmi_return_t *ret, int * error)
{
	struct ipmi_rs * rsp;
	struct ipmi_rq req;

	if (ipmi_init_bmc_get_device_guid(ret)) {
		ipmi_printf("Could not initialize return structure.\n");
		return -1;
	}

	memset(&req, 0, sizeof(req));
	req.msg.netfn = IPMI_NETFN_APP;
	req.msg.cmd = BMC_GET_DEVICE_GUID;

	rsp = intf->sendrecv(intf, &req, error);
	if (rsp == NULL) {
		ipmi_printf("Get Device GUID command failed\n");
		return -1;
	}
	if (rsp->ccode > 0) {
		ipmi_printf("Get Device GUID command failed: %s\n",
			val2str(rsp->ccode, completion_code_vals));
		return -1;
	}

	if (rsp->data_len == sizeof(ret->data.bmc_device_guid.guid)) {
		memcpy(ret->data.bmc_device_guid.guid, rsp->data, rsp->data_len);
		return 0;
	}
	else {
		ipmi_printf("Invalid GUID length %d\n", rsp->data_len);
		ipmi_set_error(error, IPMI_ERROR_INVALID_GUID_LENGTH);

		return -1;
	}
}

/* ipmi_bmc_get_system_guid  -  get the System GUID
 *
 * @intf:	ipmi interface
 *
 * returns 0 on success
 * returns -1 on error
 */
static int
ipmi_bmc_get_system_guid(struct ipmi_intf * intf, pp_ipmi_return_t *ret, int * error)
{
	struct ipmi_rs * rsp;
	struct ipmi_rq req;

	if (ipmi_init_bmc_get_system_guid(ret)) {
		ipmi_printf("Could not initialize return structure.\n");
		return -1;
	}

	memset(&req, 0, sizeof(req));
	req.msg.netfn = IPMI_NETFN_APP;
	req.msg.cmd = BMC_GET_SYSTEM_GUID;

	rsp = intf->sendrecv(intf, &req, error);
	if (rsp == NULL) {
		ipmi_printf("Get System GUID command failed\n");
		return -1;
	}
	if (rsp->ccode > 0) {
		ipmi_printf("Get System GUID command failed: %s\n",
			val2str(rsp->ccode, completion_code_vals));
		return -1;
	}

	if (rsp->data_len == sizeof(ret->data.bmc_system_guid.guid)) {
		memcpy(ret->data.bmc_system_guid.guid, rsp->data, rsp->data_len);
		return 0;
	}
	else {
		ipmi_printf("Invalid GUID length %d\n", rsp->data_len);
		ipmi_set_error(error, IPMI_ERROR_INVALID_GUID_LENGTH);

		return -1;
	}
}

/* ipmi_bmc_get_acpi_state  -  Returns host ACPI state
 *
 * @intf:	ipmi interface
 *
 * returns 0 on success
 * returns -1 on error
 */
static int
ipmi_bmc_get_acpi_state(struct ipmi_intf * intf, pp_ipmi_return_t *ipmi_ret, int * error)
{
	struct ipmi_rs * rsp;
	struct ipmi_rq req;

	if (ipmi_init_bmc_get_acpi_state(ipmi_ret)) {
		ipmi_printf("Error: Could not allocate return structure.\n");
		return -1;
	}

	memset(&req, 0, sizeof(req));
	req.msg.netfn = IPMI_NETFN_APP;
	req.msg.cmd = BMC_GET_ACPI_STATE;

	rsp = intf->sendrecv(intf, &req, error);
	if (rsp == NULL) {
		ipmi_printf("Get ACPI Power State command failed\n");
		return -1;
	}
	if (rsp->ccode > 0) {
		ipmi_printf("BMC Get ACPI Power State command failed: %s\n",
		       val2str(rsp->ccode, completion_code_vals));
		return -1;
	}

	ipmi_ret->data.bmc_acpi_state.sys_state = rsp->data[0];
	ipmi_ret->data.bmc_acpi_state.dev_state = rsp->data[1];

	switch (ipmi_ret->data.bmc_acpi_state.sys_state) {
	  case PP_IPMI_ACPI_SYS_STATE_S0:
	      pp_strappend(&ipmi_ret->data.bmc_acpi_state.sys_state_string, _("S0/G0 working"));
	      break;
	  case PP_IPMI_ACPI_SYS_STATE_S1:
	      pp_strappend(&ipmi_ret->data.bmc_acpi_state.sys_state_string, "S1");  // shall we translate this?
	      break;
	  case PP_IPMI_ACPI_SYS_STATE_S2:
	      pp_strappend(&ipmi_ret->data.bmc_acpi_state.sys_state_string, "S2");  // shall we translate this?
	      break;
	  case PP_IPMI_ACPI_SYS_STATE_S3:
	      pp_strappend(&ipmi_ret->data.bmc_acpi_state.sys_state_string, _("S3 suspend to RAM"));
	      break;
	  case PP_IPMI_ACPI_SYS_STATE_S4:
	      pp_strappend(&ipmi_ret->data.bmc_acpi_state.sys_state_string, _("S4 suspend to disk"));
	      break;
	  case PP_IPMI_ACPI_SYS_STATE_S5:
	      pp_strappend(&ipmi_ret->data.bmc_acpi_state.sys_state_string, _("S5/G2 soft off"));
	      break;	      
	  case PP_IPMI_ACPI_SYS_STATE_S45:
	      pp_strappend(&ipmi_ret->data.bmc_acpi_state.sys_state_string, _("S4/S5 soft off"));
	      break;
	  case PP_IPMI_ACPI_SYS_STATE_MECHOFF:
	      pp_strappend(&ipmi_ret->data.bmc_acpi_state.sys_state_string, _("G3 mechanical off"));
	      break;
	  case PP_IPMI_ACPI_SYS_STATE_S123:
	      pp_strappend(&ipmi_ret->data.bmc_acpi_state.sys_state_string, _("S1/S2/S3 sleeping"));
	      break;
	  case PP_IPMI_ACPI_SYS_STATE_S1234:
	      pp_strappend(&ipmi_ret->data.bmc_acpi_state.sys_state_string, _("S1/S2/S3/S4 G1 sleeping"));
	      break;
	  case PP_IPMI_ACPI_SYS_STATE_S5OVR:
	      pp_strappend(&ipmi_ret->data.bmc_acpi_state.sys_state_string, _("S5 override"));
	      break;
	  case PP_IPMI_ACPI_SYS_STATE_LEGON:
	      pp_strappend(&ipmi_ret->data.bmc_acpi_state.sys_state_string, _("Legacy On"));
	      break;
	  case PP_IPMI_ACPI_SYS_STATE_LEGOFF:
	      pp_strappend(&ipmi_ret->data.bmc_acpi_state.sys_state_string, _("Legacy Off"));
	      break;
	  case PP_IPMI_ACPI_SYS_STATE_UNKNOWN:
	      pp_strappend(&ipmi_ret->data.bmc_acpi_state.sys_state_string, _("Unknown"));
	      break;
	  default:
	      break;
	}

	switch (ipmi_ret->data.bmc_acpi_state.dev_state) {
	  case PP_IPMI_ACPI_DEV_STATE_D0:
	      pp_strappend(&ipmi_ret->data.bmc_acpi_state.dev_state_string, "D0");  // shall we translate this?
	      break;
	  case PP_IPMI_ACPI_DEV_STATE_D1:
	      pp_strappend(&ipmi_ret->data.bmc_acpi_state.dev_state_string, "D1");  // shall we translate this?
	      break;
	  case PP_IPMI_ACPI_DEV_STATE_D2:
	      pp_strappend(&ipmi_ret->data.bmc_acpi_state.dev_state_string, "D2");  // shall we translate this?
	      break;	      
	  case PP_IPMI_ACPI_DEV_STATE_D3:
	      pp_strappend(&ipmi_ret->data.bmc_acpi_state.dev_state_string, "D3");  // shall we translate this?
	      break;
	  case PP_IPMI_ACPI_DEV_STATE_UNKNOWN:
	      pp_strappend(&ipmi_ret->data.bmc_acpi_state.dev_state_string, _("Unknown"));
	      break;
	  default:
	      break;
	}
	
	return 0;
}

int ipmi_bmc_main(struct ipmi_intf * intf, int subcommand, pp_ipmi_parameter_t *params, pp_ipmi_return_t *ipmi_ret, int * error)
{
	int ret = -1;
	
	switch ((pp_ipmi_bmc_subcommand_t) subcommand) {
		case PP_IPMI_BMC_SUBCMD_RESET:
			if (!params || params->is_cmdline) {
				ipmi_printf("Error: no parameter for BMC RESET command.\n");
				ipmi_set_error(error, IPMI_ERROR_NO_PARAMETER);
			} else {
				int cmd;
				if (params->data.bmc_reset == PP_IPMI_BMC_RESET_COLD) {
					cmd = BMC_COLD_RESET;
					D("BMC reset cold.\n");
				} else if (params->data.bmc_reset == PP_IPMI_BMC_RESET_WARM) {
					cmd = BMC_WARM_RESET;
					D("BMC reset warm.\n");
				} else {
					ipmi_printf("Wrong reset command: %d\n", params->data.bmc_reset);
					break;
				}
				ret = ipmi_bmc_reset(intf, cmd, error);
			}
			break;
		case PP_IPMI_BMC_SUBCMD_INFO:
			ret = ipmi_bmc_get_deviceid(intf, ipmi_ret, error);
			break;
		case PP_IPMI_BMC_SUBCMD_GET_ENABLES:
			ret = ipmi_bmc_get_enables(intf, ipmi_ret, error);
			break;
		case PP_IPMI_BMC_SUBCMD_SET_ENABLES:
			if (!params || params->is_cmdline) {
				ipmi_printf("Error: no parameter for BMC SET_ENABLES command.\n");
				ipmi_set_error(error, IPMI_ERROR_NO_PARAMETER);
			} else {
				ret = ipmi_bmc_set_enables(intf, &params->data.bmc_enables, ipmi_ret, error);
			}
			break;
		case PP_IPMI_BMC_SUBCMD_GET_DEVICE_GUID:
			ret = ipmi_bmc_get_device_guid(intf, ipmi_ret, error);
			break;
		case PP_IPMI_BMC_SUBCMD_GET_SYSTEM_GUID:
			ret = ipmi_bmc_get_system_guid(intf, ipmi_ret, error);
			break;
		case PP_IPMI_BMC_SUBCMD_GET_ACPI_STATE:
			ret = ipmi_bmc_get_acpi_state(intf, ipmi_ret, error);
			break;
		default:
			ipmi_set_error(error, IPMI_ERROR_INVALID_COMMAND);
			ipmi_printf("Unknown BMC command: %d\n", subcommand);
	}
	
	return ret;
}

