/*
 * Copyright (c) 2003 Sun Microsystems, Inc.  All Rights Reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 
 * Redistribution of source code must retain the above copyright
 * notice, this list of conditions and the following disclaimer.
 * 
 * Redistribution in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the distribution.
 * 
 * Neither the name of Sun Microsystems, Inc. or the names of
 * contributors may be used to endorse or promote products derived
 * from this software without specific prior written permission.
 * 
 * This software is provided "AS IS," without a warranty of any kind.
 * ALL EXPRESS OR IMPLIED CONDITIONS, REPRESENTATIONS AND WARRANTIES,
 * INCLUDING ANY IMPLIED WARRANTY OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE OR NON-INFRINGEMENT, ARE HEREBY EXCLUDED.
 * SUN MICROSYSTEMS, INC. ("SUN") AND ITS LICENSORS SHALL NOT BE LIABLE
 * FOR ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING
 * OR DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.  IN NO EVENT WILL
 * SUN OR ITS LICENSORS BE LIABLE FOR ANY LOST REVENUE, PROFIT OR DATA,
 * OR FOR DIRECT, INDIRECT, SPECIAL, CONSEQUENTIAL, INCIDENTAL OR
 * PUNITIVE DAMAGES, HOWEVER CAUSED AND REGARDLESS OF THE THEORY OF
 * LIABILITY, ARISING OUT OF THE USE OF OR INABILITY TO USE THIS SOFTWARE,
 * EVEN IF SUN HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGES.
 * 
 * You acknowledge that this software is not designed or intended for use
 * in the design, construction, operation or maintenance of any nuclear
 * facility.
 */

#include <stdlib.h>
#include <string.h>
#include <stdio.h>

#include <ipmitool/helper.h>
#include <ipmitool/ipmi.h>
#include <ipmitool/ipmi_intf.h>
#include <ipmitool/ipmi_chassis.h>
#include <ipmitool/ipmi_print.h>
#include <ipmitool/ipmi_error.h>
#include <ipmitool/ipmi_strings.h>
#include <ipmi_return.h>
#include <ipmi_params.h>

#ifdef WIN32
#include <ipmitool/bswap.h>
#else
#include <pp/base.h>
#endif

int
ipmi_chassis_power_status(struct ipmi_intf * intf, int * error)
{
	struct ipmi_rs * rsp;
	struct ipmi_rq req;

	memset(&req, 0, sizeof(req));
	req.msg.netfn = IPMI_NETFN_CHASSIS;
	req.msg.cmd = 0x1;
	req.msg.data_len = 0;

	rsp = intf->sendrecv(intf, &req, error);
	if (rsp == NULL) {
		ipmi_printf("Unable to get Chassis Power Status\n");
		return -1;
	}
	if (rsp->ccode > 0) {
		ipmi_printf("Get Chassis Power Status failed: %s\n",
			val2str(rsp->ccode, completion_code_vals));
		return -1;
	}

	return rsp->data[0] & 1;
}

static int ipmi_chassis_print_power_status(struct ipmi_intf * intf, pp_ipmi_return_t *ipmi_ret, int * error)
{
	int ps = ipmi_chassis_power_status(intf, error);

	if (ipmi_init_chassis_power_status(ipmi_ret)) {
		ipmi_printf("Error: could not initialize return structure.\n");
		return -1;
	}

	if (ps < 0) {
		ipmi_printf("error in Chassis Status Command\n");
		pp_strappend(&ipmi_ret->data.chassis_power_status.status_string, _("Unknown"));
		ipmi_ret->data.chassis_power_status.status = PP_IPMI_CHASSIS_POWER_STATUS_UNKNOWN;
		return -1;
	}

	pp_strappend(&ipmi_ret->data.chassis_power_status.status_string, ps ? _("On") : _("Off"));
	ipmi_ret->data.chassis_power_status.status = ps ? PP_IPMI_CHASSIS_POWER_STATUS_ON : PP_IPMI_CHASSIS_POWER_STATUS_OFF;

	return 0;
}

int
ipmi_chassis_power_control(struct ipmi_intf * intf, uint8_t ctl, int * error)
{
	struct ipmi_rs * rsp;
	struct ipmi_rq req;

	ipmi_intf_session_set_privlvl(intf, IPMI_SESSION_PRIV_ADMIN);

	memset(&req, 0, sizeof(req));
	req.msg.netfn = IPMI_NETFN_CHASSIS;
	req.msg.cmd = 0x2;
	req.msg.data = &ctl;
	req.msg.data_len = 1;

	rsp = intf->sendrecv(intf, &req, error);

	if (rsp == NULL) {
		ipmi_printf("Unable to set Chassis Power Control to %s\n",
			val2str(ctl, ipmi_chassis_power_control_vals));
		return -1;
	}
	if (rsp->ccode > 0) {
		ipmi_printf("Set Chassis Power Control to %s failed: %s\n",
			val2str(ctl, ipmi_chassis_power_control_vals),
			val2str(rsp->ccode, completion_code_vals));
		return -1;
	}

	if (verbose) {
		ipmi_printf("Chassis Power Control: %s\n",
		       val2str(ctl, ipmi_chassis_power_control_vals));
	}

	/* sessions often get lost when changing chassis power */
	intf->abort = 1;

	return 0;
}

static int
ipmi_chassis_identify(struct ipmi_intf * intf, int timeout, int * error)
{
	struct ipmi_rq req;
	struct ipmi_rs * rsp;

	struct {
		uint8_t interval;
		uint8_t force_on;
	} identify_data;

	ipmi_intf_session_set_privlvl(intf, IPMI_SESSION_PRIV_ADMIN);

	memset(&req, 0, sizeof(req));
	req.msg.netfn = IPMI_NETFN_CHASSIS;
	req.msg.cmd = 0x4;

	if (timeout == -1){
		identify_data.interval = 0;
		identify_data.force_on = 1;
	} else {
		identify_data.interval = (uint8_t)timeout;
		identify_data.force_on = 0;
	}

	req.msg.data = (uint8_t *)&identify_data;

	/* The Force Identify On byte is optional and not
	 * supported by all devices-- if force is not specified,
	 * we pass only one data byte; if specified, we pass two
	 * data bytes and check for an error completion code
	 */	
	req.msg.data_len = (identify_data.force_on) ? 2 : 1;

	rsp = intf->sendrecv(intf, &req, error);
	if (rsp == NULL) {
		ipmi_printf("Unable to set Chassis Identify\n");
		return -1;
	}
	if (rsp->ccode > 0) {
		ipmi_printf("Set Chassis Identify failed: %s\n",
			val2str(rsp->ccode, completion_code_vals));
		if (identify_data.force_on != 0) {
			/* Intel SE7501WV2 F/W 1.2 returns CC 0xC7, but
			 * the IPMI v1.5 spec does not standardize a CC
			 * if unsupported, so we warn
			 */
			ipmi_printf("Chassis may not support Force Identify On\n");
		}
		return -1;
	}

	if (verbose) {
		ipmi_printf("Chassis identify interval: ");
		if (identify_data.force_on != 0) {
			ipmi_printf("indefinate\n");
		} else {
			if (identify_data.interval == 0) {
				ipmi_printf("off\n");
			} else {
				ipmi_printf("%i seconds\n", identify_data.interval);
			}
		}
	}
	
	return 0;
}

static int
ipmi_chassis_poh(struct ipmi_intf * intf, pp_ipmi_return_t* ipmi_ret, int * error)
{
	struct ipmi_rs * rsp;
	struct ipmi_rq req;
	uint32_t count;
	uint8_t minutes_per_count;
	u_int64_t up_minutes;
	unsigned int days, hours, minutes;

	if (ipmi_init_chassis_poh(ipmi_ret)) {
		ipmi_printf("Error: could not initialize return structure.\n");
		return -1;
	}

	memset(&req, 0, sizeof(req));
	req.msg.netfn = IPMI_NETFN_CHASSIS;
	req.msg.cmd = 0xf;

	rsp = intf->sendrecv(intf, &req, error);

	if (rsp == NULL) {
		ipmi_printf("Unable to get Chassis Power-On-Hours\n");
		pp_strappend(&ipmi_ret->data.chassis_poh, _("Unknown"));
		return -1;
	}
	if (rsp->ccode > 0) {
		ipmi_printf("Get Chassis Power-On-Hours failed: %s\n",
			val2str(rsp->ccode, completion_code_vals));
		pp_strappend(&ipmi_ret->data.chassis_poh, _("Unknown"));
		return -1;
	}
	
	minutes_per_count = rsp->data[0];
	count = buf2long(&rsp->data[1]);
	
	up_minutes = minutes_per_count * count;
	
	days = (unsigned int) up_minutes / 1440;
	up_minutes -= 1440 * days;
	
	hours = (unsigned int) up_minutes / 60;
	up_minutes -= 60 * hours;
	
	minutes = (unsigned int) up_minutes;

	pp_strappendf(&ipmi_ret->data.chassis_poh, _("%u days, %u hours, %u minutes"), days, hours, minutes);
	
	return 0;
}

static int
ipmi_chassis_restart_cause(struct ipmi_intf * intf, pp_ipmi_return_t* ipmi_ret, int * error)
{
	struct ipmi_rs * rsp;
	struct ipmi_rq req;

	if (ipmi_init_chassis_restart_cause(ipmi_ret)) {
		ipmi_printf("Error: could not initialize return structure.\n");
		return -1;
	}

	memset(&req, 0, sizeof(req));
	req.msg.netfn = IPMI_NETFN_CHASSIS;
	req.msg.cmd = 0x7;

	rsp = intf->sendrecv(intf, &req, error);

	if (rsp == NULL) {
		ipmi_printf("Unable to get Chassis Restart Cause\n");
		pp_strappend(&ipmi_ret->data.chassis_restart_cause, _("Unknown"));
		return -1;
	}
	if (rsp->ccode > 0) {
		ipmi_printf("Get Chassis Restart Cause failed: %s\n",
			val2str(rsp->ccode, completion_code_vals));
		pp_strappend(&ipmi_ret->data.chassis_restart_cause, _("Unknown"));
		return -1;
	}

	switch (rsp->data[0] & 0xf) {
	case 0:
		pp_strappend(&ipmi_ret->data.chassis_restart_cause, _("Unknown"));
		break;
	case 1:
		pp_strappend(&ipmi_ret->data.chassis_restart_cause, _("Chassis power control command"));
		break;
	case 2:
		pp_strappend(&ipmi_ret->data.chassis_restart_cause, _("Reset via pushbutton"));
		break;
	case 3:
		pp_strappend(&ipmi_ret->data.chassis_restart_cause, _("Power-up via pushbutton"));
		break;
	case 4:
		pp_strappend(&ipmi_ret->data.chassis_restart_cause, _("Watchdog expired"));
		break;
	case 5:
		pp_strappend(&ipmi_ret->data.chassis_restart_cause, _("OEM"));
		break;
	case 6:
		pp_strappend(&ipmi_ret->data.chassis_restart_cause, _("Power-up due to always-restore power policy"));
		break;
	case 7:
		pp_strappend(&ipmi_ret->data.chassis_restart_cause, _("Power-up due to restore-previous power policy"));
		break;
	case 8:
		pp_strappend(&ipmi_ret->data.chassis_restart_cause, _("Reset via PEF"));
		break;
	case 9:
		pp_strappend(&ipmi_ret->data.chassis_restart_cause, _("Power-cycle via PEF"));
		break;
	default:
		pp_strappend(&ipmi_ret->data.chassis_restart_cause, _("Invalid"));
	}
	
	return 0;
}

static int
ipmi_chassis_status(struct ipmi_intf * intf, pp_ipmi_return_t *ipmi_ret, int * error)
{
	struct ipmi_rs * rsp;
	struct ipmi_rq req;
	int have_power_event = 0;

	pp_ipmi_chassis_status_t *status;

	if (ipmi_init_chassis_status(ipmi_ret)) {
		ipmi_printf("Error: could not allocate return structure.\n");
		return -1;
	}

	status = &ipmi_ret->data.chassis_status;

	memset(&req, 0, sizeof(req));
	req.msg.netfn = IPMI_NETFN_CHASSIS;
	req.msg.cmd = 0x1;

	rsp = intf->sendrecv(intf, &req, error);
	if (rsp == NULL) {
		ipmi_printf("Error sending Chassis Status command\n");
		return -1;
	}
	if (rsp->ccode > 0) {
		ipmi_printf("Error sending Chassis Status command: %s\n",
			val2str(rsp->ccode, completion_code_vals));
		return -1;
	}

	/* byte 1 */
	pp_strappend(&ipmi_ret->data.chassis_status.power_status_string, (rsp->data[0] & 0x1) ? _("On") : _("Off"));
	status->power_status		= (rsp->data[0] & 0x01) ? PP_IPMI_CHASSIS_POWER_STATUS_ON : PP_IPMI_CHASSIS_POWER_STATUS_OFF;
	status->power_overload		= (rsp->data[0] & 0x02) ? 1 : 0;
	status->power_interlock_active	= (rsp->data[0] & 0x04) ? 1 : 0;
	status->main_power_fault	= (rsp->data[0] & 0x08) ? 1 : 0;
	status->power_control_fault	= (rsp->data[0] & 0x10) ? 1 : 0;
	switch ((rsp->data[0] & 0x60) >> 5) {
	case 0x0:
		pp_strappend(&status->power_restore_policy.string, _("always-off"));
		status->power_restore_policy.policy = PP_IPMI_CHASSIS_POLICY_ALWAYS_OFF;
		break;
	case 0x1:
		pp_strappend(&status->power_restore_policy.string, _("previous"));
		status->power_restore_policy.policy = PP_IPMI_CHASSIS_POLICY_PREVIOUS;
		break;
	case 0x2:
		pp_strappend(&status->power_restore_policy.string, _("always-on"));
		status->power_restore_policy.policy = PP_IPMI_CHASSIS_POLICY_ALWAYS_ON;
		break;
	case 0x3:
	default:
		pp_strappend(&status->power_restore_policy.string, _("unknown"));
		status->power_restore_policy.policy = PP_IPMI_CHASSIS_POLICY_UNKNOWN;
	}

	/* byte 2 */
	if (rsp->data[1] & 0x1) {
		pp_strappend(&status->last_power_event, _("ac-failed"));
		have_power_event = 1;
	}
	if (rsp->data[1] & 0x2) {
		if (have_power_event) pp_strappend(&status->last_power_event, " ");
		pp_strappend(&status->last_power_event, _("overload"));
		have_power_event = 1;
	}
	if (rsp->data[1] & 0x4) {
		if (have_power_event) pp_strappend(&status->last_power_event, " ");
		pp_strappend(&status->last_power_event, _("interlock"));
		have_power_event = 1;
	}
	if (rsp->data[1] & 0x8) {
		if (have_power_event) pp_strappend(&status->last_power_event, " ");
		pp_strappend(&status->last_power_event, _("fault"));
		have_power_event = 1;
	}
	if (rsp->data[1] & 0x10) {
		if (have_power_event) pp_strappend(&status->last_power_event, " ");
		pp_strappend(&status->last_power_event, _("command"));
		have_power_event = 1;
	}

	/* byte 3 */
	status->chassis_intrusion_active	= (rsp->data[2] & 0x1) ? 1 : 0;
	status->front_panel_lockout_active	= (rsp->data[2] & 0x2) ? 1 : 0;
	status->drive_fault			= (rsp->data[2] & 0x4) ? 1 : 0;
	status->cooling_fan_fault		= (rsp->data[2] & 0x8) ? 1 : 0;

        if (rsp->data_len > 3) {
		/* optional byte 4 */
		status->front_panel_control_info_valid = 1;
		if (rsp->data[3] == 0) {
			status->front_panel_control = 0;
		} else {
			status->front_panel_control = 1;
			status->sleep_allowed		= (rsp->data[3] & 0x80) ? 1 : 0;
			status->diag_allowed		= (rsp->data[3] & 0x40) ? 1 : 0;
			status->reset_allowed		= (rsp->data[3] & 0x20) ? 1 : 0;
			status->power_allowed		= (rsp->data[3] & 0x10) ? 1 : 0;
			status->sleep_disabled		= (rsp->data[3] & 0x08) ? 1 : 0;
			status->diag_disabled		= (rsp->data[3] & 0x04) ? 1 : 0;
			status->reset_disabled		= (rsp->data[3] & 0x02) ? 1 : 0;
			status->power_disabled		= (rsp->data[3] & 0x01) ? 1 : 0;
		}
	}
        else {
		status->front_panel_control_info_valid = 0;
        }
        
        return 0;
}

static int
ipmi_chassis_set_bootparam(struct ipmi_intf * intf, uint8_t param, uint8_t * data, int len, int * error)
{
	struct ipmi_rs * rsp;
	struct ipmi_rq req;
	uint8_t msg_data[16];

	memset(msg_data, 0, 16);
	msg_data[0] = param & 0x7f;
	memcpy(msg_data+1, data, len);

	memset(&req, 0, sizeof(req));
	req.msg.netfn = IPMI_NETFN_CHASSIS;
	req.msg.cmd = 0x8;
	req.msg.data = msg_data;
	req.msg.data_len = len + 1;

	rsp = intf->sendrecv(intf, &req, error);
	if (rsp == NULL) {
		ipmi_printf("Error setting Chassis Boot Parameter %d\n", param);
		return -1;
	}
	if (rsp->ccode > 0) {
		ipmi_printf("Set Chassis Boot Parameter %d failed: %s\n",
			param, val2str(rsp->ccode, completion_code_vals));
		return -1;
  	}

	ipmi_printf("Chassis Set Boot Param %d to %s\n", param, buf2str(data, len));
	return 0;
}

static int
ipmi_chassis_get_bootparam(struct ipmi_intf * intf, uint8_t bootparam, pp_ipmi_return_t *ipmi_ret, int * error)
{
	struct ipmi_rs * rsp;
	struct ipmi_rq req;
	uint8_t msg_data[3];
	pp_ipmi_chassis_get_bparam_t *param;

	if (ipmi_init_chassis_get_bootparam(ipmi_ret)) {
		ipmi_printf("Error: Could not allocate return structure.\n");
		return -1;
	}

	param = &ipmi_ret->data.chassis_bootparam;

	memset(msg_data, 0, 3);

	msg_data[0] = bootparam & 0x7f;
	msg_data[1] = 0;
	msg_data[2] = 0;

	memset(&req, 0, sizeof(req));
	req.msg.netfn = IPMI_NETFN_CHASSIS;
	req.msg.cmd = 0x9;
	req.msg.data = msg_data;
	req.msg.data_len = 3;

	rsp = intf->sendrecv(intf, &req, error);
	if (rsp == NULL) {
		ipmi_printf("Error Getting Chassis Boot Parameter %u\n", bootparam);
		return -1;
	}
	if (rsp->ccode > 0) {
		ipmi_printf("Get Chassis Boot Parameter %s failed: %u\n",
			bootparam, val2str(rsp->ccode, completion_code_vals));
		return -1;
	}

	param->version = rsp->data[0];
	param->boot_param = rsp->data[1] & 0x7f;
	param->invalid_or_locked = (rsp->data[1] & 0x80) ? 1 : 0;
	param->data = malloc(rsp->data_len - 2);
	if (param->data) {
		memcpy(param->data, rsp->data+2, rsp->data_len - 2);
		param->data_size = rsp->data_len - 2;
	}
	
	return 0;
}

static int
ipmi_chassis_get_bootdev(struct ipmi_intf * intf, pp_ipmi_return_t *ipmi_ret, int * error)
{
	struct ipmi_rs * rsp;
	struct ipmi_rq req;
	uint8_t msg_data[3];
	pp_ipmi_chassis_get_bflag_return_t *flag;

	if (ipmi_init_chassis_get_bootflag(ipmi_ret)) {
		ipmi_printf("Error: Could not allocate return structure.\n");
		return -1;
	}

	flag = &ipmi_ret->data.chassis_bootflag;

	memset(msg_data, 0, 3);

	msg_data[0] = 5;
	msg_data[1] = 0;
	msg_data[2] = 0;

	memset(&req, 0, sizeof(req));
	req.msg.netfn = IPMI_NETFN_CHASSIS;
	req.msg.cmd = 0x9;
	req.msg.data = msg_data;
	req.msg.data_len = 3;

	rsp = intf->sendrecv(intf, &req, error);
	if (!rsp || rsp->ccode) {
		ipmi_printf("Error: Could not read boot flag.\n");
		return -1;
	}

	flag->valid = rsp->data[0] & 0x80 ? 1 : 0;
	switch (rsp->data[0] & 0x3c) {
		case 0x04: // 00000100
			flag->flag = PP_IPMI_CHASSIS_SET_BOOTFLAG_FORCE_PXE;
			break;
		case 0x08: // 00001000
			flag->flag = PP_IPMI_CHASSIS_SET_BOOTFLAG_FORCE_DISK;
			break;
		case 0x0C: // 00001100
			flag->flag = PP_IPMI_CHASSIS_SET_BOOTFLAG_FORCE_SAFE_DISK;
			break;
		case 0x10: // 00010000
			flag->flag = PP_IPMI_CHASSIS_SET_BOOTFLAG_FORCE_DIAG;
			break;
		case 0x14: // 00010100
			flag->flag = PP_IPMI_CHASSIS_SET_BOOTFLAG_FORCE_CDROM;
			break;
		case 0x18: // 00011000
			flag->flag = PP_IPMI_CHASSIS_SET_BOOTFLAG_FORCE_BIOS;
			break;
		case 0x3C: // 00111100
			flag->flag = PP_IPMI_CHASSIS_SET_BOOTFLAG_FORCE_FLOPPY;
			break;
		default:
			flag->flag = PP_IPMI_CHASSIS_SET_BOOTFLAG_NO_CHANGE;
	}
	
	return 0;
}

static int
ipmi_chassis_set_bootdev(struct ipmi_intf * intf, pp_ipmi_chassis_bootflag_t bootflag, int clearcmos, int * error)
{
	uint8_t flags[5];
	int rc = 0;

	memset(flags, 0, sizeof(flags));
	flags[0] = 0x01;
	flags[1] = 0x01;
	rc = ipmi_chassis_set_bootparam(intf, 4, flags, 2, error);
	if (rc < 0)
		return -1;

	memset(flags, 0, sizeof(flags));

	ipmi_intf_session_set_privlvl(intf, IPMI_SESSION_PRIV_ADMIN);

	if (bootflag == PP_IPMI_CHASSIS_SET_BOOTFLAG_NO_CHANGE) {
		flags[1] = 0x00; // 00000000
	} else if (bootflag == PP_IPMI_CHASSIS_SET_BOOTFLAG_FORCE_PXE) {
		flags[1] = 0x04; // 00000100
	} else if (bootflag == PP_IPMI_CHASSIS_SET_BOOTFLAG_FORCE_DISK) {
		flags[1] = 0x08; // 00001000
	} else if (bootflag == PP_IPMI_CHASSIS_SET_BOOTFLAG_FORCE_SAFE_DISK) {
		flags[1] = 0x0C; // 00001100
	} else if (bootflag == PP_IPMI_CHASSIS_SET_BOOTFLAG_FORCE_DIAG) {
		flags[1] = 0x10; // 00010000
	} else if (bootflag == PP_IPMI_CHASSIS_SET_BOOTFLAG_FORCE_CDROM) {
		flags[1] = 0x14; // 00010100
	} else if (bootflag == PP_IPMI_CHASSIS_SET_BOOTFLAG_FORCE_BIOS) {
		flags[1] = 0x18; // 00011000
	} else if (bootflag == PP_IPMI_CHASSIS_SET_BOOTFLAG_FORCE_FLOPPY) {
		flags[1] = 0x3c; // 00111100
	} else {
		ipmi_printf("Invalid bootflag: %d\n", bootflag);
		return -1;
	}

	if (clearcmos)
		flags[1] |= 0x80;

	flags[0] = 0x80;	/* set flag valid bit */
	rc = ipmi_chassis_set_bootparam(intf, 5, flags, 5, error);
	if (rc == 0) {
		ipmi_printf("Set Boot Device to %s\n", val2str(bootflag, ipmi_boot_devices));
	}

#if 0
	flags[0] = 0x08;	/* don't automatically clear boot flag valid bit in 60 seconds */
	if (ipmi_chassis_set_bootparam(intf, 3, flags, 1, error)) {
		return -1;
	}
#endif

	return rc;
}

static int
ipmi_chassis_power_policy(struct ipmi_intf * intf, uint8_t policy, pp_ipmi_return_t *ipmi_ret, int * error)
{
	struct ipmi_rs * rsp;
	struct ipmi_rq req;

	memset(&req, 0, sizeof(req));
	req.msg.netfn = IPMI_NETFN_CHASSIS;
	req.msg.cmd = 0x6;
	req.msg.data = &policy;
	req.msg.data_len = 1;

	rsp = intf->sendrecv(intf, &req, error);
	if (rsp == NULL) {
		ipmi_printf("Error in Power Restore Policy command\n");
		return -1;
	}
	if (rsp->ccode > 0) {
		ipmi_printf("Power Restore Policy command failed: %s\n",
			val2str(rsp->ccode, completion_code_vals));
		return -1;
	}


	if (policy == IPMI_CHASSIS_POLICY_NO_CHANGE) {
		ipmi_ret->data.chassis_policies.always_off_supported = (rsp->data[0] & (1<<IPMI_CHASSIS_POLICY_ALWAYS_OFF)) ? 1 : 0;
		ipmi_ret->data.chassis_policies.always_on_supported = (rsp->data[0] & (1<<IPMI_CHASSIS_POLICY_ALWAYS_ON)) ? 1 : 0;
		ipmi_ret->data.chassis_policies.previous_supported = (rsp->data[0] & (1<<IPMI_CHASSIS_POLICY_PREVIOUS)) ? 1 : 0;
	} else if (verbose) {
		ipmi_printf("Set chassis power restore policy to ");
		switch (policy) {
		case IPMI_CHASSIS_POLICY_ALWAYS_ON:
			ipmi_printf("always-on\n");
			break;
		case IPMI_CHASSIS_POLICY_ALWAYS_OFF:
			ipmi_printf("always-off\n");
			break;
		case IPMI_CHASSIS_POLICY_PREVIOUS:
			ipmi_printf("previous\n");
			break;
		default:
			ipmi_printf("unknown\n");
		}
	}
	
	return 0;
}

int ipmi_chassis_main(struct ipmi_intf * intf, int subcommand, pp_ipmi_parameter_t *params, pp_ipmi_return_t *ipmi_ret, int * error)
{
	int ret = -1;
	
	switch ((pp_ipmi_chassis_subcommand_t) subcommand) {
		case PP_IPMI_CHASSIS_SUBCMD_STATUS:
			return ipmi_chassis_status(intf, ipmi_ret, error);
		case PP_IPMI_CHASSIS_SUBCMD_POWER_STATUS:
			return ipmi_chassis_print_power_status(intf, ipmi_ret, error);
			
		case PP_IPMI_CHASSIS_SUBCMD_POWER_CONTROL:
			if (!params || params->is_cmdline) {
				ipmi_printf("Error: No Parameter for CHASSIS POWER CONTROL.\n");
				ipmi_set_error(error, IPMI_ERROR_NO_PARAMETER);
				return -1;
			} else {
				uint8_t ctl;
				if (params->data.chassis_power == PP_IPMI_CHASSIS_POWER_DOWN) {
					ctl = IPMI_CHASSIS_CTL_POWER_DOWN;
				} else if (params->data.chassis_power == PP_IPMI_CHASSIS_POWER_UP) {
					ctl = IPMI_CHASSIS_CTL_POWER_UP;
				} else if (params->data.chassis_power == PP_IPMI_CHASSIS_POWER_CYCLE) {
					ctl = IPMI_CHASSIS_CTL_POWER_CYCLE;
				} else if (params->data.chassis_power == PP_IPMI_CHASSIS_RESET) {
					ctl = IPMI_CHASSIS_CTL_HARD_RESET;
				} else if (params->data.chassis_power == PP_IPMI_CHASSIS_POWER_DIAG) {
					ctl = IPMI_CHASSIS_CTL_PULSE_DIAG;
				} else if (params->data.chassis_power == PP_IPMI_CHASSIS_POWER_SOFT) {
					ctl = IPMI_CHASSIS_CTL_ACPI_SOFT;
				} else {
					ipmi_printf("Error: Wrong Parameter for CHASSIS POWER CONTROL.\n");
					ipmi_set_error(error, IPMI_ERROR_INVALID_PARAMETER);
					return -1;
				}

				return ipmi_chassis_power_control(intf, ctl, error);
			}
		case PP_IPMI_CHASSIS_SUBCMD_IDENTIFY:
			if (!params || params->is_cmdline) {
				ipmi_printf("Error: No Parameter for CHASSIS IDENTIFY.\n");
				ipmi_set_error(error, IPMI_ERROR_NO_PARAMETER);
				return -1;
			} else {
				return ipmi_chassis_identify(intf, params->data.chassis_identify, error);
			}
		case PP_IPMI_CHASSIS_SUBCMD_POH:
			return ipmi_chassis_poh(intf, ipmi_ret, error);
		case PP_IPMI_CHASSIS_SUBCMD_RESTART_CAUSE:
			return ipmi_chassis_restart_cause(intf, ipmi_ret, error);
		case PP_IPMI_CHASSIS_SUBCMD_GET_POLICIES:
			return ipmi_chassis_power_policy(intf, IPMI_CHASSIS_POLICY_NO_CHANGE, ipmi_ret, error);
		case PP_IPMI_CHASSIS_SUBCMD_SET_POLICY:
			if (!params || params->is_cmdline) {
				ipmi_printf("Error: No Parameter for CHASSIS SET POLICY.\n");
				ipmi_set_error(error, IPMI_ERROR_NO_PARAMETER);
				return -1;
			} else {
				uint8_t ctl;
	
				if (params->data.chassis_set_policy == PP_IPMI_CHASSIS_POLICY_ALWAYS_ON) {
					ctl = IPMI_CHASSIS_POLICY_ALWAYS_ON;
				} else if (params->data.chassis_set_policy == PP_IPMI_CHASSIS_POLICY_PREVIOUS) {
					ctl = IPMI_CHASSIS_POLICY_PREVIOUS;
				} else if (params->data.chassis_set_policy == PP_IPMI_CHASSIS_POLICY_ALWAYS_OFF) {
					ctl = IPMI_CHASSIS_POLICY_ALWAYS_OFF;
				} else if (params->data.chassis_set_policy == PP_IPMI_CHASSIS_POLICY_UNKNOWN) {
					ctl = IPMI_CHASSIS_POLICY_NO_CHANGE;
				} else {
					ipmi_printf("invalid chassis policy: %d\n", params->data.chassis_set_policy);
					ipmi_set_error(error, IPMI_ERROR_INVALID_PARAMETER);
					return -1;
				}
	
				return ipmi_chassis_power_policy(intf, ctl, ipmi_ret, error);
			}
		case PP_IPMI_CHASSIS_SUBCMD_GET_BOOTPARAM:
			if (!params || params->is_cmdline) {
				ipmi_printf("Error: No Parameter for CHASSIS GET BOOTPARAM.\n");
				ipmi_set_error(error, IPMI_ERROR_NO_PARAMETER);
				return -1;
			} else {
				ret = ipmi_chassis_get_bootparam(intf, params->data.chassis_get_bparam, ipmi_ret, error);
			}
			break;
		case PP_IPMI_CHASSIS_SUBCMD_SET_BOOTDEV:
			if (!params || params->is_cmdline) {
				ipmi_printf("Error: No Parameter for CHASSIS SET BOOTFLAG.\n");
				ipmi_set_error(error, IPMI_ERROR_NO_PARAMETER);
				return -1;
			} else {
				return ipmi_chassis_set_bootdev(intf,
					params->data.chassis_set_bootdev.device,
					params->data.chassis_set_bootdev.clear_cmos,
					error);
			}
		case PP_IPMI_CHASSIS_SUBCMD_GET_BOOTDEV:
			return ipmi_chassis_get_bootdev(intf, ipmi_ret, error);
		default:
			ipmi_printf("Invalid Chassis command: %d\n", subcommand);
			ipmi_set_error(error, IPMI_ERROR_INVALID_COMMAND);
			return -1;
	}

	return ret;
}

