/*
 * Copyright (c) 2003 Sun Microsystems, Inc.  All Rights Reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 
 * Redistribution of source code must retain the above copyright
 * notice, this list of conditions and the following disclaimer.
 * 
 * Redistribution in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the distribution.
 * 
 * Neither the name of Sun Microsystems, Inc. or the names of
 * contributors may be used to endorse or promote products derived
 * from this software without specific prior written permission.
 * 
 * This software is provided "AS IS," without a warranty of any kind.
 * ALL EXPRESS OR IMPLIED CONDITIONS, REPRESENTATIONS AND WARRANTIES,
 * INCLUDING ANY IMPLIED WARRANTY OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE OR NON-INFRINGEMENT, ARE HEREBY EXCLUDED.
 * SUN MICROSYSTEMS, INC. ("SUN") AND ITS LICENSORS SHALL NOT BE LIABLE
 * FOR ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING
 * OR DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.  IN NO EVENT WILL
 * SUN OR ITS LICENSORS BE LIABLE FOR ANY LOST REVENUE, PROFIT OR DATA,
 * OR FOR DIRECT, INDIRECT, SPECIAL, CONSEQUENTIAL, INCIDENTAL OR
 * PUNITIVE DAMAGES, HOWEVER CAUSED AND REGARDLESS OF THE THEORY OF
 * LIABILITY, ARISING OUT OF THE USE OF OR INABILITY TO USE THIS SOFTWARE,
 * EVEN IF SUN HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGES.
 * 
 * You acknowledge that this software is not designed or intended for use
 * in the design, construction, operation or maintenance of any nuclear
 * facility.
 */

#include <ipmitool/ipmi.h>
#include <ipmitool/helper.h>
#include <ipmitool/ipmi_intf.h>
#include <ipmitool/ipmi_fru.h>
#include <ipmitool/ipmi_sdr.h>
#include <ipmitool/ipmi_strings.h>
#include <ipmitool/ipmi_print.h>
#include <ipmitool/ipmi_error.h>
#include <ipmitool/dimm_spd.h>

#include <stdlib.h>
#include <string.h>
#include <ctype.h>

#include <config.h>

#include <ipmi_return.h>

static const char * chassis_type_desc[] = {
	N_("Unspecified"), N_("Other"), N_("Unknown"),
	N_("Desktop"), N_("Low Profile Desktop"), N_("Pizza Box"),
	N_("Mini Tower"), N_("Tower"),
	N_("Portable"), N_("LapTop"), N_("Notebook"), N_("Hand Held"), N_("Docking Station"),
	N_("All in One"), N_("Sub Notebook"), N_("Space-saving"), N_("Lunch Box"),
	N_("Main Server Chassis"), N_("Expansion Chassis"), N_("SubChassis"),
	N_("Bus Expansion Chassis"), N_("Peripheral Chassis"), N_("RAID Chassis"),
	N_("Rack Mount Chassis")
};

static const char * combined_voltage_desc[] = {
	"12 V", "-12 V", "5 V", "3.3 V"
};

/* get_fru_area_str  -  Parse FRU area string from raw data
 *
 * @data:	raw FRU data
 * @offset:	offset into data for area
 *
 * returns pointer to FRU area string
 */
static char *
get_fru_area_str(uint8_t * data, int * offset)
{
	static const char bcd_plus[] = "0123456789 -.:,_";
	char * str;
	int len, off, size, i, j, k, typecode;
	union {
		uint32_t bits;
		char chars[4];
	} u;

	size = 0;
	off = *offset;

	/* bits 6:7 contain format */
	typecode = ((data[off] & 0xC0) >> 6);

	/* bits 0:5 contain length */
	len = data[off++];
	len &= 0x3f;

	switch (typecode) {
	case 0:				/* 00b: binary/unspecified */
		/* hex dump -> 2x length */
		size = (len*2);
		break;
	case 2:				/* 10b: 6-bit ASCII */
		/* 4 chars per group of 1-3 bytes */
		size = ((((len+2)*4)/3) & ~3);
		break;
	case 3:				/* 11b: 8-bit ASCII */
	case 1:				/* 01b: BCD plus */
		/* no length adjustment */
		size = len;
		break;
	}

	if (size < 1)
	{
		*offset = off;
		return NULL;
	}

	str = malloc(size+1);
	if (str == NULL)
		return NULL;

	memset(str, 0, size+1);

	if (len == 0) {
		str[0] = '\0';
		*offset = off;
		return str;
	}

	switch(typecode) {
	case 0:			/* Binary */
		strncpy(str, buf2str(&data[off], len), len*2);
		break;

	case 1:			/* BCD plus */
		for (k=0; k<len; k++)
			str[k] = bcd_plus[(data[off+k] & 0x0f)];
		str[k] = '\0';
		break;

	case 2:			/* 6-bit ASCII */
		for (i=j=0; i<len; i+=3) {
			u.bits = 0;
			k = ((len-i) < 3 ? (len-i) : 3);
#ifdef WORDS_BIGENDIAN
			u.chars[3] = data[off+i];
			u.chars[2] = (k > 1 ? data[off+i+1] : 0);
			u.chars[1] = (k > 2 ? data[off+i+2] : 0);
#define CHAR_IDX 3
#else
			memcpy((void *)&u.bits, &data[off+i], k);
#define CHAR_IDX 0
#endif
			for (k=0; k<4; k++) {
				str[j++] = ((u.chars[CHAR_IDX] & 0x3f) + 0x20);
				u.bits >>= 6;
			}
		}
		str[j] = '\0';
		break;

	case 3:
		memcpy(str, &data[off], len);
		str[len] = '\0';
		break;
	}

	off += len;
	*offset = off;

	return str;
}

/* read_fru_area  -  fill in frubuf[offset:length] from the FRU[offset:length]
 *
 * @intf:	ipmi interface
 * @fru:	fru info
 * @id:		fru id
 * @offset:	offset into buffer
 * @length:	how much to read
 * @frubuf:	buffer read into
 *
 * returns -1 on error
 * returns 0 if successful
*/
static int
read_fru_area(struct ipmi_intf * intf, struct fru_info *fru, uint8_t id,
	      uint32_t offset, uint32_t length, uint8_t *frubuf,
	      int * error)
{
	static uint32_t fru_data_rqst_size = 32;
	uint32_t off = offset, tmp, finish;
	struct ipmi_rs * rsp;
	struct ipmi_rq req;
	uint8_t msg_data[4];

	if (offset > fru->size) {
		ipmi_printf("Read FRU Area offset incorrect: %d > %d\n",
			offset, fru->size);
		return -1;
	}

	finish = offset + length;
	if (finish > fru->size) {
		finish = fru->size;
		ipmi_printf("Read FRU Area length %d too large, "
			"Adjusting to %d\n",
			offset + length, finish - offset);
	}

	memset(&req, 0, sizeof(req));
	req.msg.netfn = IPMI_NETFN_STORAGE;
	req.msg.cmd = GET_FRU_DATA;
	req.msg.data = msg_data;
	req.msg.data_len = 4;

	if (fru->access && fru_data_rqst_size > 16)
		fru_data_rqst_size = 16;
	do {
		tmp = fru->access ? off >> 1 : off;
		msg_data[0] = id;
		msg_data[1] = (uint8_t)(tmp & 0xff);
		msg_data[2] = (uint8_t)(tmp >> 8);
		tmp = finish - off;
		if (tmp > fru_data_rqst_size)
			msg_data[3] = (uint8_t)fru_data_rqst_size;
		else
			msg_data[3] = (uint8_t)tmp;

		rsp = intf->sendrecv(intf, &req, error);
		if (rsp == NULL) {
			ipmi_printf("FRU Read failed\n");
			break;
		}
		if (rsp->ccode > 0) {
			ipmi_printf("FRU Read failed: %s\n",
				val2str(rsp->ccode, completion_code_vals));
			/* if we get C7 or C8 return code then we requested too
			 * many bytes at once so try again with smaller size */
			if ((rsp->ccode == 0xc7 || rsp->ccode == 0xc8 || rsp->ccode == 0xca) &&
			    ((fru_data_rqst_size >>= 1) > 8)) {
				ipmi_printf("Retrying FRU read with reduced request size %d\n",
					fru_data_rqst_size);
				continue;
			}
			break;
		}

		tmp = fru->access ? rsp->data[0] << 1 : rsp->data[0];
		memcpy((frubuf + off), rsp->data + 1, tmp);
		off += tmp;
	} while (off < finish);

	if (off < finish)
		return -1;

	return 0;
}

/* fru_area_print_chassis  -  Print FRU Chassis Area
 *
 * @intf:	ipmi interface
 * @fru:	fru info
 * @id: 	fru id
 * @offset:	offset pointer
 */
static void
fru_area_print_chassis(struct ipmi_intf * intf, struct fru_info * fru,
		       uint8_t id, uint32_t offset, pp_ipmi_return_t *ret, int * error)
{
	uint8_t * fru_area, * fru_data;
	uint32_t fru_len, area_len, i;

	i = offset;
	fru_len = 0;

	fru_data = malloc(fru->size + 1);
	if (fru_data == NULL) {
		ipmi_printf(" Out of memory!\n");
		return;
	}
	memset(fru_data, 0, fru->size + 1);

	/* read enough to check length field */
	if (read_fru_area(intf, fru, id, i, 2, fru_data, error) == 0)
		fru_len = 8 * fru_data[i + 1];
	if (fru_len <= 0) {
		free(fru_data);
		return;
	}

	/* read in the full fru */
	if (read_fru_area(intf, fru, id, i, fru_len, fru_data, error) < 0) {
		free(fru_data);
		return;
	}

	ipmi_printf(" Chassis Area (@%d, ver %d, %d bytes)\n", offset, fru_data[i], fru_data[i + 1] * 8);

	ret->data.fru_info.chassis.ver = fru_data[i++]; /* fru area version */
	area_len = fru_data[i++] * 8; /* fru area length */

	unsigned int u=0;
	for(;u<area_len;u++)
	{
	    if(isprint(fru_data[i+u]))
		printf("%c",fru_data[i+u]);
	    else printf("[%x]",fru_data[i+u]);
	}
	printf("\n");

	pp_strappend(&ret->data.fru_info.chassis.type, _(chassis_type_desc[fru_data[i++]]));

	fru_area = get_fru_area_str(fru_data, &i);
	if (fru_area) pp_strappend(&ret->data.fru_info.chassis.model, fru_area);
	free(fru_area);

	fru_area = get_fru_area_str(fru_data, &i);
	if (fru_area) pp_strappend(&ret->data.fru_info.chassis.serial, fru_area);
	free(fru_area);

	/* read any extra fields */
	while ((fru_data[i] != 0xc1) && (i < offset + area_len))
	{
		uint32_t j = i;
		fru_area = get_fru_area_str(fru_data, &i);
		if (fru_area != NULL && strlen(fru_area) > 0) {
			ipmi_printf(" Chassis Extra         : %s\n", fru_area);
			free(fru_area);
		}
		if (i == j)
			break;
	}

	free(fru_data);
}

/* fru_area_print_board  -  Print FRU Board Area
 *
 * @intf:	ipmi interface
 * @fru:	fru info
 * @id: 	fru id
 * @offset:	offset pointer
 */
static void
fru_area_print_board(struct ipmi_intf * intf, struct fru_info * fru,
		     uint8_t id, uint32_t offset, pp_ipmi_return_t *ret, int * error)
{
	uint8_t * fru_area, * fru_data;
	uint32_t fru_len, area_len, i, mfg_time;

	i = offset;
	fru_len = 0;

	fru_data = malloc(fru->size + 1);
	if (fru_data == NULL) {
		ipmi_printf(" Out of memory!\n");
		return;
	}
	memset(fru_data, 0, fru->size + 1);

	/* read enough to check length field */
	if (read_fru_area(intf, fru, id, i, 2, fru_data, error) == 0)
		fru_len = 8 * fru_data[i + 1];
	if (fru_len <= 0) {
		free(fru_data);
		return;
	}

	/* read in the full fru */
	if (read_fru_area(intf, fru, id, i, fru_len, fru_data, error) < 0) {
		free(fru_data);
		return;
	}

	ipmi_printf(" Board Area (@%d, ver %d, %d bytes)\n", offset, fru_data[i], fru_data[i + 1] * 8);

	ret->data.fru_info.board.ver = fru_data[i++]; /* fru area version */
	area_len = fru_data[i++] * 8; /* fru area length */
	unsigned int u=0;
	for(;u<area_len;u++)
	{
	    if(isprint(fru_data[i+u]))
		printf("%c",fru_data[i+u]);
	    else printf("[%x]",fru_data[i+u]);
	}
	printf("\n");

	ret->data.fru_info.board.lang = fru_data[i++]; /* fru board language */
	mfg_time = fru_data[i++]; /* mfg. date time */
	mfg_time += (uint32_t)fru_data[i++] << 8;
	mfg_time += (uint32_t)fru_data[i++] << 16;

	// mfg_time is number of minutes since 1996-01-01
	// convert to number of seconds since 1970-01-01
	struct tm origin = { .tm_mday = 1, .tm_mon = 0, .tm_year = 96, };
	ipmi_set_timestamp(mktime(&origin) + mfg_time * 60,
	                   &ret->data.fru_info.board.mfg_time);

	fru_area = get_fru_area_str(fru_data, &i);
	if (fru_area) pp_strappend(&ret->data.fru_info.board.manufacturer, fru_area);
	free(fru_area);

	fru_area = get_fru_area_str(fru_data, &i);
	if (fru_area) pp_strappend(&ret->data.fru_info.board.name, fru_area);
	free(fru_area);

	fru_area = get_fru_area_str(fru_data, &i);
	if (fru_area) pp_strappend(&ret->data.fru_info.board.serial, fru_area);
	free(fru_area);

	fru_area = get_fru_area_str(fru_data, &i);
	if (fru_area) pp_strappend(&ret->data.fru_info.board.model, fru_area);
	free(fru_area);

	fru_area = get_fru_area_str(fru_data, &i);
	if (fru_area) pp_strappend(&ret->data.fru_info.board.fru_id, fru_area);
	free(fru_area);

	/* read any extra fields */
	while ((fru_data[i] != 0xc1) && (i < offset + area_len))
	{
		uint32_t j = i;
		fru_area = get_fru_area_str(fru_data, &i);
		if (fru_area != NULL && strlen(fru_area) > 0) {
			ipmi_printf(" Board Extra           : %s\n", fru_area);
			free(fru_area);
		}
		if (i == j)
			break;
	}

	free(fru_data);
}

/* fru_area_print_product  -  Print FRU Product Area
 *
 * @intf:	ipmi interface
 * @fru:	fru info
 * @id: 	fru id
 * @offset:	offset pointer
 */
static void
fru_area_print_product(struct ipmi_intf * intf, struct fru_info * fru,
		       uint8_t id, uint32_t offset, pp_ipmi_return_t *ret, int * error)
{
	uint8_t * fru_area, * fru_data;
	uint32_t fru_len, area_len, i;

	i = offset;
	fru_len = 0;

	fru_data = malloc(fru->size + 1);
	if (fru_data == NULL) {
		ipmi_printf(" Out of memory!\n");
		return;
	}
	memset(fru_data, 0, fru->size + 1);

	/* read enough to check length field */
	if (read_fru_area(intf, fru, id, i, 2, fru_data, error) == 0)
		fru_len = 8 * fru_data[i + 1];
	if (fru_len <= 0) {
		free(fru_data);
		return;
	}

	/* read in the full fru */
	if (read_fru_area(intf, fru, id, i, fru_len, fru_data, error) < 0) {
		free(fru_data);
		return;
	}

	ipmi_printf(" Product Area (@%d, ver %d, %d bytes)\n", offset, fru_data[i], fru_data[i + 1] * 8);

	ret->data.fru_info.product.ver = fru_data[i++]; /* fru area version */
	area_len = fru_data[i++] * 8; /* fru area length */
	unsigned int u=0;
	for(;u<area_len;u++)
	{
	    if(isprint(fru_data[i+u]))
		printf("%c",fru_data[i+u]);
	    else printf("[%x]",fru_data[i+u]);
	}
	printf("\n");

	ret->data.fru_info.product.lang = fru_data[i++]; /* fru board language */

	fru_area = get_fru_area_str(fru_data, &i);
	if (fru_area) pp_strappend(&ret->data.fru_info.product.manufacturer, fru_area);
	free(fru_area);

	fru_area = get_fru_area_str(fru_data, &i);
	if (fru_area) pp_strappend(&ret->data.fru_info.product.name, fru_area);
	free(fru_area);

	fru_area = get_fru_area_str(fru_data, &i);
	if (fru_area) pp_strappend(&ret->data.fru_info.product.model, fru_area);
	free(fru_area);

	fru_area = get_fru_area_str(fru_data, &i);
	if (fru_area) pp_strappend(&ret->data.fru_info.product.version, fru_area);
	free(fru_area);

	fru_area = get_fru_area_str(fru_data, &i);
	if (fru_area) pp_strappend(&ret->data.fru_info.product.serial, fru_area);
	free(fru_area);

	fru_area = get_fru_area_str(fru_data, &i);
	if (fru_area) pp_strappend(&ret->data.fru_info.product.asset, fru_area);
	free(fru_area);

	fru_area = get_fru_area_str(fru_data, &i);
	if (fru_area) pp_strappend(&ret->data.fru_info.product.fru_id, fru_area);
	free(fru_area);

	/* read any extra fields */
	while ((fru_data[i] != 0xc1) && (i < offset + area_len))
			{
		uint32_t j = i;
		fru_area = get_fru_area_str(fru_data, &i);
		if (fru_area != NULL && strlen(fru_area) > 0) {
			ipmi_printf(" Product Extra         : %s\n", fru_area);
			free(fru_area);
		}
		if (i == j)
			break;
		}

	free(fru_data);
}
  
#define FRU_MULTIREC_CHUNK_SIZE		(255 + sizeof(struct fru_multirec_header))

/* fru_area_print_multirec  -  Print FRU Multi Record Area
 *
 * @intf:	ipmi interface
 * @fru:	fru info
 * @id: 	fru id
 * @offset:	offset pointer
 */
static void
fru_area_print_multirec(struct ipmi_intf * intf UNUSED, struct fru_info * fru UNUSED,
			uint8_t id UNUSED, uint32_t offset UNUSED, pp_ipmi_return_t *ret UNUSED, int * error UNUSED)
{
    (void)&combined_voltage_desc; // avoid warnings
/* TODO: to be implemented
 */
	uint8_t * fru_data;
	uint32_t fru_len, i;
	struct fru_multirec_header * h;
	struct fru_multirec_powersupply * ps;
	struct fru_multirec_dcoutput * dc;
	struct fru_multirec_dcload * dl;
	uint16_t peak_capacity;
	uint8_t peak_hold_up_time;
	uint32_t last_off, len;

	i = last_off = offset;
	fru_len = 0;

	fru_data = malloc(fru->size + 1);
	if (fru_data == NULL) {
		ipmi_printf(" Out of memory!\n");
		return;
	}
	memset(fru_data, 0, fru->size + 1);

	do {
		h = (struct fru_multirec_header *) (fru_data + i);

		// read area in (at most) FRU_MULTIREC_CHUNK_SIZE bytes at a time
		if ((last_off < (i + sizeof(*h))) || (last_off < (i + h->len)))
		{
			len = fru->size - last_off;
			if (len > FRU_MULTIREC_CHUNK_SIZE)
				len = FRU_MULTIREC_CHUNK_SIZE;

			if (read_fru_area(intf, fru, id, last_off, len, fru_data, error) < 0)
				break;

			last_off += len;
		}

		switch (h->type)
		{
		case FRU_RECORD_TYPE_POWER_SUPPLY_INFORMATION:
			ps = (struct fru_multirec_powersupply *)
				(fru_data + i + sizeof (struct fru_multirec_header));

#ifdef WORDS_BIGENDIAN
			ps->capacity		= BSWAP_16(ps->capacity);
			ps->peak_va		= BSWAP_16(ps->peak_va);
			ps->lowend_input1	= BSWAP_16(ps->lowend_input1);
			ps->highend_input1	= BSWAP_16(ps->highend_input1);
			ps->lowend_input2	= BSWAP_16(ps->lowend_input2);
			ps->highend_input2	= BSWAP_16(ps->highend_input2);
			ps->combined_capacity	= BSWAP_16(ps->combined_capacity);
			ps->peak_cap_ht		= BSWAP_16(ps->peak_cap_ht);
#endif
			peak_hold_up_time 	= (ps->peak_cap_ht & 0xf000) >> 12;
			peak_capacity 		= ps->peak_cap_ht & 0x0fff;

			ipmi_printf (" Power Supply Record\n");
			ipmi_printf ("  Capacity                   : %d W\n",
				ps->capacity);
			ipmi_printf ("  Peak VA                    : %d VA\n",
				ps->peak_va);
			ipmi_printf ("  Inrush Current             : %d A\n",
				ps->inrush_current);
			ipmi_printf ("  Inrush Interval            : %d ms\n",
				ps->inrush_interval);
			ipmi_printf ("  Input Voltage Range 1      : %d-%d V\n",
				ps->lowend_input1 / 100, ps->highend_input1 / 100);
			ipmi_printf ("  Input Voltage Range 2      : %d-%d V\n",
				ps->lowend_input2 / 100, ps->highend_input2 / 100);
			ipmi_printf ("  Input Frequency Range      : %d-%d Hz\n",
				ps->lowend_freq, ps->highend_freq);
			ipmi_printf ("  A/C Dropout Tolerance      : %d ms\n",
				ps->dropout_tolerance);
			ipmi_printf ("  Flags                      : %s%s%s%s%s\n",
				ps->predictive_fail ? "'Predictive fail' " : "",
				ps->pfc ? "'Power factor correction' " : "",
				ps->autoswitch ? "'Autoswitch voltage' " : "",
				ps->hotswap ? "'Hot swap' " : "",
				ps->predictive_fail ? ps->rps_threshold ?
				ps->tach ? "'Two pulses per rotation'" : "'One pulse per rotation'" :
				ps->tach ? "'Failure on pin de-assertion'" : "'Failure on pin assertion'" : "");
			ipmi_printf ("  Peak capacity              : %d W\n",
				peak_capacity);
			ipmi_printf ("  Peak capacity holdup       : %d s\n",
				peak_hold_up_time);
			if (ps->combined_capacity == 0)
				ipmi_printf ("  Combined capacity          : not specified\n");
			else
				ipmi_printf ("  Combined capacity          : %d W (%s and %s)\n",
					ps->combined_capacity,
					combined_voltage_desc [ps->combined_voltage1],
					combined_voltage_desc [ps->combined_voltage2]);
			if (ps->predictive_fail)
				ipmi_printf ("  Fan lower threshold        : %d RPS\n",
					ps->rps_threshold);
			break;

		case FRU_RECORD_TYPE_DC_OUTPUT:
			dc = (struct fru_multirec_dcoutput *)
				(fru_data + i + sizeof (struct fru_multirec_header));

#ifdef WORDS_BIGENDIAN
			dc->nominal_voltage	= BSWAP_16(dc->nominal_voltage);
			dc->max_neg_dev		= BSWAP_16(dc->max_neg_dev);
			dc->max_pos_dev		= BSWAP_16(dc->max_pos_dev);
			dc->ripple_and_noise	= BSWAP_16(dc->ripple_and_noise);
			dc->min_current		= BSWAP_16(dc->min_current);
			dc->max_current		= BSWAP_16(dc->max_current);
#endif

			ipmi_printf (" DC Output Record\n");
			ipmi_printf ("  Output Number              : %d\n",
				dc->output_number);
			ipmi_printf ("  Standby power              : %s\n",
				dc->standby ? "Yes" : "No");
			ipmi_printf ("  Nominal voltage            : %.2f V\n",
				(double) dc->nominal_voltage / 100);
			ipmi_printf ("  Max negative deviation     : %.2f V\n",
				(double) dc->max_neg_dev / 100);
			ipmi_printf ("  Max positive deviation     : %.2f V\n",
				(double) dc->max_pos_dev / 100);
			ipmi_printf ("  Ripple and noise pk-pk     : %d mV\n",
				dc->ripple_and_noise);
			ipmi_printf ("  Minimum current draw       : %.3f A\n",
				(double) dc->min_current / 1000);
			ipmi_printf ("  Maximum current draw       : %.3f A\n",
				(double) dc->max_current / 1000);
			break;

		case FRU_RECORD_TYPE_DC_LOAD:
			dl = (struct fru_multirec_dcload *)
				(fru_data + i + sizeof (struct fru_multirec_header));

#ifdef WORDS_BIGENDIAN
			dl->nominal_voltage	= BSWAP_16(dl->nominal_voltage);
			dl->min_voltage		= BSWAP_16(dl->min_voltage);
			dl->max_voltage		= BSWAP_16(dl->max_voltage);
			dl->ripple_and_noise	= BSWAP_16(dl->ripple_and_noise);
			dl->min_current		= BSWAP_16(dl->min_current);
			dl->max_current		= BSWAP_16(dl->max_current);
#endif

			ipmi_printf (" DC Load Record\n");
			ipmi_printf ("  Output Number              : %d\n",
				dl->output_number);
			ipmi_printf ("  Nominal voltage            : %.2f V\n",
				(double) dl->nominal_voltage / 100);
			ipmi_printf ("  Min voltage allowed        : %.2f V\n",
				(double) dl->min_voltage / 100);
			ipmi_printf ("  Max voltage allowed        : %.2f V\n",
				(double) dl->max_voltage / 100);
			ipmi_printf ("  Ripple and noise pk-pk     : %d mV\n",
				dl->ripple_and_noise);
			ipmi_printf ("  Minimum current load       : %.3f A\n",
				(double) dl->min_current / 1000);
			ipmi_printf ("  Maximum current load       : %.3f A\n",
				(double) dl->max_current / 1000);
			break;
		}
		i += h->len + sizeof (struct fru_multirec_header);
	} while (!(h->format & 0x80));

	free(fru_data);
	/*
*/
}

/* __ipmi_fru_print  -  Do actual work to print a FRU by its ID
 *
 * @intf:	ipmi interface
 * @id:		fru id
 *
 * returns -1 on error
 * returns 0 if successful
 * returns 1 if device not present
 */
static int
__ipmi_fru_print(struct ipmi_intf * intf, uint8_t id, pp_ipmi_return_t *ret, int * error)
{
	struct ipmi_rs * rsp;
	struct ipmi_rq req;
	struct fru_info fru;
	struct fru_header header;
	uint8_t msg_data[4];

	memset(&fru, 0, sizeof(struct fru_info));
	memset(&header, 0, sizeof(struct fru_header));

	/*
	 * get info about this FRU
	 */
	memset(msg_data, 0, 4);
	msg_data[0] = id;

	memset(&req, 0, sizeof(req));
	req.msg.netfn = IPMI_NETFN_STORAGE;
	req.msg.cmd = GET_FRU_INFO;
	req.msg.data = msg_data;
	req.msg.data_len = 1;

	rsp = intf->sendrecv(intf, &req, error);
	if (rsp == NULL) {
		ipmi_printf(" Device not present (No Response)\n");
		return -1;
	}
	if (rsp->ccode > 0) {
		ipmi_printf(" Device not present (%s)\n",
			val2str(rsp->ccode, completion_code_vals));
		return -1;
	}

	fru.size = (rsp->data[1] << 8) | rsp->data[0];
	fru.access = rsp->data[2] & 0x1;

	ipmi_printf("fru.size = %d bytes (accessed by %s)\n",
		fru.size, fru.access ? "words" : "bytes");

	if (fru.size < 1) {
		ipmi_printf(" Invalid FRU size %d\n", fru.size);
		return -1;
	}

	/*
	 * retrieve the FRU header
	 */
	msg_data[0] = id;
	msg_data[1] = 0;
	msg_data[2] = 0;
	msg_data[3] = 8;

	memset(&req, 0, sizeof(req));
	req.msg.netfn = IPMI_NETFN_STORAGE;
	req.msg.cmd = GET_FRU_DATA;
	req.msg.data = msg_data;
	req.msg.data_len = 4;

	rsp = intf->sendrecv(intf, &req, error);
	if (rsp == NULL) {
		ipmi_printf(" Device not present (No Response)\n");
		return 1;
	}
	if (rsp->ccode > 0) {
		ipmi_printf(" Device not present (%s)\n",
		       val2str(rsp->ccode, completion_code_vals));
		return 1;
	}

	if (verbose > 1)
		printbuf(rsp->data, rsp->data_len, "FRU DATA");

	memcpy(&header, rsp->data + 1, 8);

	if (header.version != 1) {
		ipmi_printf(" Unknown FRU header version 0x%02x\n",
			header.version);
		return -1;
	}

	/* offsets need converted to bytes
	 * but that conversion is not done to the structure
	 * because we may end up with offset > 255
	 * which would overflow our 1-byte offset field */

        ret->data.fru_info.ver = header.version;
        if (header.offset.internal)
            ret->data.fru_info.info_mask |= PP_IPMI_FRU_INTERNAL_INFO;
        if (header.offset.chassis)
            ret->data.fru_info.info_mask |= PP_IPMI_FRU_CHASSIS_INFO;
        if (header.offset.board)
            ret->data.fru_info.info_mask |= PP_IPMI_FRU_BOARD_INFO;
        if (header.offset.product)
            ret->data.fru_info.info_mask |= PP_IPMI_FRU_PRODUCT_INFO;
        if (header.offset.multi)
            ret->data.fru_info.info_mask |= PP_IPMI_FRU_MULTI_INFO;

	/*
	 * rather than reading the entire part
	 * only read the areas we'll format
	 */

	/* chassis area */
	if ((size_t)(header.offset.chassis*8) >= sizeof(struct fru_header))
		fru_area_print_chassis(intf, &fru, id, header.offset.chassis*8, ret, error);

	/* board area */
	if ((size_t)(header.offset.board*8) >= sizeof(struct fru_header))
		fru_area_print_board(intf, &fru, id, header.offset.board*8, ret, error);

	/* product area */
	if ((size_t)(header.offset.product*8) >= sizeof(struct fru_header))
		fru_area_print_product(intf, &fru, id, header.offset.product*8, ret, error);

	/* multirecord area */
	if ((size_t)(header.offset.multi*8) >= sizeof(struct fru_header))
		fru_area_print_multirec(intf, &fru, id, header.offset.multi*8, ret, error);

	return 0;
}

/* ipmi_fru_print  -  Print a FRU from its SDR locator record
 *
 * @intf:	ipmi interface
 * @fru:	SDR FRU Locator Record
 *
 * returns -1 on error
 */
static int
ipmi_fru_print(struct ipmi_intf * intf, struct sdr_record_fru_locator * fru, pp_ipmi_return_t *ret, int * error)
{
	char desc[17];
	uint8_t save_addr;
	int rc = 0;

	if (ipmi_init_fru_info(ret)) {
		ipmi_printf("Error: Could not allocate return structure.\n");
		return -1;
	}

	if (fru == NULL)
		return __ipmi_fru_print(intf, 0, ret, error);

	/* Logical FRU Device
	 *  dev_type == 0x10
	 *  modifier
	 *   0x00 = IPMI FRU Inventory
	 *   0x01 = DIMM Memory ID
	 *   0x02 = IPMI FRU Inventory
	 *   0x03 = System Processor FRU
	 *   0xff = unspecified
	 *
	 * EEPROM 24C01 or equivalent
	 *  dev_type >= 0x08 && dev_type <= 0x0f
	 *  modifier
	 *   0x00 = unspecified
	 *   0x01 = DIMM Memory ID
	 *   0x02 = IPMI FRU Inventory
	 *   0x03 = System Processor Cartridge
	 */
	if (fru->dev_type != 0x10 &&
	    (fru->dev_type_modifier != 0x02 ||
	     fru->dev_type < 0x08 || fru->dev_type > 0x0f))
		return -1;

	memset(desc, 0, sizeof(desc));
	memcpy(desc, fru->id_string, fru->id_code & 0x01f);
	desc[fru->id_code & 0x01f] = 0;
	ipmi_printf("FRU Device Description : %s (ID %d)\n", desc, fru->device_id);

	switch (fru->dev_type_modifier) {
	case 0x00:
	case 0x02:
		/* save current target address */
		save_addr = intf->target_addr;
		/* set new target address for bridged commands */
		intf->target_addr = fru->dev_slave_addr;

		if (intf->target_addr == intf->bmc_addr &&
		    fru->device_id == 0)
			ipmi_printf(" (Builtin FRU device)\n");
		else
			rc = __ipmi_fru_print(intf, fru->device_id, ret, error);

		/* restore previous target */
		intf->target_addr = save_addr;
		break;
	case 0x01:
		rc = ipmi_spd_print(intf, fru->device_id, error);
		break;
	default:
		if (verbose)
			ipmi_printf(" Unsupported device 0x%02x "
			       "type 0x%02x with modifier 0x%02x\n",
			       fru->device_id, fru->dev_type,
			       fru->dev_type_modifier);
		else
			ipmi_printf(" Unsupported device\n");
	}
	ipmi_printf("\n");

	return rc;
}

/* ipmi_fru_print_all  -  Print builtin FRU + SDR FRU Locator records
 *
 * @intf:	ipmi interface
 *
 * returns -1 on error
 */
static int
ipmi_fru_print_all(struct ipmi_intf * intf, pp_ipmi_return_t *ret, int * error)
{
	struct ipmi_sdr_iterator * itr;
	struct sdr_get_rs * header;
	struct sdr_record_fru_locator * fru;
	int rc;

	ipmi_printf("FRU Device Description : Builtin FRU Device (ID 0)\n");
	/* TODO: Figure out if FRU device 0 may show up in SDR records. */
	rc = ipmi_fru_print(intf, NULL, ret, error);
	ipmi_printf("\n");

	if ((itr = ipmi_sdr_start(intf, error)) == NULL) {
		ipmi_printf("Error: could not find SDR.\n");
		return -1;
	}

	while ((header = ipmi_sdr_get_next_header(intf, itr, error)) != NULL)
	{
		if (header->type != SDR_RECORD_TYPE_FRU_DEVICE_LOCATOR)
			continue;

		fru = (struct sdr_record_fru_locator *)
			ipmi_sdr_get_record(intf, header, itr, error);
		if (fru == NULL)
			continue;
		rc = ipmi_fru_print(intf, fru, ret, error);
		free(fru);
	}

	ipmi_sdr_end(intf, itr);

	return rc;
}

int ipmi_fru_main(struct ipmi_intf * intf, int subcmd, pp_ipmi_parameter_t *params UNUSED, pp_ipmi_return_t *ret, int * error)
{
	switch ((pp_ipmi_fru_subcommand_t) subcmd) {
		case PP_IPMI_FRU_SUBCMD_PRINT:
			return ipmi_fru_print_all(intf, ret, error);
		case PP_IPMI_FRU_SUBCMD_BOARD:
			return ipmi_fru_print(intf, NULL, ret, error);
		default:
			ipmi_printf("Invalid FRU command: %d\n", subcmd);
			ipmi_set_error(error, IPMI_ERROR_INVALID_COMMAND);
			return -1;
	}
}
