/*
 * ipmi_lan.c
 * 
 * Some LAN configuration commands for ipmitool.
 * (only a very limited subset for alert destination
 * configuration)
 * 
 * (c) 2005 Peppercon AG, Georg Hoesch <geo@peppercon.de>
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#ifdef WIN32

#include <assert.h>
#include <winsock2.h>
#include <windows.h>

#else // WIN32

#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <unistd.h>

#endif // WIN32

#include <errno.h>
#include <sys/types.h>
#include <signal.h>
#include <setjmp.h>

#include <config.h>

#include <ipmitool/ipmi.h>
#include <ipmitool/ipmi_intf.h>
#include <ipmitool/helper.h>
#include <ipmitool/ipmi_strings.h>
#include <ipmitool/ipmi_lan.h>
#include <ipmitool/ipmi_lanp.h>
#include <ipmitool/ipmi_print.h>
#include <ipmitool/ipmi_error.h>
#include <ipmi_params.h>
#include <ipmi_return.h>

#define PP_IPMI_LAN_CHANNEL_NUMBER        0x01
#define IPMI_CMD_SET_LAN_CONFIGURATION    0x01
#define IPMI_CMD_GET_LAN_CONFIGURATION    0x02

/* internal prototypes */
static int lan_get_destinations(struct ipmi_intf* intf, pp_ipmi_return_t* ipmi_ret, int* err);
static int lan_set_destination(struct ipmi_intf* intf, pp_ipmi_parameter_t* ipmi_rq, int* err);
static int lan_commit(struct ipmi_intf* intf, int* err);

int ipmi_lan_main(struct ipmi_intf * intf, int subcmd, pp_ipmi_parameter_t *params, pp_ipmi_return_t *ipmi_ret, int * error)
{
    int ret = -1;

    /* all the pef parameters commands need admin level */
    ipmi_intf_session_set_privlvl(intf, IPMI_SESSION_PRIV_ADMIN);

    switch ((pp_ipmi_lanp_subcommand_t) subcmd) {
    
        case PP_IPMI_LAN_DESTINATION_GET:
            ret = lan_get_destinations(intf, ipmi_ret, error);
            break;
        case PP_IPMI_LAN_DESTINATION_SET:
            ret = lan_set_destination(intf, params, error);
            break;
        default:
            ipmi_printf("libpp_impi: invalid command: LAN - %d\n", subcmd);
            ipmi_set_error(error, IPMI_ERROR_INVALID_COMMAND);
    }

    return ret;
}

static int lan_commit(struct ipmi_intf* intf, int* err) {
    struct ipmi_rs * rsp;
    struct ipmi_rq req;
    uint8_t msg_data[3];

    msg_data[0] = PP_IPMI_LAN_CHANNEL_NUMBER;
    msg_data[1] = 0;
    msg_data[2] = 2;

    memset(&req, 0, sizeof(req));
    req.msg.netfn    = IPMI_NETFN_TRANSPORT;
    req.msg.cmd      = IPMI_CMD_SET_LAN_CONFIGURATION;
    req.msg.data     = msg_data;
    req.msg.data_len = 3;

    rsp = intf->sendrecv(intf, &req, err);

    if (!rsp) {
        return -1;
    }
    
    /* the completion code is not checked as the bmc might return an
     * error if rollback is not supported. We just assume it works.  */
    return 0;
}

static int lan_get_destination_cnt(struct ipmi_intf* intf, uint8_t* cnt, int* error) {
    struct ipmi_rs * rsp;
    struct ipmi_rq req;
    uint8_t msg_data[4];

    msg_data[0] = PP_IPMI_LAN_CHANNEL_NUMBER;
    msg_data[1] = 17;
    msg_data[2] = 0;
    msg_data[3] = 0;

    memset(&req, 0, sizeof(req));
    req.msg.netfn    = IPMI_NETFN_TRANSPORT;
    req.msg.cmd      = IPMI_CMD_GET_LAN_CONFIGURATION;
    req.msg.data     = msg_data;
    req.msg.data_len = 4;

    rsp = intf->sendrecv(intf, &req, error);

    if (!rsp || rsp->ccode) {
        return -1;
    }
    
    *cnt = (rsp->data[1] & 0x0F);
    
    return 0;
}

static int lan_get_destination_by_index(struct ipmi_intf* intf,
                                   uint8_t idx,
                                   pp_ipmi_lanp_dest_t* dest,
                                   int* err)
{
    struct ipmi_rs * rsp;
    struct ipmi_rq req;
    uint8_t msg_data[4];

    /* param 18 - destination type */
    msg_data[0] = PP_IPMI_LAN_CHANNEL_NUMBER;
    msg_data[1] = 18;
    msg_data[2] = idx;
    msg_data[3] = 0;

    memset(&req, 0, sizeof(req));
    req.msg.netfn    = IPMI_NETFN_TRANSPORT;
    req.msg.cmd      = IPMI_CMD_GET_LAN_CONFIGURATION;
    req.msg.data     = msg_data;
    req.msg.data_len = 4;

    rsp = intf->sendrecv(intf, &req, err);

    if (!rsp || rsp->ccode) {
        return -1;
    }
    
    // fill *dest from rsp->data[]
    dest->id = idx;
    
    dest->acked = 0;
    if ((rsp->data[2] & 0x80) == 0x80) {
        dest->acked = 1;
    }
    dest->type = rsp->data[2] & 0x7;
    dest->timeout = rsp->data[3];
    dest->retries = rsp->data[4] & 0x7;
    
    /* param 19 - destination address */
    msg_data[0] = PP_IPMI_LAN_CHANNEL_NUMBER;
    msg_data[1] = 19;
    msg_data[2] = idx;
    msg_data[3] = 0;

    memset(&req, 0, sizeof(req));
    req.msg.netfn    = IPMI_NETFN_TRANSPORT;
    req.msg.cmd      = IPMI_CMD_GET_LAN_CONFIGURATION;
    req.msg.data     = msg_data;
    req.msg.data_len = 4;

    rsp = intf->sendrecv(intf, &req, err);

    if (!rsp || rsp->ccode) {
        return -1;
    }
    
    // fill *dest from rsp->data[]
    assert (rsp->data_len == 14);
    dest->addr_format = rsp->data[2] & 0x0F;
    dest->backup_gateway = rsp->data[3] & 0x01;
    memcpy(dest->ip_addr, &(rsp->data[4]), 4);
    memcpy(dest->mac_addr, &(rsp->data[8]), 6); 
    
    /* param 192 - destinationa address */
    msg_data[0] = PP_IPMI_LAN_CHANNEL_NUMBER;
    msg_data[1] = 192;
    msg_data[2] = idx;
    msg_data[3] = 0;

    memset(&req, 0, sizeof(req));
    req.msg.netfn    = IPMI_NETFN_TRANSPORT;
    req.msg.cmd      = IPMI_CMD_GET_LAN_CONFIGURATION;
    req.msg.data     = msg_data;
    req.msg.data_len = 4;

    rsp = intf->sendrecv(intf, &req, err);

    if (!rsp || rsp->ccode) {
        // foreign BMCs dont support the Peppercon OEM cmd
        // so, we should ignore any errors here
        dest->email[0] = '\0';
    } else {
        // fill *dest from rsp->data[]
        snprintf(dest->email, 64, "%s", rsp->data + 2);
    }

    return 0;
}

static int lan_get_destinations(struct ipmi_intf* intf, pp_ipmi_return_t* ipmi_ret, int* err) {
    /* returns ipmi_ret->lan_destination_set, a vector of pp_ipmi_lanp_dest_t */
    uint8_t cnt;
    uint8_t i;
    pp_ipmi_lanp_dest_t* dest;

    if (ipmi_init_lan_destination_list(ipmi_ret)) {
        return -1;
    }
    
    if (lan_get_destination_cnt(intf, &cnt, err) == -1) {
        pp_ipmi_cleanup_ret(ipmi_ret);
        return -1;
    }
    
    for (i=0; i<=cnt; i++) {
        dest = ipmi_init_lan_destination_list_entry(ipmi_ret);

        if (lan_get_destination_by_index(intf, i, dest, err) == -1) {
            pp_ipmi_cleanup_ret(ipmi_ret);
            return -1;
        }
    }

    return 0;
}

static int lan_set_destination(struct ipmi_intf* intf, pp_ipmi_parameter_t* ipmi_rq, int* err) {
    struct ipmi_rs * rsp;
    struct ipmi_rq req;
    pp_ipmi_lanp_dest_t* dest;
    uint8_t msg_data[70];
    
    dest = &(ipmi_rq->data.lan_destination_set);
    
    /* param 18 - destination type */
    msg_data[0] = PP_IPMI_LAN_CHANNEL_NUMBER;
    msg_data[1] = 18;
    msg_data[2] = dest->id;
    msg_data[3] = dest->type;
    if (dest->acked == 1) {
        msg_data[3] = msg_data[3] | 0x80;
    }
    msg_data[4] = dest->timeout;
    msg_data[5] = dest->retries;

    memset(&req, 0, sizeof(req));
    req.msg.netfn    = IPMI_NETFN_TRANSPORT;
    req.msg.cmd      = IPMI_CMD_SET_LAN_CONFIGURATION;
    req.msg.data     = msg_data;
    req.msg.data_len = 6;

    rsp = intf->sendrecv(intf, &req, err);

    if (!rsp || rsp->ccode) {
        return -1;
    }

    /* param 19 - destination address */
    msg_data[0] = PP_IPMI_LAN_CHANNEL_NUMBER;
    msg_data[1] = 19;
    msg_data[2] = dest->id;
    msg_data[3] = dest->addr_format;
    msg_data[4] = dest->backup_gateway;
    memcpy(msg_data+5, dest->ip_addr, 4);
    memcpy(msg_data+9, dest->mac_addr, 6);

    memset(&req, 0, sizeof(req));
    req.msg.netfn    = IPMI_NETFN_TRANSPORT;
    req.msg.cmd      = IPMI_CMD_SET_LAN_CONFIGURATION;
    req.msg.data     = msg_data;
    req.msg.data_len = 15;

    rsp = intf->sendrecv(intf, &req, err);

    if (!rsp || rsp->ccode) {
        return -1;
    }

    /* param 192 - email */
    msg_data[0] = PP_IPMI_LAN_CHANNEL_NUMBER;
    msg_data[1] = 192;
    msg_data[2] = dest->id;
    strncpy((char *)msg_data+3, (char*)(dest->email), 64);

    memset(&req, 0, sizeof(req));
    req.msg.netfn    = IPMI_NETFN_TRANSPORT;
    req.msg.cmd      = IPMI_CMD_SET_LAN_CONFIGURATION;
    req.msg.data     = msg_data;
    req.msg.data_len = 67;

    rsp = intf->sendrecv(intf, &req, err);

    if (!rsp || rsp->ccode) {
        return -1;
    }
    
    /* param 193 - SNMP community string */
    msg_data[0] = PP_IPMI_LAN_CHANNEL_NUMBER;
    msg_data[1] = 193;
    msg_data[2] = dest->id;
    strncpy((char*)msg_data+3, (char*)(dest->community), 64);

    memset(&req, 0, sizeof(req));
    req.msg.netfn    = IPMI_NETFN_TRANSPORT;
    req.msg.cmd      = IPMI_CMD_SET_LAN_CONFIGURATION;
    req.msg.data     = msg_data;
    req.msg.data_len = 67;

    rsp = intf->sendrecv(intf, &req, err);

    if (!rsp || rsp->ccode) {
        return -1;
    }
    
    return lan_commit(intf, err);
}
