/**
 * \file ipmi_oem_opma.c
 *
 * Description: OEM commands for OPMA boards
 *
 * (c) 2005 Peppercon AG, Michael Baumann <miba@peppercon.de>
 */

#include <config.h>

#include <ipmitool/ipmi.h>
#include <ipmitool/ipmi_oem_opma.h>
#include <ipmitool/ipmi_intf.h>
#include <ipmitool/ipmi_print.h>
#include <ipmitool/ipmi_error.h>
#include <ipmitool/ipmi_strings.h>

static int ipmi_oem_opma_clear_cmos(struct ipmi_intf * intf, pp_ipmi_parameter_t *params UNUSED,
				    pp_ipmi_return_t *ret UNUSED, int * error)
{
    struct ipmi_rs *rsp;
    struct ipmi_rq req;

    memset(&req, 0, sizeof(req));
    req.msg.netfn      = IPMI_NETFN_OEM_OPMA;
    req.msg.cmd	       = BMC_OEM_OPMA_CLEAR_CMOS;
    req.msg.data_len   = 0;
    req.msg.data       = NULL;

    rsp = intf->sendrecv(intf, &req, error);
    if (!rsp) {
        ipmi_printf("Error in BMC cmd '[OEM OPMA] Clear CMOS' \n");
        return -1;
    }
    if (rsp->ccode) {
        ipmi_printf("BMC cmd '[OEM OPMA] Clear CMOS' failed: %s\n",
               val2str(rsp->ccode, completion_code_vals));
        return -1;
    } 
    
    return 0;
}

static int ipmi_oem_opma_set_local_lockout(struct ipmi_intf * intf, pp_ipmi_parameter_t *params,
					   pp_ipmi_return_t *ret UNUSED, int * error)
{
    struct ipmi_rs *rsp;
    struct ipmi_rq req;

    unsigned char b[1];

    b[0] = params->data.oem_opma_set_local_lockout.lock ? 0x00 : 0x01;
	
    memset(&req, 0, sizeof(req));
    req.msg.netfn      = IPMI_NETFN_OEM_OPMA;
    req.msg.cmd	       = BMC_OEM_OPMA_SET_LOCAL_LOCKOUT;
    req.msg.data_len   = sizeof(b);
    req.msg.data       = b;

    rsp = intf->sendrecv(intf, &req, error);
    if (!rsp) {
        ipmi_printf("Error in BMC cmd '[OEM OPMA] SetLocalLockout' \n");
        return -1;
    }
    if (rsp->ccode) {
        ipmi_printf("BMC cmd '[OEM OPMA] SetLocalLockout' failed: %s\n",
               val2str(rsp->ccode, completion_code_vals));
        return -1;
    }    
    
    return 0;
}

static int ipmi_oem_opma_get_local_lockout(struct ipmi_intf * intf, pp_ipmi_parameter_t *params UNUSED,
					   pp_ipmi_return_t *ret, int * error)
{
    struct ipmi_rs *rsp;
    struct ipmi_rq req;

    memset(&req, 0, sizeof(req));
    req.msg.netfn      = IPMI_NETFN_OEM_OPMA;
    req.msg.cmd	       = BMC_OEM_OPMA_GET_LOCAL_LOCKOUT;
    req.msg.data_len   = 0;
    req.msg.data       = NULL;

    rsp = intf->sendrecv(intf, &req, error);
    if (!rsp) {
        ipmi_printf("Error in BMC cmd '[OEM OPMA] GetLocalLockout' \n");
        return -1;
    }
    if (rsp->ccode) {
        ipmi_printf("BMC cmd '[OEM OPMA] GetLocalLockout' failed: %s\n",
               val2str(rsp->ccode, completion_code_vals));
        return -1;
    }    
    if (rsp->data_len < 1) {
	ipmi_printf("Error in BMC cmd '[OEM OPMA] GetLocalLockout', response truncated1\n");
    }
    
    ret->data.oem_opma_get_local_lockout.locked = (rsp->data[0] == 0x01) ? 0 : 1;
    
    return 0;
}

int ipmi_oem_opma_main(struct ipmi_intf * intf, int subcmd, pp_ipmi_parameter_t *params,
		       pp_ipmi_return_t *ret, int * error)
{
    pp_ipmi_cleanup_ret(ret);
    switch ((pp_ipmi_kvm_subcommand_t) subcmd) {
      case PP_IPMI_OEM_OPMA_CLEAR_CMOS:
	  return ipmi_oem_opma_clear_cmos(intf, params, ret, error);
      case PP_IPMI_OEM_OPMA_SET_LOCAL_LOCKOUT:
	  return ipmi_oem_opma_set_local_lockout(intf, params, ret, error);
      case PP_IPMI_OEM_OPMA_GET_LOCAL_LOCKOUT:
	  return ipmi_oem_opma_get_local_lockout(intf, params, ret, error);
      default:
	  ipmi_printf("Invalid OEM OPMA command: %d\n", subcmd);
	  ipmi_set_error(error, IPMI_ERROR_INVALID_COMMAND);
	  return -1;
    }
}

