/*
 * ipmi_pef.c
 * 
 * Some PEF configuration commands for ipmitool.
 * 
 * (c) 2005 Peppercon AG, Georg Hoesch <geo@peppercon.de>
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#ifdef WIN32

#include <winsock2.h>
#include <windows.h>

#else // WIN32

#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <unistd.h>

#endif // WIN32

#include <errno.h>
#include <sys/types.h>
#include <signal.h>
#include <setjmp.h>

#include <config.h>

#include <ipmitool/ipmi.h>
#include <ipmitool/ipmi_intf.h>
#include <ipmitool/helper.h>
#include <ipmitool/ipmi_strings.h>
#include <ipmitool/ipmi_lanp.h>
#include <ipmitool/ipmi_print.h>
#include <ipmitool/ipmi_error.h>
#include <ipmitool/ipmi_pef.h>
#include <ipmi_params.h>
#include <ipmi_return.h>

#define IPMI_CMD_GET_PEF_CONFIGURATION    0x13
#define IPMI_CMD_SET_PEF_CONFIGURATION    0x12

/* internal prototypes */
static int pef_get_filters(struct ipmi_intf* intf, pp_ipmi_return_t* ipmi_ret, int* err);
static int pef_set_filter(struct ipmi_intf* intf, pp_ipmi_parameter_t* ipmi_rq, int* err);
static int pef_get_policies(struct ipmi_intf* intf, pp_ipmi_return_t* ipmi_ret, int* err);
static int pef_set_policy(struct ipmi_intf* intf, pp_ipmi_parameter_t* ipmi_rq, int* err);
static int pef_commit(struct ipmi_intf* intf, int* err);

int
ipmi_pef_main(struct ipmi_intf * intf, int subcmd, pp_ipmi_parameter_t *params, pp_ipmi_return_t *ipmi_ret, int * error)
{
    int ret = -1;

    /* all the pef parameters commands need admin level */
    ipmi_intf_session_set_privlvl(intf, IPMI_SESSION_PRIV_ADMIN);

    switch ((pp_ipmi_lanp_subcommand_t) subcmd) {
    
        case PP_IPMI_PEF_FILTER_GET:
            ret = pef_get_filters(intf, ipmi_ret, error);
            break;
        case PP_IPMI_PEF_FILTER_SET:
            ret = pef_set_filter(intf, params, error);
            break;
        case PP_IPMI_PEF_POLICY_GET:
            ret = pef_get_policies(intf, ipmi_ret, error);
            break;
        case PP_IPMI_PEF_POLICY_SET:
            ret = pef_set_policy(intf, params, error);
            break;
        default:
            ipmi_printf("libpp_impi: invalid command: SE - %d\n", subcmd);
            ipmi_set_error(error, IPMI_ERROR_INVALID_COMMAND);
    }

    return ret;
}

static int pef_commit(struct ipmi_intf* intf, int* err) {
    struct ipmi_rs * rsp;
    struct ipmi_rq req;
    uint8_t msg_data[2];

    msg_data[0] = 0;
    msg_data[1] = 2;

    memset(&req, 0, sizeof(req));
    req.msg.netfn    = IPMI_NETFN_SE;
    req.msg.cmd      = IPMI_CMD_SET_PEF_CONFIGURATION;
    req.msg.data     = msg_data;
    req.msg.data_len = 2;

    rsp = intf->sendrecv(intf, &req, err);

    if (!rsp) {
        return -1;
    }
    
    /* the completion code is not checked as the bmc might return an
     * error if rollback is not supported. We just assume it works.  */
    return 0;
}

static int pef_get_filter_cnt(struct ipmi_intf* intf, uint8_t* cnt, int* error) {
    struct ipmi_rs * rsp;
    struct ipmi_rq req;
    uint8_t msg_data[3];

    msg_data[0] = 5;
    msg_data[1] = 0;
    msg_data[2] = 0;

    memset(&req, 0, sizeof(req));
    req.msg.netfn    = IPMI_NETFN_SE;
    req.msg.cmd      = IPMI_CMD_GET_PEF_CONFIGURATION;
    req.msg.data     = msg_data;
    req.msg.data_len = 3;

    rsp = intf->sendrecv(intf, &req, error);

    if (!rsp || rsp->ccode) {
        return -1;
    }
    
    *cnt = (rsp->data[1] & 0x7F);
    
    return 0;
}

static int pef_get_filter_by_index(struct ipmi_intf* intf,
                                   uint8_t idx,
                                   pp_ipmi_pef_filter_t* filter,
                                   int* err)
{
    struct ipmi_rs * rsp;
    struct ipmi_rq req;
    uint8_t msg_data[3];
    uint8_t* data;

    msg_data[0] = 6;
    msg_data[1] = idx;
    msg_data[2] = 0;

    memset(&req, 0, sizeof(req));
    req.msg.netfn    = IPMI_NETFN_SE;
    req.msg.cmd      = IPMI_CMD_GET_PEF_CONFIGURATION;
    req.msg.data     = msg_data;
    req.msg.data_len = 3;

    rsp = intf->sendrecv(intf, &req, err);

    if (!rsp || rsp->ccode) {
        return -1;
    }
    
    // fill *filter from rsp->data[]
    // assert (rsp->data_len == 22);
    filter->id = idx;
    data = (uint8_t*)filter;
    memcpy(data+1, &(rsp->data[2]), rsp->data_len-2);
    
    return 0;
}

static int pef_get_filters(struct ipmi_intf* intf, pp_ipmi_return_t* ipmi_ret, int* err) {
    /* returns ipmi_ret->pef_filter_list, a vector of pp_ipmi_pef_filter_t */
    uint8_t cnt;
    uint8_t i;
    pp_ipmi_pef_filter_t* filter;

    if (ipmi_init_pef_filter_list(ipmi_ret)) {
        return -1;
    }
    
    if (pef_get_filter_cnt(intf, &cnt, err) == -1) {
        pp_ipmi_cleanup_ret(ipmi_ret);
        return -1;
    }
    
    for (i=1; i<=cnt; i++) {
        filter = ipmi_init_pef_filter_list_entry(ipmi_ret);

        if (pef_get_filter_by_index(intf, i, filter, err) == -1) {
            pp_ipmi_cleanup_ret(ipmi_ret);
            return -1;
        }
    }
    
    return 0;
}

static int pef_set_filter(struct ipmi_intf* intf, pp_ipmi_parameter_t* ipmi_rq, int* err) {
    struct ipmi_rs * rsp;
    struct ipmi_rq req;
    uint8_t msg_data[22];

    msg_data[0] = 6;
    memcpy(&(msg_data[1]), &(ipmi_rq->data.pef_filter_set), 21);   // 1 byte 'set index' + 20 bytes filter

    memset(&req, 0, sizeof(req));
    req.msg.netfn    = IPMI_NETFN_SE;
    req.msg.cmd      = IPMI_CMD_SET_PEF_CONFIGURATION;
    req.msg.data     = msg_data;
    req.msg.data_len = 22;

    rsp = intf->sendrecv(intf, &req, err);

    if (!rsp || rsp->ccode) {
        return -1;
    }
    
    return pef_commit(intf, err);
}

static int pef_get_policy_cnt(struct ipmi_intf* intf, uint8_t* cnt, int* error) {
    struct ipmi_rs * rsp;
    struct ipmi_rq req;
    uint8_t msg_data[3];

    msg_data[0] = 8;
    msg_data[1] = 0;
    msg_data[2] = 0;

    memset(&req, 0, sizeof(req));
    req.msg.netfn    = IPMI_NETFN_SE;
    req.msg.cmd      = IPMI_CMD_GET_PEF_CONFIGURATION;
    req.msg.data     = msg_data;
    req.msg.data_len = 3;

    rsp = intf->sendrecv(intf, &req, error);

    if (!rsp || rsp->ccode) {
        return -1;
    }
    
    *cnt = (rsp->data[1] & 0x7F);
    
    return 0;
}

static int pef_get_policy_by_index(struct ipmi_intf* intf,
                                   uint8_t idx,
                                   pp_ipmi_pef_policy_t* policy,
                                   int* err)
{
    struct ipmi_rs * rsp;
    struct ipmi_rq req;
    uint8_t msg_data[3];
    uint8_t* data;

    msg_data[0] = 9;
    msg_data[1] = idx;
    msg_data[2] = 0;

    memset(&req, 0, sizeof(req));
    req.msg.netfn    = IPMI_NETFN_SE;
    req.msg.cmd      = IPMI_CMD_GET_PEF_CONFIGURATION;
    req.msg.data     = msg_data;
    req.msg.data_len = 3;

    rsp = intf->sendrecv(intf, &req, err);

    if (!rsp || rsp->ccode) {
        return -1;
    }
    
    // assert(rsp->data_len == 5);
    policy->id = idx;
    data = (uint8_t*)policy;
    memcpy(data+1, &(rsp->data[2]), rsp->data_len-2);
    
    return 0;
}

static int pef_get_policies(struct ipmi_intf* intf, pp_ipmi_return_t* ipmi_ret, int* err) {
    /* returns ipmi_ret->pef_policy_list, a vector of pp_ipmi_pef_policy_t */
    uint8_t cnt;
    uint8_t i;
    pp_ipmi_pef_policy_t* policy;

    if (ipmi_init_pef_policy_list(ipmi_ret)) {
        return -1;
    }
    
    if (pef_get_policy_cnt(intf, &cnt, err) == -1) {
        pp_ipmi_cleanup_ret(ipmi_ret);
        return -1;
    }
    
    for (i=1; i<=cnt; i++) {
        policy = ipmi_init_pef_policy_list_entry(ipmi_ret);

        if (pef_get_policy_by_index(intf, i, policy, err) == -1) {
            pp_ipmi_cleanup_ret(ipmi_ret);
            return -1;
        }
    }
    
    return 0;
}

static int pef_set_policy(struct ipmi_intf* intf, pp_ipmi_parameter_t* ipmi_rq, int* err) {
    struct ipmi_rs * rsp;
    struct ipmi_rq req;
    uint8_t msg_data[5];

    msg_data[0] = 9;
    memcpy(&(msg_data[1]), &(ipmi_rq->data.pef_policy_set), 4);   // 1 byte 'set index' + 3 bytes 

    memset(&req, 0, sizeof(req));
    req.msg.netfn    = IPMI_NETFN_SE;
    req.msg.cmd      = IPMI_CMD_SET_PEF_CONFIGURATION;
    req.msg.data     = msg_data;
    req.msg.data_len = 5;

    rsp = intf->sendrecv(intf, &req, err);

    if (!rsp || rsp->ccode) {
        return -1;
    }
    
    return pef_commit(intf, err);
}
