#include <config.h>

#if !defined(WIN32)
#include <netdb.h>
#endif

#include <pp/ipmi.h>

#include <ipmitool/ipmi.h>
#include <ipmitool/ipmi_intf.h>
#include <ipmitool/ipmi_print.h>
#include <ipmitool/ipmi_error.h>
#include <ipmitool/ipmi_strings.h>

#include <ipmitool/ipmi_raritanoem.h>
#include <ipmitool/ipmi_selftest.h>

#include <ipmi_return.h>

#define IPMI_ERR_SLFTST_NOT_SUPPORTED   0x80
#define IPMI_ERR_SLFTST_FAILED          0x81
#define IPMI_ERR_SLFTST_BUSY            0x82
#define IPMI_ERR_SLFTST_NO_RESULTS      0x83

#define IPMI_OEM_PP_SLFTST_VIDEO            1
#define IPMI_OEM_PP_SLFTST_VIDEO_STATUS         1
#define IPMI_OEM_PP_SLFTST_VIDEO_CRC            2
#define IPMI_OEM_PP_SLFTST_DDC              2
#define IPMI_OEM_PP_SLFTST_DDC_INFO             1
#define IPMI_OEM_PP_SLFTST_USB              3
#define IPMI_OEM_PP_SLFTST_USB_OPERATION        1
#define IPMI_OEM_PP_SLFTST_NIC              4
#define IPMI_OEM_PP_SLFTST_NIC_STATUS           1
#define IPMI_OEM_PP_SLFTST_NIC_LOOPBACK         2
#define IPMI_OEM_PP_SLFTST_NIC_PING             3
#define IPMI_OEM_PP_SLFTST_NIC_BCASTPING        4
#define IPMI_OEM_PP_SLFTST_SERIAL           5
#define IPMI_OEM_PP_SLFTST_SERIAL_ECHO          1
#define IPMI_OEM_PP_SLFTST_IPMB             6
#define IPMI_OEM_PP_SLFTST_IPMB_BMCSTATUS      1
#define IPMI_OEM_PP_SLFTST_FML              7
#define IPMI_OEM_PP_SLFTST_FML_ESB2STATUS       1

static struct {
    int comp;
    int test;
    const char *description;
} tests[] = {
    { IPMI_OEM_PP_SLFTST_VIDEO, IPMI_OEM_PP_SLFTST_VIDEO_STATUS, "Video status" },
    { IPMI_OEM_PP_SLFTST_DDC, IPMI_OEM_PP_SLFTST_DDC_INFO, "DDC info" },
    { IPMI_OEM_PP_SLFTST_USB, IPMI_OEM_PP_SLFTST_USB_OPERATION, "USB operation" },
    { IPMI_OEM_PP_SLFTST_NIC, IPMI_OEM_PP_SLFTST_NIC_STATUS, "NIC status" },
    { IPMI_OEM_PP_SLFTST_NIC, IPMI_OEM_PP_SLFTST_NIC_LOOPBACK, "NIC loopback" },
    { IPMI_OEM_PP_SLFTST_NIC, IPMI_OEM_PP_SLFTST_NIC_PING, "Ping" },
    { IPMI_OEM_PP_SLFTST_NIC, IPMI_OEM_PP_SLFTST_NIC_BCASTPING, "Broadcast ping" },
    { IPMI_OEM_PP_SLFTST_SERIAL, IPMI_OEM_PP_SLFTST_SERIAL_ECHO, "Serial echo" },
    { IPMI_OEM_PP_SLFTST_IPMB, IPMI_OEM_PP_SLFTST_IPMB_BMCSTATUS, "IPMB BMC status" },
    { IPMI_OEM_PP_SLFTST_FML, IPMI_OEM_PP_SLFTST_FML_ESB2STATUS, "FML ESB2 status" },
    /* end of list */
    { 0, 0, NULL },
};

static struct {
    int ccode;
    int error;
} errors[] = {
    { IPMI_ERR_SLFTST_NOT_SUPPORTED, IPMI_ERROR_SELFTEST_NOT_SUPPORTED },
    { IPMI_ERR_SLFTST_FAILED, IPMI_ERROR_SELFTEST_FAILED },
    { IPMI_ERR_SLFTST_BUSY, IPMI_ERROR_SELFTEST_BUSY },
    { IPMI_ERR_SLFTST_NO_RESULTS, IPMI_ERROR_SELFTEST_NO_RESULT },
    /* end of list */
    { 0, IPMI_ERROR_UNKNOWN_ERROR },
};

static const char * get_description(int comp, int test) {
    int i = 0;

    while (tests[i].description != NULL) {
	if (tests[i].comp == comp && tests[i].test == test)
	    return tests[i].description;
	i++;
    }

    return "Unknown";
}

static int get_error_code(int ccode) {
    int i = 0;

    while (errors[i].error != IPMI_ERROR_UNKNOWN_ERROR) {
	if (errors[i].ccode == ccode)
	    return errors[i].error;
	i++;
    }

    return ccode;
}

static const char* get_error_string(int ccode) {
    return pp_ipmi_get_error_string(get_error_code(ccode));
}


#ifdef WIN32
# pragma pack(1)
#endif

// header for self test operations
typedef struct oem_pp_selftest_header_s {
    unsigned char comp; // see IPMI_OEM_PP_SLFTST_XXXX
    unsigned char test; // see IPMI_OEM_PP_SLFTST_XXXX_YYYY
} PACKED_STRUCT oem_pp_selftest_header_t;

// generic type for self test operations
typedef struct oem_pp_selftest_s {
    oem_pp_selftest_header_t header;
    unsigned char data[0]; // additional params
} PACKED_STRUCT oem_pp_selftest_t;

// specific operation structures
typedef struct {
    oem_pp_selftest_header_t header;
    unsigned char ddc_data[128];
} PACKED_STRUCT oem_pp_selftest_ddc_t;

typedef struct {
    oem_pp_selftest_header_t header;
    u_int16_t x_res;
    u_int16_t y_res;
    u_int16_t hf;
    u_int16_t vf;
} PACKED_STRUCT oem_pp_selftest_video_status_t;

typedef struct {
    oem_pp_selftest_header_t header;
    u_int32_t crc;
} PACKED_STRUCT oem_pp_selftest_video_crc_t;

typedef struct {
    oem_pp_selftest_header_t header;
    unsigned char channel;
} PACKED_STRUCT oem_pp_selftest_usb_t;

typedef struct {
    oem_pp_selftest_header_t header;
    unsigned char channel;
} PACKED_STRUCT oem_pp_selftest_nic_loopback_t;

typedef struct {
    oem_pp_selftest_header_t header;
    unsigned char channel;
    u_int32_t ip_addr;
} PACKED_STRUCT oem_pp_selftest_nic_ping_t;

typedef struct {
    oem_pp_selftest_header_t header;
    unsigned char channel;
} PACKED_STRUCT oem_pp_selftest_nic_status_send_t;

typedef struct {
    oem_pp_selftest_header_t header;
    BITFIELD3(unsigned char, speed : 4, duplex : 3, link : 1);
} PACKED_STRUCT oem_pp_selftest_nic_status_reply_t;


#ifdef WIN32
# pragma pack()
#endif

static int timedout(time_t start_time, int timeout) {
    return timeout ? ((time(NULL) - start_time) > (timeout + 1)) : 1;
}

// send a self test request
static int ipmi_selftest_send(struct ipmi_intf * intf, oem_pp_selftest_t *cmd,
			      size_t len, int *error, int timeout_sec) {
    struct ipmi_rs * rsp;
    struct ipmi_rq req;
    time_t start_time = time(NULL);

retry:
    memset(&req, 0, sizeof(req));
    req.msg.netfn = IPMI_NETFN_RARITANOEM;
    req.msg.cmd = IPMI_RARITANOEM_PERFORM_SELFTEST;
    req.msg.data = (u_int8_t *)cmd;
    req.msg.data_len = len;

    rsp = intf->sendrecv(intf, &req, error);
    if (!rsp) {
	ipmi_printf("Error in self test (%s) command\n",
	    get_description(cmd->header.comp, cmd->header.test));
        return -1;
    }
    if (rsp->ccode == IPMI_ERR_SLFTST_BUSY && !timedout(start_time, timeout_sec)) {
	ipmi_reset_error(error);
	usleep(100000);
	goto retry;
    }
    if (rsp->ccode) {
	ipmi_printf("Error in self test (%s) command: %s\n",
	    get_description(cmd->header.comp, cmd->header.test),
	    get_error_string(rsp->ccode));
	ipmi_reset_error(error);
	ipmi_set_error(error, get_error_code(rsp->ccode));
	return -1;
    }

    return 0;
}

// read self test result
static int get_selftest_result(struct ipmi_intf * intf, oem_pp_selftest_t *result_cmd,
			       oem_pp_selftest_t *request_cmd, size_t *len, int *error,
			       int timeout_sec) {
    struct ipmi_rs * rsp;
    struct ipmi_rq req;
    time_t start_time = time(NULL);

retry:
    memset(&req, 0, sizeof(req));
    req.msg.netfn = IPMI_NETFN_RARITANOEM;
    req.msg.cmd = IPMI_RARITANOEM_SELFTEST_RESULT;

    rsp = intf->sendrecv(intf, &req, error);
    if (!rsp) {
	ipmi_printf("Error in self test (%s) command\n",
	    get_description(request_cmd->header.comp, request_cmd->header.test));
        return -1;
    }
    if (rsp->ccode == IPMI_ERR_SLFTST_BUSY && !timedout(start_time, timeout_sec)) {
	ipmi_reset_error(error);
	usleep(100000);
	goto retry;
    }
    if (rsp->ccode) {
	ipmi_printf("Error in self test (%s) command: %s\n",
	    get_description(request_cmd->header.comp, request_cmd->header.test),
	    get_error_string(rsp->ccode));
	ipmi_reset_error(error);
	ipmi_set_error(error, get_error_code(rsp->ccode));
	return -1;
    }

    memcpy(result_cmd, rsp->data, rsp->data_len);
    *len = rsp->data_len - sizeof(oem_pp_selftest_t);

    if (request_cmd->header.comp != result_cmd->header.comp || request_cmd->header.test != result_cmd->header.test) {
	ipmi_printf("Self test result (%s) doesn't fit to requested test (%s)!\n",
	    get_description(result_cmd->header.comp, result_cmd->header.test),
	    get_description(request_cmd->header.comp, request_cmd->header.test));
	ipmi_reset_error(error);
	ipmi_set_error(error, IPMI_ERROR_SELFTEST_NO_MATCH);
	return -1;
    }

    return 0;
}

// perform a simple self test which neither has results nor parameters
static int ipmi_simple_selftest(struct ipmi_intf * intf, int comp, int test,
				int timeout_sec, int *error) {
    oem_pp_selftest_t send_cmd;
    oem_pp_selftest_t recv_cmd;
    size_t recv_len = sizeof(recv_cmd);
    int ret;

    memset(&send_cmd, 0, sizeof(send_cmd));
    memset(&recv_cmd, 0, sizeof(recv_cmd));

    send_cmd.header.comp = comp;
    send_cmd.header.test = test;

    if ((ret = ipmi_selftest_send(intf, &send_cmd, sizeof(send_cmd), error, timeout_sec)) != 0) {
	return ret;
    }
    if ((ret = get_selftest_result(intf, &recv_cmd, &send_cmd, &recv_len, error, timeout_sec)) != 0) {
	return ret;
    }

    return 0;
}

// perform DDC self test
static int ipmi_selftest_ddc(struct ipmi_intf * intf, pp_ipmi_return_t *ipmi_ret, int *error) {
    oem_pp_selftest_t send_cmd;
    oem_pp_selftest_ddc_t recv_cmd;
    size_t recv_len = sizeof(recv_cmd);
    int ret;

    memset(&send_cmd, 0, sizeof(send_cmd));
    memset(&recv_cmd, 0, sizeof(recv_cmd));

    send_cmd.header.comp = IPMI_OEM_PP_SLFTST_DDC;
    send_cmd.header.test = IPMI_OEM_PP_SLFTST_DDC_INFO;

    if ((ret = ipmi_selftest_send(intf, &send_cmd, sizeof(send_cmd), error, 60)) != 0) {
	return ret;
    }
    if ((ret = get_selftest_result(intf, (oem_pp_selftest_t *)&recv_cmd, &send_cmd,
	&recv_len, error, 60)) != 0) {
	return ret;
    }

    memcpy(ipmi_ret->data.self_test_ddc.ddc_data, recv_cmd.ddc_data,
	min(recv_len, sizeof(ipmi_ret->data.self_test_ddc.ddc_data)));

    return 0;
}

// perform video status self test
static int ipmi_selftest_video_status(struct ipmi_intf * intf, pp_ipmi_return_t *ipmi_ret, int *error) {
    oem_pp_selftest_t send_cmd;
    oem_pp_selftest_video_status_t recv_cmd;
    size_t recv_len = sizeof(recv_cmd);
    int ret;

    memset(&send_cmd, 0, sizeof(send_cmd));
    memset(&recv_cmd, 0, sizeof(recv_cmd));

    send_cmd.header.comp = IPMI_OEM_PP_SLFTST_VIDEO;
    send_cmd.header.test = IPMI_OEM_PP_SLFTST_VIDEO_STATUS;

    if ((ret = ipmi_selftest_send(intf, &send_cmd, sizeof(send_cmd), error, 60)) != 0) {
	return ret;
    }
    if ((ret = get_selftest_result(intf, (oem_pp_selftest_t *)&recv_cmd, &send_cmd,
	&recv_len, error, 60)) != 0) {
	return ret;
    }

    ipmi_ret->data.self_test_video_status.x_res = le16_to_cpu(recv_cmd.x_res);
    ipmi_ret->data.self_test_video_status.y_res = le16_to_cpu(recv_cmd.y_res);
    ipmi_ret->data.self_test_video_status.hor_line_freq = le16_to_cpu(recv_cmd.hf);
    ipmi_ret->data.self_test_video_status.vert_frame_freq = le16_to_cpu(recv_cmd.vf);

    return 0;
}

// perform video CRC self test
static int ipmi_selftest_video_crc(struct ipmi_intf * intf, pp_ipmi_return_t *ipmi_ret, int *error) {
    oem_pp_selftest_t send_cmd;
    oem_pp_selftest_video_crc_t recv_cmd;
    size_t recv_len = sizeof(recv_cmd);
    int ret;

    memset(&send_cmd, 0, sizeof(send_cmd));
    memset(&recv_cmd, 0, sizeof(recv_cmd));

    send_cmd.header.comp = IPMI_OEM_PP_SLFTST_VIDEO;
    send_cmd.header.test = IPMI_OEM_PP_SLFTST_VIDEO_CRC;

    if ((ret = ipmi_selftest_send(intf, &send_cmd, sizeof(send_cmd), error, 60)) != 0) {
	return ret;
    }
    if ((ret = get_selftest_result(intf, (oem_pp_selftest_t *)&recv_cmd, &send_cmd,
	&recv_len, error, 60)) != 0) {
	return ret;
    }

    ipmi_ret->data.self_test_video_crc.image_crc = le32_to_cpu(recv_cmd.crc);

    return 0;
}

// perform IPMB self test
static int ipmi_selftest_ipmb_bmc(struct ipmi_intf * intf, int *error) {
    return ipmi_simple_selftest(intf, IPMI_OEM_PP_SLFTST_IPMB,
	IPMI_OEM_PP_SLFTST_IPMB_BMCSTATUS, 60, error);
}

// perform FML self test
static int ipmi_selftest_fml_esb2(struct ipmi_intf * intf, int *error) {
    return ipmi_simple_selftest(intf, IPMI_OEM_PP_SLFTST_FML,
	IPMI_OEM_PP_SLFTST_FML_ESB2STATUS, 60, error);
}

// perform USB self test
static int ipmi_selftest_usb_operation(struct ipmi_intf * intf, unsigned char channel, int *error) {
    oem_pp_selftest_usb_t send_cmd;
    oem_pp_selftest_t recv_cmd;

    size_t recv_len = sizeof(recv_cmd);
    int ret;

    memset(&send_cmd, 0, sizeof(send_cmd));
    memset(&recv_cmd, 0, sizeof(recv_cmd));

    send_cmd.header.comp = IPMI_OEM_PP_SLFTST_USB;
    send_cmd.header.test = IPMI_OEM_PP_SLFTST_USB_OPERATION;
    send_cmd.channel = channel;

    if ((ret = ipmi_selftest_send(intf, (oem_pp_selftest_t *)&send_cmd, sizeof(send_cmd), error, 60)) != 0) {
	return ret;
    }
    if ((ret = get_selftest_result(intf, &recv_cmd, (oem_pp_selftest_t *)&send_cmd, &recv_len, error, 60)) != 0) {
	return ret;
    }

    return 0;
}

// perform NIC selftests
static int ipmi_selftest_nic_status(struct ipmi_intf * intf, unsigned char channel,
				    pp_ipmi_return_t *ipmi_ret, int *error) {
    oem_pp_selftest_nic_status_send_t send_cmd;
    oem_pp_selftest_nic_status_reply_t recv_cmd;
    
    size_t recv_len = sizeof(recv_cmd);
    int ret;

    memset(&send_cmd, 0, sizeof(send_cmd));
    memset(&recv_cmd, 0, sizeof(recv_cmd));

    send_cmd.header.comp = IPMI_OEM_PP_SLFTST_NIC;
    send_cmd.header.test = IPMI_OEM_PP_SLFTST_NIC_STATUS;
    send_cmd.channel = channel;

    if ((ret = ipmi_selftest_send(intf, (oem_pp_selftest_t *)&send_cmd, sizeof(send_cmd), error, 60)) != 0) {
	return ret;
    }
    if ((ret = get_selftest_result(intf, (oem_pp_selftest_t *)&recv_cmd,
	(oem_pp_selftest_t *)&send_cmd, &recv_len, error, 60)) != 0) {
	return ret;
    }

    ipmi_ret->data.self_test_nic_status.have_link = recv_cmd.link;
    switch (recv_cmd.duplex) {
	case 2:		// full
	    ipmi_ret->data.self_test_nic_status.duplex = PP_IPMI_NIC_STATUS_DUPLEX_FULL;
	    break;
	case 1:		// half
	    ipmi_ret->data.self_test_nic_status.duplex = PP_IPMI_NIC_STATUS_DUPLEX_HALF;
	    break;
	case 0:		// unknown
	default:
	    ipmi_ret->data.self_test_nic_status.duplex = PP_IPMI_NIC_STATUS_DUPLEX_UNKNOWN;
	    break;
    }
    switch (recv_cmd.speed) {
	case 4:		// 10 GBit
	    ipmi_ret->data.self_test_nic_status.speed = PP_IPMI_NIC_STATUS_SPEED_10_GBIT;
	    break;
	case 3:		// 1 GBit
	    ipmi_ret->data.self_test_nic_status.speed = PP_IPMI_NIC_STATUS_SPEED_1_GBIT;
	    break;
	case 2:		// 100 MBit
	    ipmi_ret->data.self_test_nic_status.speed = PP_IPMI_NIC_STATUS_SPEED_100_MBIT;
	    break;
	case 1:		// 10 MBit
	    ipmi_ret->data.self_test_nic_status.speed = PP_IPMI_NIC_STATUS_SPEED_10_MBIT;
	    break;
	case 0:		// unknown
	default:
	    ipmi_ret->data.self_test_nic_status.speed = PP_IPMI_NIC_STATUS_SPEED_UNKNOWN;
	    break;
    }

    return 0;
}

static int ipmi_selftest_nic_loopback(struct ipmi_intf * intf, unsigned char channel, int *error) {
    oem_pp_selftest_nic_loopback_t send_cmd;
    oem_pp_selftest_t recv_cmd;

    size_t recv_len = sizeof(recv_cmd);
    int ret;

    memset(&send_cmd, 0, sizeof(send_cmd));
    memset(&recv_cmd, 0, sizeof(recv_cmd));

    send_cmd.header.comp = IPMI_OEM_PP_SLFTST_NIC;
    send_cmd.header.test = IPMI_OEM_PP_SLFTST_NIC_LOOPBACK;
    send_cmd.channel = channel;

    if ((ret = ipmi_selftest_send(intf, (oem_pp_selftest_t *)&send_cmd, sizeof(send_cmd), error, 60)) != 0) {
	return ret;
    }
    if ((ret = get_selftest_result(intf, &recv_cmd, (oem_pp_selftest_t *)&send_cmd, &recv_len, error, 60)) != 0) {
	return ret;
    }

    return 0;
}

static int ipmi_selftest_nic_ping(struct ipmi_intf * intf, unsigned char channel,
				  const char *host_to_ping, int *error) {
    oem_pp_selftest_nic_ping_t send_cmd;
    oem_pp_selftest_t recv_cmd;

    size_t recv_len = sizeof(recv_cmd);
    int ret;

    memset(&send_cmd, 0, sizeof(send_cmd));
    memset(&recv_cmd, 0, sizeof(recv_cmd));

    send_cmd.header.comp = IPMI_OEM_PP_SLFTST_NIC;
    send_cmd.header.test = IPMI_OEM_PP_SLFTST_NIC_PING;
    send_cmd.channel = channel;
    if (get_cmdline_ipaddr(host_to_ping, (u_int8_t *)&send_cmd.ip_addr) != 0) {
	ipmi_set_error(error, IPMI_ERROR_COULD_NOT_PARSE_IP_ADDR);
	return -1;
    }

    if ((ret = ipmi_selftest_send(intf, (oem_pp_selftest_t *)&send_cmd, sizeof(send_cmd), error, 60)) != 0) {
	return ret;
    }
    if ((ret = get_selftest_result(intf, &recv_cmd, (oem_pp_selftest_t *)&send_cmd, &recv_len, error, 60)) != 0) {
	return ret;
    }

    return 0;
}

static int ipmi_selftest_nic_broadcast_ping(struct ipmi_intf * intf UNUSED, unsigned char channel UNUSED,
					    pp_ipmi_return_t *ipmi_ret UNUSED, int *error UNUSED) {
    ipmi_set_error(error, 0xc1);
    return -1;
}

int ipmi_selftest_main(struct ipmi_intf * intf, int subcmd,
		       pp_ipmi_parameter_t *params UNUSED,
		       pp_ipmi_return_t *ipmi_ret, int * error) {
    switch ((pp_ipmi_self_test_subcommand_t) subcmd) {
    	case PP_IPMI_SELF_TEST_SUBCMD_DDC_INFO:
    	    return ipmi_selftest_ddc(intf, ipmi_ret, error);
    	case PP_IPMI_SELF_TEST_SUBCMD_VIDEO_STATUS:
    	    return ipmi_selftest_video_status(intf, ipmi_ret, error);
    	case PP_IPMI_SELF_TEST_SUBCMD_VIDEO_CRC:
    	    return ipmi_selftest_video_crc(intf, ipmi_ret, error);
    	case PP_IPMI_SELF_TEST_SUBCMD_IPMB_BMC_STATUS:
    	    return ipmi_selftest_ipmb_bmc(intf, error);
    	case PP_IPMI_SELF_TEST_SUBCMD_FML_ESB2_STATUS:
    	    return ipmi_selftest_fml_esb2(intf, error);
    	case PP_IPMI_SELF_TEST_SUBCMD_USB_OPERATION:
	    return ipmi_selftest_usb_operation(intf, params->data.self_test_usb.channel, error);
	case PP_IPMI_SELF_TEST_SUBCMD_NIC_STATUS:
	    return ipmi_selftest_nic_status(intf,
		params->data.self_test_nic_status.channel,
		ipmi_ret, error);
	case PP_IPMI_SELF_TEST_SUBCMD_NIC_LOOPBACK:
	    return ipmi_selftest_nic_loopback(intf, params->data.self_test_nic_loopback.channel, error);
	case PP_IPMI_SELF_TEST_SUBCMD_NIC_PING:
	    return ipmi_selftest_nic_ping(intf,
		params->data.self_test_nic_ping.channel,
		params->data.self_test_nic_ping.host_to_ping, error);
	case PP_IPMI_SELF_TEST_SUBCMD_NIC_BROADCAST_PING:
	    return ipmi_selftest_nic_broadcast_ping(intf,
		params->data.self_test_nic_bcast_ping.channel,
		ipmi_ret, error);
    	default:
    	    ipmi_printf("Invalid self test command: %d\n", subcmd);
	    ipmi_set_error(error, IPMI_ERROR_INVALID_COMMAND);
    	    return -1;
    }
}
