/**
 * \file ipmi_sol.c
 *
 * Description: SOL IPMI commands
 *
 * (c) 2007 Raritan Inc, Thomas Rendelmann, <Thomas.Rendelmann@raritan.com>
 */

#include <config.h>

#include <ipmitool/ipmi.h>
#include <ipmitool/ipmi_sol.h>
#include <ipmitool/ipmi_intf.h>
#include <ipmitool/ipmi_print.h>
#include <ipmitool/ipmi_error.h>
#include <ipmitool/ipmi_strings.h>

#define SOL_PARAMETER_SET_IN_PROGRESS           0x00
#define SOL_PARAMETER_SOL_ENABLE                0x01
#define SOL_PARAMETER_SOL_AUTHENTICATION        0x02
#define SOL_PARAMETER_CHARACTER_INTERVAL        0x03
#define SOL_PARAMETER_SOL_RETRY                 0x04
#define SOL_PARAMETER_SOL_NON_VOLATILE_BIT_RATE 0x05
#define SOL_PARAMETER_SOL_VOLATILE_BIT_RATE     0x06
#define SOL_PARAMETER_SOL_PAYLOAD_CHANNEL       0x07
#define SOL_PARAMETER_SOL_PAYLOAD_PORT          0x08

static int ipmi_sol_send_params(struct ipmi_intf * intf, unsigned char *data, size_t data_len, int *error) {
    struct ipmi_rs * rsp;
    struct ipmi_rq   req;
    
    memset(&req, 0, sizeof(req));
    req.msg.netfn    = IPMI_NETFN_TRANSPORT;           /* 0x0c */
    req.msg.cmd      = IMPI_SET_SOL_CONFIG_PARAMETERS; /* 0x21 */
    req.msg.data     = data;
    req.msg.data_len = data_len;

    rsp = intf->sendrecv(intf, &req, error);
    
    if (rsp == NULL) {
    	ipmi_printf("Error setting SOL parameter\n");
    	return -1;
    }
    if (rsp->ccode > 0) {
    	ipmi_printf("Error setting SOL parameter: %s\n",
    		val2str(rsp->ccode, completion_code_vals));
    	return -1;
    }

    return 0;
}

static int ipmi_sol_set_params(struct ipmi_intf * intf, unsigned char channel,
                               unsigned char *send_data, size_t data_len, int guarded, int *error) {
    uint8_t data[3];
    int ret1 = 0, ret2 = 0;
    
    // request set-in-progress
    if (guarded) {
        memset(data, 0, sizeof(data));
        data[0] = channel;
        data[1] = SOL_PARAMETER_SET_IN_PROGRESS;
        data[2] = 0x01;
        if ((ret1 = ipmi_sol_send_params(intf, data, sizeof(data), error))) return ret1;
    }
    
    // set the parameter
    ret1 = ipmi_sol_send_params(intf, send_data, data_len, error);

    if (guarded) {
        // commit
        memset(data, 0, sizeof(data));
        data[0] = channel;
        data[1] = SOL_PARAMETER_SET_IN_PROGRESS;
        data[2] = 0x02;
        ipmi_sol_send_params(intf, data, sizeof(data), error);  // ignore the return code
        
        // set complete
        memset(data, 0, sizeof(data));
        data[0] = channel;
        data[1] = SOL_PARAMETER_SET_IN_PROGRESS;
        data[2] = 0x00;
        ret2 = ipmi_sol_send_params(intf, data, sizeof(data), error);
    }
    
    if (ret1 < 0 || ret2 < 0) {
        return -1;
    }
    return 0;
}


static int ipmi_sol_set_enable(struct ipmi_intf * intf, unsigned char channel,
                               int enabled, int *error) {
    uint8_t data[3];

    data[0] = channel;
    data[1] = SOL_PARAMETER_SOL_ENABLE;
    data[2] = enabled ? 0x01 : 0x00;
    
    return ipmi_sol_set_params(intf, channel, data, sizeof(data), 1, error);
}

static int ipmi_sol_get_enable(struct ipmi_intf * intf, unsigned char channel,
                               int * enabled, int *error) {
    struct ipmi_rs * rsp;
    struct ipmi_rq req;
    uint8_t data[4];
    
    memset(&req, 0, sizeof(req));
    req.msg.netfn    = IPMI_NETFN_TRANSPORT;
    req.msg.cmd      = IMPI_GET_SOL_CONFIG_PARAMETERS;
    req.msg.data_len = 4;
    req.msg.data     = data;

    memset(data, 0, sizeof(data));
    data[0] = channel;                  /* channel number     */
    data[1] = SOL_PARAMETER_SOL_ENABLE; /* parameter selector */
    data[2] = 0x00;                     /* set selector       */
    data[3] = 0x00;                     /* block selector     */

    rsp = intf->sendrecv(intf, &req, error);
    
    if (rsp == NULL) {
    	ipmi_printf("Error setting SOL parameter\n");
    	return -1;
    }
    if (rsp->ccode > 0) {
    	ipmi_printf("Error setting SOL parameter: %s\n",
    		val2str(rsp->ccode, completion_code_vals));
    	return -1;
    }

    if (rsp->data_len == 2) {
        *enabled = rsp->data[1];
    } else {
        ipmi_printf("Error: Unexpected data length (%d) received for SOL parameter\n", rsp->data[1]);
        return -1;
    }

    return 0;
}

static int ipmi_sol_set_user_enable(struct ipmi_intf * intf, unsigned char channel, int userid,
                                    int enabled, int *error) {
    struct ipmi_rq req;
    struct ipmi_rs *rsp;
    uint8_t data[6];
    
    memset(&req, 0, sizeof(req));
    req.msg.netfn    = IPMI_NETFN_APP;
    req.msg.cmd      = IPMI_SET_USER_PAYLOAD_ACCESS;
    req.msg.data     = data;
    req.msg.data_len = 6;
    
    memset(data, 0, 6);
    
    data[0] = channel & 0xf;	/* channel */
    data[1] = userid & 0x3f;	/* user id */
    if (!enabled)
    	data[1] |= 0x40;	/* disable */
    data[2] = 0x02;		/* payload 1 is SOL */
    
    rsp = intf->sendrecv(intf, &req, error);
    if (rsp == NULL) {
    	ipmi_printf("Error setting SOL user enable\n");
    	return -1;
    }
    if (rsp->ccode > 0) {
    	ipmi_printf("Error setting SOL user enable: %s\n",
    		val2str(rsp->ccode, completion_code_vals));
    	return -1;
    }
    
    return 0;
}

static int ipmi_sol_get_user_enable(struct ipmi_intf * intf, unsigned char channel, int userid,
                                    int * enabled, int *error) {
    struct ipmi_rq req;
    struct ipmi_rs *rsp;
    uint8_t data[2];
    
    memset(&req, 0, sizeof(req));
    req.msg.netfn    = IPMI_NETFN_APP;
    req.msg.cmd      = IPMI_GET_USER_PAYLOAD_ACCESS;
    req.msg.data     = data;
    req.msg.data_len = 2;
    
    data[0] = channel & 0xf;	/* channel */
    data[1] = userid & 0x3f;	/* user id */
    
    rsp = intf->sendrecv(intf, &req, error);
    if (rsp == NULL) {
    	ipmi_printf("Error reading SOL user enable\n");
    	return -1;
    }
    if (rsp->ccode > 0) {
    	ipmi_printf("Error reading SOL user enable: %s\n",
    		val2str(rsp->ccode, completion_code_vals));
    	return -1;
    }
    
    if (rsp->data_len >= 1) {
        *enabled = (rsp->data[0] & 0x02) ? 1 : 0;
    } else {
        ipmi_printf("Error: Unexpected data length (%d) received for SOL user enable\n", rsp->data[1]);
        return -1;
    }

    return 0;
}

int ipmi_sol_main(struct ipmi_intf * intf,
                  int subcmd,
                  pp_ipmi_parameter_t *params,
                  pp_ipmi_return_t *ret,
                  int * error) {
    switch ((pp_ipmi_sol_subcommand_t) subcmd) {
	case PP_IPMI_SOL_SUBCMD_SET_ENABLE:
	    return ipmi_sol_set_enable(intf, params->data.sol_set_enable.chan, params->data.sol_set_enable.enabled, error);
	case PP_IPMI_SOL_SUBCMD_GET_ENABLE:
	    return ipmi_sol_get_enable(intf, params->data.sol_get_enable.chan, &ret->data.sol_get_enable.enabled, error);
	case PP_IPMI_SOL_SUBCMD_SET_USER_ENABLE:
	    return ipmi_sol_set_user_enable(intf, params->data.sol_set_user_enable.chan,
	        params->data.sol_set_user_enable.uid, params->data.sol_set_user_enable.enabled, error);
	case PP_IPMI_SOL_SUBCMD_GET_USER_ENABLE:
	    return ipmi_sol_get_user_enable(intf, params->data.sol_get_user_enable.chan,
	        params->data.sol_get_user_enable.uid, &ret->data.sol_get_user_enable.enabled, error);
	default:
	    ipmi_printf("Invalid SOL command: %d\n", subcmd);
	    ipmi_set_error(error, IPMI_ERROR_INVALID_COMMAND);
	    return -1;
    }
}

