/*
 * Copyright (c) 2003 Sun Microsystems, Inc.  All Rights Reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 
 * Redistribution of source code must retain the above copyright
 * notice, this list of conditions and the following disclaimer.
 * 
 * Redistribution in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the distribution.
 * 
 * Neither the name of Sun Microsystems, Inc. or the names of
 * contributors may be used to endorse or promote products derived
 * from this software without specific prior written permission.
 * 
 * This software is provided "AS IS," without a warranty of any kind.
 * ALL EXPRESS OR IMPLIED CONDITIONS, REPRESENTATIONS AND WARRANTIES,
 * INCLUDING ANY IMPLIED WARRANTY OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE OR NON-INFRINGEMENT, ARE HEREBY EXCLUDED.
 * SUN MICROSYSTEMS, INC. ("SUN") AND ITS LICENSORS SHALL NOT BE LIABLE
 * FOR ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING
 * OR DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.  IN NO EVENT WILL
 * SUN OR ITS LICENSORS BE LIABLE FOR ANY LOST REVENUE, PROFIT OR DATA,
 * OR FOR DIRECT, INDIRECT, SPECIAL, CONSEQUENTIAL, INCIDENTAL OR
 * PUNITIVE DAMAGES, HOWEVER CAUSED AND REGARDLESS OF THE THEORY OF
 * LIABILITY, ARISING OUT OF THE USE OF OR INABILITY TO USE THIS SOFTWARE,
 * EVEN IF SUN HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGES.
 * 
 * You acknowledge that this software is not designed or intended for use
 * in the design, construction, operation or maintenance of any nuclear
 * facility.
 */

#include <stdio.h>
#include <fcntl.h>
#include <errno.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>

#include <ipmitool/ipmi.h>
#include <ipmitool/ipmi_intf.h>
#include <ipmitool/helper.h>
#include <ipmitool/ipmi_error.h>
#include <ipmitool/ipmi_print.h>

#include "imbapi.h"

#if defined(WIN32)
# include <pp/ipmi.h>
#else
# include <unistd.h>
# include <sys/ioctl.h>
#endif

#define IPMI_IMB_TIMEOUT	(1000 * 1000)
#define IPMI_IMB_MAX_RETRY	1
#ifdef WIN32
# define IPMI_IMB_DEV		"\\\\.\\Imb"
#else
# define IPMI_IMB_DEV		"/dev/imb"
#endif
#define IPMI_IMB_BUF_SIZE	64

static int initialized = 0;

static int ipmi_imb_open(struct ipmi_intf * intf, int * error) {
    if (!initialized) {
	char *devname = intf->session->device[0] ? intf->session->device : IPMI_IMB_DEV;
#if !defined(WIN32)
	struct stat stbuf;

	if (stat(devname, &stbuf) < 0) {
	    ipmi_printf("Error: no IMB driver found at %s!\n", IPMI_IMB_DEV);
	    ipmi_set_error(error, IPMI_ERROR_IMB_NOT_AVAILABLE);
	    return -1;
	}
#endif

	if (initIPMI(devname)) {
	    ipmi_printf("Error: could not initialize IMB driver.\n");
	    ipmi_set_error(error, IPMI_ERROR_IMB_NOT_AVAILABLE);
	    return -1;
	}

	initialized = 1;
    }

    intf->opened = 1;

    return 0;
}

static void ipmi_imb_close(struct ipmi_intf * intf, int force, int *error) {
    intf->opened = 0;
}

static struct ipmi_rs * ipmi_imb_send_cmd(struct ipmi_intf * intf,
					  struct ipmi_rq * req,
					  int *error) {
    IMBPREQUESTDATA imbreq;
    static struct ipmi_rs rsp;	
    int status, i;
    unsigned char ccode;
    struct ipmi_rs *ret = NULL;

    imbreq.rsSa		= IPMI_BMC_SLAVE_ADDR;
    imbreq.rsLun	= 0;
    imbreq.busType	= 0;
    imbreq.netFn	= req->msg.netfn;
    imbreq.cmdType	= req->msg.cmd;

    imbreq.data = req->msg.data;
    imbreq.dataLength = req->msg.data_len;

    if (verbose > 1) {
	ipmi_printf("IMB rsSa       : %x\n", imbreq.rsSa);
	ipmi_printf("IMB netFn      : %x\n", imbreq.netFn);
	ipmi_printf("IMB cmdType    : %x\n", imbreq.cmdType);
	ipmi_printf("IMB dataLength : %d\n", imbreq.dataLength);
    }

    rsp.data_len = IPMI_IMB_BUF_SIZE;
    memset(rsp.data, 0, rsp.data_len);

    for (i = 0; i < IPMI_IMB_MAX_RETRY; i++) {
	if (verbose > 2)
	    printbuf(imbreq.data, imbreq.dataLength, "ipmi_imb request");
	status = SendTimedImbpRequest(&imbreq, IPMI_IMB_TIMEOUT,
	    rsp.data, &rsp.data_len, &ccode);
	if (status == 0) {
	    if (verbose > 2)
		printbuf(rsp.data, rsp.data_len, "ipmi_imb response");
	    ret = &rsp;
	    break;
	}
	/* error */
	ipmi_printf("Error sending IMB request, status=%x ccode=%x\n", status, ccode);
	ipmi_set_error(error, IPMI_ERROR_IMB_ERROR);
    }

    rsp.ccode = ccode;

    return ret;
}

static int ipmi_imb_setup(struct ipmi_intf * intf, int * error) {
    intf->session = malloc(sizeof(struct ipmi_session));
    if (intf->session == NULL) {
	ipmi_printf("%s: malloc failure\n", __FUNCTION__);
	ipmi_set_error(error, IPMI_ERROR_NOT_ENOUGH_MEMORY);
	return -1;
    }
    memset(intf->session, 0, sizeof(struct ipmi_session));
    return 0;
}


struct ipmi_intf ipmi_imb_intf = {
    /* name:              */  "imb",
    /* desc:              */  "Intel IMB Interface",

    /* fd:                */  0,
    /* opened:            */  0,
    /* abort:             */  0,
    /* noanswer:          */  0,
    /* cancel:            */  0,
    /* intf_ctx:          */  NULL,

    /* cmd_wfd:           */  0,
    /* rsp_rfd:           */  0,
    /* loopi_pipe_prefix: */  0,

    /* session:           */  NULL,
    /* oem:               */  NULL,
    /* my_addr:           */  0,
    /* bmc_addr:          */  IPMI_BMC_SLAVE_ADDR,
    /* target_addr:       */  IPMI_BMC_SLAVE_ADDR,

    /* setup:             */  ipmi_imb_setup,
    /* open:              */  ipmi_imb_open,
    /* close:             */  ipmi_imb_close,
    /* do_abort:          */  NULL,
    /* sendrecv:          */  ipmi_imb_send_cmd,
    /* sendrsp:           */  NULL,
    /* recv_sol:          */  NULL,
    /* send_sol:          */  NULL,
    /* keepalive:         */  NULL,
};

