/*
 * Copyright (c) 2003 Sun Microsystems, Inc.  All Rights Reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 
 * Redistribution of source code must retain the above copyright
 * notice, this list of conditions and the following disclaimer.
 * 
 * Redistribution in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the distribution.
 * 
 * Neither the name of Sun Microsystems, Inc. or the names of
 * contributors may be used to endorse or promote products derived
 * from this software without specific prior written permission.
 * 
 * This software is provided "AS IS," without a warranty of any kind.
 * ALL EXPRESS OR IMPLIED CONDITIONS, REPRESENTATIONS AND WARRANTIES,
 * INCLUDING ANY IMPLIED WARRANTY OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE OR NON-INFRINGEMENT, ARE HEREBY EXCLUDED.
 * SUN MICROSYSTEMS, INC. ("SUN") AND ITS LICENSORS SHALL NOT BE LIABLE
 * FOR ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING
 * OR DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.  IN NO EVENT WILL
 * SUN OR ITS LICENSORS BE LIABLE FOR ANY LOST REVENUE, PROFIT OR DATA,
 * OR FOR DIRECT, INDIRECT, SPECIAL, CONSEQUENTIAL, INCIDENTAL OR
 * PUNITIVE DAMAGES, HOWEVER CAUSED AND REGARDLESS OF THE THEORY OF
 * LIABILITY, ARISING OUT OF THE USE OF OR INABILITY TO USE THIS SOFTWARE,
 * EVEN IF SUN HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGES.
 * 
 * You acknowledge that this software is not designed or intended for use
 * in the design, construction, operation or maintenance of any nuclear
 * facility.
 */

#include <stdlib.h>
#include <string.h>
#include <stdio.h>

#include <config.h>
#include <ipmitool/ipmi_intf.h>
#include <ipmitool/ipmi.h>
#include <ipmitool/ipmi_print.h>
#include <ipmitool/ipmi_sdr.h>
#include <ipmitool/ipmi_error.h>


#ifdef WIN32

#define snprintf _snprintf

extern struct ipmi_intf ipmi_lan_intf;
extern struct ipmi_intf ipmi_lanplus_intf;
#ifdef PP_FW_TYPE_KIRATOOL
extern struct ipmi_intf ipmi_imb_intf;
extern struct ipmi_intf ipmi_ppscsi_intf;
#endif

struct ipmi_intf * ipmi_intf_table[] = {
	&ipmi_lan_intf,
	&ipmi_lanplus_intf,
#ifdef PP_FW_TYPE_KIRATOOL
	&ipmi_imb_intf,
	&ipmi_ppscsi_intf,
#endif
	NULL
};

#else // WIN32

#include <pp/features.h>

extern struct ipmi_intf ipmi_lan_intf;
extern struct ipmi_intf ipmi_lanplus_intf;

#if defined (PP_FW_TYPE_ERLA)
extern struct ipmi_intf ipmi_i2c;
extern struct ipmi_intf ipmi_serial;
#endif /* PP_FW_TYPE_ERLA */
#if defined (PP_FW_TYPE_KIRATOOL) && !defined (WIN32)
extern struct ipmi_intf ipmi_open_intf;
extern struct ipmi_intf ipmi_ppscsi_intf;
#endif /* defined (PP_FW_TYPE_KIRATOOL) && !defined (WIN32) */

#if defined(PP_FEAT_IPMI_SERVER)
extern struct ipmi_intf ipmi_loopi;
#endif // PP_FEAT_IPMI_SERVER


struct ipmi_intf * ipmi_intf_table[] = {
#if defined(PP_FEAT_IPMI_SERVER)
	&ipmi_loopi,
#endif /* PP_FEAT_IPMI_SERVER */
#if !defined (PP_FEAT_IPMI_SERVER) || defined(PP_FEAT_BMC_OEMCMDS_ONLY)
	&ipmi_lan_intf,
	&ipmi_lanplus_intf,
#if defined (PP_FW_TYPE_ERLA)
	&ipmi_i2c,
	&ipmi_serial,
#endif /* PP_FW_TYPE_ERLA */
#if defined (PP_FW_TYPE_KIRATOOL) && !defined (WIN32)
	&ipmi_open_intf,
	&ipmi_ppscsi_intf,
#endif /* defined (PP_FW_TYPE_KIRATOOL) && !defined (WIN32) */
#endif /* PP_FEAT_IPMI_SERVER */
	NULL
};

#endif // WIN32

void ipmi_intf_print(void)
{
	struct ipmi_intf ** intf;
	int def = 1;

	ipmi_printf("Interfaces:\n");

	for (intf = ipmi_intf_table; intf && *intf; intf++) {
		ipmi_printf("\t%-12s  %s %s\n",
			(*intf)->name, (*intf)->desc,
			def ? "[default]" : "");
		def = 0;
	}
	ipmi_printf("\n");
}

/* Load an interface from the interface table above
 * If no interface name is given return first entry
 */
struct ipmi_intf * ipmi_intf_load(const char * name, int * error)
{
	struct ipmi_intf ** intf;
	struct ipmi_intf * i;
	struct ipmi_intf * i_new;

	if (!name && ipmi_intf_table && ipmi_intf_table[0]) {
		name = ipmi_intf_table[0]->name;
	}

	for (intf = ipmi_intf_table; intf && *intf ; intf++) {
		i = *intf;
		if (!strncmp(name, i->name, strlen(name))) {
			i_new = malloc(sizeof(struct ipmi_intf));
			memcpy(i_new, i, sizeof(struct ipmi_intf));
			if (i_new->setup && (i_new->setup(i_new, error) < 0)) {
				ipmi_printf("ERROR: Unable to setup interface %s\n", name);
				free(i_new);
				return NULL;
			}
			return i_new;
		}
	}
	ipmi_printf("ERROR: Unable to find interface %s\n", name);
	ipmi_set_error(error, IPMI_ERROR_INVALID_PARAMETER);
	return NULL;
}

void
ipmi_intf_session_set_hostname(struct ipmi_intf * intf, const char * hostname)
{
	if (intf->session == NULL)
		return;

	memset(intf->session->hostname, 0, 16);

	if (hostname != NULL) {
		memcpy(intf->session->hostname, hostname,
		       min(strlen(hostname), (size_t)64));
	}
}

void
ipmi_intf_session_set_device(struct ipmi_intf * intf, const char * device)
{
	if (intf->session == NULL)
		return;

	if (device != NULL) {
		strncpy(intf->session->device, device, sizeof(intf->session->device) - 1);
	} else {
		intf->session->device[0] = '\0';
	}
}

void
ipmi_intf_session_set_ppscsi_product(struct ipmi_intf * intf, const char * ppscsi_product)
{
	if (intf->session == NULL)
		return;

	if (ppscsi_product != NULL) {
		strncpy(intf->session->ppscsi_product, ppscsi_product, sizeof(intf->session->ppscsi_product) - 1);
	} else {
		intf->session->ppscsi_product[0] = '\0';
	}
}

void
ipmi_intf_session_set_username(struct ipmi_intf * intf, const char * username)
{
	if (intf->session == NULL)
		return;

	memset(intf->session->username, 0, 16);

	if (username == NULL)
		return;

	memcpy(intf->session->username, username, min(strlen(username), (size_t)16));
}

void
ipmi_intf_session_set_password(struct ipmi_intf * intf, const char * password)
{
	if (intf->session == NULL)
		return;

	memset(intf->session->authcode, 0, IPMI_AUTHCODE_BUFFER_SIZE);

	if (password == NULL) {
		intf->session->password = 0;
		return;
	}

	intf->session->password = 1;
	memcpy(intf->session->authcode, password,
	       min(strlen(password), (size_t)IPMI_AUTHCODE_BUFFER_SIZE));
}

void
ipmi_intf_session_set_privlvl(struct ipmi_intf * intf, uint8_t level)
{
	if (intf->session == NULL)
		return;

	intf->session->privlvl = level;
}

void
ipmi_intf_session_set_cipher_suite_id(struct ipmi_intf * intf, uint8_t cipher_suite_id)
{
	if (intf->session == NULL)
		return;

	intf->session->cipher_suite_id = cipher_suite_id;
}

void
ipmi_intf_session_set_port(struct ipmi_intf * intf, int port)
{
	if (intf->session == NULL)
		return;

	intf->session->port = port;
}

void
ipmi_intf_session_set_authtype(struct ipmi_intf * intf, uint8_t authtype)
{
	if (intf->session == NULL)
		return;

	intf->session->authtype_set = authtype;
}

void ipmi_intf_session_set_serial(struct ipmi_intf * intf, int port, int baud_rate, char *parity, int bits, int stop2, int hwf, int swf)
{
	if (intf && intf->session) {
		intf->session->serial_settings.port = port;
		intf->session->serial_settings.baud_rate = baud_rate;
		snprintf(intf->session->serial_settings.parity, sizeof(intf->session->serial_settings.parity), "%s", parity);
		intf->session->serial_settings.bits = bits;
		intf->session->serial_settings.stop2 = stop2;
		intf->session->serial_settings.hwf = hwf;
		intf->session->serial_settings.swf = swf;
	}
}

void
ipmi_intf_session_set_timeout(struct ipmi_intf * intf, uint32_t timeout)
{
	if (intf->session == NULL)
		return;

	intf->session->timeout = timeout;
}

void
ipmi_intf_session_set_retry(struct ipmi_intf * intf, int retry)
{
	if (intf->session == NULL)
		return;

	intf->session->retry = retry;
}

void
ipmi_cleanup(struct ipmi_intf * intf)
{
	ipmi_sdr_list_empty(intf);
}
