/*
 * Copyright (c) 2003 Sun Microsystems, Inc.  All Rights Reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 
 * Redistribution of source code must retain the above copyright
 * notice, this list of conditions and the following disclaimer.
 * 
 * Redistribution in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the distribution.
 * 
 * Neither the name of Sun Microsystems, Inc. or the names of
 * contributors may be used to endorse or promote products derived
 * from this software without specific prior written permission.
 * 
 * This software is provided "AS IS," without a warranty of any kind.
 * ALL EXPRESS OR IMPLIED CONDITIONS, REPRESENTATIONS AND WARRANTIES,
 * INCLUDING ANY IMPLIED WARRANTY OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE OR NON-INFRINGEMENT, ARE HEREBY EXCLUDED.
 * SUN MICROSYSTEMS, INC. ("SUN") AND ITS LICENSORS SHALL NOT BE LIABLE
 * FOR ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING
 * OR DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.  IN NO EVENT WILL
 * SUN OR ITS LICENSORS BE LIABLE FOR ANY LOST REVENUE, PROFIT OR DATA,
 * OR FOR DIRECT, INDIRECT, SPECIAL, CONSEQUENTIAL, INCIDENTAL OR
 * PUNITIVE DAMAGES, HOWEVER CAUSED AND REGARDLESS OF THE THEORY OF
 * LIABILITY, ARISING OUT OF THE USE OF OR INABILITY TO USE THIS SOFTWARE,
 * EVEN IF SUN HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGES.
 * 
 * You acknowledge that this software is not designed or intended for use
 * in the design, construction, operation or maintenance of any nuclear
 * facility.
 */

#include <stdio.h>
#include <fcntl.h>
#include <unistd.h>
#include <errno.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/time.h>

#include <config.h>
#include <ipmitool/ipmi.h>
#include <ipmitool/ipmi_intf.h>
#include <ipmitool/ipmi_print.h>
#include <ipmitool/ipmi_error.h>
#include <ipmitool/ipmi_session.h>
#include <ipmitool/ipmi_constants.h>

#include <pp/base.h>
#include <pp/features.h>
#include <pp/cfg.h>
#include <pp/um.h>

#if defined(PP_FEAT_IPMI_SERVER)

#include <pp/bmc/loopi_adapter.h>
#include <pp/bmc/loopi_adapter_chan.h>

static pthread_mutex_t mtx = PTHREAD_ERRORCHECK_MUTEX_INITIALIZER_NP;


static int loopi_send_recv(struct ipmi_intf * intf, void* sendbuf, size_t sendcnt,
                     void* recvbuf, size_t recvcnt)
{
    int ret;
    fd_set read_fd;
    struct timeval timeout;
    
    /* hm, this is quite stupid... signature should change, how?? */
    assert(recvcnt >= 128);
    
    /* this is currently quite easy and simple                             *
     * i.e. different threads using the loop channel will block each other */
    MUTEX_LOCK(&mtx);
    
    /* write ipmi msg to pipe */
    if (0 > write(intf->cmd_wfd, sendbuf, sendcnt)) {
        ipmi_printf("[loopi_client] write failed\n");
        ret = PP_ERR;
        goto bailout;
    }

    /* wait for response with select. Read should not  *
     * fail but perhaps eric might crash.              */
    FD_ZERO(&read_fd);
    FD_SET(intf->rsp_rfd, &read_fd);
    timeout.tv_sec = 10;  // current timeout is 10 seconds
    timeout.tv_usec = 0;

    /* wait for response and read it.         *
     * we assume unfragmented read from pipe. */
    if (select(intf->rsp_rfd+1, &read_fd, NULL, NULL, &timeout) == 1) {
        while (0 >= (ret = read(intf->rsp_rfd, recvbuf, recvcnt))) {
            if ((errno == EINTR)) {
                continue;
            }
            ipmi_printf("[loopi_client] read failed\n");
            ret = PP_ERR;
        }
    } else {
        // select timed out
        ipmi_printf("[loopi_client] read timed out\n");
        ret = PP_ERR;
    }

 bailout:
    MUTEX_UNLOCK(&mtx);
    return ret;
}


static struct ipmi_rs * ipmi_loopi_send_cmd(struct ipmi_intf * intf, struct ipmi_rq * req, int * perr)
{
	static struct ipmi_rs rsp;
	uint8_t send_message[req->msg.data_len + 2];
	uint8_t recv_message[255];
	int ret;

	if (!intf || !req) {
		return NULL;
	}

        if (!intf->opened && intf->open && intf->open(intf, perr) < 0) {
                return NULL;
        }

	send_message[0] = req->msg.netfn << 2;
	send_message[1] = req->msg.cmd;
	memcpy(&send_message[2], req->msg.data, req->msg.data_len);
	
#ifdef PP_IPMI_DEBUG
	{
		int i;
		D("Transmit: ");
		for (i = 0; i < req->msg.data_len + 2; i++) {
			printf("%02x ", send_message[i]);
		}
		printf("\n");
	}
#endif // PP_IPMI_DEBUG
	
	ret = loopi_send_recv(intf, send_message, req->msg.data_len + 2, recv_message, sizeof(recv_message));
	if (ret < 0) {
		ipmi_printf("Error in loopi receive.\n");
		ipmi_set_error(perr, IPMI_ERROR_NO_RESPONSE);
		return NULL;
	} else if (ret < 3) {
		ipmi_printf("Error: wrong format in loopi receive.\n");
		ipmi_set_error(perr, IPMI_ERROR_NO_RESPONSE);
		return NULL;
	}

#ifdef PP_IPMI_DEBUG
	{
		int i;
		D("Receive: ");
		for (i = 0; i < ret; i++) {
			printf("%02x ", recv_message[i]);
		}
		printf("\n");
	}
#endif // PP_IPMI_DEBUG
	memset(&rsp, 0, sizeof(struct ipmi_rs));
	rsp.ccode = recv_message[2];
	rsp.data_len = ret - 3;

	if (!rsp.ccode && rsp.data_len)
		memcpy(rsp.data, &recv_message[3], ret - 3);

	if (rsp.ccode) {
		ipmi_set_error(perr, rsp.ccode);
	}
	
	return &rsp;
}


static int ipmi_loopi_open(struct ipmi_intf * intf, int * perr)
{
    /* nothing to do but setting the right priv level */
    char *privstr = NULL;
    uint8_t privlvl = 0;
    const char* cmd_pipe = pp_bmc_loopi_chan[intf->loopi_chan].cmd_pipe;
    const char* rsp_pipe = pp_bmc_loopi_chan[intf->loopi_chan].rsp_pipe;
    
    if (!intf || !intf->session) {
	ipmi_set_error(perr, IPMI_ERROR_ESTABLISH_SESSION);
	goto error;
    }

    intf->opened = 0;

    if (0 > (intf->cmd_wfd = open(cmd_pipe, O_RDWR)) ||
        0 > (intf->rsp_rfd = open(rsp_pipe, O_RDWR)) ) {
        ipmi_printf("Can't open loopi: can't connect named pipes\n");
        goto error;
    }

    intf->opened = 1;

    /* if no user is specified, we go with admin privileges */
    if (!intf->session->username || !intf->session->username[0]) {
        intf->opened = 1;
        return 0;
    }

    if (PP_FAILED(pp_um_user_get_permission(intf->session->username,
                                            "ipmi_priv", &privstr))) {
        ipmi_printf("Can't open loopi: no privilege level for user '%s' found.\n", intf->session->username);
        goto error;
    }

    if (!strcmp(privstr, "CALLBACK"))           privlvl = IPMI_SESSION_PRIV_CALLBACK;
    else if (!strcmp(privstr, "USER"))          privlvl = IPMI_SESSION_PRIV_USER;
    else if (!strcmp(privstr, "OPERATOR"))      privlvl = IPMI_SESSION_PRIV_OPERATOR;
    else if (!strcmp(privstr, "ADMINISTRATOR")) privlvl = IPMI_SESSION_PRIV_ADMIN;
    else if (!strcmp(privstr, "OEM"))           privlvl = IPMI_SESSION_PRIV_ADMIN;

    if (privlvl == 0) {
        ipmi_printf("Can't open loopi: bad privilege level '%s' for user '%s' found.\n", privstr, intf->session->username);
        goto error;
    }

    {
        struct ipmi_rq req = {
            .msg = {
                .netfn = IPMI_NETFN_APP,
                .cmd = IPMI_GET_SESSION_PRIVILEGE_LEVEL,
                .data_len = 1,
                .data = &privlvl,
            },
        };
        struct ipmi_rs * rsp;

        rsp = ipmi_loopi_send_cmd(intf, &req, perr);
        if (!rsp || rsp->ccode) {
            ipmi_printf("Can't open loopi: error setting privilege level '%s' for user '%s' (err %x).\n",
                        privstr, intf->session->username, rsp ? rsp->ccode : 0);
            goto error;
        }
    }

    free(privstr);
    return 0;

error:
    if (intf->opened && intf->close) {
        intf->close(intf, 0, perr);
    }
    intf->opened = 0;
    free(privstr);
    return -1;
}

static void ipmi_loopi_close(struct ipmi_intf * intf, int force UNUSED, int * error UNUSED) {
    close (intf->cmd_wfd);
    close (intf->rsp_rfd);
}

static int ipmi_loopi_setup(struct ipmi_intf * intf, int * perr UNUSED)
{
        intf->cancel = 0;
        intf->session = malloc(sizeof(struct ipmi_session));
        memset(intf->session, 0, sizeof(struct ipmi_session));
        return (intf->session) ? 0 : -1;
}

struct ipmi_intf ipmi_loopi = {
	name:		"loopi",
	desc:		"Peppercon IPMI loopback adapter",
	sendrecv:	ipmi_loopi_send_cmd,
        open:           ipmi_loopi_open,
        close:          ipmi_loopi_close,
        setup:          ipmi_loopi_setup,
	bmc_addr:	IPMI_BMC_SLAVE_ADDR,
	target_addr:	IPMI_BMC_SLAVE_ADDR,
};


#endif // PP_FEAT_IPMI_SERVER
