/*
 * Copyright (c) 2003 Sun Microsystems, Inc.  All Rights Reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 
 * Redistribution of source code must retain the above copyright
 * notice, this list of conditions and the following disclaimer.
 * 
 * Redistribution in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the distribution.
 * 
 * Neither the name of Sun Microsystems, Inc. or the names of
 * contributors may be used to endorse or promote products derived
 * from this software without specific prior written permission.
 * 
 * This software is provided "AS IS," without a warranty of any kind.
 * ALL EXPRESS OR IMPLIED CONDITIONS, REPRESENTATIONS AND WARRANTIES,
 * INCLUDING ANY IMPLIED WARRANTY OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE OR NON-INFRINGEMENT, ARE HEREBY EXCLUDED.
 * SUN MICROSYSTEMS, INC. ("SUN") AND ITS LICENSORS SHALL NOT BE LIABLE
 * FOR ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING
 * OR DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.  IN NO EVENT WILL
 * SUN OR ITS LICENSORS BE LIABLE FOR ANY LOST REVENUE, PROFIT OR DATA,
 * OR FOR DIRECT, INDIRECT, SPECIAL, CONSEQUENTIAL, INCIDENTAL OR
 * PUNITIVE DAMAGES, HOWEVER CAUSED AND REGARDLESS OF THE THEORY OF
 * LIABILITY, ARISING OUT OF THE USE OF OR INABILITY TO USE THIS SOFTWARE,
 * EVEN IF SUN HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGES.
 */

#include <stdio.h>
#include <fcntl.h>
#include <unistd.h>
#include <sys/ioctl.h>
#include <errno.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>

#include <ipmitool/ipmi.h>
#include <ipmitool/ipmi_intf.h>
#include <ipmitool/ipmi_print.h>
#include <ipmitool/ipmi_error.h>
#include <ipmitool/helper.h>

#if defined(HAVE_CONFIG_H)
# include <config.h>
#endif

#if defined(HAVE_OPENIPMI_H)
# include <linux/compiler.h>
# include <linux/ipmi.h>
#else
# include "open.h"
#endif

#define IPMI_OPENIPMI_DEV	"/dev/ipmi0"
#define IPMI_OPENIPMI_DEVFS	"/dev/ipmi/0"
#define IPMI_OPENIPMI_DEVFS2	"/dev/ipmidev/0"

static int
ipmi_openipmi_open(struct ipmi_intf * intf, int * perr)
{
	int i = 0;
#ifdef INCLUDE_PICMG_GET_DEVICE_LOCATOR
	struct ipmi_rq req;
	struct ipmi_rs *rsp;
	char msg_data;
#endif

	intf->fd = open(IPMI_OPENIPMI_DEV, O_RDWR);

	if (intf->fd < 0) {
		intf->fd = open(IPMI_OPENIPMI_DEVFS, O_RDWR);
		if (intf->fd < 0) {
			intf->fd = open(IPMI_OPENIPMI_DEVFS2, O_RDWR);
		}
		if (intf->fd < 0) {
			ipmi_printf("Could not open device at %s or %s or %s\n",
				IPMI_OPENIPMI_DEV, IPMI_OPENIPMI_DEVFS , IPMI_OPENIPMI_DEVFS2);
			ipmi_set_error(perr, IPMI_ERROR_COULD_NOT_USE_OPENIPMI);
			return -1;
		}
	}

	if (ioctl(intf->fd, IPMICTL_SET_GETS_EVENTS_CMD, &i) < 0) {
		ipmi_printf("Could not enable event receiver\n");
		ipmi_set_error(perr, IPMI_ERROR_OPENIPMI_ERROR);
		return -1;
	}

	if (intf->my_addr != 0) {
		unsigned int a = intf->my_addr;
		if (ioctl(intf->fd, IPMICTL_SET_MY_ADDRESS_CMD, &a) < 0) {
			ipmi_printf("Could not set IPMB address\n");
			ipmi_set_error(perr, IPMI_ERROR_OPENIPMI_ERROR);
			return -1;
		}
		if (verbose) ipmi_printf("Set IPMB address to 0x%x", intf->my_addr);
	}
	intf->opened = 1;

#ifdef INCLUDE_PICMG_GET_DEVICE_LOCATOR
	/* PICMG hack to set right IPMB address, 
	 * we might want to do GetPICMGProperties first.
	 * In any case, on a server board or a non-picmg IpmC blade , this code 
	 * will not have any adverse side effect
	 */
	if (intf->my_addr == IPMI_BMC_SLAVE_ADDR) {
		if (verbose) ipmi_printf("Running PICMG GetDeviceLocator\n");
		memset(&req, 0, sizeof(req));
		req.msg.netfn = IPMI_NETFN_PICMG;
		req.msg.cmd = 0x01;
		msg_data    = 0x00;
		req.msg.data = &msg_data; 
		req.msg.data_len = 1;
		msg_data = 0;

		rsp = intf->sendrecv(intf, &req, perr);
		if (rsp && !rsp->ccode) {
			intf->my_addr = rsp->data[2];
			intf->target_addr = intf->my_addr;
			lprintf(LOG_DEBUG, "Discovered IPMB address = 0x%x", intf->my_addr);
		}
	}
#endif

	return 0;
}

static void
ipmi_openipmi_close(struct ipmi_intf * intf, int force, int *perr)
{
	(void)force;
	(void)perr;

	if (intf->fd >= 0) {
		close(intf->fd);
		intf->fd = -1;
	}

	intf->opened = 0;
}

static struct ipmi_rs *
ipmi_openipmi_send_cmd(struct ipmi_intf * intf, struct ipmi_rq * req, int *perr)
{
	struct ipmi_recv recv_data;
	struct ipmi_addr addr;
	struct ipmi_system_interface_addr bmc_addr = {
		addr_type:	IPMI_SYSTEM_INTERFACE_ADDR_TYPE,
		channel:	IPMI_BMC_CHANNEL,
	};
	struct ipmi_ipmb_addr ipmb_addr = {
		addr_type:	IPMI_IPMB_ADDR_TYPE,
	};
	struct ipmi_req _req;
	static struct ipmi_rs rsp;
	static int curr_seq = 0;
	fd_set rset;

	if (intf == NULL || req == NULL)
		return NULL;
	if (intf->opened == 0 && intf->open != NULL)
		if (intf->open(intf, perr) < 0)
			return NULL;    // error should be set in this case

	if (verbose > 2)
		printbuf(req->msg.data, req->msg.data_len,
			 "OpenIPMI Request Message");

	/*
	 * setup and send message
	 */

	memset(&_req, 0, sizeof(struct ipmi_req));

	if (intf->target_addr != 0 &&
	    intf->target_addr != intf->my_addr) {
		/* use IPMB address if needed */
		ipmb_addr.slave_addr = intf->target_addr;
		_req.addr = (unsigned char *) &ipmb_addr;
		_req.addr_len = sizeof(ipmb_addr);
		if (verbose) ipmi_printf("Sending request to "
			"IPMB target @ 0x%x\n", intf->target_addr);
	} else {
		/* otherwise use system interface */
		if (verbose) ipmi_printf("Sending request to System Interface\n");
		_req.addr = (unsigned char *) &bmc_addr;
		_req.addr_len = sizeof(bmc_addr);
	}

	_req.msgid = curr_seq++;
	_req.msg.netfn = req->msg.netfn;
	_req.msg.cmd = req->msg.cmd;
	_req.msg.data = req->msg.data;
	_req.msg.data_len = req->msg.data_len;

	if (ioctl(intf->fd, IPMICTL_SEND_COMMAND, &_req) < 0) {
		ipmi_printf("Unable to send command\n");
		ipmi_set_error(perr, IPMI_ERROR_OPENIPMI_ERROR);
		return NULL;
	}

	/*
	 * wait for and retrieve response
	 */

	if (intf->noanswer)
		return NULL;

	FD_ZERO(&rset);
	FD_SET(intf->fd, &rset);

	if (select(intf->fd+1, &rset, NULL, NULL, NULL) < 0) {
		ipmi_printf("I/O Error\n");
		ipmi_set_error(perr, IPMI_ERROR_OPENIPMI_ERROR);
		return NULL;
	}
	if (FD_ISSET(intf->fd, &rset) == 0) {
		ipmi_printf("No data available\n");
		ipmi_set_error(perr, IPMI_ERROR_NO_RESPONSE);
		return NULL;
	}

	recv_data.addr = (unsigned char *) &addr;
	recv_data.addr_len = sizeof(addr);
	recv_data.msg.data = rsp.data;
	recv_data.msg.data_len = sizeof(rsp.data);

	/* get data */
	if (ioctl(intf->fd, IPMICTL_RECEIVE_MSG_TRUNC, &recv_data) < 0) {
		ipmi_printf("Error receiving message\n");
		if (errno != EMSGSIZE) {
			ipmi_set_error(perr, IPMI_ERROR_OPENIPMI_ERROR);
			return NULL;
		}
	}

	if (verbose > 4) {
		ipmi_printf("Got message:");
		ipmi_printf("  type      = %d\n", recv_data.recv_type);
		ipmi_printf("  channel   = 0x%x\n", addr.channel);
		ipmi_printf("  msgid     = %ld\n", recv_data.msgid);
		ipmi_printf("  netfn     = 0x%x\n", recv_data.msg.netfn);
		ipmi_printf("  cmd       = 0x%x\n", recv_data.msg.cmd);
		if (recv_data.msg.data && recv_data.msg.data_len) {
			ipmi_printf("  data_len  = %d\n", recv_data.msg.data_len);
			ipmi_printf("  data      = %s\n",
				buf2str(recv_data.msg.data, recv_data.msg.data_len));
		}
	}

	/* save completion code */
	rsp.ccode = recv_data.msg.data[0];
	rsp.data_len = recv_data.msg.data_len - 1;

	/* save response data for caller */
	if (rsp.ccode == 0 && rsp.data_len > 0) {
		memmove(rsp.data, rsp.data + 1, rsp.data_len);
		rsp.data[recv_data.msg.data_len] = 0;
	}

	return &rsp;
}

struct ipmi_intf ipmi_open_intf = {
	name:		"open",
	desc:		"Linux OpenIPMI Interface",
	open:		ipmi_openipmi_open,
	close:		ipmi_openipmi_close,
	sendrecv:	ipmi_openipmi_send_cmd,
	my_addr:	IPMI_BMC_SLAVE_ADDR,
	target_addr:	IPMI_BMC_SLAVE_ADDR,
};

