/*
 * Copyright (c) 2006 Raritan Computer Inc.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 
 * Redistribution of source code must retain the above copyright
 * notice, this list of conditions and the following disclaimer.
 * 
 * Redistribution in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the distribution.
 * 
 * Neither the name of Sun Microsystems, Inc. or the names of
 * contributors may be used to endorse or promote products derived
 * from this software without specific prior written permission.
 * 
 * This software is provided "AS IS," without a warranty of any kind.
 * ALL EXPRESS OR IMPLIED CONDITIONS, REPRESENTATIONS AND WARRANTIES,
 * INCLUDING ANY IMPLIED WARRANTY OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE OR NON-INFRINGEMENT, ARE HEREBY EXCLUDED.
 * SUN MICROSYSTEMS, INC. ("SUN") AND ITS LICENSORS SHALL NOT BE LIABLE
 * FOR ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING
 * OR DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.  IN NO EVENT WILL
 * SUN OR ITS LICENSORS BE LIABLE FOR ANY LOST REVENUE, PROFIT OR DATA,
 * OR FOR DIRECT, INDIRECT, SPECIAL, CONSEQUENTIAL, INCIDENTAL OR
 * PUNITIVE DAMAGES, HOWEVER CAUSED AND REGARDLESS OF THE THEORY OF
 * LIABILITY, ARISING OUT OF THE USE OF OR INABILITY TO USE THIS SOFTWARE,
 * EVEN IF SUN HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGES.
 * 
 * You acknowledge that this software is not designed or intended for use
 * in the design, construction, operation or maintenance of any nuclear
 * facility.
 */

#include <windows.h>
#include <shlwapi.h>
#include <ntddscsi.h>
#include <stddef.h>

#include <stdio.h>
#include <malloc.h>

#include "ipmitool/ipmi_print.h"

#include "scsi_api.h"
#include "scsi_windows.h"

static const char *default_vfloppy_id = "Virtual Disc";

struct scsi_dev_s {
    HANDLE handle;
    int is_locked;
};

static HANDLE open_drive(const char * drive) {
    // build the device name for the scsi command
    UCHAR string[25];

    DWORD shareMode = FILE_SHARE_READ | FILE_SHARE_WRITE;  // default
    DWORD accessMode = GENERIC_WRITE | GENERIC_READ;       // default

    if (drive == NULL || drive[0] == '\0') {
	return INVALID_HANDLE_VALUE;
    }

    if (drive[0] == drive[1] == '\\') {
	_snprintf(string, sizeof(string), "%s", drive);
    } else {
	_snprintf(string, sizeof(string), "\\\\.\\%s", drive);
    }

    // open the device
    return CreateFile((char *)string, accessMode, shareMode,
	NULL, OPEN_EXISTING, 0, NULL);
}

static int lock_drive_access(HANDLE hVolume, int unlock) {
    BOOL success;
    DWORD returned = 0;

    success = DeviceIoControl(hVolume,
	unlock ? FSCTL_UNLOCK_VOLUME : FSCTL_LOCK_VOLUME,
	NULL, 0, NULL, 0, &returned, NULL);

    if (!success) {
	int err = GetLastError();
	LPVOID lpMsgBuf;
	FormatMessage(    
	    FORMAT_MESSAGE_ALLOCATE_BUFFER | 
	    FORMAT_MESSAGE_FROM_SYSTEM |     
	    FORMAT_MESSAGE_IGNORE_INSERTS, NULL,
	    err, MAKELANGID(LANG_NEUTRAL, SUBLANG_DEFAULT), // Default language
	    (LPTSTR) &lpMsgBuf, 0, NULL ); // Process any inserts in lpMsgBuf.
	ipmi_printf("Could not %slock drive: %s\n", unlock ? "un" : "", lpMsgBuf);
	LocalFree( lpMsgBuf );
    }

    return success ? 0 : -1;
}

// query the vendor and device parameter of a drive
BOOL scsi_query_drive_info(const char * drive, char* vendor, char* device) {
    BOOL retval = TRUE;
    SCSI_PASS_THROUGH_WITH_BUFFERS sptwb;
    ULONG returned = 0;
    ULONG length;
    BOOL status;
    PUCHAR data;

    HANDLE fileHandle;
    
    fileHandle = open_drive(drive);
    if (fileHandle == INVALID_HANDLE_VALUE) {
	ipmi_printf("Error opening %c.\n", drive);
	return FALSE;
    }

    // query the device
    ZeroMemory(&sptwb,sizeof(SCSI_PASS_THROUGH_WITH_BUFFERS));
    sptwb.spt.Length = sizeof(SCSI_PASS_THROUGH);
    sptwb.spt.PathId = 0;
    sptwb.spt.TargetId = 1;
    sptwb.spt.Lun = 0;
    sptwb.spt.CdbLength = CDB6GENERIC_LENGTH;
    sptwb.spt.SenseInfoLength = 24;
    sptwb.spt.DataIn = SCSI_IOCTL_DATA_IN;
    sptwb.spt.DataTransferLength = 36;
    sptwb.spt.TimeOutValue = 2;
    sptwb.spt.DataBufferOffset = offsetof(SCSI_PASS_THROUGH_WITH_BUFFERS,ucDataBuf);
    sptwb.spt.SenseInfoOffset = offsetof(SCSI_PASS_THROUGH_WITH_BUFFERS,ucSenseBuf);
    sptwb.spt.Cdb[0] = SCSIOP_INQUIRY;
    sptwb.spt.Cdb[4] = 36;

    length = offsetof(SCSI_PASS_THROUGH_WITH_BUFFERS,ucDataBuf) +
	sptwb.spt.DataTransferLength;

    status = DeviceIoControl(fileHandle, IOCTL_SCSI_PASS_THROUGH,
	&sptwb, sizeof(SCSI_PASS_THROUGH),
	&sptwb, length,
	&returned, FALSE);

    // parse the output
    if(!status) {
	// the ioctl failed
	retval = FALSE;
	goto cleanup;
    }

    // we got it
    data = (PUCHAR)sptwb.ucDataBuf;

    strncpy(vendor, (PCHAR)(data + 8), 8);
    vendor[8] = '\0';
    strncpy(device, (PCHAR)(data + 16), 16);
    device[16] = '\0';

    // cleanup
cleanup:
    CloseHandle(fileHandle);
    return retval;
}

#define MAX_DEVICES 5

scsi_drive_t *scsi_scan_devices(const char *product) {
    scsi_drive_t *ret;
    int i, count = 0;

    // find all existing drives
    DWORD devices_mask = GetLogicalDrives();

    if(devices_mask == 0) {
	ipmi_printf("No Devices found.\n");
	return NULL;
    }

    if (product == NULL || product[0] == '\0') {
	product = default_vfloppy_id;
    }

    ret = malloc((MAX_DEVICES + 1) * sizeof(scsi_drive_t));
    memset(ret, 0, (MAX_DEVICES + 1) * sizeof(scsi_drive_t));

    // go through all devices
    for(i = 0; i < sizeof(devices_mask) * 8; i++) {
	if(devices_mask & (1 << i) && count < MAX_DEVICES) {
	    char drive_path[] = { 'A' + i, ':', '\0' };
	    char dos_device_name[MAX_PATH + 1];
	    DWORD ret_qdd;
	    CHAR scsi_vendor[9];
	    CHAR scsi_device[17];

	    // query the device and continue if it is a LAN device
	    memset(dos_device_name, 0, sizeof(dos_device_name));
	    ret_qdd = QueryDosDevice(drive_path, dos_device_name, MAX_PATH);

	    if(ret_qdd == 0) {
		ipmi_printf("Cannot query Drive %s\n", drive_path);
		continue;
	    }

	    if(StrStrI(dos_device_name, "LanManRedirector")) {
		continue;
	    }

	    // query the scsi settings
	    if(!scsi_query_drive_info(drive_path, scsi_vendor, scsi_device)) {
		// this happens if the device is no SCSI device, so don't show error here
		continue;
	    }

	    // match the info ...
	    if (!strncmp(scsi_device, product, strlen(product))) {
		ret[count].path = strdup(drive_path);
		ret[count].name = strdup(scsi_device);
		count++;
	    }
	}
    }

    return ret;
}

void scsi_free_device_list(scsi_drive_t *list) {
    scsi_drive_t *elem = list;
    if (!list) return;
    while (elem->path) {
	free(elem->path);
	free(elem->name);
	elem++;
    }
    free(list);
}

scsi_dev_t *scsi_open(const char *path) {
    scsi_dev_t *dev;
    HANDLE handle = open_drive(path);
    if (handle == INVALID_HANDLE_VALUE) return NULL;

    dev = malloc(sizeof(scsi_dev_t));
    memset(dev, 0, sizeof(scsi_dev_t));

    dev->handle = handle;
    if (!lock_drive_access(handle, 0)) {
	dev->is_locked = 1;
    }
    return dev;
}

void scsi_close(scsi_dev_t *dev) {
    if (dev->is_locked) {
	lock_drive_access(dev->handle, 1);
    }
    CloseHandle(dev->handle);
    free(dev);
}

int scsi_send_request(scsi_dev_t *dev, scsi_ioreq_t *req) {
    SCSI_PASS_THROUGH_WITH_BUFFERS sptwb;
    ULONG returned = 0;
    ULONG in_length, out_length;
    BOOL status;
    int ret = -1;

    ZeroMemory(&sptwb, sizeof(SCSI_PASS_THROUGH_WITH_BUFFERS));

    sptwb.spt.Length = sizeof(SCSI_PASS_THROUGH);
    sptwb.spt.PathId = 0;
    sptwb.spt.TargetId = 1;
    sptwb.spt.Lun = 0;
    sptwb.spt.CdbLength = req->cmd_len;
    sptwb.spt.SenseInfoLength = sizeof(sptwb.ucSenseBuf);
    switch (req->datadir) {
	case SCSI_DATADIR_TO_DEVICE:
	    sptwb.spt.DataIn = SCSI_IOCTL_DATA_OUT;
	    sptwb.spt.DataTransferLength = req->data_len;
	    memcpy(sptwb.ucDataBuf, req->data_buf, req->data_len);
	    out_length = offsetof(SCSI_PASS_THROUGH_WITH_BUFFERS, ucDataBuf) +
		sptwb.spt.DataTransferLength;
	    in_length = sizeof(SCSI_PASS_THROUGH_WITH_BUFFERS);
	    break;
	case SCSI_DATADIR_FROM_DEVICE:
	    sptwb.spt.DataIn = SCSI_IOCTL_DATA_IN;
	    sptwb.spt.DataTransferLength = req->data_len;
	    in_length = offsetof(SCSI_PASS_THROUGH_WITH_BUFFERS, ucDataBuf) +
		sptwb.spt.DataTransferLength;
	    out_length = sizeof(SCSI_PASS_THROUGH);
	    break;
	case SCSI_DATADIR_NONE:
	    sptwb.spt.DataIn = SCSI_IOCTL_DATA_UNSPECIFIED;
	    in_length = sizeof(SCSI_PASS_THROUGH_WITH_BUFFERS);
	    out_length = sizeof(SCSI_PASS_THROUGH);
	    break;
    }
    // NOTE: Windows is stupid! Don't know how it is implemented internally,
    // but a timeout of "1 second" means "it is timed out after the internal
    // clock switched to the next second" which could happen after one micro
    // second and lead to immediate timeouts.
    sptwb.spt.TimeOutValue = (req->timeout + 999) / 1000 + 1;
    sptwb.spt.DataBufferOffset = offsetof(SCSI_PASS_THROUGH_WITH_BUFFERS, ucDataBuf);
    sptwb.spt.SenseInfoOffset = offsetof(SCSI_PASS_THROUGH_WITH_BUFFERS, ucSenseBuf);

    memcpy(sptwb.spt.Cdb, req->cmd_buf, req->cmd_len);

    status = DeviceIoControl(dev->handle, IOCTL_SCSI_PASS_THROUGH,
	&sptwb, out_length,
	&sptwb, in_length,
	&returned, FALSE);

    // parse the output
    if(!status) {
	// the ioctl failed
	int err = GetLastError();
	LPVOID lpMsgBuf;
	FormatMessage(    
	    FORMAT_MESSAGE_ALLOCATE_BUFFER | 
	    FORMAT_MESSAGE_FROM_SYSTEM |     
	    FORMAT_MESSAGE_IGNORE_INSERTS, NULL,
	    err, MAKELANGID(LANG_NEUTRAL, SUBLANG_DEFAULT), // Default language
	    (LPTSTR) &lpMsgBuf, 0, NULL ); // Process any inserts in lpMsgBuf.
	ipmi_printf("scsi_send_request(): Error: ioctl failed (%s)\n", lpMsgBuf);
	LocalFree( lpMsgBuf );
	goto bail;
    }

    if (req->datadir == SCSI_DATADIR_FROM_DEVICE) {
	if (returned == 0) {
	    ipmi_printf("scsi_send_request(): Error: nothing returned in ioctl\n");
	    goto bail;
	}
	req->actual_len = sptwb.spt.DataTransferLength;
	memcpy(req->data_buf, sptwb.ucDataBuf, sptwb.spt.DataTransferLength);
    }

    ret = 0;

bail:
    return ret;
}

