/*
 * Copyright (c) 2003 Sun Microsystems, Inc.  All Rights Reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 
 * Redistribution of source code must retain the above copyright
 * notice, this list of conditions and the following disclaimer.
 * 
 * Redistribution in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the distribution.
 * 
 * Neither the name of Sun Microsystems, Inc. or the names of
 * contributors may be used to endorse or promote products derived
 * from this software without specific prior written permission.
 * 
 * This software is provided "AS IS," without a warranty of any kind.
 * ALL EXPRESS OR IMPLIED CONDITIONS, REPRESENTATIONS AND WARRANTIES,
 * INCLUDING ANY IMPLIED WARRANTY OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE OR NON-INFRINGEMENT, ARE HEREBY EXCLUDED.
 * SUN MICROSYSTEMS, INC. ("SUN") AND ITS LICENSORS SHALL NOT BE LIABLE
 * FOR ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING
 * OR DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.  IN NO EVENT WILL
 * SUN OR ITS LICENSORS BE LIABLE FOR ANY LOST REVENUE, PROFIT OR DATA,
 * OR FOR DIRECT, INDIRECT, SPECIAL, CONSEQUENTIAL, INCIDENTAL OR
 * PUNITIVE DAMAGES, HOWEVER CAUSED AND REGARDLESS OF THE THEORY OF
 * LIABILITY, ARISING OUT OF THE USE OF OR INABILITY TO USE THIS SOFTWARE,
 * EVEN IF SUN HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGES.
 * 
 * You acknowledge that this software is not designed or intended for use
 * in the design, construction, operation or maintenance of any nuclear
 * facility.
 */

#include <stdio.h>
#include <fcntl.h>
#include <unistd.h>
#include <errno.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>

#include <pp/base.h>
#include <pp/termios.h>

#include <config.h>
#include <ipmitool/ipmi.h>
#include <ipmitool/ipmi_intf.h>
#include <ipmitool/ipmi_print.h>
#include <ipmitool/ipmi_error.h>

#define SERIAL_START		0xA0
#define SERIAL_STOP		0xA5
#define SERIAL_HANDSHAKE	0xA6
#define SERIAL_DATA_ESCAPE	0xAA
#define SERIAL_ESCAPE		0x1B

#include <pp/base.h>

static int curr_seq = 0;

static int ipmi_to_serial(char *in_msg, char *out_msg, int len);
static int serial_to_ipmi(char *in_msg, char *out_msg, int len);

static struct ipmi_rq_entry *
ipmi_req_add_entry(struct ipmi_intf * intf, struct ipmi_rq * req, int * error)
{
    struct ipmi_rq_entry * e = malloc(sizeof(struct ipmi_rq_entry));

    if (e == NULL) {
	ipmi_set_error(error, IPMI_ERROR_NOT_ENOUGH_MEMORY);
	ipmi_printf("WARNING: no memory!\n");
    }
    else {
	memset(e, 0, sizeof(struct ipmi_rq_entry));
	memcpy(&e->req, req, sizeof(struct ipmi_rq));

	e->intf = intf;
    }
    return e;
}

static struct ipmi_rq_entry *
ipmi_serial_build_cmd(struct ipmi_intf * intf, struct ipmi_rq * req, int * error)
{
    uint8_t * ipmi_msg, * serial_msg;
    int cs, tmp;
    int len = 0;
    struct ipmi_rq_entry * entry;
    struct ipmi_session * s = intf->session;

    if (!intf->session) {
	return NULL;
    }

    if (curr_seq >= 64)
	curr_seq = 0;

    entry = ipmi_req_add_entry(intf, req, error);
    if (!entry)
	return NULL;

    len = req->msg.data_len + 7;
    ipmi_msg = malloc(len);
    memset(ipmi_msg, 0, len);

    /* ipmi message header */
    len = 0;
    cs = len;

    ipmi_msg[len++] = intf->target_addr;
    ipmi_msg[len++] = req->msg.netfn << 2;
    tmp = len - cs;
    ipmi_msg[len++] = ipmi_csum(ipmi_msg+cs, tmp);
    cs = len;
    ipmi_msg[len++] = intf->my_addr ? intf->my_addr : IPMI_REMOTE_SWID;

    entry->rq_seq = curr_seq++;
    ipmi_msg[len++] = entry->rq_seq << 2;
    ipmi_msg[len++] = req->msg.cmd;

    /* message data */
    if (req->msg.data_len) {
	memcpy(ipmi_msg+len, req->msg.data, req->msg.data_len);
	len += req->msg.data_len;
    }

    /* second checksum */
    tmp = len - cs;
    ipmi_msg[len++] = ipmi_csum(ipmi_msg+cs, tmp);

#ifdef PP_IPMI_DEBUG
    {
	int i;
	D("ipmi message: ");
	for (i = 0; i < len; i++) {
	    printf("%02x ", ipmi_msg[i]);
	}
	printf("\n");
    }
#endif // PP_IPMI_DEBUG

    /* add escape characters */
    serial_msg = malloc(2 * len + 2);
    serial_msg[0] = SERIAL_START;
    len = ipmi_to_serial(ipmi_msg, &serial_msg[1], len) + 1;
    serial_msg[len++] = SERIAL_STOP;

    free(ipmi_msg);

    if (s->in_seq) {
	s->in_seq++;
	if (!s->in_seq)
	    s->in_seq++;
    }

    entry->msg_len = len;
    entry->msg_data = serial_msg;

    return entry;
}

static struct ipmi_rs* ipmi_serial_build_response(char *serial_data, int len) {
    int x = 0;
    static struct ipmi_rs rsp;
    char *ipmi_data;
    int start_byte = -1;
    int i;

    if (len < 9) {
	ipmi_printf("Serial package too small.\n");
	return NULL;
    }

    for (i = 0; i < len; i++) {
    	if (serial_data[i] == SERIAL_START) {
    	    start_byte = i;
    	}
    	// don't break because restarting is allowed
    }
    
    if (start_byte == -1 || serial_data[len - 1] != SERIAL_STOP) {
	ipmi_printf("Serial package has wrong format.\n");
	return NULL;
    }

    /* delete escaped characters */
    ipmi_data = malloc(len - 2);
    len = serial_to_ipmi(&serial_data[1 + start_byte], ipmi_data, len - 2 - start_byte);

#ifdef PP_IPMI_DEBUG
    {
	int k;
	D("ipmi message (%d bytes): ", len);
	for (k = 0; k < len; k++) {
	    printf("%02x ", ipmi_data[k]);
	}
	printf("\n");
    }
#endif // PP_IPMI_DEBUG

    memset(&rsp, 0, sizeof(rsp));
    x++;		/* own ipmi address */	
    rsp.payload.ipmi_response.netfn   = ipmi_data[x] >> 2;
    rsp.payload.ipmi_response.rq_lun  = ipmi_data[x++] & 0x3;
    x++;		/* checksum */
    rsp.payload.ipmi_response.rs_addr = ipmi_data[x++];
    rsp.payload.ipmi_response.rq_seq  = ipmi_data[x] >> 2;
    rsp.payload.ipmi_response.rs_lun  = ipmi_data[x++] & 0x3;
    rsp.payload.ipmi_response.cmd     = ipmi_data[x++]; 
    rsp.ccode                         = ipmi_data[x++];

    rsp.data_len = len - 7 - start_byte;
    if (!rsp.ccode && rsp.data_len) {
	memcpy(&rsp.data, &ipmi_data[x], rsp.data_len);
    }

    free(ipmi_data);

#ifdef PP_IPMI_DEBUG
    printf("CC: %02x\n", rsp.ccode);
    printf("len: %d\n", rsp.data_len);
    {
	int k;
	D("rsp.data: ");
	for (k = 0; k < rsp.data_len; k++) {
	    printf("%02x ", rsp.data[k]);
	}
	printf("\n");
    }
#endif // PP_IPMI_DEBUG

    /* checksum follows */

    return &rsp;
}

static int serial_write(struct ipmi_intf * intf, char *buf, int len) {
    int i = 0, j;
    while (i < len) {
    	j = write(intf->fd, buf + i, len - i);
    	if (j < 0) {
    	    return j;
    	}
    	i += j;
    }
    return i;
}

static struct ipmi_rs * ipmi_serial_send_cmd(struct ipmi_intf * intf, struct ipmi_rq * req, int * error)
{
    struct ipmi_rq_entry * entry;
    uint8_t recv_message[512];
    int ret = 0, len = 0;
    struct timeval to;
    fd_set serialset;
    int have_response = 0;
    char handshake = SERIAL_HANDSHAKE;

    if (!intf->opened && intf->open && intf->open(intf, error) < 0)
	return NULL;

/* generate the request message */
    entry = ipmi_serial_build_cmd(intf, req, error);
    if (!entry) {
	ipmi_set_error(error, IPMI_ERROR_INTERNAL);
	ipmi_printf("Aborting send command, unable to build.\n");
	return NULL;
    }

#ifdef PP_IPMI_DEBUG
    {
	int i;
	D("Transmit: ");
	for (i = 0; i < entry->msg_len; i++) {
	    printf("%02x ", entry->msg_data[i]);
	}
	printf("\n");
    }
#endif // PP_IPMI_DEBUG

    /* send the data via serial */
    /* write the data */
    if (serial_write(intf, &handshake, 1) != 1 ||
    	serial_write(intf, entry->msg_data, entry->msg_len) != entry->msg_len) {
	
	ipmi_printf("Error: could not write data to serial interface.\n");
	ipmi_set_error(error, IPMI_ERROR_SEND_FAILED);
	goto bail;
    }

    /* wait for the answer and read it */
    len = 0;
    memset(recv_message, 0, sizeof(recv_message));
    while (len < (int)sizeof(recv_message)) {
    	to.tv_sec = len == 0 ? 5 : 1;	/* wait some time */
    	to.tv_usec = 0;
    	FD_ZERO(&serialset);
    	FD_SET(intf->fd, &serialset);
    	
    	if (select(intf->fd + 1, &serialset, NULL, NULL, &to) < 0) {
	    ipmi_set_error(error, IPMI_ERROR_NO_RESPONSE);
	    ipmi_printf("Error waiting for serial response.\n");
	    goto bail;

   	} else if (FD_ISSET(intf->fd, &serialset)) {
    	    ret = read(intf->fd, &recv_message[len], sizeof(recv_message) - len);
    	    if (ret > 0) {
#ifdef PP_IPMI_DEBUG
    	    	D("Received %d bytes.\n", ret);
    	        {
		    int i;
		    D("Received: ");
		    for (i = 0; i < ret; i++) {
		    	printf("%02x ", recv_message[len+i]);
		    }
		    printf("\n");
	    	}
#endif // PP_IPMI_DEBUG
    	    	len += ret;
    	    } else {
    	    	goto bail;
    	    }
	} else {
	    ipmi_set_error(error, IPMI_ERROR_NO_RESPONSE);
	    ipmi_printf("Timeout waiting for serial response.\n");
	    goto bail;
	}
	
	if (recv_message[len - 1] == SERIAL_STOP) {
	    break;
	}
    }
    
    have_response = 1;
    
#ifdef PP_IPMI_DEBUG
    if (have_response) {
	int i;
	D("Receive: ");
	for (i = 0; i < len; i++) {
	    printf("%02x ", recv_message[i]);
	}
	printf("\n");
    }
#endif // PP_IPMI_DEBUG

 bail:
    free(entry->msg_data);
    free(entry);

    return have_response ? ipmi_serial_build_response(recv_message, len) : NULL;
}

static int ipmi_serial_setup(struct ipmi_intf * intf, int * perr UNUSED)
{
    intf->cancel = 0;
    intf->session = malloc(sizeof(struct ipmi_session));
    memset(intf->session, 0, sizeof(struct ipmi_session));
    return (intf->session) ? 0 : -1;
}

static int ipmi_serial_open(struct ipmi_intf * intf, int * perr)
{
    char filename[50];
    
    if (!intf || !intf->session) {
	ipmi_set_error(perr, IPMI_ERROR_ESTABLISH_SESSION);
	return -1;
    }

    snprintf(filename, sizeof(filename), "/dev/ttyS%d", intf->session->serial_settings.port);
    D("Opening serial adapter %s.\n", filename);
    intf->fd = open(filename, O_RDWR);

    if (intf->fd == -1) {
	ipmi_printf("Could not open %s.\n", filename);
	ipmi_set_error(perr, IPMI_ERROR_NO_CONNECTION);
	return -1;
    }
    
    if (PP_FAILED(pp_base_set_tty_params(intf->fd,
    	intf->session->serial_settings.baud_rate,
    	intf->session->serial_settings.parity,
    	intf->session->serial_settings.bits,
    	intf->session->serial_settings.stop2,
    	intf->session->serial_settings.hwf,
    	intf->session->serial_settings.swf))) {
	
	ipmi_printf("Could not set tty params for %s.\n", filename);
	ipmi_set_error(perr, IPMI_ERROR_NO_CONNECTION);
	close(intf->fd);
	intf->fd = -1;
	return -1;
    }

    intf->opened = 1;
    curr_seq = 0;
    return 0;
}

static void ipmi_serial_close(struct ipmi_intf * intf UNUSED, int force UNUSED, int * perr UNUSED)
{
    D("Closing serial adapter.\n");
    close (intf->fd);
    intf->fd = -1;
    intf->opened = 0;
}

struct ipmi_intf ipmi_serial = {
    name:		"serial",
    desc:		"Peppercon IPMI Serial adapter",
    setup:		ipmi_serial_setup,
    open:		ipmi_serial_open,
    close:		ipmi_serial_close,
    sendrecv:		ipmi_serial_send_cmd,
    bmc_addr:		IPMI_BMC_SLAVE_ADDR,
    target_addr:	IPMI_BMC_SLAVE_ADDR,
};

/**** helper functions ****/

static int ipmi_to_serial(char *in_msg, char *out_msg, int len) {
    int i, len_out;
    for (i = 0, len_out = 0; i < len; i++) {
    	if (in_msg[i] == SERIAL_START ||
    	    in_msg[i] == SERIAL_STOP ||
    	    in_msg[i] == SERIAL_HANDSHAKE ||
    	    in_msg[i] == SERIAL_DATA_ESCAPE) {
    	    
    	    out_msg[len_out++] = 0xAA;
    	    out_msg[len_out++] = in_msg[i] + 0x10;
    	} else if (in_msg[i] == SERIAL_ESCAPE) {
    	    out_msg[len_out++] = 0xAA;
    	    out_msg[len_out++] = in_msg[i] + 0x20;
    	} else {
    	    out_msg[len_out++] = in_msg[i];
    	}
    }
    
    return len_out;
}

static int serial_to_ipmi(char *in_msg, char *out_msg, int len) {
    int i, len_out;
    for (i = 0, len_out = 0; i < len; i++) {
    	if (in_msg[i] == SERIAL_DATA_ESCAPE && (i < (len - 1))) {
    	    i++;
    	    if (in_msg[i] == (SERIAL_START + 0x10) ||
    	    	in_msg[i] == (SERIAL_STOP + 0x10) ||
    	    	in_msg[i] == (SERIAL_HANDSHAKE + 0x10) ||
    	    	in_msg[i] == (SERIAL_DATA_ESCAPE + 0x10)) {
    	    	
    	    	out_msg[len_out++] = in_msg[i] - 0x10;
    	    } else if (in_msg[i] == (SERIAL_ESCAPE + 0x20)) {
    	    	out_msg[len_out++] = in_msg[i] - 0x20;
    	    }
    	} else {
    	    out_msg[len_out++] = in_msg[i];
    	}
    }
    
    return len_out;
}
