/******************************************************************************\
* cat_condensation.c                                                           *
*                                                                              *
* Implementation of a particle filter algorithm for intelligent online mouse   *
* cursor tracking (CAT)                                                        *
*                                                                              *
* This filter only uses pointer moves for sampling and image understanding for *
* resampling                                                                   *
*                                                                              *
* Copyright 2005 Peppercon AG                                                  *
* Thomas Weber tweb@peppercon.de                                               *
\******************************************************************************/

/* TODO!!!
 * - strip off interface data only used for assertions... (fbinfo);
 */

#include <math.h> // for sqrt
#include <string.h> // for ffs

#include <pp/base.h>
#include <pp/vsc.h>
#include <pp/grab.h>

#include "cat_debug.h"
#include "cat_internal.h"
#include "iipptr_bitmap.h"
#include "iipptr_internal.h"
#if defined(PP_FEAT_KITTY_CAT)
#include "iipptr_cursor.h"
#endif /* PP_FEAT_KITTY_CAT */

static void reset_samples(cat_condensation_data_t *ccd);
static void initialize_samples(cat_condensation_data_t *ccd,
                               const fb_format_info_t *fb_info);
static void initialize_sample_random(cat_sample32_t *sample, 
                                     const fb_format_info_t *fb_info);
static inline int prob_cond(float prob); 

#if defined(CAT_DEBUG)
static int sample_sort(const void *p1, const void *p2);
#endif /* CAT_DEBUG */

static const float p_min = 1.0 / CAT_CONDENSATION_SAMPLE_COUNT;

#define RANDOM_NOISE(range)     (random() & range) 
#define RANDOM_NOISE_PM(range)  ((random() % (range << 1)) - range)
#define RANDOM_NOISE_PM_(range) ((random() & ((range << 1) | 1)) - range)

#define DEBUG_XY 96
#define noCAT_DEBUG_DUMP_SAMPLES

#define CAT_COND_SAMPLE_TIMEOUT_NS 100000000 // 100 ms

#if defined(CAT_CONDENSATION_BACKTRACE_PTR_MOVE)
#define RESAMPLE_INVALIDATE_BY_PTR_MOVE
#else /* CAT_CONDENSATION_BACKTRACE_PTR_MOVE */
#define RESAMPLE_INVALIDATE_BY_PTR_MOVE { \
            cat_queue_lock(ptr_move_queue); \
            if(!cat_queue_is_empty(ptr_move_queue)) { \
                cat_queue_unlock(ptr_move_queue); \
                CDMSG(CAT_CDMSG_CONDENSATION_INFO, \
                      "resampling invalidated due to pointer move\n"); \
                goto bail; \
            } \
            cat_queue_unlock(ptr_move_queue); \
        }
#endif /* CAT_CONDENSATION_BACKTRACE_PTR_MOVE */

/**
 * condensation lifecycle
 */
int cat_condensation_init(driver_t *obj) {
    cat_condensation_data_t *ccd = NULL;
    data_imouse_t* imouse_data;
    
    assert(obj);
    
    imouse_data = (data_imouse_t*)obj->data_conv_mouse.data;
    assert(imouse_data);
    
    if(imouse_data->tr_data || imouse_data->tr_sample ||
       imouse_data->tr_resample || imouse_data->tr_reset ||
       imouse_data->get_pos_estim || imouse_data->set_pos_estim) {
        goto error;
    }
    
    /* initialize cursors */
    /* TODO!!!!!! not only one but set of cursors and hotspots */
#if !defined(CAT_FB_DIRECT)
    {
        t_bitmap img = {0, 0, NULL};
        u_int16_t left, right, top, bottom;
        data_cat_t *mouse_data = (data_cat_t*)imouse_data;
        
        mouse_data->mask = (t_bitmap*)calloc(sizeof(t_bitmap), 1);
        mouse_data->shape = (t_bitmap*)calloc(sizeof(t_bitmap), 1);
        if(PP_ERR == read_bitmap(mouse_data->mask,
#if defined(__powerpc__)
                                 "/root/cursor_mask.bmp"
#else
"/home/tweb/src/eric_firmware/libpp_km/src/cat_test/matching/cursor_mask.bmp"
#endif
                                 )) {
            CDMSG(CAT_CDMSG_ERROR, "could not read mask file");
            goto error;
        }
        if(PP_ERR == threshold_bitmap_by_color(&img, mouse_data->mask, 
                                      (t_rgb){CAT_CROP_CHANNEL_THRESHOLD, 
                                              CAT_CROP_CHANNEL_THRESHOLD, 
                                              CAT_CROP_CHANNEL_THRESHOLD}, 1)) {
            CDMSG(CAT_CDMSG_ERROR, "could not threshold mask file");
            goto error;
        }
        if(PP_ERR == crop_bitmap_by_color(mouse_data->mask, 
                                          &left, &right, &top, &bottom, 
                                          &img, (t_rgb){0, 0, 0})) {
            CDMSG(CAT_CDMSG_ERROR, "could not crop mask file");
            goto error;
        }
        
        if(PP_ERR == read_bitmap(&img, 
#if defined(__powerpc__)
                                 "/root/cursor_shape.bmp"
#else
"/home/tweb/src/eric_firmware/libpp_km/src/cat_test/matching/cursor_shape.bmp"
#endif
                                 )) {
            CDMSG(CAT_CDMSG_ERROR, "could not read shape file");
            goto error;
        }
        if(PP_ERR == crop_bitmap(mouse_data->shape, &img, 
                                 left, right, top, bottom)) {
            CDMSG(CAT_CDMSG_ERROR, "could not crop shape file");
            goto error;
        }
    }
#elsif defined(PP_FEAT_CAT) /* CAT_FB_DIRECT */
    mouse_data->shape = &imouse_data->current_cursor_shape;
    mouse_data->mask = &imouse_data->current_cursor_mask;
#endif /* CAT_FB_DIRECT */
    
    /* initialize condensation tracking */
    ccd = (cat_condensation_data_t*)malloc(sizeof(cat_condensation_data_t));
    ccd->samples = (cat_sample32_t*)calloc(sizeof(cat_sample32_t),
                                           CAT_CONDENSATION_SAMPLE_COUNT);
    ccd->hypos = (cat_sample32_t*)calloc(sizeof(cat_sample32_t),
                                         CAT_CONDENSATION_SAMPLE_COUNT);
    ccd->initialize = 1;
    
    imouse_data->tr_data = (void*)ccd;
    imouse_data->tr_sample = cat_condensation_sample;
    imouse_data->tr_resample = cat_condensation_resample;
    imouse_data->tr_reset = cat_condensation_reset;
    imouse_data->get_pos_estim = cat_condensation_get_position_estimate;
    imouse_data->set_pos_estim = cat_condensation_set_position_estimate;
#if defined(PP_FEAT_CAT)
    mouse_data->get_pos_diff = cat_condensation_get_position_diff;
#endif /* PP_FEAT_CAT */

#if defined(CONDENSATION_DEBUG_SAMPLES)
    dsf = fopen(dsfn, "w");
#endif /* CONDENSATION_DEBUG_SAMPLES */
    
    return PP_SUC;
 
 error:
    CDMSG(CAT_CDMSG_ERROR, "Failed to initialize condensation tracking\n");
/*
    if(ccd) {
        free(ccd->samples);
        free(ccd->hypos);
        free(ccd);
        ccd = NULL;
    }
    mouse_data->tr_data = NULL;
    mouse_data->tr_sample = NULL;
    mouse_data->tr_resample = NULL;
    mouse_data->get_pos_estim = NULL; 
*/
#if !defined(CAT_FB_DIRECT)
    {
        data_cat_t *mouse_data = (data_cat_t*)imouse_data;

        if(mouse_data->mask) {
            free(mouse_data->mask->rgb);
            free(mouse_data->mask);
            mouse_data->mask = NULL;
        }
        if(mouse_data->shape) {
            free(mouse_data->shape->rgb);
            free(mouse_data->shape);
            mouse_data->shape = NULL;
        }
    }
#endif /* CAT_FB_DIRECT */
    abort();
    return PP_ERR;
}

void cat_condensation_cleanup(driver_t *obj) {
    cat_condensation_data_t *ccd;
    data_imouse_t* imouse_data;
#if defined(PP_FEAT_CAT) || !defined(CAT_FB_DIRECT)
    data_cat_t *mouse_data = (data_cat_t*)imouse_data;
#endif /* PP_FEAT_CAT || !CAT_FB_DIRECT */

#if defined(CONDENSATION_DEBUG_SAMPLES)
    fclose(dsf);
#endif /* CONDENSATION_DEBUG_SAMPLES */

    assert(obj);
    
    imouse_data = (data_imouse_t*)obj->data_conv_mouse.data;
    assert(imouse_data);
    
    ccd = (cat_condensation_data_t*)imouse_data->tr_data;
    if(ccd) {
        free(ccd->samples);
        free(ccd->hypos);
        free(ccd);
        ccd = NULL;
    }
#if !defined(CAT_FB_DIRECT)
    if(mouse_data->mask) {
        free(mouse_data->mask->rgb);
        free(mouse_data->mask);
        mouse_data->mask = NULL;
    }
    if(mouse_data->shape) {
        free(mouse_data->shape->rgb);
        free(mouse_data->shape);
        mouse_data->shape = NULL;
    }
#endif /* CAT_FB_DIRECT */
    imouse_data->tr_data = NULL;
    imouse_data->tr_sample = NULL;
    imouse_data->tr_resample = NULL;
    imouse_data->tr_reset = NULL;
    imouse_data->get_pos_estim = NULL; 
    imouse_data->set_pos_estim = NULL; 
#if defined(PP_FEAT_CAT)
    imouse_data->get_pos_diff = NULL; 
#endif /* PP_FEAT_CAT */
}

/*************************** condensation callbacks ***************************/
 
/**
 * sampling:
 * - dequeue ptr moves
 * - map absolute movement to relative predicion
 * - apply prediction to samples
 */
void cat_condensation_sample(driver_t *obj) {
    cat_condensation_data_t *ccd;
    data_imouse_t *imouse_data;
    cat_queue_t *diffmap_queue, /* *ptr_pos_queue, */ *ptr_move_queue;
    fb_format_info_t fb_info;
    cat_queue_entry_t *cqe;
    pp_stopwatch_t timeout;
#if defined(CAT_CONDENSATION_INTELLI_SAMPLE)
    cat_queue_diffmap_entry_t diffmap, last_diffmap;
    int diffmap_initialized, count, shape_width, shape_height;
#endif /* CAT_CONDENSATION_INTELLI_SAMPLE */
#if defined(PP_FEAT_KITTY_CAT)
    t_mouse_data *md;
#endif /* PP_FEAT_KITTY_CAT */
#if CAT_COND_PROFILE_CONDENSATION
    pp_stopwatch_t watch;
    
    CDMSG(CAT_COND_PROFILE_CONDENSATION, "called\n");
    pp_hrtime_start(&watch);
#endif /* CAT_COND_PROFILE_CONDENSATION */

    assert(obj);
    
    imouse_data = (data_imouse_t*)obj->data_conv_mouse.data;
    assert(imouse_data);
    
    diffmap_queue = imouse_data->diffmap_queue;
    assert(diffmap_queue);
    
//    ptr_pos_queue = mouse_data->ptr_pos_queue;
//    assert(ptr_pos_queue);
    
    ptr_move_queue = imouse_data->ptr_move_queue;
    assert(ptr_move_queue);
    
    ccd = (cat_condensation_data_t*)imouse_data->tr_data;
    assert(ccd);

    if(pp_vsc_get_fb_format(0, &fb_info) == PP_ERR) {
        CDMSG(CAT_CDMSG_WARNING, "No framebuffer info yet\n");
        return;
    }

#if defined(CAT_CONDENSATION_INTELLI_SAMPLE)
    diffmap_initialized = cat_get_merged_diffmap(&diffmap, &last_diffmap,
                                                 diffmap_queue) == PP_SUC;
#endif /* CAT_CONDENSATION_INTELLI_SAMPLE */

    if(ccd->initialize || ccd->reset) {
        while(ccd->initialize || ccd->reset) {
            if(PP_ERR != cat_queue_trylock(diffmap_queue)) {
                if(PP_ERR != cat_queue_trylock(ptr_move_queue)) {
                    cat_queue_clear(diffmap_queue);
                    cat_queue_clear(ptr_move_queue);
                    if(ccd->reset) {
                        reset_samples(ccd);
                        ccd->reset = 0;
                    } else {
                        /* initialize samples by random, clear all queues */
                        initialize_samples(ccd, &fb_info);
                    }
                    ccd->initialize = 0;
                    cat_queue_unlock(diffmap_queue);
                    cat_queue_unlock(ptr_move_queue);
                    break;
                }
//else { printf("tweb: 2nd search, moveq already locked\n"); }
                cat_queue_unlock(diffmap_queue);
            }
//else { printf("tweb: 2nd search, diffmapq already locked\n"); }
            usleep(25000);
        }
//        cat_queue_lock(ptr_pos_queue);
//        cat_queue_clear(ptr_pos_queue);
//        cat_queue_unlock(ptr_pos_queue);
    } else {
        /* get samples from hypos and predict position for each sample */
        
        int dx = 0, dy = 0;
        int rcdx = 0, rcdy = 0, i, sampled;
        float f_max;
        
        cat_queue_lock(ptr_move_queue);
        for(i = 0; NULL != (cqe = cat_queue_dequeue(ptr_move_queue)); ++i) {
            /* get translated offset from data_conf_mouse_cat.c: send_ptrmove */
            
            assert(cqe->type == CAT_QUEUE_PTR_MOVE_ENTRY);
            
            rcdx += cqe->data.ptr_move.x;
            rcdy += cqe->data.ptr_move.y;
/*
            dx += cat_translation_lookup(obj, cqe->data.ptr_move.x);
            dy += cat_translation_lookup(obj, cqe->data.ptr_move.y);
*/
            
            cat_queue_destroy_entry(cqe);
        }
        cat_queue_unlock(ptr_move_queue);
        
        ccd->relative.x = rcdx;
        ccd->relative.y = rcdy;
#warning translate by lookup table...
        dx = rcdx;
        dy = rcdy;
        
        sampled = 0;
//        pp_hrtime_start(&timeout);
        
#if defined(CAT_CONDENSATION_INTELLI_SAMPLE)
        {
            /* intelli sample... */

#if defined(PP_FEAT_CAT)
            data_cat_t *mouse_data = (data_cat_t*)imouse_data;
            
            assert(mouse_data->shape);
            shape_width = mouse_data->shape->width;
            shape_height = mouse_data->shape->height;
#endif /* PP_FEAT_CAT */

#if defined(PP_FEAT_KITTY_CAT)
            shape_width = imouse_data->current_cursor_shape.width;
            shape_height = imouse_data->current_cursor_shape.height;

            assert(imouse_data->current_kvm_port);
            MUTEX_LOCK(&imouse_data->kvm_port_mtx);
            if(imouse_data->current_kvm_port == NULL) {
                CDMSG(CAT_CDMSG_ERROR, "no current kvm port selected\n");
                return;
            }
            md = &imouse_data->current_kvm_port->mouse_data;
            MUTEX_UNLOCK(&imouse_data->kvm_port_mtx);
            assert(md);

#if defined(CAT_CONDENSATION_USE_ERIC_MOUSE_CURSOR_STATE)
            if(md->cursor_state != ERIC_MOUSE_CURSOR_IS_VALID) {
                CDMSG(CAT_CDMSG_INFO, "pointer shape not yet initialized\n");
                return;
            }
#else
            if(md->cursor_state != ERIC_MOUSE_CURSOR_IS_VALID && 
               PP_ERR == get_mousecursor_shape(obj)) {
                CDMSG(CAT_CDMSG_ERROR, 
                      "pointer shape could not be initialized\n");
                return;
            }
#endif
    
            ccd->absolute.x = md->cur_x / MOUSE_RASTER;
            ccd->absolute.y = md->cur_y / MOUSE_RASTER;
#endif /* PP_FEAT_KITTY_CAT */

/*
            diffmap_initialized = cat_get_merged_diffmap(&diffmap, 
                                                     diffmap_queue) == PP_SUC;
*/

            if(ccd->rreliable) {
                /* position was reliable on last resampling */
                
                int last_x, last_y;
            
                last_x = ccd->estim.x;
                last_y = ccd->estim.y;
                
                /* get some samples around last known position */
                for(count = 0; count < CAT_CONDENSATION_SAMPLE_COUNT / 8;
                    ++count, ++sampled) {
                    u_int32_t bias_x, bias_y;
                    int tmp_i;

                    tmp_i = last_x + RANDOM_NOISE_PM(5) * shape_width;
                    bias_x = SET_WITHIN(tmp_i, 0, (int)fb_info.g_w - 1);
                    tmp_i = last_y + RANDOM_NOISE_PM(5) * shape_height;
                    bias_y = SET_WITHIN(tmp_i, 0, (int)fb_info.g_h - 1);

                    if(!diffmap_initialized || 
                       PP_ERR == cat_diffmap_initialize_sample_biased(
                                                         &ccd->samples[sampled],
                                                         bias_x, bias_y,
                                                         &diffmap, &fb_info)) {
                        CDMSG(CAT_CDMSG_CONDENSATION_INTELLI_INFO, 
                              "could not init biased around last known "
                              "position\n");
                        ccd->samples[sampled].x = bias_x;
                        ccd->samples[sampled].y = bias_y;
                    }
//                    assert(ccd->samples[sampled].x < fb_info.g_w);
//                    assert(ccd->samples[sampled].y < fb_info.g_h);
                }

#if CAT_COND_PROFILE_CONDENSATION
                pp_hrtime_stop(&watch);
                CDMSG(CAT_COND_PROFILE_CONDENSATION,
                      "init biased around last known position finished, "
                      "took %llu usec, %d samples initialized\n",
                      pp_hrtime_read(&watch) / 1000, sampled);
#endif    
                
#if !defined(PP_FEAT_KITTY_CAT)
                {
                    float f_x, f_y, f;
                    /* some more samples in moving direction */
                    f_x = dx > 0 ? (int)fb_info.g_w - last_x : last_x;
                    f_x /= abs(dx);
                    f_y = dy > 0 ? (int)fb_info.g_h - last_y : last_y;
                    f_y /= abs(dy);
                    f = drand48() * MIN(f_x, f_y);
    
                    for(count = 0; count < CAT_CONDENSATION_SAMPLE_COUNT / 8;
                        ++count, ++sampled) {
                        u_int32_t bias_x, bias_y;
    
                        bias_x = SET_WITHIN(last_x + (int)floor((float)dx * f), 
                                            0, (int)fb_info.g_w - 1);
                        bias_y = SET_WITHIN(last_y + (int)floor((float)dy * f), 
                                            0, (int)fb_info.g_h - 1);
    
                        if(!diffmap_initialized || 
                           PP_ERR == cat_diffmap_initialize_sample_biased(
                                                         &ccd->samples[sampled],
                                                         bias_x, bias_y,
                                                         &diffmap, &fb_info)) {
                            CDMSG(CAT_CDMSG_CONDENSATION_INTELLI_INFO,
                                  "could not init biased in movement vector "
                                  "direction\n");
                            ccd->samples[sampled].x = bias_x;
                            ccd->samples[sampled].y = bias_y;
                        }
//                        assert(ccd->samples[sampled].x < fb_info.g_w);
//                        assert(ccd->samples[sampled].y < fb_info.g_h);
                    }

#if CAT_COND_PROFILE_CONDENSATION
                    pp_hrtime_stop(&watch);
                    CDMSG(CAT_COND_PROFILE_CONDENSATION,
                          "init biased in movement vector direction, "
                          "took %llu usec, %d samples initialized\n",
                          pp_hrtime_read(&watch) / 1000, sampled);
#endif /* CAT_COND_PROFILE_CONDENSATION  */
                }
#endif /* !PP_FEAT_KITTY_CAT  */
            }

            /* get some samples around last absolute position */
/*
            cat_queue_lock(ptr_pos_queue);
            for(i = 0; NULL != (cqe = cat_queue_dequeue(ptr_pos_queue)); ++i) {
                assert(cqe->type == CAT_QUEUE_PTR_POS_ENTRY);
                
                ax += cqe->data.ptr_pos.x;
                ay += cqe->data.ptr_pos.y;
                
                cat_queue_destroy_entry(cqe);
            }
            cat_queue_unlock(ptr_pos_queue);

            if(ax >= 0) {
*/
                for(count = 0; count < CAT_CONDENSATION_SAMPLE_COUNT / 8;
                    ++count, ++sampled) {
                    u_int32_t bias_x, bias_y;
                    int tmp_i;

                    tmp_i = ccd->absolute.x + RANDOM_NOISE_PM(5) * shape_width;
                    bias_x = SET_WITHIN(tmp_i, 0, (int)fb_info.g_w - 1);
                    tmp_i = ccd->absolute.y + RANDOM_NOISE_PM(5) * shape_height;
                    bias_y = SET_WITHIN(tmp_i, 0, (int)fb_info.g_h - 1);

                    if(!diffmap_initialized || 
                       PP_ERR == cat_diffmap_initialize_sample_biased(
                                                         &ccd->samples[sampled],
                                                         bias_x, bias_y,
                                                         &diffmap, &fb_info)) {
                        CDMSG(CAT_CDMSG_CONDENSATION_INTELLI_INFO,
                              "could not init biased around last known "
                              "absolute position\n");
                        ccd->samples[sampled].x = bias_x;
                        ccd->samples[sampled].y = bias_y;
                    }
//                    assert(ccd->samples[sampled].x < fb_info.g_w);
//                    assert(ccd->samples[sampled].y < fb_info.g_h);
                }

#if CAT_COND_PROFILE_CONDENSATION
                pp_hrtime_stop(&watch);
                CDMSG(CAT_COND_PROFILE_CONDENSATION,
                      "init biased around last known absolute position "
                      "finished, took %llu usec, %d samples initialized\n",
                      pp_hrtime_read(&watch) / 1000, sampled);
#endif    
//            }

            /* init some samples by chance */
            for(count = 0;
#if defined(PRODUCT_ERIC2)
                count < CAT_CONDENSATION_SAMPLE_COUNT / 8;
#else /* PRODUCT_ERIC2 */
                /* some more random samples for analog devices as diffmap is
                 * not that reliable due to noise */
                count < CAT_CONDENSATION_SAMPLE_COUNT / 3;
#endif /* PRODUCT_ERIC2 */
                ++count, ++sampled) {
                if(!diffmap_initialized ||
                   cat_diffmap_initialize_sample(&ccd->samples[sampled],
                                                 &diffmap, &fb_info)
                       == PP_ERR) {
                    CDMSG(CAT_CDMSG_CONDENSATION_INTELLI_INFO, 
                          "could not random sample from diffmap\n");
                    initialize_sample_random(&ccd->samples[sampled], &fb_info);
                }
 //               assert(ccd->samples[sampled].x < fb_info.g_w);
//                assert(ccd->samples[sampled].y < fb_info.g_h);
            }

#if CAT_COND_PROFILE_CONDENSATION
            pp_hrtime_stop(&watch);
            CDMSG(CAT_COND_PROFILE_CONDENSATION,
                  "init random by diffmap finished, took %llu usec, "
                  "%d samples initialized\n", 
                  pp_hrtime_read(&watch) / 1000, sampled);
#endif

#if defined(PRODUCT_ERIC2)
            /* init some samples by chance in last diffmap
             * makes no sense for analog devices, as last diffmap will
             * probably be caused by noise */
            for(count = 0; count < CAT_CONDENSATION_SAMPLE_COUNT / 8;
                ++count, ++sampled) {
                if(!diffmap_initialized ||
                   cat_diffmap_initialize_sample(&ccd->samples[sampled],
                                                 &last_diffmap, &fb_info)
                       == PP_ERR) {
                    CDMSG(CAT_CDMSG_CONDENSATION_INTELLI_INFO, 
                          "could not random sample from last diffmap\n");
                    initialize_sample_random(&ccd->samples[sampled], &fb_info);
                }
 //               assert(ccd->samples[sampled].x < fb_info.g_w);
//                assert(ccd->samples[sampled].y < fb_info.g_h);
            }
#endif /* PRODUCT_ERIC2 */

#if CAT_COND_PROFILE_CONDENSATION
            pp_hrtime_stop(&watch);
            CDMSG(CAT_COND_PROFILE_CONDENSATION,
                  "init random by last diffmap finished, took %llu usec, "
                  "%d samples initialized\n", 
                  pp_hrtime_read(&watch) / 1000, sampled);
#endif

            /* finished intelli sampling, now go for the hypos! */
        }
#endif /* CAT_CONDENSATION_INTELLI_SAMPLE */
        
        /**
         * scale the noise...
         * we want at least 20, usually twice the max of dx and dy
         */
        f_max = (float)MAX(dx, dy) / MAX(20, MAX(dx, dy) * 2);
        pp_hrtime_start(&timeout);

        for(; sampled < CAT_CONDENSATION_SAMPLE_COUNT; ) {
            int r = random() & CAT_CONDENSATION_SAMPLE_COUNT;
            float f = drand48() * f_max;

            /* sample according to weight */
            pp_hrtime_stop(&timeout);
            if(!prob_cond(ccd->hypos[r].w) && 
               pp_hrtime_read(&timeout) < CAT_COND_SAMPLE_TIMEOUT_NS) {
                continue;
            }
            
#if defined(CAT_CONDENSATION_INTELLI_SAMPLE)
            /* initialize random sample from diffmap */
            if(!ccd->hypos[r].w && (!diffmap_initialized ||
               cat_diffmap_initialize_sample(&ccd->samples[sampled], &diffmap,
                                             &fb_info) == PP_ERR)) {
                    CDMSG(CAT_CDMSG_CONDENSATION_INTELLI_INFO, 
                          "could not random sample from diffmap\n");
                initialize_sample_random(&ccd->samples[sampled], &fb_info);
            } else
#endif /* CAT_CONDENSATION_INTELLI_SAMPLE */
            {
                /* set real hypo */
            
                ccd->samples[sampled].x = SET_WITHIN((int)ccd->hypos[r].x + 
                                                     (int)floor((float)dx * f), 
                                                     0, (int)fb_info.g_w - 1);
                ccd->samples[sampled].y = SET_WITHIN((int)ccd->hypos[r].y + 
                                                     (int)floor((float)dy * f), 
                                                     0, (int)fb_info.g_h - 1);
            }
            
//            assert(ccd->samples[sampled].x < fb_info.g_w);
//            assert(ccd->samples[sampled].y < fb_info.g_h);
            ++sampled;
        }
            
#if CAT_COND_PROFILE_CONDENSATION
    pp_hrtime_stop(&watch);
    CDMSG(CAT_COND_PROFILE_CONDENSATION,
          "init from hypos finished, took %llu usec, %d samples initialized\n",
          pp_hrtime_read(&watch) / 1000, sampled);
#endif    
    }

#if defined(CAT_CONDENSATION_INTELLI_SAMPLE)
    if(diffmap_initialized) {
        free(diffmap.data);
    }
#endif /* CAT_CONDENSATION_INTELLI_SAMPLE */

#if CAT_COND_PROFILE_CONDENSATION
    pp_hrtime_stop(&watch);
    CDMSG(CAT_COND_PROFILE_CONDENSATION,
          "stopped, took %llu usec\n", pp_hrtime_read(&watch) / 1000);
#endif    
}

void cat_condensation_resample(driver_t *obj) {
    cat_condensation_data_t *ccd;
    data_imouse_t *imouse_data;
#if defined(PP_FEAT_CAT)
    data_cat_t *mouse_data = (data_cat_t*)imouse_data;
#endif  /* PP_FEAT_CAT */
    int cluster_ids[CAT_CONDENSATION_SAMPLE_COUNT];
    int i, j, clusters, diffmap_initialized = 0;
    fb_format_info_t fb_info;
#if defined(CAT_FB_DIRECT)
    u_int16_t *shape_fb_rgb = NULL;
#else /* CAT_FB_DIRECT */
    t_bitmap screen = {0, 0, NULL};
#endif /* CAT_FB_DIRECT */
    cat_queue_diffmap_entry_t diffmap;
    float weight_sum = 0, weight_f, p_max;
    int weight_count = 0;
    cat_sample32_t cluster_data[CAT_CONDENSATION_SAMPLE_COUNT];
    u_int32_t shape_width, shape_height;
#if !defined(CAT_CONDENSATION_INTELLI_SAMPLE)
    cat_queue_t *diffmap_queue;
#endif /* CAT_CONDENSATION_INTELLI_SAMPLE */
    cat_queue_t *ptr_move_queue;
    t_bitmap *shape, *mask;
#if defined(PP_FEAT_KITTY_CAT)
    t_mouse_data *md;
#endif /* PP_FEAT_KITTY_CAT */

    static const u_int32_t threshold = 32; /* max mean div per px and channel */ 
    static const u_int32_t t_squared = 1024; /* threshold squared (for MSE) */ 

#if CAT_COND_PROFILE_CONDENSATION
    pp_stopwatch_t watch;
    
    CDMSG(CAT_COND_PROFILE_CONDENSATION, "called\n");
    pp_hrtime_start(&watch);
#endif    

    assert(obj);
    
    imouse_data = (data_imouse_t*)obj->data_conv_mouse.data;
    assert(imouse_data);
    assert(imouse_data->grab_client);
    
    ptr_move_queue = imouse_data->ptr_move_queue;
    assert(ptr_move_queue);

    ccd = (cat_condensation_data_t*)imouse_data->tr_data;
    assert(ccd);
    assert(ccd->samples);
    
    if(ccd->initialize) {
        /* not initialized yet */
        return;
    }
    
    if(pp_vsc_get_fb_format(0, &fb_info) == PP_ERR) {
        CDMSG(CAT_CDMSG_WARNING, "No framebuffer info yet\n");
        return;
    }

#if defined(PP_FEAT_CAT)
    assert(mouse_data->shape);
    shape = mouse_data->shape;
    
    assert(mouse_data->mask);
    mask = mouse_data->mask;

    ccd->absolute.x = mouse_data->rc_abs.x;
    ccd->absolute.y = mouse_data->rc_abs.y;
#endif /* PP_FEAT_CAT */
#if defined(PP_FEAT_KITTY_CAT)
    assert(imouse_data->current_kvm_port);
    MUTEX_LOCK(&imouse_data->kvm_port_mtx);
    if(imouse_data->current_kvm_port == NULL) {
	CDMSG(CAT_CDMSG_ERROR,"no current kvm port selected\n");
	return;
    }
    md = &imouse_data->current_kvm_port->mouse_data;
    MUTEX_UNLOCK(&imouse_data->kvm_port_mtx);
    assert(md);

    shape = &imouse_data->current_cursor_shape;
    mask = &imouse_data->current_cursor_mask;

#if defined(CAT_CONDENSATION_USE_ERIC_MOUSE_CURSOR_STATE)
    if(md->cursor_state != ERIC_MOUSE_CURSOR_IS_VALID) {
        CDMSG(CAT_CDMSG_INFO, "pointer shape not yet initialized\n");
printf("cursor state %p is invalid!!!\n", &md->cursor_state);sleep(15);
        return;
    }
    
    /* if cursor is valid, shape and mask have to be initialized */
    assert(shape->rgb);
    assert(mask->rgb);
#else
    if(md->cursor_state != ERIC_MOUSE_CURSOR_IS_VALID && 
       PP_ERR == get_mousecursor_shape(obj)) {
        CDMSG(CAT_CDMSG_ERROR, 
              "pointer shape could not be initialized\n");
        return;
    }
#endif
#endif /* PP_FEAT_KITTY_CAT */

//#warning DEBUG
//fb_info.g_w = DEBUG_XY;fb_info.g_h = DEBUG_XY;
#if defined(CAT_FB_DIRECT)
#warning initialize obj only once!!!
    if(((!shape->rgb || !mask->rgb) &&
#if defined(PP_FEAT_CAT)
        PP_ERR == cat_intelli_get_mousecursor_shape(obj)) ||
#endif /* PP_FEAT_CAT */
#if defined(PP_FEAT_KITTY_CAT)
#if defined(CAT_CONDENSATION_USE_ERIC_MOUSE_CURSOR_STATE)
       0 && // shape and mask have to be initialized or something is wrong!
#endif /* CAT_CONDENSATION_USE_ERIC_MOUSE_CURSOR_STATE */
       PP_ERR == get_mousecursor_shape(obj)) ||
#endif /* PP_FEAT_CAT */
       PP_ERR == cat_bmp2fb_rgb(&shape_fb_rgb, shape)) {
        CDMSG(CAT_CDMSG_ERROR,
              "could not convert shape to frame buffer RGB!\n");
        abort();
        goto error;
    }
#else /* CAT_FB_DIRECT */
    screen.width = fb_info.g_w;
    screen.height = fb_info.g_h;
    screen.rgb = (t_rgb*)malloc(screen.width * screen.height * sizeof(t_rgb));

    if(!screen.rgb ||
       PP_ERR == fb_to_rgb_area(imouse_data->grab_client, &screen, 
                                0, 0, screen.width, screen.height)) {
        CDMSG(CAT_CDMSG_WARNING, "could not get screenshot!\n");
        free(screen.rgb);
//        abort();
        goto error;
    }
#endif /* CAT_FB_DIRECT */

#if !defined(CAT_CONDENSATION_INTELLI_SAMPLE)
    diffmap_queue = imouse_data->diffmap_queue;
    assert(diffmap_queue);
    
    diffmap_initialized = cat_get_merged_diffmap(&diffmap, NULL,
                                                 diffmap_queue) == PP_SUC;
#endif /* !CAT_CONDENSATION_INTELLI_SAMPLE */

    shape_width = shape->width;
    shape_height = shape->height;
    
    clusters = cat_cluster_simple_density(cluster_ids, ccd->samples, 
                                          shape_width * 2, shape_height * 2);
    CDMSG(CAT_CDMSG_CONDENSATION_INFO, 
          "clustering finished, now processing %d clusters\n", clusters + 1);
//    if(diffmap_initialized) DEBUG_CQDE(diffmap);
    
    for(i = 0; i < clusters; ++i) {
        int count, ret, s_left, s_top, s_width, s_height;
        t_point found;
        u_int64_t matching_time;
        
        RESAMPLE_INVALIDATE_BY_PTR_MOVE;
        
        if(PP_ERR == (count = cat_cluster_get_search_space_for_cluster(
                                  &s_left, &s_top, &s_width, &s_height,
                                  i, cluster_ids, ccd->samples,
                                  shape_width, shape_height, 
                                  PP_FB_TILE_WIDTH, PP_FB_TILE_HEIGHT,
                                  fb_info.g_w, fb_info.g_h))) {
            CDMSG(CAT_CDMSG_ERROR, "ERROR occured while getting search space "
                                   "dimensions for cluster %d!\n", i);
            break;
        }
#if 0
        {
            /* TODO! calc search space from cluster dimensions */
/*
            s_left = x_min;
            s_top = y_min;
            s_width = x_max - x_min + shape->width;
            s_height = y_max - y_min + shape->height;
            
            if(s_left + s_width > (int)screen.width) {
                s_width = screen.width - s_left;
            }
            if(s_top + s_height > (int)screen.height) {
                s_height = screen.height - s_top;
            }
*/            
            int scr_max_x, scr_max_y, tmp;
            int fuzzy = PP_FB_TILE_WIDTH / 2;
            
            s_left = x_min > fuzzy ? x_min - fuzzy : 0;
            s_top = y_min > fuzzy ? y_min - fuzzy : 0;
            scr_max_x = fb_info.g_w - s_left;
            scr_max_y = fb_info.g_h - s_top;
            tmp = x_max - x_min + shape->width + fuzzy;
            s_width = tmp < scr_max_x ? tmp : scr_max_x;
            tmp = y_max - y_min + shape->height + fuzzy;
            s_height = tmp < scr_max_y ? tmp : scr_max_y;
        }
#endif
        
        matching_time = pp_hrtime();
#if defined(CAT_FB_DIRECT)
        if(PP_ERR == (ret = cat_mse32_fb(imouse_data->grab_client,
                                         shape_fb_rgb, mask,
                                         s_left, s_top, s_width, s_height,
                                         threshold, 0, &found))) {
#else /* CAT_FB_DIRECT */
        if(PP_ERR == (ret = cat_mse32(&screen, shape, mask,
                                      s_left, s_top, s_width, s_height,
                                      threshold, &found))) {
#endif /* CAT_FB_DIRECT */
            /* we didn't find a mouse pointer at current location,
               init new hypo from diffmap, fall back to random */
            for(j = 0; j < CAT_CONDENSATION_SAMPLE_COUNT; ++j) {
#if defined(CAT_CONDENSATION_INTELLI_SAMPLE)
                if(cluster_ids[j] == i) {
                    ccd->hypos[j].p = 0;
                    ccd->hypos[j].w = 0;
                }
#else /* CAT_CONDENSATION_INTELLI_SAMPLE */
#if 1 // init from diffmap, fallback to random
                if(cluster_ids[j] == i && (!diffmap_initialized ||
                   cat_diffmap_initialize_sample(&ccd->hypos[j], &diffmap,
                                                 &fb_info) == PP_ERR)) {
//printf("tweb: hypo %d: falling back to random...\n", j);
#else // init random
                if(cluster_ids[j] == i) {
#endif
                    initialize_sample_random(&ccd->hypos[j], &fb_info);
//                    ccd->hypos[j].w = t_squared; /* maximum error */
                    ccd->hypos[j].w = 0; /* maximum error */
                }
#endif /* CAT_CONDENSATION_INTELLI_SAMPLE */
            }
        } else {
            /* we succeeded... set hypo to detected coordinates */
            float p = (float)count / CAT_CONDENSATION_SAMPLE_COUNT;
            int w = t_squared - ret;
            
            cluster_data[weight_count].x = found.pos_x;
            cluster_data[weight_count].y = found.pos_y;
            cluster_data[weight_count].p = p;
            cluster_data[weight_count].w = w;
            cluster_data[weight_count].t = matching_time;
            for(j = 0; j < CAT_CONDENSATION_SAMPLE_COUNT; ++j) {
                if(cluster_ids[j] == i) {
                    ccd->hypos[j].x = found.pos_x;
                    ccd->hypos[j].y = found.pos_y;
                    ccd->hypos[j].p = p;
                    ccd->hypos[j].w = w;
                    ccd->hypos[j].t = cluster_data[weight_count].t;
                }
            }
            weight_sum += w;
            ++weight_count;
        }            
    }
    
    if(!weight_count) {
        /* no hit, return */
        CDMSG(CAT_CDMSG_CONDENSATION_INFO,
              "could not locate pointer, bailing out\n");
        CAT_INVALIDATE_POSITION_ESTIMATE;
        ccd->rreliable = 0;
        goto bail;
    }
    
#if 0
    /**
     * we summed up weights, now calc probabilities
     * - sum over all weights is weight_sum, prob is 100%
     * - block matching weights are indirect, small weight means good result
     * - maximum weight sum could be (for mse) threshold^2 * weight_count
     * - we want p(threshold^2) = 0% and p(0) = max
     * - weight sum is maximum weight sum - weight_sum
     * - weighting factor f = 1 / weight_sum
     * for single weight w
     * - correct value w = threshold^2 - w
     * - multiply w with factor f
     */
    weight_sum = t_squared * weight_count - weight_sum;
    weight_f = 1.0 / weight_sum;
    for(j = 0; j < CAT_CONDENSATION_SAMPLE_COUNT; ++j) {
//printf("converted w = %f to %f\n", ccd->hypos[j].w, (t_squared - ccd->hypos[j].w) * weight_f);
        ccd->hypos[j].w = (t_squared - ccd->hypos[j].w) * weight_f;
    }
#endif
    weight_f = 1.0 / weight_sum;
    for(j = 0; j < CAT_CONDENSATION_SAMPLE_COUNT; ++j) {
//printf("converted w = %f to %f\n", ccd->hypos[j].w, ccd->hypos[j].w * weight_f);
        ccd->hypos[j].w *= weight_f;
    }
    for(j = 0, p_max = 0; j < weight_count; ++j) {
        cluster_data[j].w *= weight_f;
        if(cluster_data[j].p > p_max) {
            p_max = cluster_data[j].p;
            i = j;
        }
    }
    
    RESAMPLE_INVALIDATE_BY_PTR_MOVE;
    if((weight_count == 1 || p_max > 0.75)) {
        /* position is only reliable if we didn't move
         * only one cluster is certainly reliable ;-) 
         * p > 0.75 should also be reliable enough... */
        
        CDMSG(CAT_CDMSG_CONDENSATION_INFO, "position (%4d, %3d) is reliable\n", 
              cluster_data[i].x, cluster_data[i].y);
#if 0 // impossible?
        if(ccd->rreliable) {
            /* position was reliable on last resampling */
            int dx_abs, dy_abs;
            
            dx_abs = cluster_data[i].x - ccd->estim.x;
            dy_abs = cluster_data[i].y - ccd->estim.y;
            
/*
            cat_translation_adjust(obj, ccd->relative.x, dx_abs);
            cat_translation_adjust(obj, ccd->relative.y, dy_abs);
*/
        }
#endif

#if !defined(CAT_CONDENSATION_BACKTRACE_PTR_MOVE)
        cat_set_position_estimate(obj, 1, cluster_data[i].x, cluster_data[i].y,
                                  cluster_data[i].t);
#else /* CAT_CONDENSATION_BACKTRACE_PTR_MOVE */
        {
            cat_queue_entry_t *cqe;
            int ptr_pos_x = cluster_data[i].x;
            int ptr_pos_y = cluster_data[i].y;
            u_int64_t now = pp_hrtime();
            long dq_offset = 5 /* ms */ << 20 /* ns ;-) */;
int i = 0;
            
            cat_queue_lock(ptr_move_queue);
            while(NULL != 
                  (cqe = cat_queue_dequeue_timed(ptr_move_queue,
                                                 cluster_data[i].t - dq_offset,
                                                 CAT_QUEUE_TIME_BEFORE))) {
                /* skip old moves */
                cat_queue_destroy_entry(cqe);
i++;
            }
printf("tweb: %d entries dropped\n", i);
i = 0;
            while(NULL != 
                  (cqe = cat_queue_dequeue_timed(ptr_move_queue,
                                                 now - dq_offset,
                                                 CAT_QUEUE_TIME_BEFORE))) {
                /* apply new moves (older than ???ms) */
                assert(cqe->type == CAT_QUEUE_PTR_MOVE_ENTRY);
                
                ptr_pos_x += cqe->data.ptr_move.x;
                ptr_pos_y += cqe->data.ptr_move.y;
                cat_queue_destroy_entry(cqe);
            }
printf("tweb: %d entries applied (%d, %d)\n", i, ptr_pos_x - cluster_data[i].x, ptr_pos_y - cluster_data[i].y);

            if(cat_queue_is_empty(ptr_move_queue)) {
                cat_set_position_estimate(obj, 1, 
                                          SET_WITHIN(ptr_pos_x, 0, 
                                                     (int)fb_info.g_w - 1),
                                          SET_WITHIN(ptr_pos_y, 0, 
                                                     (int)fb_info.g_h - 1),
                                          cluster_data[i].t);
            }
            cat_queue_unlock(ptr_move_queue);
        }
#endif /* CAT_CONDENSATION_BACKTRACE_PTR_MOVE */
        
        ccd->rreliable = 1;
    } else {
        CAT_INVALIDATE_POSITION_ESTIMATE;
        ccd->rreliable = 0;
    }
    
#if defined(CAT_DEBUG)
    {
        int top = CAT_CONDENSATION_SAMPLE_COUNT >> 1;
        u_int32_t cx = -1, cy = -1;
        float cp = -1, cw = -1;
        
#if 1
        /* sort samples and print top n */
        qsort(cluster_data, weight_count,
              sizeof(cat_sample32_t), sample_sort);
        for(i = 0, j = 0; i < weight_count && j < top &&
                          cluster_data[i].w > p_min; ++i) {
            if(cluster_data[i].w == cw && cluster_data[i].p == cp &&
               cluster_data[i].x == cx && cluster_data[i].y == cy) {
                /* still the same... */
                continue;
            }
            cw = cluster_data[i].w;
            cp = cluster_data[i].p;
            cx = cluster_data[i].x;
            cy = cluster_data[i].y;
            ++j;
            CDMSG(CAT_CDMSG_CONDENSATION_INFO,
                  "cluster %3d: x = %4d, y = %3d, p = %f, w = %f\n", 
                  j, cx, cy, cp, cw); 
        }
#else
        /* sort samples and print top n */
        qsort(ccd->hypos, CAT_CONDENSATION_SAMPLE_COUNT,
              sizeof(cat_sample32_t), sample_sort);
        for(i = 0, j = 0; i < CAT_CONDENSATION_SAMPLE_COUNT && j < top &&
                          ccd->hypos[i].w > p_min; ++i) {
            if(ccd->hypos[i].w == cw && ccd->hypos[i].p == cp &&
               ccd->hypos[i].x == cx && ccd->hypos[i].y == cy) {
                /* still the same... */
                continue;
            }
            cw = ccd->hypos[i].w;
            cp = ccd->hypos[i].p;
            cx = ccd->hypos[i].x;
            cy = ccd->hypos[i].y;
            ++j;
            CDMSG(1, "cluster %3d: x = %4d, y = %3d, p = %f, w = %f\n", 
                  j, cx, cy, cp, cw); 
        }
#endif
    }
#endif /* CAT_DEBUG */
    
 bail:
#if CAT_COND_PROFILE_CONDENSATION
    pp_hrtime_stop(&watch);
    CDMSG(CAT_COND_PROFILE_CONDENSATION,
          "stopped, took %llu usec\n", pp_hrtime_read(&watch) / 1000);
#endif    

#if defined(CAT_DEBUG) && defined(CAT_DEBUG_DUMP_SAMPLES)
    {
        /* print samples to screenshot and save */
        /* SCREENSHOT WILL BE MODIFIED, DO NOT USE IT ANY MORE!!! */
        
        char hypos_fn[100];
        static int hypos_iter = 1;
        
        if(diffmap_initialized) {
            DEBUG_DRAW_CQDE(diffmap,
                            &screen, ((t_rgb){r: 0xfd, g: 0xac, b: 0x38}));
        }

        for(i = 0; i < clusters; ++i) {
            int s_left, s_top, s_width, s_height;
            
            if(PP_ERR == cat_cluster_get_search_space_for_cluster(
                             &s_left, &s_top, &s_width, &s_height,
                             i, cluster_ids, ccd->samples,
                             shape_width, shape_height, 
                             PP_FB_TILE_WIDTH, PP_FB_TILE_HEIGHT,
                             fb_info.g_w, fb_info.g_h)) {
                break;
            }
            
            cat_debug_draw_rect(s_left, s_top,
                                s_left + s_width, s_top + s_height,
                                &screen, (t_rgb){r: 0xff, g: 0xff, b: 0x00});
        }

        cat_debug_clustering_draw_clusters(clusters, cluster_ids, ccd->samples,
                                           &screen, 
                                           (t_rgb){r: 0x00, g: 0xff, b: 0x00});

        cat_debug_draw_samples(ccd->samples, &screen, 
                               (t_rgb){r: 0xff, g: 0x00, b: 0x00});

        cat_debug_draw_samples(ccd->hypos, &screen, 
                               (t_rgb){r: 0x00, g: 0x00, b: 0xff});

        snprintf(hypos_fn, 100, "debug_samples_%s%d.bmp", 
                 hypos_iter < 10 ? "0" : "", hypos_iter);
        write_bitmap(&screen, hypos_fn);
        hypos_iter = (hypos_iter % 64) + 1;
        CDMSG(1, "wrote %s\n", hypos_fn);
    }
#endif /* CAT_DEBUG && CAT_DEBUG_DUMP_SAMPLES */

 error:
#if defined(CAT_FB_DIRECT)
    free(shape_fb_rgb);
#else /* CAT_FB_DIRECT */
    free(screen.rgb);
#endif /* CAT_FB_DIRECT */
    if(diffmap_initialized) {
        free(diffmap.data);
    }
};

int64_t cat_condensation_get_position_estimate(const driver_t *obj,
                                               u_int16_t *estim_x,
                                               u_int16_t *estim_y) {
    cat_condensation_data_t *ccd;
    data_imouse_t *imouse_data;

    assert(obj);
    
    imouse_data = ((data_imouse_t*)obj->data_conv_mouse.data);
    assert(imouse_data);
    
    ccd = (cat_condensation_data_t*)imouse_data->tr_data;
    assert(ccd);

    if(!ccd->reliable) {
        return PP_ERR;
    }
    
    if(estim_x && estim_y) {
        *estim_x = ccd->estim.x;
        *estim_y = ccd->estim.y;
    }
    
    return ccd->estim_time;
}

int cat_condensation_set_position_estimate(const driver_t *obj,
                                           u_char reliable,
                                           u_int16_t estim_x, 
                                           u_int16_t estim_y,
                                           u_int64_t timestamp) {
    cat_condensation_data_t *ccd;
    data_imouse_t *imouse_data;

    assert(obj);
    
    /**
     * later on, we use 63 bit timestamps... 
     * assert, we are not running out of time ;-)
     */
    assert((int64_t)timestamp >= 0);
    
    imouse_data = ((data_imouse_t*)obj->data_conv_mouse.data);
    assert(imouse_data);
    
    ccd = (cat_condensation_data_t*)imouse_data->tr_data;
    assert(ccd);

    ccd->reliable = reliable == 1;

    if(ccd->reliable) {
        ccd->estim.x = estim_x;
        ccd->estim.y = estim_y;
    }
    
    return PP_SUC;
}

#if defined(PP_FEAT_CAT) // absolute position only defined for CAT
int cat_condensation_get_position_diff(const driver_t *obj,
                                       int16_t *dx, int16_t *dy) {
    cat_condensation_data_t *ccd;
    data_imouse_t *imouse_data;

    assert(obj);
    
    imouse_data = ((data_imouse_t*)obj->data_conv_mouse.data);
    assert(imouse_data);
    
    ccd = (cat_condensation_data_t*)imouse_data->tr_data;
    assert(ccd);

    if(!ccd->reliable) {
        return PP_ERR;
    }
    
    if(dx && dy) {
        *dx = ccd->absolute.x - ccd->estim.x;
        *dy = ccd->absolute.y - ccd->estim.y;
    }
    
    return PP_SUC;
}
#endif /* PP_FEAT_CAT */


/****************************** local functions *******************************/

static void reset_samples(cat_condensation_data_t *ccd) {
    int i;
    
    assert(ccd);
    
    for(i = 0; i < CAT_CONDENSATION_SAMPLE_COUNT; ++i) {
        ccd->samples[i].x = 0;
        ccd->samples[i].y = 0;
        ccd->samples[i].p = p_min;
    }
}

static void initialize_samples(cat_condensation_data_t *ccd,
                               const fb_format_info_t *fb_info) {
    int i;
    
    assert(ccd);
    assert(fb_info);
    
    for(i = 0; i < CAT_CONDENSATION_SAMPLE_COUNT; ++i) {
        initialize_sample_random(&ccd->samples[i], fb_info);
    }
}

static void initialize_sample_random(cat_sample32_t *sample, 
                                     const fb_format_info_t *fb_info) {
    assert(sample);
    assert(fb_info);
    
    /* TODO: something "faster" than mod? */
    sample->x = random() % fb_info->g_w;
    sample->y = random() % fb_info->g_h;
//#warning init samples debug!!!
//sample->x = random() % DEBUG_XY; sample->y = random() % DEBUG_XY;
    sample->p = p_min;
}

static inline int prob_cond(float prob) {
    return random() < (0x7fffffff * prob);
}


#if defined(CAT_DEBUG)
/**
 * sort by weight
 */
static int sample_sort(const void *p1, const void *p2) {
    assert(p1);
    assert(p2);

    return (((cat_sample32_t*)p1)->w) < (((cat_sample32_t*)p2)->w) ||
           (((cat_sample32_t*)p1)->p) < (((cat_sample32_t*)p2)->p) ||
           (((cat_sample32_t*)p1)->x) < (((cat_sample32_t*)p2)->x) ||
           (((cat_sample32_t*)p1)->y) < (((cat_sample32_t*)p2)->y) ? 1 : -1;
}
#endif /* CAT_DEBUG */

void cat_condensation_reset(const driver_t *obj) {
    cat_condensation_data_t *ccd;
    data_imouse_t *imouse_data;
    
    assert(obj);
    
    imouse_data = (data_imouse_t*)obj->data_conv_mouse.data;
    assert(imouse_data);

    ccd = (cat_condensation_data_t*)imouse_data->tr_data;
    assert(ccd);
    
    CAT_INVALIDATE_POSITION_ESTIMATE;
    ccd->reset = 1;
}

