/******************************************************************************\
* cat_intelli.c                                                                *
*                                                                              *
* Routines for intelligent syncing CAT driven mouse pointers                   *
* This adapts funtionality already implemented for mouse_intelli               *
*                                                                              *
* Copyright 2005 Peppercon AG                                                  *
* Thomas Weber tweb@peppercon.de                                               *
\******************************************************************************/

#include <math.h> // for fabs

#include <pp/vsc.h>

#include "iipptr_cursor.h"
#include "cat_internal.h"
#include "cat_debug.h"

#if defined(PP_KM_CAT_INTELLI)

//#define PPDU_MAX_LENGTH		4
//#define BUILD_INITIAL_SLEEP	10000
#define MOUSE_SEGMENT_MIN_SIZE  3
#define REGRESSION_COUNT	25
#define BUILD_SLEEP		100000
#define MOUSE_MAX_DIFF_COUNT    5
#define MAX_M                   15.0

extern t_mouse_func non_mouse_corr;

static int cat_detect_mouse_universe(driver_t* obj) {
    t_point mouse_cursor_point={0,0}, mouse_cursor_expect={0,0};
    int	i, step, prev;
    int ret = ERIC_MOUSE_UNIVERSE_OK;
    char debug_msg[100];
    data_imouse_t* imouse_data;
    t_mouse_data* md;
#if defined(CAT_FB_DIRECT)
    u_int16_t *shape_fb_rgb = NULL;
#endif /* CAT_FB_DIRECT */
#if defined(CAT_ADAPTIVE_LOOKUP_TABLE)
    data_cat_t* mouse_data;
#endif /* CAT_ADAPTIVE_LOOKUP_TABLE */
    
    assert(obj);
    
    imouse_data = (data_imouse_t*)obj->data_conv_mouse.data;
    assert(imouse_data);
    assert(imouse_data->current_kvm_port);
    
    md = &imouse_data->current_kvm_port->mouse_data;
    assert(md);

    md->state = ERIC_MOUSE_GETTING;
    md->accel_steps = 0;

    /* get mouse movements */
    if(0 != move_mouse_corner(obj, 0))
	goto bailout;
	
    mouse_cursor_expect.pos_x = 0;

#if defined(CAT_FB_DIRECT)
    if(PP_ERR == cat_bmp2fb_rgb(&shape_fb_rgb,
                                &imouse_data->current_cursor_shape)) {
        CDMSG(CAT_CDMSG_ERROR,
              "could not convert shape to frame buffer RGB!\n");
        abort();
        goto bailout;
    }
#endif /* CAT_FB_DIRECT */

#if 1
    {
        int found = 0, sync_iter = 0, __min_found__TODO__ = 25;
        
        cat_lookup_table_reset(obj);
    
        while(sync_iter++ < 10 && found < __min_found__TODO__) {
            for(i = step = prev = 0; step <= PP_KM_CAT_MAX_TICKS_DIRECT; 
                step += (step / 16) + sync_iter) {
                // make step 
                
                /* assert we do not send 0xAA to mouse */
                /* step == 0xAA */
                step += step == 0xAA;
                /* -2 * step == - 0xAA */
                step += step == 0x2B;
                
//                if(0 != move_mouse_direct(obj, step, 0, 0, 0, 1)) {
                if(PP_SUC != (obj->data_unit_mouse.send_mouse_data(obj, step, 
                                                                    0, 0, 0))) {
                    goto bailout;
                }
                usleep(BUILD_SLEEP);
        
                // determine step 
                if(step > 0) {
                    mouse_cursor_expect.pos_x = prev;
                }
                
assert(!mouse_cursor_expect.pos_y);
//	mouse_cursor_expect.pos_y = 0;
        
                if(PP_ERR == 
#if defined(CAT_FB_DIRECT)
                   cat_find_cursor_for_table_fb(obj, shape_fb_rgb,
                                                &mouse_cursor_expect, 
                                                &mouse_cursor_point)
#else /* CAT_FB_DIRECT */
                   find_cursor_for_table(obj, &mouse_cursor_expect,                               
                                         &mouse_cursor_point)
#endif /* CAT_FB_DIRECT */
                   || mouse_cursor_point.pos_x < prev) {
                    /* we didn't find pointer */
                    CDMSG(CAT_CDMSG_SYNC_INFO, 
                          "sync failed for iteration %d, step was %d\n", 
                          i, step);
                    if(PP_SUC != (obj->data_unit_mouse.send_mouse_data(obj,
                                                         step * -2, 0, 0, 0))) {
                        goto bailout;
                    }
                    if(++i < 5) {
                        /* one more chance */
                        continue;
                    } else {
                        break;
                    }
                } else {
                    /* reset counter */
                    i = 0;
                }
        
//            cat_lookup_table_set(obj, step, mouse_cursor_point.pos_x);
                cat_lookup_table_add(obj, step, mouse_cursor_point.pos_x);
                ++found;
        
                // back to the roots
//            if(0 != move_mouse_direct(obj, mouse_cursor_point.pos_x * -2,
//                                      0, 0, 0, 0))
                if(PP_SUC != (obj->data_unit_mouse.send_mouse_data(obj,
                                                         step * -2, 0, 0, 0))) {
                    goto bailout;
                }
        
                usleep(BUILD_SLEEP);
        
                snprintf(debug_msg, sizeof(debug_msg), 
                         "Step=%d Dist=%d Scaling=%f",
                         step, mouse_cursor_point.pos_x, (float)mouse_cursor_point.pos_x / step);
                CDMSG(CAT_CDMSG_SYNC_INFO, "%s\n", debug_msg);
                
                prev = mouse_cursor_point.pos_x;
            }
        }
        if(found < __min_found__TODO__) {
            /* seems we didn't find enough supporting points */
            return ERIC_MOUSE_UNIVERSE_FAILED;
        }
    }
#else // old!
    for(i = step = prev = 0; i < REGRESSION_COUNT; prev++, step += 2) {
	// make step 
	if(0 != move_mouse_direct(obj, step, 0, 0, 0, 1)) {
	    goto bailout;
        }
	usleep(BUILD_SLEEP);

	// determine step 
	if(i > 0) {
            mouse_cursor_expect.pos_x = prev;
        }
        
assert(!mouse_cursor_expect.pos_y);
//	mouse_cursor_expect.pos_y = 0;

#if defined(CAT_FB_DIRECT)
        cat_find_cursor_for_table_fb(obj, shape_fb_rgb,
                                     &mouse_cursor_expect, &mouse_cursor_point);
#else /* CAT_FB_DIRECT */
	find_cursor_for_table(obj, &mouse_cursor_expect, &mouse_cursor_point);
#endif /* CAT_FB_DIRECT */

        cat_lookup_table_add(obj, step, mouse_cursor_point.pos_x);

	// back to the roots
	if(0 != move_mouse_direct(obj, mouse_cursor_point.pos_x * -2,
				  0, 0, 0, 0))
	    goto bailout;

	snprintf(debug_msg, sizeof(debug_msg), "Step=%d Dist=%d",
		 step, mouse_cursor_point.pos_x);
	CDMSG(CAT_CDMSG_SYNC_INFO, "%s\n", debug_msg);
    }
#endif // old

    cat_lookup_table_adjust(obj);

    if(0 != move_mouse_corner(obj, 0))
	goto bailout;

    md->cur_x = MOUSE_RASTER/2;
    md->cur_y = -MOUSE_RASTER/2;
    md->state = ERIC_MOUSE_IS_VALID;
    cat_set_position_estimate(obj, 1, 0, 0, pp_hrtime());

#if defined(CAT_ADAPTIVE_LOOKUP_TABLE)
    mouse_data = (data_cat_t*)obj->data_conv_mouse.data;

    cat_queue_lock(mouse_data->ll_move_queue);
    cat_queue_clear(mouse_data->ll_move_queue);
    cat_queue_unlock(mouse_data->ll_move_queue);
    
    mouse_data->ll_move_pos.x = 0;
    mouse_data->ll_move_pos.y = 0;
    
    pp_hrtime_start(&mouse_data->ll_move_timer);
#endif /* CAT_ADAPTIVE_LOOKUP_TABLE */    
    
#if defined(CAT_FB_DIRECT)
    free(shape_fb_rgb);
#endif /* CAT_FB_DIRECT */

#if 0
#warning DEBUG!!!!!
for(i = 0; i < 56; ++i) printf("tweb: cat_lookup_table_in2out(%3d) = %4d (factor = %f)\n", i, cat_lookup_table_in2out(obj, i), ((float)cat_lookup_table_in2out(obj, i) / MOUSE_RASTER) / (float)i);
sleep(1);
#endif

    return ret;
    
 bailout:
#if defined(CAT_FB_DIRECT)
    free(shape_fb_rgb);
#endif /* CAT_FB_DIRECT */

    return ERIC_MOUSE_UNIVERSE_FAILED;
}

/**
 * call get_mousecursor_shape and crop mask and shape afterwards
 */
int cat_intelli_get_mousecursor_shape(driver_t* obj) {
    data_imouse_t *imouse_data;
    t_bitmap img = {0, 0, NULL};
    u_int16_t left, right, top, bottom;

    assert(obj);
    
    imouse_data = (data_imouse_t*) obj->data_conv_mouse.data;
    assert(imouse_data);
    
    if(PP_ERR == get_mousecursor_shape(obj) ||
       PP_ERR == threshold_bitmap_by_color(
                     &img, &imouse_data->current_cursor_mask,
                     (t_rgb){CAT_CROP_CHANNEL_INTELLI_THRESHOLD,
                             CAT_CROP_CHANNEL_INTELLI_THRESHOLD,
                             CAT_CROP_CHANNEL_INTELLI_THRESHOLD}, 1) ||
       PP_ERR == crop_bitmap_by_color(&imouse_data->current_cursor_mask,
                                      &left, &right, &top, &bottom, 
                                      &img, (t_rgb){0, 0, 0}) ||
       PP_ERR == crop_bitmap(&img, &imouse_data->current_cursor_shape,
                             left, right, top, bottom)) {
        CDMSG(CAT_CDMSG_SYNC_INFO, 
              "could not get mouse cursor shape or mask\n");
        free(imouse_data->current_cursor_shape.rgb);
        imouse_data->current_cursor_shape.rgb = NULL;
        free(imouse_data->current_cursor_mask.rgb);
        imouse_data->current_cursor_mask.rgb = NULL;
        free(img.rgb);
        
        return PP_ERR;
    }
    
    free(imouse_data->current_cursor_shape.rgb);
    imouse_data->current_cursor_shape = img;
    
#if defined(CAT_DEBUG)
    DBG_WRITE_BITMAP(&imouse_data->current_cursor_shape,
                     "debug_sync_shape.bmp");
    DBG_WRITE_BITMAP(&imouse_data->current_cursor_mask, "debug_sync_mask.bmp");
#endif /* CAT_DEBUG */
    
    return PP_SUC;
}

int cat_intelli_get_mouse_universe(driver_t* obj) {
    int ret = ERIC_MOUSE_UNIVERSE_FAILED;
    data_imouse_t* imouse_data = (data_imouse_t*) obj->data_conv_mouse.data;

    if (cat_intelli_get_mousecursor_shape(obj) == 0) {
        imouse_data->current_kvm_port->mouse_data.fm_corr = non_mouse_corr;
        ret = cat_detect_mouse_universe(obj);
    }

    return ret;
}

int cat_intelli_move_mouse_diff(driver_t* obj, long diff_x, long diff_y, 
                                const unsigned char buttonmask, int* moved) {
    return cat_intelli_move_mouse_diff_core(obj, diff_x, diff_y, buttonmask,
                                            moved, NULL);
}

int cat_intelli_move_mouse_diff_core(driver_t* obj, long diff_x, long diff_y, 
                                     const unsigned char buttonmask, int* moved,
                                     cat_queue_ll_move_entry_t *ctx) {
    long moved_x, moved_y, ticks_x, ticks_y, euclidean, ticks;
    int count = 0, ret = 0;
    data_imouse_t* imouse_data = (data_imouse_t*) obj->data_conv_mouse.data;
    t_mouse_data* md = &imouse_data->current_kvm_port->mouse_data;
    long diff_orig_x = diff_x, diff_orig_y = diff_y;
    fb_format_info_t fb_info;
    
    *moved = 0;

    if(pp_vsc_get_fb_format(0, &fb_info) == PP_ERR) {
        CDMSG(CAT_CDMSG_WARNING, "No framebuffer info yet\n");
        return PP_ERR;
    }

    while ((diff_x || diff_y) && (++count < MOUSE_MAX_DIFF_COUNT)) {
        /* calc euclidean distance */
#warning TODO! fixpoint magic?
        euclidean = floor(sqrt((double)diff_x * diff_x + 
                               (double)diff_y * diff_y) + 0.5);
        ticks = cat_lookup_table_out2in(obj, euclidean);
        
	/* X direction */
	ticks_x = (ticks * diff_x) / euclidean;

	/* Y direction */
	ticks_y = (ticks * diff_y) / euclidean;
        
        CDMSG(CAT_CDMSG_INTELLI_MOVE_INFO,
              "out2in: x: %ld => %ld, y: %ld => %ld, d: %ld => %ld\n", 
              diff_x, ticks_x, diff_y, ticks_y, euclidean, ticks);

	/* avoid sending 0xAA to host, because it may lead to a mouse reset */
        ticks_x -= ticks_x == 0xAA;
        ticks_x += ticks_x == -86;
        ticks_y -= ticks_y == 0xAA;
        ticks_y += ticks_y == -86;

	/* return, we cannot reach the target exactly from our position */
	if (ticks_x == 0 && ticks_y == 0) {
            ret = PP_SUC;
	    break;
	}

        if (0 != (ret = obj->data_unit_mouse.send_mouse_data(obj,
                                                    ticks_x, ticks_y * -1,
                                                    0, buttonmask))) {
    	    break;
	}
	
	moved_x = cat_lookup_table_in2out(obj, ticks_x);
	md->cur_x = md->cur_x + moved_x;
	diff_x = SET_WITHIN(diff_x - moved_x, 
                            0, (long)fb_info.g_w * MOUSE_RASTER);
        
	moved_y = cat_lookup_table_in2out(obj, ticks_y);
	md->cur_y = md->cur_y + moved_y;
	diff_y = SET_WITHIN(diff_y - moved_y,
                            0, (long)fb_info.g_h * MOUSE_RASTER);

#if defined(CAT_ADAPTIVE_LOOKUP_TABLE)
        pp_km_cat_enqueue_ll_move(moved_x, moved_y, ticks_x, ticks_y);
#endif /* CAT_ADAPTIVE_LOOKUP_TABLE */

        CDMSG(CAT_CDMSG_INTELLI_MOVE_INFO,
              "set cur_x = %ld (+ %ld), cur_y = %ld (+ %ld)\n", 
              md->cur_x, moved_x, md->cur_y, moved_y);
	
	/* we reached our original diff value again, so
	   pointer is oscillating, set to 0 to stop movement */
	if (diff_x == diff_orig_x) diff_x = 0;
	if (diff_y == diff_orig_y) diff_y = 0;
	
	*moved = 1;
        
        if(ctx && ctx->step) {
            /* one step mode, required for lookup table filling */
            ctx->ticks_x = ticks_x;
            ctx->ticks_y = ticks_y;
            ctx->moved_x = moved_x;
            ctx->moved_y = moved_y;
            
            break;
        }
    }
    return ret;
}

/* obj function moves the mouse pointer using our GUT and
   the virtual mouse universe */
int cat_intelli_move_mouse(driver_t *obj, int x, int y, int z ,
                           const unsigned char buttonmask) {
    int ret, moved;
    long target_diff_x, target_diff_y;
    data_imouse_t* imouse_data;
#if defined(CAT_ADAPTIVE_LOOKUP_TABLE)
    data_cat_t* mouse_data;
#endif /* CAT_ADAPTIVE_LOOKUP_TABLE */
    t_mouse_data* md;
    
    assert(obj);
    imouse_data = (data_imouse_t*)obj->data_conv_mouse.data;
    assert(imouse_data);
    assert(imouse_data->current_kvm_port);
    md = &imouse_data->current_kvm_port->mouse_data;
    assert(md);

    target_diff_x = (x * MOUSE_RASTER) - md->cur_x;
    target_diff_y = (y * MOUSE_RASTER) - md->cur_y;

#if defined(CAT_APPROACH_ON_THE_FLY)
    {
        u_int16_t estim_x, estim_y;
        int64_t estim_time;
        
        if(PP_ERR != (estim_time = cat_get_position_estimate(obj, &estim_x,
                                                             &estim_y))) {
            long diff_x, diff_y, diff_cb;
            int diff_scaling = 2;
#if defined(CAT_ADAPTIVE_LOOKUP_TABLE)
            cat_queue_entry_t *cqe;
            long moved_x = 0, moved_y = 0, ticks_min = 111111, ticks_max = 0;
            long ll_diff_x, ll_diff_y;
#endif /* CAT_ADAPTIVE_LOOKUP_TABLE */

            assert(diff_scaling > 0);

            /* invalidate applied position estimate! */
            CAT_INVALIDATE_POSITION_ESTIMATE;
            
            diff_x = (md->cur_x - (estim_x * MOUSE_RASTER));
            diff_y = (md->cur_y - (estim_y * MOUSE_RASTER));
            
#if defined(CAT_ADAPTIVE_LOOKUP_TABLE)
            mouse_data = (data_cat_t*)obj->data_conv_mouse.data;
            
            pp_hrtime_stop(&mouse_data->ll_move_timer);
            if(pp_hrtime_read(&mouse_data->ll_move_timer) >> 6 > 5) {
                /* about five seconds... */
                pp_hrtime_start(&mouse_data->ll_move_timer);
    
                cat_queue_lock(mouse_data->ll_move_queue);
                for(; NULL != 
                      (cqe = cat_queue_dequeue_timed(mouse_data->ll_move_queue,
                                                     estim_time,
                                                     CAT_QUEUE_TIME_BEFORE));) {
                    long tmp_abs;
                    
                    assert(cqe->type == CAT_QUEUE_LL_MOVE_ENTRY);
                    
                    moved_x += cqe->data.ll_move.moved_x;
                    moved_y += cqe->data.ll_move.moved_y;
                    
                    tmp_abs = abs(cqe->data.ll_move.ticks_x);
                    if(tmp_abs && tmp_abs < ticks_min) {
                        ticks_min = tmp_abs;
                    } 
                    if(tmp_abs > ticks_max) {
                        ticks_max = tmp_abs;
                    }
                    
                    tmp_abs = abs(cqe->data.ll_move.ticks_y);
                    if(tmp_abs && tmp_abs < ticks_min) {
                        ticks_min = tmp_abs;
                    } 
                    if(tmp_abs > ticks_max) {
                        ticks_max = tmp_abs;
                    }
                   
                    cat_queue_destroy_entry(cqe);
                }
                cat_queue_unlock(mouse_data->ll_move_queue);
                
                ll_diff_x = estim_x - mouse_data->ll_move_pos.x;
                ll_diff_y = estim_y - mouse_data->ll_move_pos.y;
                mouse_data->ll_move_pos.x = estim_x;
                mouse_data->ll_move_pos.y = estim_y;
                
                if((moved_x || moved_y) && (ll_diff_x || ll_diff_y)) {
                    float moved_scaling;
                    
#warning fixpoint thingy!
                    moved_scaling = sqrt((float)ll_diff_x * ll_diff_x + 
                                         (float)ll_diff_y * ll_diff_y) *
                                    MOUSE_RASTER /
                                    sqrt((float)moved_x * moved_x + 
                                         (float)moved_y * moved_y);
//printf("tweb: real is %f, expected is %f (%ld, %ld), moved_scaling is %f, min = %ld, max = %ld\n", sqrt(ll_diff_x * ll_diff_x + ll_diff_y * ll_diff_y), sqrt(moved_x * moved_x + moved_y * moved_y) / MOUSE_RASTER, moved_x, moved_y, moved_scaling, ticks_min, ticks_max);usleep(11111);
    
                    /**
                     * do not scale, if
                     * - scaling is less than 5%
                     * - movement is smaller than 100 pixel
                     * - scaling is larger than 100%
                     */
                     
                    if((moved_scaling < 0.95 || moved_scaling > 1.05) && 
                       abs(ll_diff_x) + abs(ll_diff_y) > 100) {
                        if(moved_scaling > 2.0) {
                            CDMSG(CAT_CDMSG_SYNC_INFO ||
                                  CAT_CDMSG_LOOKUP_TABLE_INFO,
                                  "resized scaling from %f to %f\n",
                                  moved_scaling, 2.0);
                            moved_scaling = 2.0;
                        } else if(moved_scaling < 0.5) {
                            CDMSG(CAT_CDMSG_SYNC_INFO ||
                                  CAT_CDMSG_LOOKUP_TABLE_INFO,
                                  "resized scaling from %f to %f\n",
                                  moved_scaling, 0.5);
                            moved_scaling = 0.5;
                        }
                        
                        cat_lookup_table_scale(obj, moved_scaling, 
                                               ticks_min, ticks_max);
                            
                        CDMSG(CAT_CDMSG_SYNC_INFO ||
                              CAT_CDMSG_LOOKUP_TABLE_INFO, 
                              "scaled lookup table [%3ld:%3ld] by %f\n",
                              ticks_min, ticks_max, moved_scaling);
                    }
                }
            }
#endif /* CAT_ADAPTIVE_LOOKUP_TABLE */

            diff_cb = abs(diff_x) + abs(diff_y);
            if(diff_cb > PP_KM_CAT_MAX_ASYNC_CB * MOUSE_RASTER) {
                if(diff_cb > PP_KM_CAT_MAX_ASYNC_CB * 3 * MOUSE_RASTER) {
                    diff_x /= diff_scaling;
                    diff_y /= diff_scaling;
                }
                
                target_diff_x += diff_x;
                target_diff_y += diff_y;

                CDMSG(CAT_CDMSG_INTELLI_MOVE_INFO, 
                      "current(%3ld, %3ld), tracked (%3d, %3d), "
                      "diff %ld, %ld, applied %ld, %ld to movement\n",
                      md->cur_x / MOUSE_RASTER, md->cur_y / MOUSE_RASTER, 
                      estim_x, estim_y, 
                      diff_x * diff_scaling, diff_y * diff_scaling, 
                      diff_x, diff_y);

                md->cur_x -= diff_x;
                md->cur_y -= diff_y;
            }
        }
    }
#endif /* CAT_APPROACH_ON_THE_FLY */
    
    KD("move_mouse: (%ld,%ld)->(%ld,%ld)\n", 
       md->cur_x, md->cur_y, (long)x*MOUSE_RASTER, (long)y*MOUSE_RASTER);
    
    if (0 != (ret = cat_intelli_move_mouse_diff(obj, 
                                                target_diff_x, target_diff_y,
				                buttonmask, &moved)))
	goto bailout;

    /* now send buttons
       this is necessary when no movement is made but the
       buttons must be transmitted anyway */

    if (!moved)
	ret = obj->data_unit_mouse.send_mouse_data(obj, 0, 0, 0, buttonmask);
    
 bailout:
    return ret;
}
#endif /* PP_KM_CAT_INTELLI */
