/******************************************************************************\
* cat_queue.c                                                                  *
*                                                                              *
* Queue for intelligent online mouse cursor detection and tracking (CAT)       *
*                                                                              *
* Copyright 2005 Peppercon AG                                                  *
* Thomas Weber tweb@peppercon.de                                               *
\******************************************************************************/

#include <sys/time.h>
#include "cat_debug.h"
#include "cat_internal.h"

/**
 * init a new cat queue for size entries
 * catq is set to a pointer to obj queue
 */
int cat_queue_init(cat_queue_t **catq, size_t size, void (*notify)(void),
                   char *name) {
    cat_queue_t *_catq = (cat_queue_t*)malloc(sizeof(cat_queue_t));
    _catq->size = size;
    _catq->elems = 0;
    _catq->first = 0;
    _catq->last = 0;
    _catq->notify = notify; 
    _catq->data = (cat_queue_entry_t**)calloc(size, sizeof(cat_queue_entry_t*));
    pthread_mutex_init(&_catq->mtx, NULL);
    *catq = _catq;
    _catq->name = name ? strdup(name) : "";
    return PP_SUC;
}

/**
 * free cat queue and all of its elements
 */
void cat_queue_free(cat_queue_t *catq) {
    if(!catq) {
        return;
    }
    
    MUTEX_LOCK(&catq->mtx);
    
    cat_queue_clear(catq);
    free(catq->data);
    free(catq->name);
    /* destroy mutex */
    pthread_mutex_destroy(&catq->mtx);
    free(catq);
    catq = NULL;
    
    return;
}

/**
 * dequeue and destroy all entries of queue
 */
void cat_queue_clear(cat_queue_t *catq) {
    size_t i;
    
    assert(catq);
    
    /* don't forget to lock explicitely ;-) */
    assert(pthread_mutex_trylock(&catq->mtx) != 0);

    for(i = 0; i < catq->size; ++i) {
        /* destroy all elements */
        cat_queue_destroy_entry(catq->data[i]);
        catq->data[i] = NULL;
    }

    catq->elems = 0;
    catq->first = 0;
    catq->last = 0;
    
    CDMSG(CAT_CDMSG_CAT_QUEUE_INFO, "catq '%s' cleared\n", catq->name);
}

/**
 * cat queue mutex locking
 */
void cat_queue_lock(cat_queue_t *catq) {
    register int mutex_error;
#if !defined(NDEBUG)
    int loop = 0;
#endif /* NDEBUG */
    
    assert(catq);
    CDMSG(CAT_CDMSG_CAT_QUEUE_INFO, "lock queue '%s'\n", catq->name);
    
    /* don't use MUTEX_LOCK to avoid dead locks */
 retry:
    if(0 != (mutex_error = pthread_mutex_trylock(&catq->mtx))) {
        if(mutex_error == EBUSY) {
            assert(loop++ < 400);
            CDMSG(CAT_CDMSG_CAT_QUEUE_INFO || CAT_CDMSG_MTX, 
                  "queue '%s' already locked, retrying\n", catq->name);
            usleep(10000 + (random() & 0x3fff));
            goto retry;
        } else {
            abort();
            exit(1);
        }
    }
}

int cat_queue_trylock(cat_queue_t *catq) {
    register int mutex_error;
    
    assert(catq);
    CDMSG(CAT_CDMSG_CAT_QUEUE_INFO, "try lock queue '%s'\n", catq->name);
    
    /* don't use MUTEX_LOCK to avoid dead locks */
    if(0 != (mutex_error = pthread_mutex_trylock(&catq->mtx))) {
        if(mutex_error == EBUSY) {
            CDMSG(CAT_CDMSG_CAT_QUEUE_INFO || CAT_CDMSG_MTX, 
                  "queue '%s' already locked\n", catq->name);
            return PP_ERR;
        } else {
            abort();
            exit(1);
            return PP_ERR;
        }
    }
    
    return PP_SUC;
}

void cat_queue_unlock(cat_queue_t *catq) {
    assert(catq);
    MUTEX_UNLOCK(&catq->mtx);
    CDMSG(CAT_CDMSG_CAT_QUEUE_INFO, "queue '%s' unlocked\n", catq->name);
}

/**
 * query if cat queue is empty
 */
int cat_queue_is_empty(cat_queue_t *catq) {
    /* don't forget to lock explicitely ;-) */
    assert(pthread_mutex_trylock(&catq->mtx) != 0);
    
    return catq->elems == 0;
}

/**
 * enqueue cat queue entry in cat queue
 */
int cat_queue_enqueue(cat_queue_t *catq, cat_queue_entry_t *cqe) {
    assert(catq);
    assert(catq->data);
    assert(cqe);
    
    if(catq->size == catq->elems) {
        /* queue is full! */
        CDMSG(CAT_CDMSG_CAT_QUEUE_INFO, "queue '%s' is FULL!!!\n", catq->name);
        return PP_ERR;
    }
    
    /* don't forget to lock explicitely ;-) */
    assert(pthread_mutex_trylock(&catq->mtx) != 0);

    assert(!catq->data[catq->last]); // has to be empty
    catq->data[catq->last] = cqe;
    if(++catq->last == catq->size) {
        catq->last = 0; // wrap around
    }
    ++catq->elems;
    
    if(catq->notify) {
        catq->notify();
    }
    
    return PP_SUC;
}

/**
 * enqueue a diffmap in cat queue
 */
int cat_queue_enqueue_diffmap(cat_queue_t *catq, u_int32_t *diffmap, 
                              u_int size, u_int line_len,
                              u_int width, u_int height) {
    int ret;
    cat_queue_entry_t *cqe;
    
    cqe = cat_queue_create_diffmap_entry(diffmap, size, line_len, 
                                         width, height);
    assert(cqe);
//    DEBUG_CQDE(cqe->data.diffmap);
    if(PP_ERR == (ret = cat_queue_enqueue(catq, cqe))) {
        cat_queue_destroy_entry(cqe);
    }

    return ret;
}

/**
 * enqueue a pointer position in cat queue
 */
int cat_queue_enqueue_ptr_pos(cat_queue_t *catq, u_int16_t x, u_int16_t y) {
    int ret;
    cat_queue_entry_t *cqe;
    
    cqe = cat_queue_create_ptr_pos_entry(x, y);
    assert(cqe);
    if(PP_ERR == (ret = cat_queue_enqueue(catq, cqe))) {
        cat_queue_destroy_entry(cqe);
    }
                                            
    return ret;
}

/**
 * enqueue a pointer move in cat queue
 */
int cat_queue_enqueue_ptr_move(cat_queue_t *catq, int16_t x, int16_t y) {
    int ret;
    cat_queue_entry_t *cqe;
    
    cqe = cat_queue_create_ptr_move_entry(x, y);
    assert(cqe);
    if(PP_ERR == (ret = cat_queue_enqueue(catq, cqe))) {
        cat_queue_destroy_entry(cqe);
    }
                                            
    return ret;
}

#if defined(CAT_ADAPTIVE_LOOKUP_TABLE)
/**
 * enqueue a low level move in cat queue
 */
int cat_queue_enqueue_ll_move(cat_queue_t *catq, long moved_x, long moved_y,
                              long ticks_x, long ticks_y) {
    int ret;
    cat_queue_entry_t *cqe;
    
    cqe = cat_queue_create_ll_move_entry(moved_x, moved_y, ticks_x, ticks_y);
    assert(cqe);
    if(PP_ERR == (ret = cat_queue_enqueue(catq, cqe))) {
        cat_queue_destroy_entry(cqe);
    }
                                            
    return ret;
}
#endif /* CAT_ADAPTIVE_LOOKUP_TABLE */

/**
 * dequeues first entry of cat queue
 */
cat_queue_entry_t* cat_queue_dequeue(cat_queue_t *catq) {
    cat_queue_entry_t *cqe;
    
    assert(catq);
    assert(catq->data);
    
    /* don't forget to lock explicitely ;-) */
    assert(pthread_mutex_trylock(&catq->mtx) != 0);
    
    if(!catq->elems) {
        return NULL;
    }
    
    cqe = catq->data[catq->first];
    assert(cqe);
    catq->data[catq->first] = NULL;
    --catq->elems;
    if(++catq->first == catq->size) {
        catq->first = 0; // wrap around
    }
    
    return cqe;
}

/**
 * dequeues first entry of cat queue
 */
cat_queue_entry_t* cat_queue_dequeue_timed(cat_queue_t *catq, u_int64_t time,
                                           cat_queue_time_type_t comparator) {
    cat_queue_entry_t *cqe;
    
    assert(catq);
    assert(catq->data);
    
    /* don't forget to lock explicitely ;-) */
    assert(pthread_mutex_trylock(&catq->mtx) != 0);
    
    if(!catq->elems) {
        return NULL;
    }
    
    cqe = catq->data[catq->first];
    assert(cqe);
    if((comparator == CAT_QUEUE_TIME_BEFORE && cqe->timestamp >= time) ||
       (comparator == CAT_QUEUE_TIME_IS && cqe->timestamp != time) ||
       (comparator == CAT_QUEUE_TIME_AFTER && cqe->timestamp <= time)) {
        /* found elem with improper timestamp */
        return NULL;
    }
    
    catq->data[catq->first] = NULL;
    --catq->elems;
    if(++catq->first == catq->size) {
        catq->first = 0; // wrap around
    }
    
    return cqe;
}

static cat_queue_entry_t* cat_queue_create_entry(cat_queue_entry_type_t type) {
    cat_queue_entry_t *cqe = (cat_queue_entry_t*)
                             malloc(sizeof(cat_queue_entry_t));
    cqe->type = type;
    cqe->timestamp = pp_hrtime();
    
    return cqe;
}

/**
 * create diffmap entry for cat queue
 * if size is not 0, size bytes of diffmap are copied, diffmap pointer is used
 * otherwhise 
 */
cat_queue_entry_t* cat_queue_create_diffmap_entry(u_int32_t *diffmap, 
                                                  u_int size, u_int line_len,
                                                  u_int width, u_int height) {
    cat_queue_entry_t *cqe;
    u_int32_t line_len32 = OHDIV(line_len, 32);

    assert(diffmap);
    
    cqe = cat_queue_create_entry(CAT_QUEUE_DIFFMAP_ENTRY);
    if(size) {
        /* copy the diffmap */
        u_int32_t *dm = (u_int32_t*)malloc(size);
        memcpy(dm, diffmap, size);
        cqe->data.diffmap.data = dm;
        cqe->data.diffmap.size = size;
    } else {
        /* use diffmap pointer and calculate (minimum) size */
        cqe->data.diffmap.data = diffmap;
        cqe->data.diffmap.size = line_len32 * 4 * height;
    }
    cqe->data.diffmap.line_len = line_len32 * 32;
    cqe->data.diffmap.width = width;
    cqe->data.diffmap.height = height;
    
    return cqe;
}

/**
 * create pointer position entry for cat queue
 */
cat_queue_entry_t* cat_queue_create_ptr_pos_entry(u_int16_t x, u_int16_t y) {
    cat_queue_entry_t *cqe;

    cqe = cat_queue_create_entry(CAT_QUEUE_PTR_POS_ENTRY);
    cqe->data.ptr_pos.x = x;
    cqe->data.ptr_pos.y = y;
    
    return cqe;
}

/**
 * create pointer move entry for cat queue
 */
cat_queue_entry_t* cat_queue_create_ptr_move_entry(int16_t x, int16_t y) {
    cat_queue_entry_t *cqe;

    cqe = cat_queue_create_entry(CAT_QUEUE_PTR_MOVE_ENTRY);
    cqe->data.ptr_move.x = x;
    cqe->data.ptr_move.y = y;
    
    return cqe;
}

#if defined(CAT_ADAPTIVE_LOOKUP_TABLE)
/**
 * create tick entry for cat queue
 */
cat_queue_entry_t* cat_queue_create_ll_move_entry(long moved_x, long moved_y,
                                                  long ticks_x, long ticks_y) {
    cat_queue_entry_t *cqe;

    cqe = cat_queue_create_entry(CAT_QUEUE_LL_MOVE_ENTRY);
    cqe->data.ll_move.moved_x = moved_x;
    cqe->data.ll_move.moved_y = moved_y;
    cqe->data.ll_move.ticks_x = ticks_x;
    cqe->data.ll_move.ticks_y = ticks_y;
    
    return cqe;
}
#endif /* CAT_ADAPTIVE_LOOKUP_TABLE */

/**
 * free cat queue entry and stored data
 */
void cat_queue_destroy_entry(cat_queue_entry_t *cqe) {
    if(!cqe) {
        return;
    }
    
    switch(cqe->type) {
        case CAT_QUEUE_DIFFMAP_ENTRY:
            return cat_queue_destroy_diffmap_entry(cqe);
            break;
        case CAT_QUEUE_PTR_POS_ENTRY:
            return cat_queue_destroy_ptr_pos_entry(cqe);
            break;
        case CAT_QUEUE_PTR_MOVE_ENTRY:
            return cat_queue_destroy_ptr_move_entry(cqe);
            break;
#if defined(CAT_ADAPTIVE_LOOKUP_TABLE)
        case CAT_QUEUE_LL_MOVE_ENTRY:
            return cat_queue_destroy_ll_move_entry(cqe);
            break;
#endif /* CAT_ADAPTIVE_LOOKUP_TABLE */
        default:
            CDMSG(CAT_CDMSG_ERROR, 
                  "unknown cat queue entry type %d\n", cqe->type);
            abort();
    }
}

/**
 * free cat queue diffmap entry and stored data
 */
void cat_queue_destroy_diffmap_entry(cat_queue_entry_t *cqe) {
    if(!cqe) {
        return;
    }
    
    assert(cqe->type == CAT_QUEUE_DIFFMAP_ENTRY);
    
    free(cqe->data.diffmap.data);
    free(cqe);
}

/**
 * free cat queue pointer position entry and stored data
 */
void cat_queue_destroy_ptr_pos_entry(cat_queue_entry_t *cqe) {
    if(!cqe) {
        return;
    }
    
    assert(cqe->type == CAT_QUEUE_PTR_POS_ENTRY);

    free(cqe);
}

/**
 * free cat queue pointer move entry and stored data
 */
void cat_queue_destroy_ptr_move_entry(cat_queue_entry_t *cqe) {
    if(!cqe) {
        return;
    }
    
    assert(cqe->type == CAT_QUEUE_PTR_MOVE_ENTRY);

    free(cqe);
}

#if defined(CAT_ADAPTIVE_LOOKUP_TABLE)
/**
 * free cat queue low level move entry and stored data
 */
void cat_queue_destroy_ll_move_entry(cat_queue_entry_t *cqe) {
    if(!cqe) {
        return;
    }
    
    assert(cqe->type == CAT_QUEUE_LL_MOVE_ENTRY);

    free(cqe);
}
#endif /* CAT_ADAPTIVE_LOOKUP_TABLE */

